module TextUtilities;

{*****************************************}
{
{        Pepper:  Spice Interim Editor
{        Text Manipulation Utilities
{        Richard Cohn
{        April 1, 1981
{
{*****************************************}


exports

imports RegionEditor from RegionEdit;
imports Perq_String from Perq_String;

  
{-- disk/memory paging routines --}
 
 procedure SetUpSwapFile;
 function GetPFile (D: DiskPage):  FileIndex;
 function GetChFile (First, Last:  pChunk):  FileIndex;
 function Mem (D: DiskPage):  MemPage; 
 procedure CreateEmptyPage; 
 
 
{-- position and cursor routines --}
 
 
 procedure Add1( var P: Position ); 
 procedure Sub1( var P: Position ); 
 function Add( P: Position; N: integer):  Position; 
 procedure Detach( var C: Cursor ); 
 procedure Attach( var C: Cursor; P: Position; RW: ReadWrite ); 
 procedure ReAttach( var C: Cursor; P: Position ); 
 procedure Add1C( var C: Cursor ); 
 procedure Sub1C( var C: Cursor ); 
 procedure AddC( var C: Cursor; N: integer ); 
 function  Subtract( P1, P2: Position ): integer; 
 function  LT( P1, P2: Position ): boolean; 
 function  LE( P1, P2: Position ): boolean; 
 function  EQ( P1, P2: position ): boolean; 
 function  NE( P1, P2: position ): boolean; 
 function  GT( P1, P2: Position ): boolean; 
 function  GE( P1, P2: Position ): boolean; 
 function  Bot( P: Position ): boolean; 
 function  Eot( P: Position ): boolean; 
 
 
{-- text maintenance routines --}
 
 procedure UnSelect;
 procedure CheckCRLF (var P, Q:  Position);
 procedure FixUp (OldC, NewC: pChunk; Adjust: integer); 
 procedure Split (var P: Position); 
 procedure Join (var P, Q: Position); 
 procedure Collect (P, Q: Position); 
 procedure Copy (First, Last: Position); 
 procedure ReOrder (Start: Position); 
 procedure NewChunk (var P: pChunk);
 
 
{********************************} private {********************************}

 imports IO from IO;
 imports IOErrors from IOErrors;
 imports Memory from Memory;
 imports FileSystem from FileSystem;
 imports Canvas from EdCanvas;
 imports Perq_String from Perq_String;
 imports ScreenUtilities from EdScreen; 


{****************************************************************}

procedure SetUpSwapFile;

{ set up the swap file for writing out pages }

var i:  integer;

begin
if DEBUG [3] then Status ('Enter SetUpSwapFile');
IdFile [SwapFile].Index := FSEnter ('>Editor.Swap$');
if DEBUG [3] then
    StatusNumber ('SetUpSwapFile:  IdFile =', IdFile [SwapFile].Index);
if IdFile [SwapFile].Index = 0 then
    begin
    Warn ('Cannot open swap file');
    NewEvent.Cmd := ExitCmd;
    ReturnKey (inputQueue, newEvent);
    exit (EditRegion)
    end;
if DEBUG [3] then Prompt ('Swapping started');
SwapOn := true;
if DEBUG [3] then Status ('Exit SetUpSwapFile')
end; { SetUpSwapFile }


{****************************************************************}
 
function GetPFile (D: DiskPage):  FileIndex;

{ Return the index of the file that D belongs to.  Remember that the swap
{ file is the last file index, so there is no need for an upper bound on 
{ the while loop.  }

var f:  FileIndex;

begin
f := MinFile;
while (D < IdFile [f].MinPage) or (D > IdFile [f].MaxPage) do
    f := f + 1;
GetPFile := f
end; { GetPFile }


{****************************************************************}
 
function GetChFile (First, Last:  pChunk):  FileIndex;

{ Return the index of the file that the chunk list belongs to. }

begin
while (GetPFile (First^.CPage) = SwapFile) and (First <> Last) do
    First := First^.Next;
GetChFile := GetPFile (First^.CPage)
end; { GetChFile }


{****************************************************************}
 
 function CleanPage: MemPage;

{ Find a page to swap out.  The Multics second-chance algorithm is used to 
{ determine which page should be swapped out.  See Habermann, Intro to OS
{ Design for details.  }

 var M: MemPage;
     OutPage: Integer;  { -1 or MemPage }
     UnReserved: set of MemPage;
     OldWin:  pTextWindow;
     DirtyFile:  FileIndex;

 begin { CleanPage }
  if DEBUG [2] then Status('Enter CleanPage');
  M := LastMemPage;
  OutPage := -1;
  UnReserved := [0..MaxMemPage] -
                [DrawCursor.ChPage, Cursor1.ChPage, Cursor2.ChPage];
  repeat
   if M = MaxMemPage then M := 0 else M := M + 1;
   if M in UnReserved then
    with Txt[M] do
     begin
      if Used then
          Used := false
      else
          if Dirty then
              begin
              RealDirty := true;
              Dirty := false
              end
          else
              OutPage := M
     end
  until OutPage >= 0;
  with Txt[OutPage] do
   if RealDirty then
    begin
     DirtyFile := GetPFile (TPage);
     if not SwapOn and (DirtyFile = SwapFile) then
         SetUpSwapFile;
     if DEBUG [2] then
         StatusNumber ('Swap out page ', TPage-IdFile[DirtyFile].MinPage);
     if DEBUG [2] then StatusNumber ('to file ', DirtyFile);
     FSBlkWrite(IdFile[DirtyFile].Index, TPage-IdFile[DirtyFile].MinPage,
          Recast(Buffer,PDirBlk));
     Dirty := False
    end;
  CleanPage := OutPage;
  if DEBUG [2] then Status('Exit CleanPage')
 end { CleanPage };
 

{****************************************************************}

 function Mem (D:  DiskPage):  MemPage;

{ Given a disk page, find the associated MemPage.  If the page is not
{ currently in the editor's array of text pages, it is swapped in.  }

 var State: (Scanning, NotFound, Found);
     M: MemPage;
     OldWin:  pTextWindow;
     X,Y:  integer;
     NeededFile:  FileIndex;

 begin { Mem }
  if DEBUG [2] then Status('Enter Mem');
  State := Scanning;
  M := LastMemPage;
  repeat
   if M = MaxMemPage then M := 0 else M := M + 1;
   if D = Txt[M].TPage then
    begin State := Found; Txt[M].Used := true end
   else
    if M = LastMemPage then State := NotFound
  until State <> Scanning;
  if State = NotFound then
   begin
    M := CleanPage;
    with Txt[M] do
     begin
      TPage := D;
      NeededFile := GetPFile (TPage);
      Used := true;
      Dirty := false;
      RealDirty := false;
      if DEBUG [3] then
          begin
          SReadCursor (X,Y);
          OldWin := CurWin;
          ChangeTextWindow (PromptWindow);
          ClearLine (1,0,0);
          write ('Swap in page ', TPage-IdFile[NeededFile].MinPage,
               ' from file ', NeededFile);
          ChangeTextWindow (OldWin);
          SSetCursor (X,Y)
          end;
      if not SwapOn and (NeededFile = SwapFile) then
          SetUpSwapFile;
      FSBlkRead(IdFile[NeededFile].Index, D-IdFile[NeededFile].MinPage,
           Recast(Buffer,PDirBlk))
     end
   end;
  LastMemPage := M;
  Mem := M;
  if DEBUG [2] then Status('Exit Mem')
 end { Mem };
 
 
{****************************************************************}

 procedure CreateEmptyPage;

{ If there are no empty pages left on which to write new text, this
{ procedure is called to create one.  }

 var P: pChunk;
     M: MemPage;

 begin { CreateEmptyPage }
  if DEBUG [2] then Status('Enter CreateEmptyPage');
  NewChunk (P);
  with P^ do
   begin CPage := Pages;
    Pages := Pages + 1;
    First := 0;
    Length := MaxLength;
    OrderC := (EmptyLast.Chunk^.OrderC + EmptyLast.Chunk^.Length)
        mod (MaxLength - 1);
    if OrderC > EmptyLast.Chunk^.OrderC then
        OrderP := EmptyLast.Chunk^.OrderP
    else
        OrderP := EmptyLast.Chunk^.OrderP + 1;
    Next := nil;
    Prev := EmptyLast.Chunk
   end;
  EmptyLast.Chunk^.Next := P;
  EmptyLast.Chunk := P;
  M := CleanPage;
  with Txt[M] do
   begin TPage := P^.CPage;
    Dirty := false;
    RealDirty := false;
    Used := false
   end;
  if DEBUG [2] then Status('Exit CreateEmptyPage')
 end { CreateEmptyPage };
 
 
{****************************************************************}

 procedure Add1 (var P: Position);

{ Self-explanatory.  Note that if P = FilledLast of some window, then
{ the editor will probably bomb soon.  }

 begin { Add1 }
 if P.Offset + 1 = P.Chunk^.Length then
     begin
     if P.Chunk^.Next <> nil then
         begin 
         P.Chunk := P.Chunk^.Next;
         P.Offset := 0
         end
     end
 else P.Offset := P.Offset + 1
 end { Add1 };
 
 
{****************************************************************}

 procedure Sub1 (var P: Position);

{ Self-explanatory.  Note that if P = FilledFirst of some window, then
{ the editor will probably bomb soon.  }

 begin { Sub1 }
 if P.Offset = 0 then
     begin
     if P.Chunk^.Prev <> nil then
         begin
         P.Chunk := P.Chunk^.Prev;
         P.Offset := P.Chunk^.Length - 1
         end
     end 
 else P.Offset := P.Offset - 1
 end { Sub1 };
 
 
{****************************************************************}

 function Add (P: Position; N: integer):  Position;

{ Self-explanatory.  Note that it's possible for the result to be in
{ no-man's-land--before FilledFirst or beyond FilledLast.  If so, the editor
{ will probably bomb soon.  }

 var Offset, Length: integer;
     Q:  Position;

 begin { Add }
 with CurWin^ do begin
  Q.Chunk := P.Chunk;
  Offset := P.Offset + N;
  if Offset < 0 then
   begin
    repeat Q.Chunk := Q.Chunk^.Prev;
     if Q.Chunk = nil then Offset := 0
     else Offset := Offset + Q.Chunk^.Length
    until Offset >= 0;
    if Q.Chunk = nil then Q := FilledFirst else Q.Offset := Offset
   end
  else
   if Offset >= Q.Chunk^.Length then
    begin Length := Q.Chunk^.Length;
     repeat Offset := Offset - Length;
      Q.Chunk := Q.Chunk^.Next;
      if Q.Chunk = nil then Length := Offset + 1
      else Length := Q.Chunk^.Length
     until Offset < Length;
     if Q.Chunk = nil then Q := FilledLast else Q.Offset := Offset
    end
   else Q.Offset := Offset;
 end; { with }
 Add := Q
 end { Add };
 
 
{****************************************************************}

 procedure Attach (var C: Cursor; P: Position; RW: ReadWrite);

{ Attach a cursor to a position.  The position must be unattached at the time. 
{ Does a lookup on the page pointed at by the position's chunk.
{ Warn should be Error but string is then too long }

 begin { Attach }
  with C, Pos do
   begin
   if Attached then
     Warn ('internal error: attempt to Attach an attached cursor');
    Pos := P;
    ChPage := Mem(Chunk^.CPage);
    ChOffset := Chunk^.First + Offset;
    Attached := true;
    Writing := RW = WriteCursor;
    if Writing then Txt[ChPage].Dirty := true;
    Ch := Txt[ChPage].Buffer^[ChOffset];
 if DEBUG [1] then
     begin
     writeln ('    Ch = ', Ch, ', CPage =', ChPage, ', Offset =', ChOffset);
     with Chunk^ do
     writeln('    CPage =', CPage, ', OrderP =', OrderP, ', OrderC =', OrderC);
     Status ('Attach');
     end
   end;
 end { Attach };
 
 
{****************************************************************}

 procedure ReAttach (var C: Cursor; P: Position);

{ Attach a cursor to a position.  The position should be attached at the
{ time.
{ Does a lookup on the page pointed at by the position's chunk.  }

 begin { ReAttach }
  with C, Pos do
   begin
   if Chunk^.CPage = P.Chunk^.CPage then
    begin Pos := P;
     ChOffset := Chunk^.First + Offset;
     Ch := Txt[ChPage].Buffer^[ChOffset]
    end
   else
    begin Detach(C);
     if Writing then Attach(C,P,WriteCursor)
     else Attach(C,P,ReadCursor);
    end;
 if DEBUG [1] then
     begin
     writeln ('    Ch = ', Ch, ', CPage =', ChPage, ', Offset =', ChOffset);
     with Chunk^ do
     writeln('    CPage =', CPage, ', OrderP =', OrderP, ', OrderC =', OrderC);
     Status ('ReAttach')
     end
    end;
 end { ReAttach };
 
 
{****************************************************************}

 procedure Detach(* var C: Cursor *);

{ Detach an attached cursor.
{ Warn should be Error but string is then too long.  }

 begin { Detach }
 if DEBUG [2] then Status ('Enter Detach');
  with C do
   begin
   if DEBUG [2] then StatusNumber ('Detach:  Cursor Page =', Pos.Chunk^.OrderP);
   if Attached then Attached := false
   else Warn ('internal error: attempt to Detach an unattached cursor')
   end
 end { Detach };
 
 
{****************************************************************}

 procedure Add1C (var C: Cursor);

{ Advance C one position.  Note that if C.Pos = FilledLast of some window, then
{ the editor will probably bomb soon.  The character of the current position
{ is overwritten if the cursor is a writing cursor.  }

 begin { Add1C }
  with C, Pos do
    begin
    if Writing then Txt[ChPage].Buffer^[ChOffset] := Ch;
    if Offset + 1 < Chunk^.Length then
     begin Offset := Offset + 1;
      ChOffset := ChOffset + 1
     end
    else
     if Chunk^.Next <> nil then
      begin Chunk := Chunk^.Next;
       Offset := 0;
       ChPage := Mem(Chunk^.CPage);
       ChOffset := Chunk^.First;
       if Writing then Txt[ChPage].Dirty := true
      end;
    Ch := Txt[ChPage].Buffer^[ChOffset];
 if DEBUG [1] then
     begin
     writeln ('    Ch = ', Ch, ', CPage =', ChPage, ', Offset =', ChOffset);
     with Chunk^ do
     writeln('    CPage =', CPage, ', OrderP =', OrderP, ', OrderC =', OrderC);
     Status ('Add1C')
     end
    end; { with }
 end { Add1C };
 
 
{****************************************************************}

 procedure Sub1C (var C: Cursor);

{ Move C back one position.  Note that if C.Pos = FilledFirst of some window, 
{ then the editor will probably bomb soon.  The character of the current
{ position is overwritten if the cursor is a writing cursor.  }

 begin { Sub1C }
  with C, Pos do
    begin
    if Writing then Txt[ChPage].Buffer^[ChOffset] := Ch;
    if Offset > 0 then
     begin Offset := Offset - 1;
      ChOffset := ChOffset - 1
     end
    else
     if Chunk^.Prev <> nil then
      begin Chunk := Chunk^.Prev;
       Offset := Chunk^.Length - 1;
       ChPage := Mem(Chunk^.CPage);
       ChOffset := Chunk^.First + Offset;
       if Writing then Txt[ChPage].Dirty := true
      end;
    Ch := Txt[ChPage].Buffer^[ChOffset];
 if DEBUG [1] then
     begin
     write ('    Ch = ', Ch, ', CPage =', ChPage, ', Offset =', ChOffset);
     with Chunk^ do
     writeln('    CPage =', CPage, ', OrderP =', OrderP, ', OrderC =', OrderC);
     Status ('Sub1C');
     end
    end; { with }
 end { Sub1C };


{****************************************************************}

procedure AddC (var C: Cursor; N: integer);

{ Move C backward or forward N positions.  
{ Note that it's possible for the result to be in
{ no-man's-land--before FilledFirst or beyond FilledLast.  If so, the editor
{ will probably bomb soon.  The character of the current position
{ is overwritten if the cursor is a writing cursor.
{ This is a procedure and not a function since the only cursors in the editor
{ should be Cursor1, Cursor2, and DrawCursor.  Otherwise FixUp may cause
{ problems.  }

var NewOffset, Length: integer;

begin { AddC }
if DEBUG [2] then Status ('Enter AddC');
with CurWin^, C, Pos do
    begin
    if Writing then 
        Txt[ChPage].Buffer^[ChOffset] := Ch;
    NewOffset := Offset + N;
    if NewOffset < 0 then
        begin
        repeat 
            Chunk := Chunk^.Prev;
            if Chunk = nil then NewOffset := 0
            else NewOffset := NewOffset + Chunk^.Length
        until NewOffset >= 0;
        if Chunk = nil then Pos := FilledFirst else Offset := NewOffset;
        ChPage := Mem(Chunk^.CPage);
        ChOffset := Chunk^.First + NewOffset;
        if Writing then Txt[ChPage].Dirty := true
        end
    else
        if NewOffset >= Chunk^.Length then
             begin 
             Length := Chunk^.Length;
             repeat 
                 NewOffset := NewOffset - Length;
                 Chunk := Chunk^.Next;
                 if Chunk = nil then Length := NewOffset + 1
                 else Length := Chunk^.Length
             until NewOffset < Length;
             if Chunk = nil then Pos := FilledLast else Offset := NewOffset;
             ChPage := Mem(Chunk^.CPage);
             ChOffset := Chunk^.First + NewOffset;
             if Writing then Txt[ChPage].Dirty := true
             end
         else
             begin 
             Offset := NewOffset;
             ChOffset := ChOffset + N
             end;
    Ch := Txt[ChPage].Buffer^[ChOffset]
    end; { with }
end { AddC };
 
 
{****************************************************************}

 function Subtract (P1, P2: Position):  integer;

{ Return the number of characters between P1 and P2.  (Negative if P2 is
{ before P1.)  Can be used only if the distance from P1 to P2 is < 32768.  }

 var N: integer;

 begin { Subtract }
  N := -P2.Offset;
  while (P2.Chunk <> P1.Chunk) and (P2.Chunk <> nil) do
   begin N := N + P2.Chunk^.Length;
    if N > 0 then P2.Chunk := P2.Chunk^.Next
    else P2.Chunk := nil
   end;
  if P2.Chunk = nil then N := -1
  else N := N + P1.Offset;
  Subtract := N
 end { Subtract };
 
 
{****************************************************************}

 function LT (P1, P2: Position):  boolean;

{ Evaluate P1 < P2.  }

 begin { LT }
  if P1.Chunk = P2.Chunk then LT := P1.Offset < P2.Offset
  else
   if P1.Chunk^.OrderP = P2.Chunk^.OrderP then
    LT := P1.Chunk^.OrderC < P2.Chunk^.OrderC
   else LT := P1.Chunk^.OrderP < P2.Chunk^.OrderP
 end { LT };
 
 
{****************************************************************}

 function LE (P1, P2: Position):  boolean;

{ Evaluate P1 <= P2.  }

 begin { LE }
  if P1.Chunk = P2.Chunk then LE := P1.Offset <= P2.Offset
  else
   if P1.Chunk^.OrderP = P2.Chunk^.OrderP then
    LE := P1.Chunk^.OrderC <= P2.Chunk^.OrderC
   else LE := P1.Chunk^.OrderP < P2.Chunk^.OrderP
 end { LE };
 
 
{****************************************************************}

 function EQ (P1, P2: Position):  boolean;

{ Evaluate P1 = P2.  This could be done in-line in Perq Pascal.  }

 begin { EQ }
  EQ := (P1.Chunk = P2.Chunk) and (P1.Offset = P2.Offset)
 end { EQ };
 
 
{****************************************************************}

 function NE (P1, P2: Position):  boolean;

{ Evaluate P1 <> P2.  }

 begin { NE }
  NE := (P1.Chunk <> P2.Chunk) or (P1.Offset <> P2.Offset)
 end { NE };
 
 
{****************************************************************}

 function GT (P1, P2: Position):  boolean;

{ Evaluate P1 > P2.  }

 begin { GT }
  if P1.Chunk = P2.Chunk then GT := P1.Offset > P2.Offset
  else
   if P1.Chunk^.OrderP = P2.Chunk^.OrderP then
    GT := P1.Chunk^.OrderC > P2.Chunk^.OrderC
   else GT := P1.Chunk^.OrderP > P2.Chunk^.OrderP
 end { GT };
 
    
{****************************************************************}

 function GE (P1, P2: Position):  boolean;

{ Evaluate P1 >= P2.  }

 begin { GE }
  if P1.Chunk = P2.Chunk then GE := P1.Offset >= P2.Offset
  else
   if P1.Chunk^.OrderP = P2.Chunk^.OrderP then
    GE := P1.Chunk^.OrderC >= P2.Chunk^.OrderC
   else Ge := P1.Chunk^.OrderP > P2.Chunk^.OrderP
 end { GE };
 
 
{****************************************************************}

 function Bot (P: Position):  boolean;

{ Return true if P is at the first character of the buffer or before.
{ Recall that three characters precede the first real character in the buffer:
{ beg-of-file, CR, and LF.  }

 var b:  boolean;

 begin { Bot }
  b := EQ(P,CurWin^.FilledFirst);
  if not b then 
      begin
      Sub1(P);
      b := EQ(P,CurWin^.FilledFirst);
      if not b then
          begin 
          Sub1(P);
          b := EQ(P,CurWin^.FilledFirst)
          end
      end;
  Bot := b
 end { Bot };
 
 
{****************************************************************}

 function Eot (P: Position):  boolean;

{ Return true if P is at the last character of the buffer.  The last
{ character is the special end-of-file character.  }

 begin { Eot }
  Eot := EQ(P,CurWin^.FilledLast)
 end { Eot };


{****************************************************************}

procedure UnSelect;

{ UnSelect the currently selected text.  It is used by the delete, insert,
{ and select all commands.  }

var OldWin:  pTextWindow;

begin
OldWin := nil;
with Bufary [SelectB] do
    if SelectWindow = CurWin then
        Underline (First, Last, Erase)
    else if SelectWindow <> nil then
        begin
        OldWin := CurWin;
        ChangeTextWindow (SelectWindow);
        Underline (First, Last, Erase)
        end;
SelectWindow := nil;
DrawLn (SelectB);
UpdateThumbBar;
if OldWin <> nil then
    ChangeTextWindow (OldWin)
end; { UnSelect }
   
  
{****************************************************************}

  procedure CheckCRLF (var P, Q: Position);

{ Recall that end-of-lines are represented by CR LF pairs.  If P points to the
{ LF of a CR LF pair, it is moved to the CR, while if Q points to the CR of a
{ CR LF pair, it is moved to the LF.  }

  begin { CheckCRLF }
   Attach(Cursor1,P,ReadCursor);
   if Cursor1.Ch = LF then
    begin Sub1C(Cursor1);
     if Cursor1.Ch = CR then P := Cursor1.Pos
    end;
   ReAttach(Cursor1,Q);
   if Cursor1.Ch = CR then
    begin Add1C(Cursor1);
     if Cursor1.Ch = LF then Q := Cursor1.Pos
    end;
   Detach(Cursor1)
  end { CheckCRLF };
 
 
{****************************************************************}

 procedure FixUp (OldC, NewC: pChunk; Adjust: integer);

{ Fix up the positions and cursor positions that may have been messed up by
{ inserting or deleting characters in the buffer.  Only the positions that
{ are in the chunk OldC are changed.  If the position's offset > Adjust, then
{ the position's chunk is changed to NewC and its offset is fixed.  Adjust
{ represents the point at which the chunk structure has been changed.  }

 var L:  LineIndex;
     b:  BufRange;
 
{********************************}
 
  procedure F( var P: Position );

  begin { F }
   with P do
    if Chunk = OldC then
     if Offset >= Adjust then
      begin Chunk := NewC;
       if Adjust > Offset then Offset := 0
       else Offset := Offset - Adjust
      end
  end { F };
  
{********************************}
  
begin { FixUp }
with CurWin^ do
    begin
    F(FilledFirst);  F(FilledLast);
    F(ScreenFirst);  F(ScreenLast);
    F(Mark);
    for L := 0 to LastLine do
        begin 
        with Ln[L].Start do
            if Chunk = OldC then
                if Offset >= Adjust then
                    begin 
                    Chunk := NewC;
                    if Adjust > Offset then 
                        Offset := 0
                    else 
                        Offset := Offset - Adjust
                    end;
        with Ln[L].Finish do
            if Chunk = OldC then
                if Offset >= Adjust then
                    begin 
                    Chunk := NewC;
                    if Adjust > Offset then 
                        Offset := 0
                    else 
                        Offset := Offset - Adjust
                    end
        end { for }
    end; { with }
F(EmptyFirst);   F(EmptyLast);
F(PrevKill);     F(NextKill);
F(SourceFirst);  F(SourceLast);
F(Display);
F(LeftPart);     F(RightPart);
F(PFirst);       F(PLast);
F(Tmp);
F(savedPos);
F(justP);
F(DrawCursor.Pos);
F(Cursor1.Pos);
F(Cursor2.Pos);
for b := minBuffer to maxBuffer do
    begin
    F (Bufary [b].First);
    F (Bufary [b].Last)
    end
end { FixUp };
    
 
{****************************************************************}

 procedure Split (var P: Position);

{ Split text before P.  Assume Sub1(P) has been saved if necessary. }

 var OldC, NewC: pChunk;
     Adjust: integer;
 begin { Split }
  if DEBUG [2] then Status('Enter Split');
  with P do
   if Offset = 0 then
    begin
     if Chunk^.Prev <> nil then Chunk^.Prev^.Next := nil;
     Chunk^.Prev := nil
    end
   else
    begin OldC := Chunk;
     NewChunk (Chunk);
     NewC := Chunk;
     Adjust := Offset;
     with Chunk^ do
      begin Prev := nil;
       Next := OldC^.Next;
       if Next <> nil then Next^.Prev := Chunk;
       Length := OldC^.Length - Adjust;
       OrderC := OldC^.OrderC + Adjust;
       OrderP := OldC^.OrderP + OrderC div MaxLength;
       OrderC := OrderC mod MaxLength;
       CPage := OldC^.CPage;
       First := OldC^.First + Adjust
      end;
     with OldC^ do
      begin Length := Offset; Next := nil end;
     Offset := 0;
     FixUp(OldC,NewC,Adjust)
    end;
  if DEBUG [2] then Status('Exit Split')
 end { Split };
 
 
{****************************************************************}

 procedure Join (var P, Q: Position);

{ Join P (left) to Q (right). Assume P points to last character in chunk
{ and Q to first character in chunk. }

 var PC, QC: pChunk;
     Adjust: integer;

 begin { Join }
  if DEBUG [2] then Status('Enter Join');
  PC := P.Chunk;
  QC := Q.Chunk;
  if (PC^.CPage = QC^.CPage) and (PC^.First + PC^.Length = QC^.First) then
   begin Adjust := PC^.Length;
    Q.Offset := Adjust;
    PC^.Next := QC^.Next;
    if PC^.Next <> nil then PC^.Next^.Prev := PC;
    PC^.Length := PC^.Length + QC^.Length;
    { Dispose(Q.Chunk); }
    Q.Chunk := PC;
    FixUp(QC,PC,-Adjust)
   end
  else
   begin PC^.Next := Q.Chunk;
    QC^.Prev := P.Chunk
   end;
  ReOrder(P);
  if DEBUG [2] then Status('Exit Join')
 end { Join };
 
 
{****************************************************************}

 procedure Collect (P, Q: Position);

{ Garbage collect the chunk structure delimited by P and Q.  }

 var R: Position;

 begin { Collect }
 with CurWin^ do
  begin
  if NE(P,NilPosition) then
   if NE (P,FilledLast) then
   begin
    if P.Offset <> 0 then Split(P);
    if Q.Offset <> Q.Chunk^.Length - 1 then
     begin R := Add (Q,1);
      Split(R)
     end;
    Q.Chunk^.Next := FreeChunk;
    FreeChunk := P.Chunk 
   end
  end { with }
 end { Collect };


{****************************************************************}

procedure Copy (First, Last: Position);

{ Place a copy of the chunk structure delimited by First and Last into PFirst
{ and PLast.  Only the pointers need be updated.  }

 var Done: boolean;
     Scan, T: pChunk;

begin { Copy }
  Scan := First.Chunk; 
  PFirst.Chunk := nil;
  Done := false;
  repeat
   NewChunk (T);
   T^ := Scan^;
   T^.OrderP := 0;
   T^.OrderC := 0;
   if PFirst.Chunk = nil then PFirst.Chunk := T
   else
    begin PLast.Chunk^.Next := T;
     T^.Prev := PLast.Chunk
    end;
   PLast.Chunk := T;
   if Scan = Last.Chunk then Done := true
   else
    begin Scan := Scan^.Next;
     if Scan = nil then Done := true
    end
  until Done;
  PFirst.Chunk^.Prev := nil;
  PFirst.Chunk^.First := PFirst.Chunk^.First + First.Offset;
  PFirst.Chunk^.Length := PFirst.Chunk^.Length - First.Offset;
  PFirst.Offset := 0;
  if PFirst.Chunk = PLast.Chunk then
   PLast.Offset := Last.Offset - First.Offset
  else PLast.Offset := Last.Offset;
  PLast.Chunk^.Length := PLast.Offset + 1;
  PLast.Chunk^.Next := nil
end; { Copy }
 
 
{****************************************************************}

 procedure ReOrder (Start: Position);

{ Fix up the OrderP and OrderC fields of the chunk structure starting at
{ Start.  }

 var P: pChunk;
     OP, OC: integer;

 begin { ReOrder }
  with Start.Chunk^ do
   begin P := Next;
    OP := OrderP;
    OC := OrderC + Length
   end;
  while P <> nil do with P^ do
   begin OP := OP + OC div MaxLength;
    OC := OC mod MaxLength;
    OrderP := OP;
    OrderC := OC;
    OC := OC + Length;
    P := Next
   end
 end; { ReOrder }


{****************************************************************}

 procedure NewChunk(var P: pChunk);

{ Create a new chunk.  }

 label 1;
 
  handler FullSegment;
  var
    i: integer;
  begin { FullSegment }
   i := 1;
   CreateSegment(ChunkSeg,2,2,10);
   Goto 1;
  end;

 begin { NewChunk }
 if FreeChunk = nil then
1:   New(ChunkSeg,1,P)
 else
     begin
     P := FreeChunk;
     FreeChunk := FreeChunk^.Next
     end;
 p^.prev := nil;
 p^.next := nil;
 end. { NewChunk }
