module ScreenUtilities;

{*****************************************}
{
{        Pepper:  Spice Interim Editor
{        Buffer Manipulation Utilities
{        Richard Cohn
{        April 1, 1981
{
{*****************************************}


exports

imports RegionEditor from RegionEdit;
imports Perq_String from Perq_String;


 {-- screen routines --}

 procedure OnPointer;
 procedure OffPointer;
 procedure DisplPointer (x,y:  integer);
 procedure MoveTextPointer (L:  LineIndex; C:  ColumnIndex);

 procedure MovePencil( L: LineIndex; C: ColumnIndex ); 
 procedure WriteChar( Ch: Char );
 procedure EraseChar;
 procedure DeleteChar( L: LineIndex; C: ColumnIndex ); 
 procedure InsertChar( L: LineIndex; C: ColumnIndex ); 

 procedure ClearLine(  L: LineIndex; C, LastC: ColumnIndex ); 
 procedure ClearWindow ( L: LineIndex; C: ColumnIndex ); 
 procedure ChangeTextWindow (NewWin:  pTextWindow);

 procedure Prompt( S: PString ); 
 procedure Status( S: PString ); 
 procedure StatusNumber( S: PString; n: integer ); 
 procedure StatusCh (s: PString; ch: char);
 procedure Error( S: PString ); 
 procedure Warn( S: PString ); 
 function GetAnswer (pr: string):  string;
 
 
 {-- display routines --}
 
 function  GetPosition ( L: LineIndex; C: ColumnIndex):  Position; 
 procedure GetLC ( P: Position; var L: LineIndex; var C: ColumnIndex ); 
 function  OnScreen( P: Position; L1, L2: LineIndex ): boolean;

 procedure DrawLn( BufLine:  BufRange);
 procedure UpdateThumbBar;
 procedure UnderLine( First, Last: Position; U: UnderKind ); 

 procedure ScrollUp (L1, L2:  LineIndex; Distance:  integer);
 procedure ScrollDown (L1, L2:  LineIndex; Distance:  integer);
 procedure Draw( P, Q: Position; L: LineIndex; C: ColumnIndex ); 
 procedure JoinScreen (P: Position; L: LineIndex; C: ColumnIndex); 
 procedure Show( P: Position; L1, L2: LineIndex ); 

{-- window utilities --} 

procedure SetUpPromptWindow (PromptW:  pWindow);
function SetUpWindow (WinBound:  pWindow; WindowName:  PString; DoPrompt,
    DoScroll, DoThumb, DoBox, AutoSave:  boolean):  pTextWindow;
procedure RefreshPromptWindow;
procedure RefreshTextWindow;
procedure RefreshName;

 
private

 imports Screen from Screen;
 imports Raster from Raster;
 imports IO_Unit from IO_Unit;
 imports IO_Others from IO_Others;
 imports IOErrors from IOErrors;
 imports Memory from Memory;
 imports FileSystem from FileSystem;
 imports TextUtilities from EdText;
 imports Canvas from EdCanvas;
 
var
  oldCpX, oldCpY: integer;
  lastTime: long;
  ptrIsOff: boolean;
 
 
{****************************************************************}

 procedure OnPointer;

{ Turn cursor on.  }

 begin { OnPointer }
 PtrDisplayed := true;
 IOCursorMode(IndepCursor);
 IOSetModeTablet (RelTablet);
 ptrIsOff := false;
 end { OnPointer };
 
    
{****************************************************************}

 procedure OffPointer;

{ Turn cursor off.  }

 begin { OffPointer }
 PtrDisplayed := false;
 IOCursorMode(OffCursor);
 IOSetModeTablet (OffTablet);
 ptrIsOff := true;
 end { OffPointer };
 
 
{****************************************************************}

procedure ChangePointer (NewPtr:  PtrRange);

{ Change the displayed cursor to NewPtr.  Called by DisplPointer and
{ MoveTextPointer only.  }

begin
if NewPtr <> CurrentPtr then
    begin
    CurrentPtr := NewPtr;
    with CursorPtr [CurrentPtr] do
        IOLoadCursor (Pattern, xOrigin, yOrigin)
    end
end; { ChangePointer }

{**********************************************************************}

procedure DisplPointer (* x,y:  integer *);

{ Display the pointer (cursor) at the closest legal point to x,y (in pixels).
{ If necessary, change the pointer first.  }
var
  cpX, cpY: integer;
  curTime: long;
  timeOut: boolean;

begin
with CurWin^ do
    begin
    if y < HTextBound then
        { in thumb area }
        begin
        ChangePointer (ThumbPtr);
        if oldCpY <> yThumb then
          yPointer := HTextBound - 4;
        IOSetCursorPos (x, yThumb);
        oldCpX := -1;
        oldCpY := yThumb;
        end
    else
        begin
        if y <= yHome then
            CurLine := 0
        else if y >= yLastLine then
            CurLine := LastLine
        else if (y >= (oldCpY + lineHeight)) or
                (y <= (oldCpY - lineHeight)) then
            CurLine := (y - yHome + (lineHeight div 2)) div LineHeight;
        if x < VTextBound then
            { Scroll area}
            begin
            ChangePointer (ScrollPtr);
            cpX := Bound^.xMin + xScroll;
            cpY := LineHeight * CurLine + yHome;
            if cpX <> oldCpX then
              xPointer := VTextBound - 4;
            if cpY <> oldCpY then
              yPointer := cpY;
            IOSetCursorPos (Bound^.xMin + xScroll, cpY);
            oldCpX := cpX;
            oldCpY := cpY;
            end
        else { Text area }
            begin
            if x <= xHome then 
                CurCol := 0
            else if x >= xLastColumn then
                CurCol := LastColumn
            else if (x >= (oldCpX + charWidth)) or
                    (x <= (oldCpX - charWidth)) then
                CurCol := (x - xHome + (charWidth div 2)) div CharWidth;
            ChangePointer (TextPtr);
            cpX := CharWidth * CurCol + xHome;
            cpY := LineHeight * CurLine + yHome;
            IOGetTime(recast(curTime, double));
            timeOut := curTime > (lastTime + 6); { 1/10 second timeout }
            if timeOut then
              begin
                lastTime := 2147483620;
                xPointer := cpX;
                yPointer := cpY;
              end;
            IOSetCursorPos (cpX, cpY);
            if (cpX <> oldCpX) or (cpY <> oldCpY) then
              lastTime := curTime;
            oldCpX := cpX;
            oldCpY := cpY;
            end
        end; { else }
    end; { with }
end; { DisplPointer }


{**********************************************************************}

procedure MoveTextPointer (* L:  LineIndex; C:  ColumnIndex *);

{ Display the pointer at the specified line and column.  If necessary, change
{ the pointer to the text pointer.  }

var xDisplayed, yDisplayed:  integer;

begin
with CurWin^ do
    begin
    CurLine := L;
    CurCol := C;
    xDisplayed := C * CharWidth + xHome;
    yDisplayed := L * LineHeight + yHome;
    xPointer := xDisplayed;
    yPointer := yDisplayed;
    ChangePointer (TextPtr);
    IOSetCursorPos (xDisplayed, yDisplayed);
    end { with }
end; { MoveTextPointer }


{**********************************************************************}

 procedure MovePencil (L: LineIndex; C: ColumnIndex);

{ Move writing pencil (cursor).  }

 begin { MovePencil }
 with CurWin^ do
     SSetCursor( C * CharWidth + xHome,
              L * LineHeight + CharHeight + yHome )
 end { MovePencil };
 
 
{****************************************************************}

 procedure WriteChar (Ch: Char);

{ Put the given character at the current writing position.  }

 var XC, YC: Integer;

 begin { WriteChar }
  if Ch >= ' ' then SPutChr(Ch)
  else
   begin
    SReadCursor(XC,YC);
    SPutChr( Chr(Ord(Ch) + Ord('@')) );
    RasterOp(RNot, CharWidth, CharHeight + 2,
         XC - 1, YC-CharHeight - 1, SScreenW, SScreenP,
         XC - 1, YC-CharHeight - 1, SScreenW, SScreenP)
   end
 end { WriteChar };
 
 
{****************************************************************}

 procedure EraseChar;

{ Erase the character at the current writing position.  }

 var XC, YC: Integer;

 begin { EraseChar }
  SReadCursor(XC,YC);
  RasterOp(RXor, CharWidth, LineHeight,
           XC - 1, YC - CharHeight - 1, SScreenW, SScreenP,
           XC - 1, YC - CharHeight - 1, SScreenW, SScreenP)
 end { EraseChar };
 
 
{****************************************************************}

 procedure ClearLine (L: LineIndex; C, LastC: ColumnIndex);

{ Erase line L from column C to column LastC.  If LastC = 0 then the line is
{ erased to the margin.  }

 var XC, YC, xWidth: integer;

 begin { ClearLine }
 with CurWin^ do
  begin
  if C < 0 then
      C := 0;
  if L < 0 then
      L := 0;
  XC := C * CharWidth + xHome;
  YC := L * LineHeight + yHome;
  if LastC = 0 then
      xWidth := Bound^.xMax - XC
  else
      xWidth := (LastC - C + 1) * CharWidth;
  CtrlSPending := false;
  RasterOp(RXor, xWidth, LineHeight,
           XC - 1, YC - 1, SScreenW, SScreenP,
           XC - 1, YC - 1, SScreenW, SScreenP);
  MovePencil(L,C)
  end
 end { ClearLine };
 
 
{****************************************************************}

procedure ChangeTextWindow (* NewWin:  pTextWindow *);

var OldWin: pTextWindow;

begin
if NewWin = nil then
    Warn ('Internal error--attempt to change to non-existent window')
else
    begin
    OldWin := CurWin;
    CurWin := NewWin;
    with CurWin^ do
        if not IsPrompt then
            begin
            if OldWin <> nil then
                if OldWin^.IsPrompt then
                    exit (ChangeTextWindow);
            MoveTextPointer (CurLine, CurCol);
            end
    end
end; { ChangeTextWindow }
    

{********************************************************************}

 procedure DeleteChar (L: LineIndex; C: ColumnIndex);

{ Delete the character at the specified line and column by shifting the
{ line left one character.  }

 var XC, YC: integer;

 begin { DeleteChar }
 with CurWin^ do
  begin
  XC := C * CharWidth + xHome;
  YC := L * LineHeight + yHome;
  RasterOp(RRpl, Bound^.xMax - XC - CharWidth, LineHeight, 
           XC - 1, YC - 1, SScreenW, SScreenP,
           XC + CharWidth - 1, YC - 1, SScreenW, SScreenP);
  MovePencil(L,LastColumn);
  EraseChar
  end
 end { DeleteChar };
 
 
{****************************************************************}

 procedure InsertChar (L: LineIndex; C: ColumnIndex);

{ Prepare to insert a character by shifting the line to the right one 
{ character, starting at column C.  }

 var XC, YC: integer;

 begin { InsertChar }
 with CurWin^ do
  begin
  XC := C * CharWidth + xHome;
  YC := L * LineHeight + yHome;
  RasterOp(RRpl, Bound^.xMax - XC - CharWidth, LineHeight, 
           XC + CharWidth - 1, YC - 1, SScreenW, SScreenP,
           XC - 1, YC - 1, SScreenW, SScreenP);
  MovePencil(L,C);
  Write(' ')
  end
 end { InsertChar };
  
    
{****************************************************************}

 procedure ClearWindow (L: LineIndex; C: ColumnIndex); 

{ Erase the contents of the current window from line L, column C to the 
{ bottom of the screen.  }

 var i, j: integer;
     XC, YC: integer;

 begin { ClearScreen }
 with CurWin^ do
  begin
  if L < 0 then L := 0;
  if C < 0 then C := 0;
  ClearLine(L,C,0);
  if L <> LastLine then
   begin
   if L >= 0 then
       begin
       XC := xHome;
       YC := (L+1) * LineHeight + yHome
       end
   else
       begin
       L := 0;
       XC := Bound^.xMin + 1;
       YC := Bound^.yMin + 1
       end;
     RasterOP(RXor, Bound^.xMax - XC, Bound^.yMax - YC,
              XC - 1, YC - 1, SScreenW, SScreenP,
              XC - 1, YC - 1, SScreenW, SScreenP)
    end;
  MovePencil(L,C)
  end { with }
 end { ClearWindow };
 
    
{****************************************************************************}

 procedure Prompt (S: PString); 

{ Print the string S on the prompt line.  }

 var X, Y: Integer;
     OldWin:  pTextWindow;
 
 begin { Prompt }
 if PromptSize <> NoPrompt then
    begin
    SReadCursor (X,Y); 
    OldWin := CurWin;
    ChangeTextWindow (PromptWindow);
    ClearLine (0,0,0);
    if Length (s) > curWin^.nColumn then
        begin
        Adjust (s, curWin^.nColumn - 1);
        AppendChar (s, '$');
        end;
    write (S);
    ChangeTextWindow (OldWin);
    NeedPrompt := false;
    ImmedPrompt := false;
    SSetCursor (X,Y)
    end
end { Prompt };
 
 
{****************************************************************}

 procedure Status (S: PString);

{ Uses SSetCursor and not Prompt to allow use by procedures before
{ the prompt window is initialized.  Puts S at top left corner of 
{ the screen.  Note that there must be a CurWin for this procedure to work,
{ althought it need not have any legitimate values.  }

var 
    X, Y: Integer;
    Ch:  char;

begin { Status }
SReadCursor(X,Y);
SSetCursor (ScreenMargin+BoxMargin, 3*LineHeight+BoxMargin-1);
write (S, '     ');
repeat
until IOCRead (KeyBoard,Ch) = IOEIOC;
SSetCursor(X,Y);
end { Status };
 
 
{****************************************************************}

procedure StatusNumber (S: PString; n:  integer);

{ Similar to Status except that is prints out S and n.  }

var 
    X, Y: Integer;
    Ch:  char;

begin { StatusNumber }
SReadCursor(X,Y);
SSetCursor (ScreenMargin+BoxMargin, 3*LineHeight+BoxMargin-1);
write (S, n, '     ');
repeat
until IOCRead (KeyBoard,Ch) = IOEIOC;
SSetCursor(X,Y);
end; { StatusNumber }


{****************************************************************}

procedure StatusCh (s: PString; ch: char);

{ Print a message and a character.  If the character is a control
{ character, print the ord of the character.  }

begin  { StatusCh }
if (ch < ' ') or (ch >= DEL) then
    Status (Concat (s, IntToStr (Ord (ch))))
else
    begin
    AppendChar (s, ch);
    Status (s);
    end;
end;  { StatusCh }
 
 
{****************************************************************}

 procedure Error (S: PString);

{ Prints an error message in the prompt window and beeps.  Waits until the
{ user types a space before continuing.  }

 var X, Y:  integer;

begin { Error }
SReadCursor(X,Y);
Prompt (Concat (Concat ('*** ', S), ', <space> to continue ***'));
write (BEL);
if macroRunning then
  begin
    freeQueue(inputQueue);
    macroRunning := false;
  end;
repeat LookForCommand (CanvKeybd)
until NewEvent.Cmd = NullCmd;
repeat
    LookForCommand (CanvKeybd);
    if (NewEvent.Cmd <> NullCmd) and (NewEvent.Ch <> ' ') then
        write (BEL);  
until NewEvent.Ch = ' ';
NeedPrompt := true;
ImmedPrompt := true;
SSetCursor(X,Y);
end { Error };
 
 
{****************************************************************}

 procedure Warn (S: PString);

{ Prints an error message in the prompt window and beeps.  }

begin { Warn }
if replay = notReplaying then
  begin
  Prompt (Concat (Concat ('*** ', S), ' ***'));
  write (BEL);
  NeedPrompt := true;
  ImmedPrompt := false;
  if macroRunning then
    begin
      freeQueue(inputQueue);
      macroRunning := false;
    end;
  end
else
  Error (S);
end { Warn };


{****************************************************************}

function GetAnswer (pr: string):  string;

{ Prompt for a string and return whatever's entered.  Eventually to
{ be replaced by an line editing procedure.  The characters entered
{ are not saved for the replay.  }

var
    ans:  string;
    oldWin:  pTextWindow;

begin  { GetAnswer }
oldWin := CurWin;
OffPointer;
Prompt (pr);
ChangeTextWindow (promptWindow);
MovePencil (0, Length (pr));
readln (ans);
ChangeTextWindow (oldWin);
OnPointer;
GetAnswer := ans;
end;  { GetAnswer }
 
    
{****************************************************************}

 function GetPosition (L: LineIndex; C: ColumnIndex):  Position;

{ Translates a line, column pair into a text position.  Returns ScreenLast
{ if the cursor is off the screen.  }

 begin { GetPosition }
 with CurWin^ do
  begin
  if DEBUG [2] then Status('Enter GetPosition');
  if Ln[L].Length <= 0 then { off screen } GetPosition := ScreenLast
  else
      with Ln[L] do
          if Eot (Start) then
              GetPosition := FilledLast
          else
              if C >= Length then
                  GetPosition := Add(Start,Length - 1)
              else GetPosition := Add(Start,C);
   end; { with }
 if DEBUG [2] then Status('Exit GetPosition')
 end { GetPosition };
 

{****************************************************************}

 procedure GetLC (P: Position; var L: LineIndex; var C: ColumnIndex);

{ Translates a text position into a line, column pair.  Returns L = -1 
{ or lastLine + 1 if P is not on the screen.  }

 var CheckC:  integer;

 begin { GetLC }
 with CurWin^ do
  begin
  if DEBUG [2] then Status('Enter GetLC');
  L := -1;
  C := 0;
  repeat
      L := L + 1;
      while Ln[L].Start = NilPosition do
          L := L + 1
  until LE(P,Ln[L].Finish) or (L = LastLine);
  if GE(P,Ln[L].Start) then
   if LE(P,Ln[L].Finish) then 
       begin
       CheckC := Subtract(P,Ln[L].Start);
       if CheckC > LastColumn then { must do long way }
           C := Ln[L].Length - Subtract (Ln[L].Finish, P)
       else
           C := CheckC
       end
   else { OffScreen }
       L := LastLine + 1
  else { OffScreen }
      L := -1
  end; { with }
  if DEBUG [2] then Status ('Exit GetLC')
 end { GetLC };
 
 
{****************************************************************}

 procedure DrawLn (BufLine:  BufRange); 

{ Draw one of the four buffer lines--search, replace, kill, or select.  }

  var GatherWidth:  integer;

{********************************}

  procedure D (BufType: string; GatherL:  LineIndex; GatherC:  ColumnIndex);

  const Margin = 9;

  var R, S, T: Position;
      i: integer;
      LastCh: Char;
      X,Y:  integer;
      OldWin:  pTextWindow;

{****************}
  
   procedure WriteCh;

   var Ch: char;

   begin { WriteCh }
    Ch := DrawCursor.Ch;
    if LastCh = CR then
     if Ch = LF then Write(EolMarker)
     else
      begin WriteChar(CR);
       if Ch <> CR then WriteChar(Ch)
      end
    else
     if Ch <> CR then WriteChar(Ch);
    LastCh := Ch
   end { WriteCh };
   
{****************}
    
  begin { D }
  if DEBUG [2] then Status ('Enter D of DrawLn');
  SReadCursor (X,Y);
  if DEBUG [2] then Status (Concat ('CurWin = ', CurWin^.Name));
  OldWin := CurWin;
  ChangeTextWindow (PromptWindow);
  with CurWin^, Bufary[BufLine] do
   begin
   ClearLine(GatherL,GatherC,GatherC+LastColumn div 2);
   CtrlSPending := false;
   write (BufType);
   WriteChar('{');
   if (SelectWindow <> nil) or (BufType <> 'Select ') then
    if NE(First,NilPosition) then
     begin R := First;
     i := LastColumn div 4 - Margin;
     while (i > 0) and NE(R,Last) do
      begin i := i - 1; Add1(R) end;
     T := Last;
     i := LastColumn div 4 - Margin;
     while (i > 0) and NE(T,First) do
      begin i := i - 1; Sub1(T) end;
     S := T;
     i := Margin;
     while (i > 0) and NE(S,First) do
      begin i := i - 1; Sub1(S) end;
     Attach(DrawCursor,First,ReadCursor);
     LastCh := ' ';
     while NE(DrawCursor.Pos,R) and NE(DrawCursor.Pos,S) do
      begin WriteCh; Add1C(DrawCursor) end;
     if NE(DrawCursor.Pos,S) then
      begin Write(DotDotDot);
       ReAttach(DrawCursor,T);
       Sub1C(DrawCursor);
       LastCh := DrawCursor.Ch;
       Add1C(DrawCursor)
      end;
     while NE(DrawCursor.Pos,Last) do
      begin WriteCh; Add1C(DrawCursor) end;
     WriteCh;
     if LastCh = CR then WriteChar(CR);
     Detach(DrawCursor)
    end; { if NE (First, NilPosition) }
   WriteChar('}')
   end; { with }
  ChangeTextWindow (OldWin);
  SSetCursor (X,Y);
  if DEBUG [2] then Status ('Exit D of DrawLn')
  end { D };
 
{********************************}
 
 begin { DrawLn }
 if PromptSize = FullPrompt then
  with PromptWindow^, Bound^ do
       case BufLine of
           KillBMin..KillBMax:  D('Kill   ', GatherL1, LastColumn div 2 + 1);
           SearchB:  D('Search ', GatherL1, 0);
           ReplaceB: D('Replace', GatherL1+1, 0);
           SelectB:  D('Select ', GatherL1+1, LastColumn div 2 + 1);
           end
 end { DrawLn };
   
   
{****************************************************************}

function Page (P:  Position):  integer;
begin 
Page := P.Chunk^.OrderP
end;

{********************************}
 
procedure UpdateThumbBar;

{ Redraw the thumb bar:  recalculate the size and position of the black hole
{ and redraw the select and mark marks if necessary.  }

var nChars, fSChar, lSChar:  real;
    TBFillOrigin, temp:  integer;
    xSelect, ySelect, selectWidth:  integer;
    xMark, yMark:  integer;

begin { UpdateThumbBar }
if DEBUG [2] then Status ('Enter UpdateThumbBar');
with CurWin^ do
if TBarOn and ThumbOn then
    begin
    if DEBUG [2] then
      Status (Concat ('UpdateThumbBar:  CurWin =', Name));
    { erase old }
    FillRectangle (White, xTBarOrigin, yTBarOrigin, TBarWidth, TBarHeight);
    with filledLast.chunk^ do
      begin
        temp := length;
        nChars := (float(orderP) * 512.0) + float(orderC) + float(temp) - 3.0;
      end;
    with screenFirst.chunk^ do
      begin
        temp := screenFirst.offset;
        fSChar := (float(orderP) * 512.0) + float(orderC) + float(temp) - 2.0;
      end;
    with screenLast.chunk^ do
      begin
        temp := screenLast.offset;
        lSChar := (float(orderP) * 512.0) + float(orderC) + float(temp) - 2.0;
      end;
    if nChars >= 1.0 then
      begin
        TBFillOrigin := xTBarOrigin +
                        round(fSChar * float(TBarWidth) / nChars);
        TBFillWidth := round((lSChar - fSChar + 1.0) *
                       float(TBarWidth) / nChars);
      end
    else
      begin
        TBFillOrigin := xTBarOrigin;
        TBFillWidth := TBarWidth;
      end;
    if TBFillOrigin < xTBarOrigin then
      TBFillOrigin := xTBarOrigin;
    if TBFillWidth < MinFillWidth then
      TBFillWidth := MinFillWidth;
    { if at end of thumb bar, adjust size }
    if  (TBFillOrigin + TBFillWidth) > (xTBarOrigin + TBarWidth) then
      TBFillWidth := xTBarOrigin + TBarWidth - TBFillOrigin;
    if DEBUG [2] then
      begin
        StatusNumber ('TBFillOrigin =', TBFillOrigin);
        StatusNumber ('TBFillWidth =', TBFillWidth);
      end;
    FillRectangle(Black, TBFillOrigin, yTBarOrigin, TBFillWidth+1, TBarHeight);
    if (SelectWindow = CurWin) and (nChars > 0.9) then
      begin
        with BufAry[SelectB].First do
          begin
            temp := offset;
            fSChar := (float(chunk^.orderP) * 512.0) + float(chunk^.orderC) +
                      float(temp) - 2.0;
          end;
        with BufAry[SelectB].Last do
          begin
            temp := offset;
            lSChar := (float(chunk^.orderP) * 512.0) + float(chunk^.orderC) +
                      float(temp) - 2.0;
          end;
        xSelect := xTBarOrigin + round(fSChar * float(TBarWidth) / nChars);
        selectWidth := round((lSChar - fSChar + 1.0) *
                             float(TBarWidth) / nChars);
        if selectWidth < 4 then
          begin
            xSelect := xSelect - ((4 - selectWidth) div 2);
            selectWidth := 4;
          end;
        if xSelect < xTBarOrigin then
          xSelect := xTBarOrigin;
        if (xSelect + selectWidth) > (xTBarOrigin + TBarWidth) then
          if xSelect >= (xTBarOrigin + TBarWidth - 4) then
            begin
             xSelect := xTBarOrigin + TBarWidth - 5;
             selectWidth := 4;
            end
          else
            selectWidth := xTBarOrigin + TBarWidth - xSelect;
        ySelect := yTBarOrigin + (TBarHeight div 2);
        fillRectangle(invert, xSelect, ySelect, selectWidth, 2);
      end;
    if Mark <> NilPosition then
      begin
        with Mark do
          begin
            temp := offset;
            fSChar := (float(chunk^.orderP) * 512.0) + float(chunk^.orderC) +
                      float(temp) - 2.0;
          end;
        if nChars >= 0.9 then
          xMark := xTBarOrigin + round(fSChar * float(TBarWidth) / nChars)
        else
          xMark := xTBarOrigin + 1;
        if xMark < xTBarOrigin + 1 then
            xMark := xTBarOrigin + 1
        else if (xMark + 2) >= (xTBarOrigin + TBarWidth) then
          xMark := xTBarOrigin + TBarWidth - 2;
        fillRectangle(invert, xMark, yTBarOrigin + 2, 2, TBarHeight - 4);
      end
    end { with }
end; { UpdateThumbBar }


{*************************************************************************}

 procedure DrawUnderLine (First,Last: Position; L: LineIndex; C: ColumnIndex;
                          U: UnderKind);

{ Underline from First to Last starting at screen coordinates L, C. }

 var FirstC: ColumnIndex;
     LastChar: boolean;
     XC, YC, Func1, Func2, Width: integer;
     LastCh: Char;
     Eol: Boolean;
 begin { DrawUnderLine }
 CtrlSPending := false;
 with CurWin^ do
  begin
  FirstC := C;
  XC := C * CharWidth + xHome;
  YC := L * LineHeight + CharHeight + 1 + yHome;
  if U = Erase then
   begin Func1 := RXor; Func2 := RXor end
  else
   begin Func1 := RXnor;
    if U = ThinBlack then Func2 := RXor
    else Func2 := RXnor
   end;
  Attach(DrawCursor,First,ReadCursor);
  if DrawCursor.Ch = LF then
   begin Sub1C(DrawCursor);
    if DrawCursor.Ch <> CR then Add1C(DrawCursor)
   end;
  LastCh := ' ';
  repeat LastChar := EQ(DrawCursor.Pos,Last);
   Eol := (DrawCursor.Ch = LF) and (LastCh = CR);
   LastCh := DrawCursor.Ch;
   if not Eol then C := C + 1;
   if (Eol and (C <> 0)) or (C = NColumn) or LastChar then
    begin Width := (C - FirstC) * CharWidth;
     if U = ThickBlack then
      begin XC := XC + 1; Width := Width - 2 end;
     RasterOp(Func1, Width, 1,
              XC - 1, YC, SScreenW, SScreenP,
              XC - 1, YC, SScreenW, SScreenP);
     YC := YC + 1;
     RasterOp(Func2, Width, 1,
              XC - 1, YC, SScreenW, SScreenP,
              XC - 1, YC, SScreenW, SScreenP);
     L := L + 1;
     C := 0;
     XC := xHome;
     YC := YC + LineHeight - 1;
     FirstC := 0
    end;
   Add1C(DrawCursor)
  until LastChar;
  Detach(DrawCursor)
  end { with }
 end { DrawUnderLine };


{****************************************************************}

procedure ScrollUp (* L1, L2:  LineIndex; Distance:  integer *);

{ Scroll up lines L1 to L2, Distance lines.  If Distance > L2 then the
{ editor will bomb.  The cursor is not moved by this operation.  }

 var i, j: integer;
     P: Position;
     XC, YC, Height: integer;

begin { ScrollUp }
if DEBUG [2] then Status('Enter ScrollUp');
if L2 > L1 then
    with CurWin^, Bound^ do
        begin
        if (L1 = 0) and Eot(ScreenLast) then
            while Eot(Ln[Distance-1].Finish) and (Distance > 0) do
                Distance := Distance - 1;
        if Distance > 0 then
            begin
            XC := xHome;
            YC := L1 * LineHeight + yHome;
            Height := (L2 - L1 - Distance + 1) * LineHeight;
            if DEBUG [2] then Status ('RasterOp 1');
            if Height > 0 then
                RasterOp(RRpl, xMax - xHome, Height,
                 XC - 1, YC - 1, SScreenW, SScreenP,
                 XC - 1, YC + Distance * LineHeight - 1, SScreenW, SScreenP);
            YC := (L2 - Distance + 1) * LineHeight + yHome;
            if DEBUG [2] then Status ('RasterOp 2');
            RasterOp(RXor, xMax - xHome, Distance * LineHeight,
             XC - 1, YC - 1, SScreenW, SScreenP,
             XC - 1, YC - 1, SScreenW, SScreenP);
            for i := L1 to L2-Distance do Ln[i] := Ln[i+Distance];
            if DEBUG [2] then Status ('Before Draw');
            if (L2 = LastLine) and not Eot(ScreenLast) then
                begin
                P := Add (ScreenLast,1);
                Draw(P,FilledLast,LastLine-Distance+1,0)
                end
            else
                for i := L2 - Distance + 1 to L2 do
                    with Ln[i] do
                        begin
                        Start := FilledLast;
                        Finish := FilledLast;
                        Length := 1;
                        end;
            if L1 = 0 then
                 ScreenFirst := Ln[0].Start
            else
                L1 := L1 - 1;
            updateThumbBar;
            if CurrentPtr = TextPtr then
                if CurLine > 0 then
                    MoveTextPointer (CurLine-1, CurCol)
                else
                    if CurCol <= Ln [0].Length then
                        MoveTextPointer (0, CurCol)
                    else
                        MoveTextPointer (0, Ln[0].Length)
            end { if Distance > 0 }
        end; { with }
if DEBUG [2] then Status('Exit ScrollUp');
end { ScrollUp };
 
 
{**********************************************************************}

 procedure ScrollDown(* L1, L2: LineIndex; Distance: integer *);

{ Scroll down lines L1 to L2, Distance lines.  If L1 + Distance > LastLine then
{ the editor will bomb.  The cursor is not moved by this operation.  }

 var i, j: integer;
     L: LineIndex;
     C: ColumnIndex;
     P, Q: Position;
     XC, YC, Height: integer;

 begin { ScrollDown }
 with CurWin^, Bound^ do
  begin
  if DEBUG [2] then Status('Enter ScrollDown');
  if L2 > L1 then
   begin
    if L1 = 0 then
     if Bot(ScreenFirst) then Distance := 0
     else
      begin 
       Q := Add (ScreenFirst,-1);
       Attach(DrawCursor,Q,ReadCursor);
       j := 0;
       repeat 
        if j = 0 then 
            i := -1
        else
            i := 0;
        repeat
         repeat Sub1C(DrawCursor);
          i := i + 1
         until DrawCursor.Ch = LF;
        Sub1C(DrawCursor);
        if DrawCursor.Ch <> CR then Add1C(DrawCursor);
       until DrawCursor.Ch = CR;
        j := j + (i + LastColumn) div NColumn
       until (j >= Distance) or Bot(DrawCursor.Pos);
       P := Add (DrawCursor.Pos,2); { skip CR LF }
       Detach(DrawCursor);
       while j > Distance do
        begin 
         P := Add (P,NColumn);
         i := i - NColumn;
         j := j - 1
        end;
       Distance := j
      end;
    if Distance > 0 then
     begin XC := xHome;
      YC := L1 * LineHeight + yHome;
      Height := (L2 - L1 - Distance + 1) * LineHeight;
      if Height > 0 then
       RasterOp(RRpl, xMax - xHome, Height,
                XC - 1, YC + Distance * LineHeight - 1, SScreenW, SScreenP,
                XC - 1, YC - 1, SScreenW, SScreenP);
      RasterOp(RXor, xMax - xHome, Distance * LineHeight,
               XC - 1, YC - 1, SScreenW, SScreenP,
               XC - 1, YC - 1, SScreenW, SScreenP);
      for i := L2 downto L1+Distance do Ln[i] := Ln[i-Distance];
      if L2 = LastLine then
        ScreenLast := Ln[LastLine].Finish;
      if L1 = 0 then Draw(P,Q,0,0)
      else
       for i := L1 to L1+Distance-1 do
        with Ln[i] do
         begin Start := ScreenFirst;
          Finish := ScreenFirst;
          Length := 1
         end
     end
   end;
  if DEBUG [2] then Status('Exit ScrollDown')
  end
 end { ScrollDown };


{**********************************************************************}

 procedure Draw (P, Q: Position; L: LineIndex; C: ColumnIndex);

{ Draw from P to Eot, set Ln, and set ScreenLast if possible.  Uses
{ microcode to write a line at a time.  }

 var Done, NewLine, SelectOnScreen, ChangeThumbBar: boolean;
     Ch: char;
     SL, SelFirst, SelLast: Position;
     SelL: LineIndex;
     SelC: ColumnIndex;
     FontP: FontPtr;
     Termination: (CharCount, ScreenWidth, ControlChar);
     X, Y: Integer;
     NewChOffset, AdvanceC, Max, TMax: Integer;
     Redrawing: Boolean;
     CharP :pSwapBuffer;                { for Raster Op hack }
     FirstChar, LastCharP1 :integer;    { for Raster Op hack }
  
  
{********************************}

  procedure Advance( N: Integer );

  begin { Advance }
   if EQ(DrawCursor.Pos,Q) then Done := true
   else
    begin AddC(DrawCursor,N);
     if DrawCursor.Pos.Chunk = nil then Done := true
    end
  end { Advance };
  
  
{********************************}

 begin { Draw }
 with CurWin^, Bufary[SelectB] do
  begin
  if DEBUG [2] then Status('Enter Draw');
  FontP := GetFont;
  ChangeThumbBar := false;
  if L <= 0 then
      L := 0;
  if C <= 0 then
   begin Ln[L].Start := P;
    if L = 0 then
     begin ScreenFirst := P;
      ChangeThumbBar := true
     end
   end;
  if C < 0 then
   begin
    Redrawing := False;
    ClearWindow (L,0);
    C := 0;
    MovePencil(L,0)
   end
  else
   begin
    Redrawing := True;
    ClearLine(L,C,0)
   end;
  NewLine := false;
  Done := false;
  Attach(DrawCursor,P,ReadCursor);
  if SelectWindow <> CurWin then
      SelectOnScreen := false
  else
    if LT(DrawCursor.Pos,First) then SelectOnScreen := false
  else
   if GT(DrawCursor.Pos,Last) then SelectOnScreen := false
   else
    begin SelFirst := DrawCursor.Pos;
     SelL := L;
     SelC := C;
     SelectOnScreen := true
    end;
{  OffPointer;  }
  repeat
   CtrlSPending := false;
   if EQ(DrawCursor.Pos,Q) then Done := true;
   Ch := DrawCursor.Ch;
   if SelectWindow = CurWin then
    if EQ(DrawCursor.Pos,First) then
     begin SelFirst := DrawCursor.Pos;
      SelL := L;
      SelC := C;
      SelectOnScreen := true
     end;
   if Done or (Chr(Land(Ord(Ch), #177)) < ' ') then { draw a single character }
    begin
     SL := DrawCursor.Pos;
     if Ch = CR then
      begin
       Advance(1);
       if not Done and (DrawCursor.Ch = LF) then
        begin SL := DrawCursor.Pos;
         WriteChar(' ');
         Advance(1);
         NewLine := true
        end
       else WriteChar(Ch)
      end
     else
      begin
       if EQ(DrawCursor.Pos,FilledLast) then SPutChr(EotMarker)
       else WriteChar(Ch);
       Advance(1)
      end;
     C := C + 1
    end
   else
    begin
     Max := DrawCursor.Pos.Chunk^.Length - DrawCursor.Pos.Offset;
     if DrawCursor.Pos.Chunk = Q.Chunk then { stop before end of text }
      begin TMax := Q.Offset - DrawCursor.Pos.Offset;
       if TMax < Max then Max := TMax
      end;
     if DrawCursor.Pos.Chunk = First.Chunk then
      if First.Offset > DrawCursor.Pos.Offset then { stop before select }
       begin
        TMax := First.Offset - DrawCursor.Pos.Offset;
        if TMax < Max then Max := TMax
       end;
     if C + Max > NColumn then { stop at end of line }
      Max := NColumn - C;
{ new rasterOp code }
     CharP := Txt[DrawCursor.ChPage].Buffer;  { RasterOp hack }
     FirstChar := DrawCursor.ChOffset;
     LastCharP1 := FirstChar + Max;
     SReadCursor(X, Y);
     {$R-}
     LoadExpr(RRpl);               { Raster-Op function }
     LoadExpr(X);                  { destination X }
     LoadExpr(Y);                  { destination Y }
     LoadExpr(ScreenSeg);          { destination base }
     LoadExpr(0);
     LoadAdr(FontP);
     InLineByte( 239 {LDDW} );
     LoadAdr(CharP^);
     LoadExpr(FirstChar);
     LoadExpr(LastCharP1);
     LoadExpr(768);
     LoadExpr( LOr( Shift(#4010,8), Shift(#4010,-8) ) );
     InLineByte( 240 {STLATE} );
     InLineByte( #165 {7,,5} );
     InLineByte( 240 {STLATE} );
     InLineByte( #7 {0,,7} );
     {$R=}
{ end of new rasterOp code }
{
     (Tos)   = Maximum X-coordinate + 1.
     (Tos-1) = Maximum byte offset + 1.
     (Tos-2) = Byte offset from the beginning of the byte array.
     (Tos-3) = Address of the byte array as an offset from the base of the
               memory stack.
     (Tos-4) = Character set address as an offset from the base of the
               memory stack.
     (Tos-5) = Destination base address as an offset from the base of the
               memory stack.
     (Tos-6) = Destination Y-coordinate.
     (Tos-7) = Destination X-coordinate.
     (Tos-8) = Raster-op function.
}
     InLineByte( 191 {JCS} );
{
     (Tos)   = Current X-Coordinate.
     (Tos-1) = Next byte offset.
     (Tos-2) = Termination condition:
                 0 - Character count exhausted.
                 1 - Screen width exhausted.
                 2 - Control character encountered.
}
     StorExpr(X);
     StorExpr(NewChOffset);
     StorExpr(Termination);
     AdvanceC := NewChOffset - DrawCursor.ChOffset;
     SL := Add(DrawCursor.Pos, AdvanceC - 1);
     Advance(AdvanceC);
     C := C + AdvanceC;
     MovePencil(L,C)
    end;
   if C = NColumn then NewLine := true;
   if NewLine or Done then
    begin NewLine := false;
     Ln[L].Finish := SL;
     Ln[L].Length := C;
     L := L + 1;
     C := 0;
     if L > LastLine then Done := true;
     if not Done then
      begin Ln[L].Start := DrawCursor.Pos;
       if Redrawing then ClearLine(L,0,0) else MovePencil(L,0)
      end
    end
  until Done;
  if ptrIsOff then
    OnPointer;
  if (L > LastLine) or Eot(DrawCursor.Pos) then
   begin ScreenLast := SL;
    ChangeThumbBar := true;
    for L := L to LastLine do with Ln[L] do
     begin Start := FilledLast;
      Finish := FilledLast;
      Length := 1
     end
   end;
  Detach(DrawCursor);
  if SelectOnScreen then
   begin ChangeThumbBar := true;
    if LE(Last,SL) then SelLast := Last
    else SelLast := SL;
    DrawUnderLine(SelFirst,SelLast,SelL,SelC,ThinBlack)
   end;
  if ChangeThumbBar then UpdateThumbBar;
  end; { with }
  if DEBUG [2] then Status('Exit Draw')
 end { Draw };
 
 
{****************************************************************}

 procedure UnderLine (First, Last: Position; U: UnderKind);

{ Underline the portion of screen from First to Last.  }

 var L: LineIndex;
     C: ColumnIndex;

 begin { UnderLine }
 with CurWin^ do
  begin
  if DEBUG [2] then Status('Enter UnderLine');
  if LT(First,ScreenFirst) then First := ScreenFirst;
  if GT(Last,ScreenLast) then Last := ScreenLast;
  if LE(First,Last) then
   begin GetLC (First,L,C); DrawUnderLine(First,Last,L,C,U) end;
  if DEBUG [2] then Status('Exit UnderLine')
  end { with }
 end { UnderLine };
 
 
{****************************************************************}

function OnScreen  (P: Position; L1, L2: LineIndex ): boolean;

{ Return true if the position P is on the screen between line L1 and
{ line L2.  Automatically fails if L1 > L2.  }

 var L: LineIndex;
     C: ColumnIndex;

begin { OnScreen }
 with CurWin^ do
  begin
  if DEBUG [2] then Status('enter OnScreen');
  if (L1 > LastLine) or (L2 < 0) then OnScreen := False
  else
   begin
    GetLC (P, L, C);
    if L < L1 then OnScreen := Bot(ScreenFirst)
    else OnScreen := L <= L2
   end;
  end; { with }
  if DEBUG [2] then Status ('Enter OnScreen debug');
  if DEBUG [1] then
   begin
    with CurWin^ do
     begin
     if (L1 <= LastLine) and (L2 >= 0) then
      StatusNumber ('LC =', 100*L+C);
     StatusNumber ('L1L2 =', 100*L1+L2);
     if (L1 > LastLine) or (L2 < 0) then Status ('  Off')
     else
      if L < L1 then
       if Bot(ScreenFirst) then Status('  On')
       else Status('  Off')
      else
       if L <= L2 then Status('  On')
       else Status('  Off');
    end; { with }
   end;
 if DEBUG [2] then Status ('Exit OnScreen')
end { OnScreen }; 
 
  
{****************************************************************}

 procedure Show (P: Position; L1, L2: LineIndex);

{ If P is not on the screen then redraw, placing P midway between L1 and L2.
{ Then move the cursor to P.
{ Show depends on the fact that Add (FilledFirst,1).Ch = CR.  }

 var L:  LineIndex;
     C:  ColumnIndex;
     nL, nC:  integer;  { can be > MaxColumn if long lines }
     Q:  Position;

begin { Show }
if DEBUG [2] then Status('enter Show');
 with CurWin^ do
  begin
  if not OnScreen(P,L1,L2) then
   begin
    nL := LastLine div 2;
    Q := P;
    Attach(DrawCursor,Q,ReadCursor);
    repeat nC := 0;
     repeat
      repeat
           Sub1C(DrawCursor);
           nC := nC + 1
      until (DrawCursor.Ch = LF);
      Sub1C(DrawCursor);
      if DrawCursor.Ch <> CR then Add1C(DrawCursor)
     until DrawCursor.Ch = CR;
     nL := nL - (nC + LastColumn) div NColumn
    until (nL <= 0) or Bot(DrawCursor.Pos);
    Q := Add (DrawCursor.Pos,2);  { skip CR LF }
    Detach(DrawCursor);
    while nL < 0 do
     begin Q := Add (Q,NColumn);
      nL := nL + 1
     end;
    Draw(Q,FilledLast,0,-1);
   end; {if not OnScreen }
   GetLC (P,L,C);
   MoveTextPointer (L,C)
  end; { with }
if DEBUG [2] then Status('exit Show')
end { Show };
  
 
{****************************************************************}

 procedure JoinScreen (P: Position; L: LineIndex; C: ColumnIndex); 

{ Move the text starting at position P to line L, column C.  }

 var Q: Position;
     UpperL, LowerL: LineIndex;
 begin { JoinScreen }
 with CurWin^ do
  begin
  Attach(DrawCursor,P,ReadCursor);
  if not Eot(DrawCursor.Pos) then
   repeat
    while (DrawCursor.Ch <> CR) and not Eot(DrawCursor.Pos) do
        Add1C (DrawCursor);
    if (DrawCursor.Ch = CR) and not Eot(DrawCursor.Pos) then
     begin Add1C(DrawCursor);
      if DrawCursor.Ch <> LF then Sub1C(DrawCursor)
     end
   until (DrawCursor.Ch = LF) or Eot(DrawCursor.Pos);
  Q := DrawCursor.Pos;
  Detach(DrawCursor);
  Draw(P,Q,L,C);
  if OnScreen(Q,0,LastLine-1) then
   begin GetLC(Q,UpperL,C);
    UpperL := UpperL + 1;
    LowerL := LastLine;
    if Eot(Q) then ClearWindow(UpperL,0)
    else
     begin Add1(Q);
      while (LowerL > 0) and NE(Ln[LowerL].Start,Q) do LowerL := LowerL - 1;
      if EQ(Ln[LowerL].Start,Q) and (LowerL >= UpperL) and not Eot(Q) then
       ScrollUp(UpperL,LastLine,LowerL-UpperL)
      else
       Draw(Q,FilledLast,UpperL,-1)
     end
   end
  end { with }
 end; { JoinScreen }


{****************************************************************}

procedure CheckBounds (NewWindow:  pWindow);

{ See if bounds are on the screen.  If not, make them so.  Does not check if
{ bounds are within the editor's portion of the screen.  }

begin { CheckBounds }
with NewWindow^ do
    begin
    if xMin < 0 then
        xMin := 0
    else if xMax > xScreenMax then
        xMax := xScreenMax;
    if yMin < 0 then
        yMin := 0
    else if yMax > yScreenMax then
        yMax := yScreenMax
    end 
end; { CheckBounds }

    
{**********************************************************************}

procedure SetUpPromptWindow (* PromptW:  pWindow *);

{ Set up the data structure that holds information about the prompt window.
{ Does NOT refresh the screen.  Should only be called by EditInit.  }

var
    PromptHeight:  integer;
   
begin { SetUpPromptWindow }
if DEBUG [2] then Status  ('Enter SetUpPromptWindow');
PromptWindow := nil;
if PromptW = nil then
    PromptSize := NoPrompt
else
    begin
    CheckBounds (PromptW);
    PromptHeight := PromptW^.yMax - PromptW^.yMin;
    if PromptHeight < minPromptHeight then
        PromptSize := NoPrompt
    else if PromptHeight < maxPromptHeight then
            { one line prompt }
            { NOT YET IMPLEMENTED }
        PromptSize := NoPrompt
    else { full prompt window }
        begin
        PromptSize := FullPrompt;
        with PromptW^ do
            yMax := yMin + maxPromptHeight - 1;
        PromptWindow := SetUpWindow(PromptW,'',true,false, false, true, false);
        PromptWindow^.WinNum := PromptNum
        end; 
    end;
if DEBUG [2] then Status  ('Exit SetUpPromptWindow');
end; { SetUpPromptWindow }


{**********************************************************************}

function SetUpWindow (WinBound: pWindow; WindowName:  PString; DoPrompt,
    DoScroll, DoThumb, DoBox, AutoSave:  boolean):  pTextWindow;

{ Sets up a text window data structure, including prompt windows.  Does NOT
{ redraw the screen.  }

var Margin:  integer;
    HTB, VTB:  integer; { needed since Prompt windows don't have them }
    NewWin:  pTextWindow;
    l:  LineIndex;
 
begin { SetUpWindow }
if DEBUG [2] then Status  ('Enter SetUpWindow');
Margin := 0; 
CheckBounds (WinBound);
{
if DoPrompt then
}
    new (WindowSeg, 1, NewWin, false, true)
{
else
    if DoThumb then
        new (WindowSeg, 1, NewWin, false, true)
    else
        new (WindowSeg, 1, NewWin, false, true)
};
with NewWin^ do
    begin
    Name := WindowName;
    new (Bound);
    Bound^ := WinBound^;
    HTB := Bound^.yMin;
    VTB := Bound^.xMin;
    BoxOn := DoBox or DoScroll or DoThumb;
    if BoxOn then
        Margin := BoxMargin;
    if DoScroll then
        VTB := VTB + ScrollWidth;
    if DoThumb then
        HTB := HTB + ThumbHeight;
    xHome := VTB + Margin;
    yHome := HTB + Margin;
    LastLine := (Bound^.yMax - yHome - Margin) div LineHeight - 1;
    yLastLine := (LastLine + 1) * LineHeight + yHome;
    nColumn := (Bound^.xMax - xHome - Margin) div CharWidth;
    LastColumn := nColumn - 1;
    xLastColumn := nColumn * CharWidth + xHome;
    leftMargin := dfltLeftMargin;
    rightMargin := dfltRightMargin;
    IsPrompt := DoPrompt;
    if not IsPrompt then
        begin
        HTextBound := HTB;
        VTextBound := VTB;
        ScrollOn := DoScroll;
        CurLine := 0;
        CurCol := 0;
        FilledFirst := NilPosition;
        FilledLast := NilPosition;
        ScreenFirst := NilPosition;
        ScreenLast := NilPosition;
        Mark := NilPosition;
        for L := 0 to LastLine do
            begin 
            Ln[L].Start := NilPosition;
            Ln[L].Finish := NilPosition;
            end; { for }
        if AutoSave then
            SaveAfter := NormSaveAfter
        else
            SaveAfter := -1;
        nChanges := 0;
        nMoves := 0;
        Prev := nil;
        Next := nil;
        ThumbOn := DoThumb;
        if DoThumb then
            begin
            xTBarOrigin := Bound^.xMin + xTBarOffset;
            yTBarOrigin := Bound^.yMin + yTBarOffset;
            yThumb := yTBarOrigin + TBarHeight
            end
        end { if not IsPrompt }
    end; { with }
SetUpWindow := NewWin;
if DEBUG [2] then Status  ('Exit SetUpWindow');
end; { SetUpWindow }


{**********************************************************************}

procedure RefreshPromptWindow;

var OldWin:  pTextWindow;
    Height, Width:  integer;
    b:  BufRange;

begin
if DEBUG [2] then Status  ('Enter RefreshPromptWindow');
if PromptSize <> NoPrompt then
    begin
    OldWin := CurWin;
    ChangeTextWindow (PromptWindow);
    with PromptWindow^ do
        begin
        with Bound ^ do
            begin
            Width := xMax - xMin + 1;
            Height := yMax - yMin + 1;
            FillRectangle (White, xMin, yMin, Width, Height);
            if BoxOn then
                PutRectangle (xMin, yMin, Width, Height, 1, Black)
            end;
        ChangeTextWindow (OldWin);
        if PromptSize = FullPrompt then
            begin
            Prompt (CmdPrompt);
            DrawLn (KillB);
            for b := SearchB to SelectB do
                DrawLn (b);
            end
        end; { with }
    end; { if PromptSize... }
if DEBUG [2] then Status  ('Exit RefreshPromptWindow')
end; { RefreshPromptWindow }


{****************************************************************}

procedure RefreshName;

{ Write the file name in the thumbbar region, prepended with a '*' if the }
{ file has been modified.  }

var
    xName, NameWidth:  integer;

begin
with CurWin^, Bound^ do
    begin
    { first erase old name with gray }
    xName := xMin + NameOffset;
    NameWidth := (Length (Name) + 5) * CharWidth;
    FillRectangle (Gray, xName, yMin+1, xTBarOrigin-2-xName, ThumbHeight-1);
    FillRectangle (White, xName, yTBarOrigin, NameWidth, TBarHeight);
    PutRectangle (xName, yTBarOrigin-1, NameWidth, TBarHeight+2, 1, Black);

    { now write new name }
    SSetCursor (xMin + NameOffset + CharWidth div 2, yTBarOrigin + TBarHeight);
    SChrFunc (ROr);
    if nChanges = 0 then
        write (chr(WinNum + ord('0')), ':  ', Name)
    else
        write (chr(WinNum + ord('0')), ': *', Name);
    SChrFunc (RRpl);
    end;
end;  { RefreshName }    


{****************************************************************}

procedure RefreshTextWindow;


procedure DrawRegions;

var 
    Width, Height:  integer;

begin
with CurWin^, Bound^ do
    begin
    Width := xMax - xMin + 1;
    Height := yMax - yMin + 1;
    FillRectangle (White, xMin, yMin, Width, Height);
    if DEBUG [2] then StatusNumber ('RefreshTextWindow:  FilledFirst Page =',
        FilledFirst.Chunk^.OrderP);
    if BoxOn then
        PutRectangle (xMin, yMin, Width, Height, 1, Black);
    if ThumbOn then
        begin
        FillRectangle (Black, xMin, yMin + ThumbHeight, Width, 1);
        FillRectangle (Gray, xMin+1, yMin+1, Width-2, ThumbHeight-1);
        RefreshName;
        PutRectangle (xTBarOrigin-1, yTBarOrigin-1, TBarWidth+2,
            TBarHeight+2, 1, Black)
        end;
    if ScrollOn then
        FillRectangle (Black, ScrollWidth, yMin+ThumbHeight, 1, 
            Height-ThumbHeight)
    end { with }
end; { DrawRegions }


{****************************}

begin { RefreshTextWindow }
if DEBUG [2] then Status  ('Enter RefreshTextWindow');
with CurWin^ do
    begin
    DrawRegions;
    Draw (ScreenFirst, FilledLast, 0, 0);
    MoveTextPointer (CurLine, CurCol);
    NeedPrompt := true;
    end; { with }
if DEBUG [2] then Status  ('Exit RefreshTextWindow')
end. { RefreshTextWindow }
