module RealFunctions;
{-----------------------------------------------------------------------------
{
{ RealFunctions - Standard functions for reals.
{ J. Strait   27 Nov 81.
{ Copyright (C) Three Rivers Computer Corporation, 1981.
{
{ Abstract:
{       RealFunctions implements many of the standard functions whose
{       domain and/or range is the set of real numbers.  The
{       implementation of these functions was guided by the book
{
{       Software Manual for the Elementary Functions,
{       William J. Cody, Jr.  and  William Waite,
{       (C) 1980 by Prentice-Hall, Inc.
{
{       The domain (inputs) and range (outputs) of the functions are
{       given in their abstract.  The following notation is used.
{       Parentheses () are used for open intervals (those that do not
{       include the endpoints), and brackets [] are used for closed
{       intervals (those that do include their endpoints).  The closed
{       interval [RealMLargest, RealPLargest] is used to mean all real
{       numbers, and the closed interval [-32768, 32767] is used to
{       mean all integer numbers.
{
{       Currently all functions described by Cody and Waite are
{       implemented with the exception of the hyperbolic functions
{       (SinH, CosH, TanH).
{
{       DISCLAIMER:
{
{       Only the most cursory testing of these functions has been done.
{       No guarantees are made as to the accuracy or correctness of the
{       functions.  Validation of the functions must be done, but at
{       some later date.
{
{ Design:
{       AdX, IntXp, SetXp, and Reduce are implemented as Pascal
{       functions.  It is clear that replacing the calls with in-line
{       code (perhaps through a macro expansion) would improve the
{       efficiency.
{
{       Many temporary variables are used.  Elimination of unnecessary
{       temporaries would also improve the efficiency.
{
{       Many limit constants have been chosen conservatively, thus
{       trading a small loss in range for a guarantee of correctness.
{       The choice of these limits should be re-evaluated by someone
{       with a better understanding of the issues.
{
{       Some constants are expressed in decimal (thus losing the
{       guarantee of precision).  Others are expressed as Sign,
{       Exponent, and Significand and are formed at execution time.
{       Converting these two 32-bit constants which are Recast into
{       real numbers would improve the correctness and efficiency.
{
{       More thought needs to be given to the values which are returned
{       after resuming from an exception.  The values that are returned
{       now are the ones recommended by Cody and Waite.  It seems that
{       Indefinite values (NaNs in the IEEE terminology) might make
{       more sense in some cases.
{
{-----------------------------------------------------------------------------}

{-----------------------------------------------------------------------------
{
{ Change history:
{
{ 27 Nov 81  V1.0  J. Strait
{ Start module.
{
{-----------------------------------------------------------------------------}

exports


const RealPInfinity =   Recast(#17740000000,Real);   {  1.0 / 0.0 }
      RealMInfinity =   Recast(#37740000000,Real);   { -1.0 / 0.0 }
      RealPIndefinite = Recast(#00000000001,Real);   {  0.0 / 0.0 }
      RealMIndefinite = Recast(#20000000001,Real);   { -0.0 / 0.0 }
      RealPLargest =    Recast(#17737777777,Real);   { largest positive }
      RealMLargest =    Recast(#37737777777,Real);   { largest negative }
      RealPSmallest =   Recast(#00040000000,Real);   { smallest positive }
      RealMSmallest =   Recast(#20040000000,Real);   { smallest negative }


function Sqrt( X: Real ): Real;
function Exp( X: Real ): Real;
function Ln( X: Real ): Real;
function Log10( X: Real ): Real;
function Power( X, Y: Real ): Real;
function PowerI( X: Real; Y: Integer ): Real;
function Sin( X: Real ): Real;
function Cos( X: Real ): Real;
function Tan( X: Real ): Real;
function CoTan( X: Real ): Real;
function ArcSin( X: Real ): Real;
function ArcCos( X: Real ): Real;
function ArcTan( X: Real ): Real;
function ArcTan2( Y, X: Real ): Real;




exception SqrtNeg( X: Real );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       SqrtNeg is raised when Sqrt is passed a negative argument.
{       You may resume from this exception, in which case Sqrt returns
{       Sqrt(Abs(X)).
{
{ Parameters:
{       X - Argument of Sqrt.
{
{-----------------------------------------------------------------------------}




exception ExpLarge( X: Real );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       ExpLarge is raised when Exp is passed an argument which is too
{       large.  You may resume from this exception, in which case Exp
{       returns RealPInfinity.
{
{ Parameters:
{       X - Argument of Exp.
{
{-----------------------------------------------------------------------------}




exception ExpSmall( X: Real );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       ExpLarge is raised when Exp is passed an argument which is too
{       small.  You may resume from this exception, in which case Exp
{       returns 0.0.
{
{ Parameters:
{       X - Argument of Exp.
{
{-----------------------------------------------------------------------------}




exception LogSmall( X: Real );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       LogSmall is raise when Ln or Log10 is passed an argument which
{       is too small.  You may resume from this exception in which case
{       Ln or Log10 returns RealMInfinity if X is zero or the log of
{       Abs(X) if X is non-zero.
{
{ Parameters:
{       X - Argument of Ln or Log10.
{
{-----------------------------------------------------------------------------}




exception PowerZero( X, Y: Real );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       PowerZero is raised when Power or PowerI is called with X = 0.0
{       and Y = 0.0.  You may resume from this exception in which case
{       Power or PowerI returns RealPInfinity.
{
{ Parameters:
{       X - Argument of Power or PowerI.
{       Y - Argument of Power or PowerI.
{
{-----------------------------------------------------------------------------}




exception PowerNeg( X, Y: Real );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       PowerNeg is raised when Power is called with X < 0.0 or with
{       X = 0.0 and Y < 0.0, or PowerI is called with X = 0.0 and
{       Y < 0.  You may resume from this exception in which case Power
{       or PowerI returns Power(Abs(X),Y) in the case of X < 0.0 or
{       returns RealPInfinity in the case of X = 0.0 and Y < 0.0.
{
{ Parameters:
{       X - Argument of Power or PowerI.
{       Y - Argument of Power or PowerI.
{
{-----------------------------------------------------------------------------}




exception PowerBig( X, Y: Real );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       PowerBig is raised when Power or PowerI is called with X and Y
{       for which X raised to the Y power is too large to be represented.
{       You may resume from this exception in which case Power or PowerI
{       returns RealPInfinity.
{
{ Parameters:
{       X - Argument of Power or PowerI.
{       Y - Argument of Power or PowerI.
{
{-----------------------------------------------------------------------------}




exception PowerSmall( X, Y: Real );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       PowerSmall is raised when Power or PowerI is called with X and Y
{       for which X raised to the Y is too close to zero to be represented.
{       You may resume from this exception in which case Power or PowerI
{       returns 0.0.
{
{ Parameters:
{       X - Argument of Power or PowerI.
{       Y - Argument of Power or PowerI.
{
{-----------------------------------------------------------------------------}




exception SinLarge( X: Real );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       SinLarge is raised when Sin is called with an argument which is
{       too large.  You may resume from this exception in which case
{       Sin returns 0.0.
{
{ Parameters:
{       X - Argument of Sin.
{
{-----------------------------------------------------------------------------}




exception CosLarge( X: Real );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       CosLarge is raised when Cos is called with an argument which is
{       too large.  You may resume from this exception in which case
{       Cos returns 0.0.
{
{ Parameters:
{       X - Argument of Cos.
{
{-----------------------------------------------------------------------------}




exception TanLarge( X: Real );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       CosLarge is raised when Tan or CoTan is called with an argument
{       which is too large.  You may resume from this exception in which
{       case Tan or CoTan returns 0.0.
{
{ Parameters:
{       X - Argument of Tan or CoTan.
{
{-----------------------------------------------------------------------------}




exception ArcSinLarge( X: Real );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       ArcSinLarge is raised when ArcSin is called with an argument
{       which is too large.  You may resume from this exception in which
{       case ArcSin returns RealPInfinity.
{
{ Parameters:
{       X - Argument of ArcSin.
{
{-----------------------------------------------------------------------------}




exception ArcCosLarge( X: Real );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       ArcCosLarge is raised when ArcCos is called with an argument
{       which is too large.  You may resume from this exception in which
{       case ArcCos returns RealPInfinity.
{
{ Parameters:
{       X - Argument of ArcCos.
{
{-----------------------------------------------------------------------------}




exception ArcTan2Zero( Y, X: Real );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       ArcTan2Zero is raised when ArcTan2 is called with both X and Y
{       equal to zero.  You may resume from this exception in which
{       case ArcTan2 returns RealPInfinity.
{
{ Parameters:
{       Y - Argument of ArcTan2.
{       X - Argument of ArcTan2.
{
{-----------------------------------------------------------------------------}




private
