/*
    CONFIGUR - This program will configure the serial and parallel ports
	on the Z-150 computer.
	    
    Author: RJM
    Date:   9/21/83
    File:   CONFIGUR.C

    Version 1.75 - Initial version
    Version 1.76 - Fix selection of an option marked (not available in
		   current configuration)
    Version 1.77 - Fix #define addresses of SERIAL_TIME_OUT and
    		   PRINTER_TIME_OUT in z150rom.h
    Version 1.78 - Make compatible with MS-DOS 1.25
    Version 1.90 - Add Compatability handshake in user defined.
		   Bring all utilities to V1.90
    Version 2.00 - Bring all utilities to V2.00, fix selection of DTR neg. in user defined
    Version 2.01 - 4/29/84 Undo mapping if selection is configur LPT device
		   Also 2.0 BIOS moves to 70H (in z150bios.h)
    Version 2.02 - 2/21/85 Add configurable auto-assign capability in the
		   BIOS.  New BIOS version number.
    Version 2.03 - 8/26/85 Take off R/O attribute on IO.SYS while updating.
		   Add alternate menu support.
*/

/*
		RESTRICTED RIGHTS LEGEND
		------------------------
	
	    "Use, duplication, or disclosure by the
	Government is subject to restrictions as set forth
	in paragraph (b) (3) (B) of the Rights in Technical
	Data and Computer Software clause in DAR
	7-104.9(a).  Contractor/manufacturer is Zenith
	Data Systems Corporation of Hilltop Road, St.
	Joseph, Michigan 49085.
*/

#include "stdio.h"
#include "ctype.h"
#include "conio.h"
#include "z150bios.h"
#include "z150rom.h"
#include "msdos.h"

#define FALSE 0
#define TRUE 1


/* Flags for signons, only one should be true */
#define	ZENITH	TRUE
#define	NBI	FALSE
#define	SYNTREX	FALSE


/* Version and release of CONFIGUR */
#define VERSION 2
#define RELEASE 3

#define NUM_SCT_DEFAULT 7	/* Number of default serial configurations */

/* Some useful system call numbers */
#define CHDIR		0x3b
#define GET_VERSION	0x30

#define NUM_ALT 8		/* Number of alternate menu choices */

/* Externals defined by the Z-150 interface module Z150INT */

extern unsigned rax, rbx, rcx, rdx;
extern char ral, rah, rbl, rbh, rcl, rch, rdl, rdh;

/* globals to count the number of LPT and COM devices currently attached */
unsigned numlpt, numcom;
char max_drive;			/* Maximum drive number */

/* Local copy of the BIOS configuration tables */
struct io_table config_table;

struct menu_choice {		/* alternate menu choices */
    char name[80];
    struct sct parms;
} alt_table[NUM_ALT];

/* Flag for if running under DOS 2.0 */
char dos2;

char a_flag;			/* auto-assign flag */

unsigned bios_seg;		/* segment of the BIOS */
unsigned config_ptr;		/* offset from bios_seg of configuration info. */

/* Define the externals used in the z150int() module */
extern char ral, rah, rbl, rbh, rcl, rch, rdl, rdh;
extern unsigned rax, rbx, rcx, rdx, rsi, rdi, rsp, rbp;
extern unsigned rcs, rds, res, rss, flags;
    
main()
{
    char c, flag;

    init();				/* Do some initialization */
    flag = FALSE;			/* first time through flag */

    /* Main menu loop. Print menu and process commands */
    while(TRUE)
    {
	signon();
    
	puts("\r\n\nUse one of the following options to configure a device\r\n\n");
	puts("A. Configure LPT device\r\n");
	puts("B. Configure COM device");
	if (numcom == 0) puts(" (Not possible in current configuration)");
	puts("\r\nC. Change automatic partition assignment flag");
	puts("\r\nD. Modify alternate menu");
	puts("\r\n");
	puts("\r\n");

	/* Variable "flag" selects menu for modified or unmodified tables */
	if (flag)
	{
	    puts("  Use one of the following to modify an existing system\r\n\n");
	    puts("E. Exit program\r\n");
	    puts("F. Make changes to disk\r\n");
	    puts("G. Make changes to memory\r\n");
	    puts("H. Make changes to both disk and memory\r\n");
	    puts("\r\nEnter selection (A-H): ");
	}
	else
	{
	    puts("\nE. Exit with no changes\r\n");
	    puts("\r\nEnter selection (A-E): ");
	}

	/* Get a valid menu selection */    
	while(TRUE)
	{
	    if (flag)
	    {
		c = gc('H');
		if (numcom == 0 && c == 'B')
		{
		    puts("\010 \010");
		    beep();
		}
		else break;
	    }
	    else
	    {
		c = gc('E');
		if (numcom == 0 && c == 'B') 
		{
		    puts("\010 \010");
		    beep();
		}
		else break;
	    }
	}
    
	/* Dispatch command */
	switch(c)
	{
	    /* LPT configuration */
	    case 'A':
		lptdev();
		break;
	
	    /* COM configuration */
	    case 'B':
		comdev();
		break;
  
	    /* Auto-assign flag */
	    case 'C':
		auto_assign();
		break;

	    /* Modify menu */
	    case 'D':
		mod_menu();
		break;

	    /* Exit */
	    case 'E':
		exit(0);

	    /* Modify disk */
	    case 'F':
		disk();
		break;

	    /* Modify memory */
	    case 'G':
		memory();
		break;

	    /* Modify disk and memory */
	    case 'H':
		disk();
		memory();
		break;
	}
	flag = TRUE;		/* Tables have been modified */
    }
}    


/*
    signon() - This routine prints the signon message
*/
signon()
{
    float ver;
    
    ver = VERSION + ((float)RELEASE) / 100;
    cls();
    cursor(0,0);

#if ZENITH
    printf("                        CONFIGUR Version %2.2f\r\n", ver);
    puts("              Copyright(C) 1985, Zenith Data Systems Inc.\r\n");
#endif

#if NBI
    printf("                        CONFIGUR Version %2.2f\r\n", ver);
    puts("                     (C)Copyright NBI, Inc. 1984\r\n");
#endif

#if SYNTREX
    printf("                          CONFIGUR Version %2.2f\r\n", ver);
    puts("                     (C)Copyright SYNTREX, Inc. 1986\r\n");
#endif

    return;
}


/*
    init() - This routine does some initialization
*/
init()
{
    unsigned getindex(), getword();

    /* Set a flag if running under DOS 2 */
    dos2 = bdos(GET_VERSION);			/* dos2 = TRUE if version 2 */

    if (dos2) bios_seg = BIOS2_SEG;	/* Segment of 2.0 BIOS */
    else bios_seg = BIOS1_SEG;		/* Segment of 1.25 BIOS */
    /* Get pointer to config. table at start of BIOS */
    config_ptr = getword(bios_seg, CONFIG_PTR);

    a_flag = getbyte(bios_seg,config_ptr+7);

    /* Determine equipment */
    equip();


    /* Make sure versions are OK and get pointer to configuration info. */
    if (getindex() == 0)
    {
	puts("\r\nVersion mismatch with IO.SYS\r\n");
	exit(0);
    }

    if (!get_tbl()) exit(0);		/* Get the table from memory */

    return;
}




/*
    lptdev() - This routine is called when a parallel configuration
    has been selected. It will determine the type of configuration
    desired and process it.
*/
lptdev()
{
    char c;
    
    while(TRUE)
    {
	cls();
	cursor(0, 0);
	puts("  Use one of the following options to select the type of configuration\r\n\n");
	puts("A. Map parallel output to serial output");
	if (numcom == 0) puts(" (Not possible in current configuration)");
	puts("\r\n");
	puts("B. Configure parallel device");
	if (numlpt == 0) puts(" (Not possible in current configuration)");
	puts("\r\n\n");
	puts("C. Exit\r\n\n");

	puts("Enter selection (A-C): ");
    
	while(TRUE)
	{
	    c = gc('C');
	    if ((numcom == 0 && c == 'A') || (numlpt == 0 && c == 'B'))
	    {
		beep();
		puts("\010 \010");	/* Back up over bad character */
	    }
	    else break;
	}

	/* Dispatch the request */
	switch (c)
	{
	    /* Re-map the parallel device to serial */
	    case 'A':
		remap();
		break;
	
	    /* Configure a LPT (parallel) device */
	    case 'B':
		lptconfig();
		break;
	    
	    /* Exit */
	    case 'C':
		return;
	}
    }
}

/*
    comdev() - This routine is called when a serial configuration
	has been selected. It will determine the type of configuration
	desired and process it.
*/
comdev()
{
    char c, i;
    int device;

    /* Print the device selection menu */
    cls();
    cursor(0, 0);
    puts("Select the serial port to be configured\r\n\n");
    for (i = 0; i < numcom; ++i) printf("%c. COM%d\r\n", i+'A', i+1);
    printf("\n%c. Exit\r\n", numcom+'A');
    printf("\nEnter selection (A-%c): ", numcom+'A');

    /* Get the users selection */
    c = gc(numcom+'A');
    if (c == numcom+'A') return;
    device = c - 'A';

    /* Print the serial configuration menu */
    cls();
    cursor(0, 0);
    puts("  Use one of the following options to select the appropriate configuration\r\n\n");
    puts("A. Compatibility mode (2400 baud, DTR and RTS pos.)\r\n");
    puts("B. MX-80  (4800 baud, DTR pos. (pin 20))\r\n");
    puts("C. H/Z-25 (4800 baud, RTS pos. (pin 4))\r\n");
    puts("D. H-14/WH-24 (4800 baud, RTS Neg. (pin 4))\r\n");
    puts("E. Diablo 630/1640 (1200 baud, ETX/ACK)\r\n");
    puts("F. WH-23/WH-33/WH-43 modem (300 baud, No handshake)\r\n");
    puts("G. WH-12 Votrax Type-N-Talk (4800 baud, RTS Pos. (pin 4))\r\n");
    puts("H. User Defined\r\n");
    puts("I. Alternate menu\r\n");
    puts("J. Exit with no changes\r\n\n");
    puts("Enter selection (A-J): ");
    
    /* Get character A-J */
    c = gc(NUM_SCT_DEFAULT+2+'A');
    if (c == NUM_SCT_DEFAULT+2+'A') return;

    /* Check for user defined configuration */
    if (c == NUM_SCT_DEFAULT+'A') user_def(&(config_tables.sct_tbl[device]),device);
    else if (c == NUM_SCT_DEFAULT+1+'A') alt_menu(device);
    else copy_tbl(device, c);		/* Copy user selection */
    
    return;
}


/*
    copy_tbl() - This routine will copy one of the pre-defined
	tables that has been selected by the user into the
	local configuration tables.
*/
copy_tbl(device, c)
int device;
char c;
{
    static struct sct default_tables[NUM_SCT_DEFAULT] =
	{
	/* Compatibility mode */
	sct_default+sct_dtr+sct_dtr_polar+sct_rts+sct_rts_polar,
	0, 0, 0, 0, 0, 0xa3, 1,
	
	/* MX-80 */
	sct_dtr+sct_dtr_polar, 0, 0, 0, 0, 0, 0xc3, 2,
	
	/* H/Z-25 */
	sct_rts+sct_rts_polar, 0, 0, 0, 0, 0, 0xc3, 2,
	
	/* H-14/WH-24 */
	sct_rts, 0, 0, 0, 0, 0, 0xc3, 2,
	
	/* Diablo */
	sct_protocol, 0, 0, 0, 32, 32, 0x83, 2,
	
	/* WH-23/WH-33/WH-43 */
	0, 0, 0, 0, 0, 0, 0x43, 2,
	
	/* WH-12 Votrax */
	sct_rts+sct_rts_polar, 0, 0, 0, 0, 0, 0xc3, 2
    };

    config_tables.sct_tbl[device].sct_handshake = default_tables[c-'A'].sct_handshake;
    config_tables.sct_tbl[device].sct_attr      = default_tables[c-'A'].sct_attr;
    config_tables.sct_tbl[device].sct_padchar   = default_tables[c-'A'].sct_padchar;
    config_tables.sct_tbl[device].sct_numpad    = default_tables[c-'A'].sct_numpad;
    config_tables.sct_tbl[device].sct_flag      = default_tables[c-'A'].sct_flag;
    config_tables.sct_tbl[device].sct_count     = default_tables[c-'A'].sct_count;
    config_tables.sct_tbl[device].sct_init      = default_tables[c-'A'].sct_init;
    config_tables.sct_tbl[device].sct_time_out  = default_tables[c-'A'].sct_time_out;
    return;
}


/*
    user_def() - This routine is for the user defined serial device.
	It prompts for all the information needed to configure a
	serial device.
*/
user_def(c_table,device)
struct sct *c_table;
int device;
{
    char c;
    int num;
    
    cls();
    cursor(0, 0);

    /* Get the attribute information */
    c_table->sct_attr = 0;
    puts("Answer the following questions with Y for Yes and N for No\r\n\n");
    puts("Strip parity on input? (Y/N) <N> ");
    if (yes_no('N')) c_table->sct_attr |= sct_spi;
    
    /* Strip parity? */
    puts("Strip parity on output? (Y/N) <N> ");
    if (yes_no('N')) c_table->sct_attr |= sct_spo;
    
    /* Map input? */
    puts("Map lower case to upper on input? (Y/N) <N> ");
    if (yes_no('N')) c_table->sct_attr |= sct_mli;
    
    /* Map output? */
    puts("Map lower case to upper on output? (Y/N) <N> ");
    if (yes_no('N')) c_table->sct_attr |= sct_mlo;

    /* Get the initialization information */
    c_table->sct_init = 0;
    baud(c_table);    			/* Baud rate info.   */
    stopbit(c_table);			/* stop bit info.    */
    parity(c_table);			/* parity info.      */
    wordlen(c_table);			/* word length info. */
    
    /* Determine handshake info. */
    handshake(c_table);

    cls();
    cursor(0, 0);

    /* Get the pad character and the number of pad characters */
    c_table->sct_padchar = padchar();
    c_table->sct_numpad = padcount();
    
    /*
	Get the user specified time out value for the printer.
	Only useful for hardware handshake.
    */
    if (!(c_table->sct_handshake & sct_protocol) &&
	c_table->sct_handshake != 0)
    {
	time_msg();
	printf("\r\n\nEnter time out value for COM%c:", device+'1');
	if ((num = get_num()) == 0) num = 1;
	c_table->sct_time_out = num;
    }
    return;
}


/*
    wordlen() - This routine will update the wordlength info. for a given
	serial device.
*/
wordlen(c_table)
struct sct *c_table;
{
    char c;

    cls();
    cursor(0, 0);
    puts("Use one of the following to select the word length\r\n");
    puts(" NOTE: Word length is exclusive of stop bits and parity\r\n\n");
    puts("A. 7 bit words\r\n");
    puts("B. 8 bit words\r\n");
    puts("\r\nEnter one of the word length values: ");
    c = gc('B');
    c_table->sct_init |= 2+(c-'A');
    return;
}


/*
    handshake() - This routine will update the handshake information for
    a given serial device
*/
handshake(c_table)
struct sct *c_table;
{
    static char hndshk[8] = {
		0,
		sct_protocol,
		sct_protocol+sct_etx_dc1,
    		sct_rts+sct_rts_polar+sct_dtr+sct_dtr_polar,
		sct_rts+sct_rts_polar,
		sct_rts,
		sct_dtr+sct_dtr_polar,
		sct_dtr
		};

    char c;
    cls();
    cursor(0, 0);
    puts("Use the following to select a handshaking protocol\r\n\n");
    puts("A. No Handshaking\r\n");
    puts("B. ETX/ACK\r\n");
    puts("C. DC1/DC3\r\n");
    puts("D. Compatibility mode, DTR and RTS Positive\r\n");
    puts("E. RTS Positive (pin 4)\r\n");
    puts("F. RTS Negative (pin 4)\r\n");
    puts("G. DTR Positive (pin 20)\r\n");
    puts("H. DTR Negative (pin 20)\r\n");
    puts("\r\nEnter one of the handshake values: ");
    
    c = gc('@'+sizeof(hndshk));
    c_table->sct_handshake = hndshk[c-'A'];
    if (c == 'B')
    {
	puts("\r\nNumber of characters between the ETX/ACK handshake (0-255): ");
	/* Set both the burst count and the initial count */
	c_table->sct_count = get_num();
	c_table->sct_flag  = c_table->sct_count;
    }
    return;
}

/*
    parity() - This routine will get the parity information from the user.
*/
parity(c_table)
struct sct *c_table;
{
    char c, par;
    
    cls();
    cursor(0, 0);
    puts("Use one of the following parity selections\r\n\n");
    puts("A. No parity\r\n");
    puts("B. Odd parity\r\n");
    puts("C. Even parity\r\n");
    puts("\r\nEnter one of the parity values: ");
    c = gc('C');
    switch(c)
    {
	/* No parity */
	case 'A':
	    par = 0;
	    break;
	    
	/* Odd parity */
	case 'B':
	    par = 1;
	    break;
	    
	/* Even parity */
	case 'C':
	    par = 3;
	    break;
    }
    
    /* Update the init parameter in the table */
    c_table->sct_init |= par << 3;
    return;
}


/*
    stopbit() - Get the number of stop bits from the user.
*/
stopbit(c_table)
struct sct *c_table;
{
    char c;
    
    cls();
    cursor(0, 0);
    puts("Use one of the following stop bit values\r\n\n");
    puts("A. 1 Stop bit\r\n");
    puts("B. 2 Stop bits\r\n");
    puts("\r\nEnter one of the stop bit values: ");
    c = gc('B');
    c_table->sct_init |= (c - 'A') << 2;
    return;
}


/*
    baud() - Get the baud rate from the user.
*/
baud(c_table)
struct sct *c_table;
{
    char c;
    
    /* Get initialization parameters */
    cls();
    cursor(0, 0);
    puts("Select one of the following baud rates\r\n\n");
    puts("A. 110\r\n");
    puts("B. 150\r\n");
    puts("C. 300\r\n");
    puts("D. 600\r\n");
    puts("E. 1200\r\n");
    puts("F. 2400\r\n");
    puts("G. 4800\r\n");
    puts("H. 9600\r\n");
    
    puts("\r\nEnter one of the baud rate values: ");
    c = gc('H');
    c_table->sct_init |= (c - 'A') << 5;
    return;
}


/*
    equip() - This routine will set the two globals "numlpt" and
	"numcom" to the number of parallel and the number of
	serial devices respectively. It also determines the number
	of drives for prompting the user.
*/
equip()
{
    z150int(EQUIPMENT_INTR);
    numlpt = (rax & 0xc000) >> 14;	/* isolate num. of parallel devices */
    numcom = (rax & 0x0e00) >> 9;	/* isolate num. of serial devices */
    
    /* Determine the number of block devices */
    max_drive = bdos(SEL_DISK, bdos(CUR_DISK));
    return;
}


/*
    remap() - This routine will prompt the user for the re-map value
	desired.
*/
remap()
{
    char c, i, device;

    cls();
    cursor(0, 0);

    /* Print the parallel selections */
    puts("Select the parallel port to be mapped.\r\n\n");
    for (i = 0; i < 3; ++i) printf("%c. LPT%d\r\n", i+'A', i+1);
    puts("\nD. Exit\r\n");
    puts("\nEnter selection (A-D): ");

    /* Get the parallel device to map */
    c = gc('D');
    if (c == 'D') return; 
    device = c-'A';
    
    cls();
    cursor(0, 0);
    
    /* Get the type of mapping for this parallel device */
    printf("Use one of the following to select a mapping for LPT%d\r\n\n", device+1);
    puts("A. No mapping\r\n");
    for (i = 0; i < numcom; ++i) printf("%c. Map to COM%d\r\n", i+'A'+1, i+1);
    puts("\r\nEnter one of the mapping values: ");
    c = gc(numcom+'A');

    config_table.pct_tbl[device].pct_remap &= 0xf0;	/* Get rid of old info. */
    config_table.pct_tbl[device].pct_remap |= c - 'A';	/* Pack in remap value */

    return;
}


/*
    mod_menu() - This routine allows the user to modify the alternate
		menu.
 */

mod_menu()
{
    char have_file = TRUE;
    char current_dir[65];
    int i = 0;
    char c;
    int menufile;
    struct buffer {
	int count;
	char text[80];
    } buf;

    cls();
    cursor(0,0);
    if (dos2) {
	save_dir(current_dir, bdos(CUR_DISK)+1);
	bdos(CHDIR, "\\");
    }
    menufile = open("ALTMENU.SYS",0x8000);	/* try to open data file */
    if (menufile == -1) {
	puts("No ALTMENU.SYS file.");
	have_file = FALSE;
    } else {
	read(menufile, alt_table, sizeof(alt_table));
	close(menufile);
	puts("The current menu is\r\n\r\n");
	for (i=0; i<NUM_ALT; i++) {
	    if (alt_table[i].name[0] == 0)
		break;
	    printf("%c. ",'A'+i);
	    printf("%s\r\n",alt_table[i].name);
	}
    }
    if (dos2) bdos(CHDIR, current_dir);
    puts("\r\n\r\nUse one of the following options to modify the menu\r\n");
    puts("\r\nA. Exit with no changes");
    puts("\r\nB. Create a new menu item");
    if (i >= NUM_ALT)			/* menu full */
	puts("   (Not possible, menu full)");
    if (have_file) {
	puts("\r\nC. Modify existing menu item");
	puts("\r\n\r\nEnter selection (A-C) ");
	c = gc('C');
    } else {
	puts("\r\n\r\nEnter selection (A-B) ");
	c = gc('B');
    }
    switch (c) {

	case 'A':
	    return;

	case 'B':
	    if (i >= NUM_ALT) {
		puts("\r\n\r\nMenu is full.\r\n\r\nPress RETURN to continue...");
		getchar();
		return;
	    }
	    cls();
	    cursor(0,0);
	    puts("Enter name of printer (60 characters or less)\r\n\r\n> ");
	    buf.count = 60;
	    gets(&buf);
	    strcpy(alt_table[i].name,buf.text);
	    user_def(&alt_table[i].parms,0);
	    break;

	case 'C':
	    printf("\r\n\r\nWhich item would you like to modify (A-%c) ",i-1+'A');
	    c = gc(i-1+'A');
	    cls();
	    cursor(0,0);
	    puts("Enter name of printer (60 characters or less)\r\n\r\n> ");
	    buf.count = 60;
	    gets(&buf);
	    strcpy(alt_table[c-'A'].name,buf.text);
	    user_def(&alt_table[c-'A'].parms,0);
	    break;
    }

    cls();
    cursor(0,0);
    puts("Menu item complete.\r\n\r\n");
    puts("Make changes to disk? (Y/N) <Y> ");
    if (yes_no('Y')) {
	if (dos2) {
	    save_dir(current_dir, bdos(CUR_DISK)+1);
	    bdos(CHDIR, "\\");
	}
	menufile = creat("ALTMENU.SYS",0x8000);
	if (menufile == -1) {
	    puts("\r\nError writing ALTMENU.SYS\r\n\r\nPress RETURN to continue...");
	    getchar();
	} else {
	    if (write(menufile, &alt_table, sizeof(alt_table)) != sizeof(alt_table)) {
		puts("\r\nError writing ALTMENU.SYS\r\n\r\nPress RETURN to continue...");
		getchar();
	    }
	    close(menufile);
	}
	if (dos2) bdos(CHDIR, current_dir);
    }

}


/*
    alt_menu(device) - This routine displays the alternate user definable
			menu, and allows a selection to be made from it.
 */

alt_menu(device)
int device;
{
    int menufile;
    int i;
    char c;
    char current_dir[65];

    cls();
    cursor(0,0);
    if (dos2) {
	save_dir(current_dir, bdos(CUR_DISK)+1);	/* save current directory */
	bdos(CHDIR,"\\");		/* change to root */
    }
    menufile = open("ALTMENU.SYS",0x8000);	/* try to open data file */
    if (menufile == -1) {
	puts("No ALTMENU.SYS file.");
	puts("\r\n\r\nPress RETURN to continue...");
	getchar();
	if (dos2) bdos(CHDIR, current_dir);
	return;
    }
    read(menufile, alt_table, sizeof(alt_table));
    close(menufile);
    if (dos2)
	bdos(CHDIR, current_dir);	/* restore to original directory */

    puts("  Use one of the following options to select the appropriate configuration");
    puts("\r\n\r\n");
    for (i=0; i<NUM_ALT; i++) {
	if (alt_table[i].name[0] == 0)
	    break;
	printf("%c. ",'A'+i);
	printf("%s\r\n",alt_table[i].name);
    }
    printf("%c. Exit with no changes\r\n",'A'+i);
    printf("\r\nEnter selection (A-%c) ",'A'+i);
    c = gc('A'+i);
    if (c == 'A'+i)
	return;

    config_tables.sct_tbl[device].sct_handshake = alt_table[c-'A'].parms.sct_handshake;
    config_tables.sct_tbl[device].sct_attr      = alt_table[c-'A'].parms.sct_attr;
    config_tables.sct_tbl[device].sct_padchar   = alt_table[c-'A'].parms.sct_padchar;
    config_tables.sct_tbl[device].sct_numpad    = alt_table[c-'A'].parms.sct_numpad;
    config_tables.sct_tbl[device].sct_flag      = alt_table[c-'A'].parms.sct_flag;
    config_tables.sct_tbl[device].sct_count     = alt_table[c-'A'].parms.sct_count;
    config_tables.sct_tbl[device].sct_init      = alt_table[c-'A'].parms.sct_init;
    config_tables.sct_tbl[device].sct_time_out  = alt_table[c-'A'].parms.sct_time_out;

}


/*
    lptconfig() - This routine will prompt for the parallel
	configuration information.
*/
lptconfig()
{
    char i, c, device;
    int count, temp, n, num;
    
    cls();
    cursor(0, 0);

    /* Get the device selection */
    puts("Select the parallel device to be configured\r\n\n");
    for (i = 0; i < numlpt; ++i) printf("%c. LPT%d\r\n", i+'A', i+1);
    printf("\n%c. Exit\r\n", numlpt+'A');
    printf("\nEnter selection (A-%c): ", numlpt+'A');

    c = gc(numlpt+'A');
    if (c == numlpt+'A') return;
    device = c - 'A';
    config_tables.pct_tbl[device].pct_remap = 0;	/* clear attributes  and mapping */
    
    cls();
    cursor(0, 0);
    
    /* Check for strip parity on output */
    puts("Answer the following questions with Y for Yes and N for No\r\n\n");
    puts("Strip parity on output? (Y/N) <N> ");

    if (yes_no('N')) config_tables.pct_tbl[device].pct_remap |= pct_spo;
    
    /* Check for map lower to upper on output */
    puts("\r\nMap lower case to upper on output? (Y/N) <N> ");

    if (yes_no('N')) config_tables.pct_tbl[device].pct_remap |= pct_mlo;

    cls();
    cursor(0, 0);
    
    config_tables.pct_tbl[device].pct_padchar = padchar();
    
    config_tables.pct_tbl[device].pct_numpad = (char) padcount();
    
    time_msg();
    printf("\r\n\nEnter time out value for LPT%c:", device+'1');
    if ((num = get_num()) == 0) num = 1;
    config_tables.pct_tbl[device].pct_time_out = num;
    
    return;
}


/*
    auto_assign() - This routine will prompt the user for a value
	for the auto-assign flag.
*/
auto_assign()
{
    char c;

    cls();			/* clear the screen */
    cursor(0,0);
    puts("\r\n\r\nThe system is currently set for ");
    if (a_flag)
	puts("automatic partition assignment.\r\n");
    else
	puts("manual partition assignment.\r\n");
    puts("\r\n");
    puts("\r\nUse one of the following to select assignment mode.\r\n\r\n");
    puts("A. Automatic partition assignment.\r\n");
    puts("B. Manual partition assignment.\r\n");
    puts("\r\nC. No change, exit to main menu.\r\n\r\n");
    puts("Enter selection (A-C): ");
    c = gc('C');
    switch (c) {
	case 'A':
	    a_flag = TRUE;
	    break;
	case 'B':
	    a_flag = FALSE;
	    break;
	case 'C':
	    break;
    }
}



/*
    gc() - This routine will get a character from the user
	in a specified range.
*/
char gc(limit)
char limit;
{
    char c;
    
    while(TRUE)
    {
	c = getchar();
	if (islower(c)) c = toupper(c);
	if (c < 'A' || c > limit) beep();
	else
	{
	    putchar(c);
	    return(c);
	}
    }
}

/*
    yes_no() - This routine will return TRUE if the user types a 'Y'
	or FALSE if the user types a 'N'.
*/
yes_no(c)
char c;
{
    char c1;
    
    while(TRUE)
    {
	c1 = getchar();
	if (islower(c1)) c1 = toupper(c1);
	if (c1 == '\r') c1 = c;
	if (c1 != 'N' && c1 != 'Y') beep();
	else
	{
	    putchar(c1);
	    puts("\r\n");
	    return(c1 == 'Y');
	}
    }
}


/*
    padchar() - This routine will return the users pad character.
*/
padchar()
{
        
    /* Get pad char and count */
    puts("\r\n\nIf you do not want a pad character, simply\r\n");
    puts(" press the RETURN key, and then enter a zero\r\n");
    puts(" as the number of pad characters, otherwise type\r\n");
    puts(" the actual key character you want to pad\r\n");
    puts("\nFor example, to pad after all carriage returns, type\r\n");
    puts(" the RETURN key.\r\n");
    puts("\nType the key corresponding to your desired pad character: ");
    return(getchar());
}


/*
    padcount() - This routine will return the count of desired pad
	characters.
*/
padcount()
{
    puts("\r\nEnter the number of pad characters to send (0-255): ");
    return(get_num());
}


/*
    get_num() - This routine will prompt the user for a decimal number
	between 0 and 255. If a null is input, 0 is returned.
*/
get_num()
{
    int count, temp;
    char c;

    count = 0;    
    while(TRUE)
    {
	c = getchar();
	if (c == '\r') return(count);
	if (c < '0' || c > '9') beep();
	else
	{
	    temp = 10 * count + c - '0';
	    if (temp > 255) beep();
	    else
	    {
		count = temp;
		putchar(c);
	    }
	}
    }
}

/*
    memory() - This routine will modify the memory configuration tables
    in the BIOS
*/
memory()
{
    unsigned getindex(), loc;
    int i;

    /* Make sure versions are OK and get pointer to configuration info. */
    if ((loc = getindex()) == 0) return;

    /* Modify memory */
     movbyte(mydsreg(), &config_tables, bios_seg, loc, sizeof(struct io_table));
    
    /* Initialize the parallel devices */
    for (i = 0; i < numlpt; ++i)
    {
	rah = PIO_INIT;
	rdx = i;
	z150int(PRINTER_IO_INTR);
	putbyte(ROM_DATA, PRINT_TIME_OUT+i, config_tables.pct_tbl[i].pct_time_out);
    }
    
    /* Re-program the serial devices */
    for (i = 0; i < numcom; ++i)
    {
	rah = SIO_INIT;
	ral = config_tables.sct_tbl[i].sct_init;
	rdx = i;
	z150int(SERIAL_IO_INTR);
	putbyte(ROM_DATA, SERIAL_TIME_OUT+i, config_tables.sct_tbl[i].sct_time_out);
    }

    return;
}


/*
    get_tbl() - This routine will get the configuration tables from
	the BIOS and return them into local storage. TRUE is returned
	if the BIOS version is OK else FALSE is returned.
*/
get_tbl()
{
    unsigned getindex(), loc;

    if ((loc = getindex()) == 0) return(FALSE);

    movbyte(bios_seg, loc, mydsreg(), &config_tables, sizeof(struct io_table));
    return(TRUE);
}


/*
    getindex() - This routine will check if the version of the BIOS
	is correct and return a pointer to the configuration tables
	if so, else 0 is returned.
*/
unsigned getindex()
{
    unsigned getword();
    char getbyte(), ver;
    struct config1_vector *p1;
    struct config2_vector *p2;
    
    if (dos2)
    {
	p2 = (struct config2_vector *)config_ptr;
	ver = getbyte(bios_seg, &p2->bios_version);
	if (ver < 20 || BIOS2_CVER < ver) return(0);	/* 20 to VER for 2.0 */
	return(getword(bios_seg, &p2->io_address));
    }
    else
    {
	p1 = (struct config1_vector *) config_ptr;
	if (BIOS1_CVER < getbyte(bios_seg, &p1->bios_version)) return(0);
	return(getword(bios_seg, &p1->io_address));
    }
}


/*
    beep() - This routine will beep the keyboard to signal
	an erroneous input.
*/
beep()
{
#define BELL '\007'
    putchar(BELL);
    return;
}


/*
    disk() - This routine will modify the configuration tables in
	the file IO.SYS. It first checks version numbers to ensure
	proper operation.
*/
disk()
{
    char c, drv, last_drv, current_dir[64];

    printf("\r\n\nEnter drive name with system to modify (A-%c): ", '@'+max_drive);
    c = gc('@'+max_drive);
    
    drv = c-'@';

    /* Save the last drive and log in the specified drive */
    last_drv = bdos(CUR_DISK);
    bdos(SEL_DISK, drv-1);
    
    /* If DOS 2 save the current directory and change to the root */
    if (dos2)
    {
        save_dir(&current_dir[0], drv);
	bdos(CHDIR, "\\");
    }

    /* Try to modify the disk */
    if (!do_disk(drv)) badfile();
    
    /* Restore the directory and restore the logged drive */
    if (dos2) bdos(CHDIR, &current_dir[0]); /* Restore to old directory */
    bdos(SEL_DISK, last_drv);
    return;
}

/*
    do_disk() - This routine will do the actual modificatio of the
	disk configuration parameters.
*/
do_disk(drv)
char drv;
{
    unsigned loc;
    struct ext_fcb bios;
    struct config2_vector *cv2;
    struct config1_vector *cv1;
    static char fname[] = {"IO      SYS"};
    char *bios_name = "X:\\IO.SYS";
    char attrib;
    char vers, verflag, *p;
    int i;

    /* Prepare an extended FCB */
    bios.extend = 0xff;
    for (i = 0; i < 5; ++i) bios.zero[i] = 0;
    if (dos2) bios.attr = 0x20;
    else bios.attr = 0;
    bios.drive = drv;
    p = (char *)&bios.name;
    for (i = 0; i < 11; ++i) *p++ = fname[i];

    /* turn off read-only bit */

    if (dos2) {
	*bios_name = drv + 'A' - 1;
        attrib = zchmd(bios_name,0,);	/* get file attribute */
        zchmd(bios_name,1,attrib & 0xfe);	/* turn off read-only bit */
    }

    /* Attempt to open a normal file first */
    if (bdos(FILE_OPEN, &bios) == 0xff)
    {
	/* If failed, turn on hidden and system bits */
	if (dos2) bios.attr = 0x26;
	else bios.attr = 6;
	
	if (bdos(FILE_OPEN, &bios) == 0xff) return(FALSE);
    }

    bios.rec_size = 1;
    
    /* Get the version number */
    if (dos2)
    {
	cv2 = (struct config2_vector *) config_ptr;
	bios.ranrec = (long)&cv2->bios_version;
    }
    else
    {
	cv1 = (struct config1_vector *) config_ptr;
	bios.ranrec = (long)&cv1->bios_version;
    }
    
    if (rread(&bios, &vers, 1) != 1) return(FALSE);

    /* For 2.0 version must be 20 <= ver <= BIOS2_CVER */
    verflag = FALSE;		/* Assume version OK */
    if (dos2)
    {
	if (vers < 20 || vers > BIOS2_CVER) verflag = TRUE;
    }
    else
    {
	if (vers > BIOS1_CVER) verflag = TRUE;
    }
    
    if (verflag)
    {
	puts("\r\n\nIO.SYS file version incorrect\r\n");
	puts("Strike any key to continue...");
	getchar();
	return(TRUE);	/* Error done here, pretend none at higher level */
    }

   if (dos2) {
	bios.ranrec = (long)&cv2->auto_flag;
	if(rwrite(&bios, &a_flag, 1) != 1)
	    return(FALSE);
    }

    if (dos2) bios.ranrec = (long) &cv2->io_address;
    else bios.ranrec = (long) &cv1->io_address;
    if (rread(&bios, &loc, 2) != 2) return(FALSE);

    bios.ranrec = (long) loc;
    if (rwrite(&bios, &config_tables, sizeof(struct io_table))
	!= sizeof(struct io_table)) return(FALSE);
	
    bdos(FILE_CLOSE, &bios);
    if (dos2)
	zchmd(bios_name,1,attrib);	/* restore old attribs */
    return(TRUE);
}




/*
    badfile() - Report an error in modifying IO.SYS to the user.
*/
badfile()
{
    puts("\r\n\nCan not locate file IO.SYS\r\n");
    puts("Strike any key to continue...");
    getchar();
    return;
}

/*
    time_msg() - This routine will explain the purpose of the time
	out value.
*/
time_msg()
{
    puts("\r\n\nThe time out value is used to give slow devices time to respond\r\n");
    puts(" to Input/Output requests. A small value is usually sufficient,\r\n");
    puts(" but a number 0 to 255 can be entered.\r\n");
    return;
}

