/* sysLib.c - Motorola MVME-135 system dependent library */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which UniWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "vme.h"
#include "memLib.h"
#include "sysLib.h"
#include "ioLib.h"
#include "config.h"
#include "mv135.h"
#include "iv68k.h"

IMPORT char end;	/* end of system, created automatically by ld */

IMPORT VOID logMsg ();


#define	HZ	2048000		/* clock rate */

/* globals */

int sysBus = BUS;			/* system bus type (VME_BUS, etc) */
int sysCpu = CPU;			/* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int sysProcNum;				/* processor number of this cpu */
int sysFlags;				/* boot flags */
char sysBootHost [BOOT_FIELD_LEN];	/* name of host from which we booted */
char sysBootFile [BOOT_FIELD_LEN];	/* name of file from which we booted */
int sysBusAdrs = 0;			/* our DRAM bus address */

/* locals */

LOCAL int	sysClkTicksPerSecond = 60;
LOCAL int	auxClkTicksPerSecond = 60;
LOCAL int	sysClkRunning	     = FALSE;
LOCAL int	auxClkRunning	     = FALSE;
LOCAL BOOL	sysClkIsConnected    = FALSE;
LOCAL FUNCPTR 	sysClkRoutine        = NULL;
LOCAL int 	sysClkArg 	     = NULL;
LOCAL FUNCPTR 	sysAuxClkRoutine     = NULL;
LOCAL int 	sysAuxClkArg 	     = NULL;
LOCAL BOOL	sysMailboxIsConnected= FALSE;
LOCAL FUNCPTR 	sysMailboxRoutine    = logMsg;
LOCAL int	sysMailboxArg	     = (int) "mailbox interrupt\n";

/* forward declarations */

LOCAL VOID sysClkInt ();		/* clock handler */
LOCAL VOID sysMailboxInt ();		/* mailbox handler */
LOCAL VOID sysAbort ();			/* abort button handler */


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "Motorola MVME-135"
*/

char *sysModel ()

    {
    return ("Motorola MVME-135");
    }
/*******************************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board.
* It is normally called from usrInit (2) in usrConfig (1).
* This routine must be called in supervisor mode.
*
* The timers are initialized and turned off.
* The CIO is setup.
* The bus address is determined.
*/

VOID sysHwInit ()

    {
    char zero = 0;

    /* board level initialization */

    if (*STAT2_ADRS);			/* clear status bits by reading */

    *CNT4_ADRS	= CNT4_GLBRES;		/* dont reset board */
    *CNT1_ADRS	= CNT1_WWP;		/* turn off parity interrupts */
    *CNT2_ADRS	= ((char) 0xff);	/* no vme ints; allow vme memory
					   access  */
    *CNT3_ADRS	= CNT3_SYSFIEN + CNT3_SHPIEN + CNT3_SLPIEN +
       		  CNT3_LMIEN + CNT3_BRIQO + CNT3_VSBIRQO;
    *CNT5_ADRS	= CNT5_WD0 + CNT5_WD1 + CNT5_LTMO + CNT5_VTMO +
		  CNT5_OPT0 + CNT5_32_16;

    /* duart initialization */

    *DUART_IMR  = 0;			/* mask serial interrupts */
    *DUART_IVR	= SIO_INT_VEC;		/* set interrupt vector */
    *DUART_ACR  = BRG_SELECT | TMR_EXT_CLK; /* brg #2, xtal clock */

    /* CIO initialization */

    /* CIO interrupt configuration */
    *Z8036_VEC_TIM(CIO_ADRS) 	= CIO_INT_VEC;

    *Z8036_A_DIR(CIO_ADRS) = 0xff;	/* set port A as an input */

    /* timer 1 initialize */
    *Z8036_TIM1_MODE(CIO_ADRS) 	 = Z8036_TIM_MODE_CONT;
    *Z8036_TIM1_TC_MSB(CIO_ADRS) = 0xa0;
    *Z8036_TIM1_TC_LSB(CIO_ADRS) = zero;

    /* timer 2 initialize */
    *Z8036_TIM2_MODE(CIO_ADRS)   = Z8036_TIM_MODE_CONT;
    *Z8036_TIM2_TC_MSB(CIO_ADRS) = 0xa0;
    *Z8036_TIM2_TC_LSB(CIO_ADRS) = zero;

    /* CIO interrupt enable */

    *Z8036_A_CONT(CIO_ADRS) 	= 0x20;	    /* clear port a interrupts */
    *Z8036_A_CONT(CIO_ADRS) 	= Z8036_CONT_INT_DIS;

    *Z8036_B_CONT(CIO_ADRS) 	= 0x20;	    /* clear port b interrupts */
    *Z8036_B_CONT(CIO_ADRS) 	= Z8036_CONT_INT_DIS;

    *Z8036_TIM1_CONT(CIO_ADRS)	= Z8036_CONT_CLR_IP_IUS;
    *Z8036_TIM1_CONT(CIO_ADRS)	= Z8036_CONT_INT_DIS;
    
    *Z8036_TIM2_CONT(CIO_ADRS)	= Z8036_CONT_CLR_IP_IUS;
    *Z8036_TIM2_CONT(CIO_ADRS)	= Z8036_CONT_INT_DIS;

    *Z8036_CONF(CIO_ADRS) 	= 0;
    
    *Z8036_INT(CIO_ADRS)	= Z8036_INT_MIE | Z8036_INT_TIM_INC_STAT |
				  Z8036_INT_RIGHT_JUST;

    /* compute our bus address once, then store it for optimization */

    sysBusAdrs = (*MPCSR_ID(MPCSR_ADRS) & 0x1f) << 20;
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
*
* It starts probing at the end of bss, then tries to read a byte
* at every 4K boundary until it finds one that can't be read.
* This routine must be called in supervisor mode, the first time only.
* After that, it can be called in user mode as well.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    static char *memTop;
    char bitBucket;

#define	PAGE_SIZE	0x1000

    if (memTop == NULL)
	{
	/* Look for the top of memory starting at the first even page
	 * boundary after _end. */

	memTop = (char *) ((((int) &end) + (PAGE_SIZE - 1)) &
			    (~ (PAGE_SIZE - 1)));

	/* search a possible 4 Mbytes */

	for (; (memTop < (char *) 0x400000); memTop += PAGE_SIZE)
	    {
	    if (vxMemProbe (memTop, READ, 1, &bitBucket) != OK)
		break;
	    }

	/* memTop is now pointing to the first non-existent address */
	}

    return (memTop);
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-x, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility could be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor)
*
* INTERNAL
* Note that the "WARM" restart address is at (ROM_BASE_ADRS + 16) bytes.
*/

STATUS sysToMonitor (startType)
    int startType;  /* parameter is passed to ROM to tell it how to boot */
                    /* possible types are defined in sysLib.h            */
    {
    (* ((FUNCPTR) (ROM_BASE_ADRS + 16))) (startType);

    return (OK);	/* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* NOTE:
* The abort switch is also setup at this time.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each system clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    /* set-up interupt vector */

    if (!sysClkIsConnected &&
	(intConnect (INUM_TO_IVEC (CIO_INT_VEC), sysClkInt, 0)    == ERROR ||
	intConnect (INUM_TO_IVEC (ABORT_INT_VEC), sysAbort, NULL) == ERROR))
	{
	return (ERROR);
	}
    
    sysClkIsConnected = TRUE;

    sysClkRoutine     = routine;
    sysClkArg         = arg;

    return (OK);
    }
/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/

VOID sysClkDisable ()

    {
    /* stop interrupts */

    *Z8036_CONF (CIO_ADRS) &= (~Z8036_CONF_1_ENA);

    /* clear interrupt enable mask */

    *Z8036_TIM1_CONT(CIO_ADRS)	= Z8036_CONT_INT_DIS;

    sysClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysClkEnable - turn system clock interrupts on
*/

VOID sysClkEnable ()

    {
    unsigned int tc;		/* time constant */

    /* Initialize timer A */
    /* the timer chip has been prepared in sysHwInit */

    tc = HZ / sysClkTicksPerSecond;

    /* ensure reasonable rates */

    if (tc < 1 || tc > 65535)		/* 16 bits */
	{
	printErr ("sysClkEnable: %d out of range.\n", sysClkTicksPerSecond);
	return;
	}

    /* disable interrupts */
    *Z8036_TIM1_CONT(CIO_ADRS) = Z8036_CONT_INT_DIS;

    /* set time constant */
    *Z8036_TIM1_TC_MSB(CIO_ADRS) = (char)((tc & 0xff00) >> 8);
    *Z8036_TIM1_TC_LSB(CIO_ADRS) = (char)(tc & 0x00ff);

    /* enable interrupts */
    *Z8036_CONF(CIO_ADRS) 	|= Z8036_CONF_1_ENA;

    /* Start timer */
    *Z8036_TIM1_CONT(CIO_ADRS)  = Z8036_CONT_INT_EN | Z8036_TIM_CONT_GATE |
				  Z8036_TIM_CONT_TRIG;

    sysClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysClkInt - clock interrupt handler
*/

LOCAL VOID sysClkInt ()

    {
    char mask;    

    /* see which clock interupted */

    mask = *Z8036_VEC_TIM(CIO_ADRS) & 0x06;

    if (mask & 0x04)			/* main system clock */
	{
	/* invoke user system clock routine */

	if (sysClkRoutine != NULL)
	    (*(FUNCPTR) sysClkRoutine) (sysClkArg);	

	/* clear pending interupt and under service flags */
	*Z8036_TIM1_CONT(CIO_ADRS) = Z8036_CONT_CLR_IP_IUS;	

	/* restart timer */
	*Z8036_TIM1_CONT(CIO_ADRS) = Z8036_CONT_INT_EN | Z8036_TIM_CONT_GATE |
				  Z8036_TIM_CONT_TRIG;
	}
    else
	{
	/* invoke auxiliary clock routine */

	if (sysAuxClkRoutine != NULL)
	    (*(FUNCPTR) sysAuxClkRoutine) (sysAuxClkArg); 

	/* clear pending interupt and under service flags */
	*Z8036_TIM2_CONT(CIO_ADRS) = Z8036_CONT_CLR_IP_IUS;	

	/* restart timer */
	*Z8036_TIM2_CONT(CIO_ADRS) = Z8036_CONT_INT_EN | Z8036_TIM_CONT_GATE |
				  Z8036_TIM_CONT_TRIG;
	}
    }
/*******************************************************************************
*
* sysClkGetRate - get rate of system clock
*
* Finds out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkSetRate (2), sysClkEnable (2)
*/

int sysClkGetRate ()

    {
    return (sysClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkSetRate - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* System clock interrupts are not enabled.
* It is normally called in usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkGetRate (2), sysClkEnable (2)
*/

VOID sysClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	sysClkTicksPerSecond = ticksPerSecond;

    if (sysClkRunning)
	{
	sysClkDisable ();
	sysClkEnable ();
	}
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    /* set-up interupt vector */

    if (!sysClkIsConnected &&
	intConnect (INUM_TO_IVEC (CIO_INT_VEC), sysClkInt, 0) == ERROR)
	{
	return (ERROR);
	}

    sysClkIsConnected = TRUE;

    sysAuxClkRoutine  = routine;
    sysAuxClkArg      = arg;

    return (OK);
    }
/**********************************************************************
*
* sysAuxClkDisconnect - disconnect a routine from the auxiliary clock interrupt
*
* This routine disables the auxiliary clock interrupt and disconnects
* the routine currently connected to the auxiliary clock interrupt.
*/

VOID sysAuxClkDisconnect ()

    {
    /* disable the auxiliary clock interrupt */

    sysAuxClkDisable ();

    /* connect the dummy routine, just in case */

    sysAuxClkConnect (logMsg, (int)"auxiliary clock interrupt\n");
    }
/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/

VOID sysAuxClkDisable ()

    {
    /* stop timer */
    *Z8036_CONF(CIO_ADRS)      &= ~(Z8036_CONF_2_ENA);

    /* clear interrupt mask */
    *Z8036_TIM2_CONT(CIO_ADRS)	= Z8036_CONT_INT_DIS;

    auxClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysAuxClkEnable - turn auxiliary clock interrupts on
*/

VOID sysAuxClkEnable ()

    {
    unsigned int tc;		/* time constant */

    /* Initialize timer B */
    /* the timer chip has been prepared in sysHwInit */

    tc = HZ / auxClkTicksPerSecond;

    /* ensure reasonable rate */

    if (tc < 1 || tc > 65535)	/* 16 bits */
	{
	printErr ("sysAuxClkEnable: %d out of range.\n", auxClkTicksPerSecond);
	return;
	}

    /* disable interrupts */
    *Z8036_TIM2_CONT(CIO_ADRS) = Z8036_CONT_INT_DIS;

    /* set time constant */
    *Z8036_TIM2_TC_MSB(CIO_ADRS) = (char)((tc & 0xff00) >> 8);
    *Z8036_TIM2_TC_LSB(CIO_ADRS) = (char)(tc & 0x00ff);

    /* enable timer B interrupts */
    *Z8036_CONF(CIO_ADRS)      |= Z8036_CONF_2_ENA;

    /* Start timer B */
    *Z8036_TIM2_CONT(CIO_ADRS) 	= Z8036_CONT_INT_EN | Z8036_TIM_CONT_GATE |
				  Z8036_TIM_CONT_TRIG;

    auxClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysAuxClkGetRate - get rate of auxiliary clock
*
* Finds out the auxiliary clock speed.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkSetRate (2)
*/

int sysAuxClkGetRate ()
    
    {
    return (auxClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysAuxClkSetRate - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
*
* NOTE MVME135:
* The auxiliary clock is of higher priority than the system clock.
*
* SEE ALSO: sysAuxClkConnect (2), sysAuxClkGetRate (2)
*/

VOID sysAuxClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	auxClkTicksPerSecond = ticksPerSecond;
	
    if (auxClkRunning)
	{
	sysAuxClkDisable ();
	sysAuxClkEnable ();
	}
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the bus
*
* SEE ALSO: sysBusToLocalAdrs (2)
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,     */
                        /* use address modifier codes as defined in vme.h, */
                        /* such as VME_AM_STD_SUP_DATA                     */
    char *localAdrs;    /* local address to convert                        */
    char **pBusAdrs;    /* where to return bus address                     */

    {
    if ((localAdrs < LOCAL_MEM_LOCAL_ADRS) ||
	(localAdrs >= (char *) (sysMemTop ())))
	{
	/* this is off-board memory - just return local address */

	*pBusAdrs = localAdrs;
	}
    else
	{
	/* this is on-board memory - map to bus address space;
	 *   - the memory is placed in STD space at address LOCAL_MEM_BUS_ADRS.
	 */

	if (((adrsSpace != VME_AM_STD_SUP_PGM) &&
	     (adrsSpace != VME_AM_STD_SUP_DATA) &&
	     (adrsSpace != VME_AM_STD_USR_PGM) &&
	     (adrsSpace != VME_AM_STD_USR_DATA)))
	    return (ERROR);

	*pBusAdrs = (char *) ((int) localAdrs - (int) LOCAL_MEM_LOCAL_ADRS
			      + (int) LOCAL_MEM_BUS_ADRS);
	}

    return (OK);
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,     */
                        /* use address modifier codes as defined in vme.h, */
                        /* such as VME_AM_STD_SUP_DATA                     */
    char *busAdrs;      /* bus address to convert                          */
    char **pLocalAdrs;  /* where to return local address                   */

    {
    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pLocalAdrs = (char *) (0xffff0000 | (int) busAdrs);
	    return (OK);

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    if ((busAdrs < (char *) LOCAL_MEM_BUS_ADRS) ||
		(busAdrs >= (char *) (sysMemTop () - LOCAL_MEM_BUS_ADRS)))
		/* this is off-board memory - just return bus address */
		*pLocalAdrs = busAdrs;
	    else
		*pLocalAdrs = (char *)
		              (((int) busAdrs - (int) LOCAL_MEM_BUS_ADRS)
				        + (int) LOCAL_MEM_LOCAL_ADRS);
	    return (OK);

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* sysIntDisable - disable interrupt level
*
* This routine disables the specified interrupt level.
*
* RETURNS: OK or ERROR if intLevel not in range 1 - 7
*
* ARGSUSED
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    if ((intLevel < 1) || (intLevel > 7))
	return (ERROR);

    /* disable means turn the bit on */

    *CNT2_ADRS |= (1 << intLevel);

    return (OK);
    }
/*******************************************************************************
*
* sysIntEnable - enable interrupt level
*
* This routine enables the specified VME interrupt level.
*
* RETURNS: OK or ERROR if intLevel not in range 1 - 7
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    char stash;

    if (intLevel < 1 || intLevel > 7)
	return (ERROR);

    /* enable means turn the bit off in the mask */

    stash = *CNT2_ADRS;
    stash &= (char) (0xff ^ (1 << intLevel));

    *CNT2_ADRS = stash;

    return (OK);
    }
/*******************************************************************************
*
* sysIntAck - acknowledge VME bus interrupt
*
* This routine acknowledges the specified interrupt.
*
* NOTE MVME135:
* If the interrupt level is enabled, it gets acknowledged automatically.
* This routine has no effect.
*
* RETURNS: NULL
*
* ARGSUSED
*/

int sysIntAck (intLevel)
    int intLevel;	/* interrupt level to acknowledge */

    {
    return (NULL);
    }
/*******************************************************************************
*
* sysIntGen - generate interrupt
*
* This routine generates a VME bus interrupt.
*
* NOTE MVME135:
* This board is unable to generate a VME bus interrupt.
* This routine has no effect.
*
* RETURNS: ERROR unable to generate a VME bus interrupt
*
* ARGSUSED
*/

STATUS sysIntGen (level, vector)
    int level;
    int vector;

    {
    return (ERROR);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
*
* NOTE MVME135:
* Uses the SIGHP high priority mailbox interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2)
*/

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each mailbox interrupt */
    int arg;		/* argument with which to call routine */

    {
    if (!sysMailboxIsConnected &&
       intConnect (INUM_TO_IVEC (SIGHP_INT_VEC), sysMailboxInt, 0) == ERROR)
	{
	return (ERROR);
	}

    sysMailboxIsConnected = TRUE;

    sysMailboxRoutine     = routine;
    sysMailboxArg         = arg;

    return (OK);
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interupt
*
* This routine enables the mailbox interrupt.
*
* NOTE MVME135:
* The SIGHP high priority mailbox interrupt is used.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;	/* address of mailbox, unused as there
			/* is only one location */

    {
    *CNT3_ADRS &= ~CNT3_SHPIEN;		/* turn the bit off to enable */
    return (OK);
    }
/*******************************************************************************
*
* sysMailboxInt - mailbox interrupt handling
*
* This routine is a wrapper fo mailbox interrupt service routines
* connected in with sysMailboxConnect.  It acknowledges the mailbox
* interrupt, and then calls the previously supplied routine.
*/

LOCAL VOID sysMailboxInt ()

    {
    /* acknowledge the interrupt */

    *MPCSR_SIGHP(MPCSR_ADRS) = (char) 0x00;

    if (sysMailboxRoutine != NULL)
	(*sysMailboxRoutine) (sysMailboxArg); 
    }
/*******************************************************************************
*
* sysGetProcNum - get processor number
*
* This routine returns the processor number previously set with 
* sysSetProcNum (2).
*
* RETURNS: processor number
*/

int sysGetProcNum ()

    {
    return (sysProcNum);
    }
/*******************************************************************************
*
* sysSetProcNum - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*/

VOID sysSetProcNum (procNum)
    int procNum;		/* processor number */

    {
    sysProcNum = procNum;
    }
/*******************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* RETURNS: TRUE (successful set) or FALSE (failure)
*/

BOOL sysBusTas (addr)
    char *addr;		/* address to be tested and set */

    {
    return (vxTas (addr));
    }

/* miscellaneous support routines */

/*******************************************************************************
*
* sysDIP - read front panel dip switches
*
* The dip switches are user definable.
*
* RETURNS: 8 bits of data
*/

int sysDIP ()
    {
    return ((*STAT1_ADRS) & 0xff);
    }
/*******************************************************************************
*
* sysAbort - handle abort switch
*
* When the ABORT switch is hit UniWorks will jump to the monitor.
*/

LOCAL VOID sysAbort ()

    {
    while ((*STAT2_ADRS) & 0x20)
	;/* wait for button to be released */

    sysToMonitor (BOOT_WARM_NO_AUTOBOOT);
    }
