/* sysLib.c - MVME-133/133A/133XT/134 system dependent library */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which Uniorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts

MVME133 VARIATIONS
This library supports several variations of the MVME133 board,
the MVME133, MVME133A, MVME133XT and the MVME134.
When hardware is initialized in sysHwInit(2), the appropriate
memory map addresses are determined.

The MVME133 and MVME133A use the MM58274 real time clock,
have 1 Meg. of RAM and are the same except that the
MVME133 has only a 24 bit address space.

The MVME133XT and MVME134 use the MT48TO2 real time clock,
have 4 Meg. of RAM and are the same except that the
MVME133XT has a Floating Pointing Coprocessor MC68882, while the
MVME134 has a Paged Memory Management Unit MC68851.
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "vme.h"
#include "ioLib.h"
#include "memLib.h"
#include "sysLib.h"
#include "config.h"
#include "iv68k.h"


IMPORT char end;		/* end of system, created automatically by ld */
IMPORT VOID logMsg ();


/* MVME133: special 16 bit access with no interrupt on level 3 */
#define SPECIAL_ACCESS(addr) ((char *)(((int)addr | (1 << 24)) & ~(1 << 25)))

#define	HZ	1230769		/* clock rate, Motorola says 1.23 */
#define	PRESCALE	200	/* some may prefer 100 */


/* globals */

int sysBus        = BUS;		/* system bus type (VME_BUS, etc) */
int sysCpu        = CPU;		/* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int sysProcNum;				/* processor number of this cpu */
int sysFlags;				/* boot flags */
char sysBootHost [BOOT_FIELD_LEN];	/* name of host from which we booted */
char sysBootFile [BOOT_FIELD_LEN];	/* name of file from which we booted */

char *MFP_BASE_ADR;		/* multi-function peripheral address */
char *RTC_BASE_ADR;		/* real time clock address */


/* locals */

LOCAL int sysClkTicksPerSecond = 60;
LOCAL int auxClkTicksPerSecond = 60;
LOCAL int sysClkRunning        = FALSE;
LOCAL int auxClkRunning        = FALSE;

LOCAL int sysTarget;		/* MV133, MV133A, MV133XT, or MV134 */


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "Motorola MVME-133", or whatever
*/

char *sysModel ()

    {
    switch (sysTarget)
	{
	case MV133:
	    return ("Motorola MVME-133");

	case MV133A:
	    return ("Motorola MVME-133A");

	case MV133XT:
	case MV134:
	    return ("Motorola MVME-133XT / 134");

	default:
	    return ("Motorola MVME-UNKNOWN");
	}
    }
/*******************************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board.
* It is normally called from usrInit (2) in usrConfig (1).
* This routine must be called in supervisor mode.
*
* Determine board type: MVME133, MVME133A, MVME133XT, or MVME134.
* The timers are initialized and turned off.
* On the MVME133 and MVME133A the real time clock is turned off.
* Multi-Function Peripheral chip is initialized.
* On the MVME134 the MMU is turned off.
*
* NOTE:
* This routine determines whether the target is an MVME133, MVME133A,
* MVME133XT, or MVME134.  It relies on the MVME133 and MVME133A having
* only 1 Megabyte of contiguous memory.
*/

VOID sysHwInit ()

    {
    char test;

    if (sysTarget == 0)
	{
	/* target is unknown, play games to figure it out */

	if (sysMemTop () == (char *) 0x100000)
	    {
	    /* only 1 Meg., must be an MVME133 or MVME133A */

	    sysTarget    = MV133;
	    MFP_BASE_ADR = MV133_MFP_BASE_ADR;
	    RTC_BASE_ADR = MV133_RTC_BASE_ADR;

	    /* test for RTC at MVME133 location (and clear power up bit) */

	    if (vxMemProbe (SPECIAL_ACCESS (RTC_CONTROL(RTC_BASE_ADR)),
			    READ, 1, &test) == ERROR)
		{
		/* bad guess, must be an MVME133A */
		sysTarget     = MV133A;
		MFP_BASE_ADR  = MV133A_MFP_BASE_ADR;
		RTC_BASE_ADR  = MV133A_RTC_BASE_ADR;
		if (*RTC_CONTROL(RTC_BASE_ADR)); /* clear power up bit */
		}

	    /* initialize real time clock so it won't interrupt us */

	    /* stop clock, interrupt, select interrupt register */
	    *RTC_CONTROL(RTC_BASE_ADR) = 0xf;
	    *RTC_SET_INT(RTC_BASE_ADR) = 0x0; /* clear all interrupts */
	    /* stop clock, interrupt, select clock setting */
	    *RTC_CONTROL(RTC_BASE_ADR) = 0x5;
	    }
	else
	    {
	    /* must be an MVME133XT or MVME134 */

	    /* MVME133XT_MFP_BASE_ADR is the same */

	    MFP_BASE_ADR = MV134_MFP_BASE_ADR;
	    RTC_BASE_ADR = MV134_RTC_BASE_ADR;    /* MVME133XT is the same */

	    sysTarget = MV134;

	    /* real time clock creates no interrupts */ 

	    /* The MVME133XT has an MC68882 on board,
	     * probing for it would be a useful way to
	     * differentiate it from the MVME134, however
	     * we don't want to drag in fppLib.
	     *
	     * if (fppProbe ())
	     *	sysTarget = MV133XT;
	     */

	    /* clear MC68851 power/up bus error flag;
	     * accessing this location on the MVME133XT clears PWRUP */

	    if (*PUPMMU_ADR);
	    }
	}

    /* disable interrupts from MFP chip */
    *MFP_IERA(MFP_BASE_ADR) = 0;	/* clear interrupt enable register A */
    *MFP_IERB(MFP_BASE_ADR) = 0;	/* clear interrupt enable register B */

    *MFP_IMRA(MFP_BASE_ADR) = 0;	/* clear interrupt mask register A */
    *MFP_IMRB(MFP_BASE_ADR) = 0;	/* clear interrupt mask register B */

    /* clear interrupt pending and in-service registers */

    *MFP_IPRA(MFP_BASE_ADR) = 0;	/* interrupt pending register A */
    *MFP_IPRB(MFP_BASE_ADR) = 0;	/* interrupt pending register B */
    *MFP_ISRA(MFP_BASE_ADR) = 0;	/* interrupt in-service register A */
    *MFP_ISRB(MFP_BASE_ADR) = 0;	/* interrupt in-service register B */

    /* initialize vector register on MFP chip */

    *MFP_VR(MFP_BASE_ADR) = MFP_INT_VECT_NUM;

    /* initialize data direction register */

    *MFP_DDR(MFP_BASE_ADR) = 0x38;	/* gpio 5,4,3 are outputs */
    *MFP_AER(MFP_BASE_ADR) = 0x85;	/* this is from 134bug */

    *MFP_GPIP(MFP_BASE_ADR) = 0;	/* enable interrupts,turn off sysfail */

    /* initialize timers */

#if	(PRESCALE == 100)
    *MFP_TACR(MFP_BASE_ADR) = 0x06;	/* timer A delay mode, with prescale */
#else
    *MFP_TACR(MFP_BASE_ADR) = 0x07;	/* timer A delay mode, with prescale */
#endif	PRESCALE

    *MFP_TBCR(MFP_BASE_ADR) = 0;	/* clear timer B control register */
    *MFP_TCDCR(MFP_BASE_ADR) = 0;	/* clear timer C & D control regs. */

    *MFP_TADR(MFP_BASE_ADR) = 205;	/* timer A data register */
					/* - 205==60Hz w/200 prescale */

    *MFP_TBDR(MFP_BASE_ADR) = 0;	/* clear timer B data register */
    *MFP_TCDR(MFP_BASE_ADR) = 0;	/* clear timer C data register */
    *MFP_TDDR(MFP_BASE_ADR) = 0;	/* clear timer D data register */
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
*
* It starts probing at the end of bss, then tries to read a byte
* at every 4K boundary until it finds one that can't be read.
* This routine must be called in supervisor mode, the first time only.
* After that, it can be called in user mode as well.
*
* RETURNS: address of the first missing byte of memory
*
* INTERNAL
* This routine is used by sysHwInit (2) to differentiate between models.
*/

char *sysMemTop ()

    {
    static char *memTop = NULL;	/* top of memory address */
    char bitBucket;

#define	PAGE_SIZE	0x1000

    if (memTop == NULL)
	{
	/* Look for the top of memory starting at the first even page
	 * boundary after _end. */

	memTop = (char *) ((((int) &end) + (PAGE_SIZE - 1)) &
			    (~ (PAGE_SIZE - 1)));

	for (; vxMemProbe (memTop, READ, 1, &bitBucket) == OK;
	     memTop += PAGE_SIZE)
	    ;

	/* memTop is now pointing to the first non-existent address */
	}

    return (memTop);
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-X, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility could be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor)
*
* INTERNAL
* Note that the "WARM" restart address is at (ROM_BASE_ADRS + 16) bytes.
*/

STATUS sysToMonitor (startType)
    int startType;   /* parameter is passed to ROM to tell it how to boot, */
                     /* possible type are defined in sysLib.h              */
    {
    (* ((FUNCPTR) (ROM_BASE_ADRS + 16))) (startType);

    return (OK);	/* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURN: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine to be called at each clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    return (intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), routine, arg));
    }
/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/

VOID sysClkDisable ()

    {
    int temp;

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp &= ~MFP_TIMER_A;
    *MFP_IERA(MFP_BASE_ADR) = temp;

    sysClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysClkEnable - turn system clock interrupts on
*/

VOID sysClkEnable ()

    {
    int temp;

    /* Write the timer value */

#if	(PRESCALE == 100)
    *MFP_TACR(MFP_BASE_ADR) = 0x06;	/* timer A delay mode, with prescale */
#else
    *MFP_TACR(MFP_BASE_ADR) = 0x07;	/* timer A delay mode, with prescale */
#endif	PRESCALE

    *MFP_TADR(MFP_BASE_ADR) = HZ / (PRESCALE * sysClkTicksPerSecond);

    /* set mask */

    temp = *MFP_IMRA(MFP_BASE_ADR);
    temp |= MFP_TIMER_A;
    *MFP_IMRA(MFP_BASE_ADR) = temp;

    /* enable interrupts */

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp |= MFP_TIMER_A;
    *MFP_IERA(MFP_BASE_ADR) = temp;

    sysClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysClkGetRate - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkSetRate (2), sysClkEnable (2)
*/

int sysClkGetRate ()
    
    {
    return (sysClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkSetRate - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* System clock interrupts are not enabled.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkGetRate (2), sysClkEnable (2)
*/

VOID sysClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	sysClkTicksPerSecond = ticksPerSecond;

    if (sysClkRunning)
	{
	sysClkDisable ();
	sysClkEnable ();
	}
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    /* connect the routine to the interrupt vector */

    return (intConnect (INUM_TO_IVEC (MFP_INT_VECT_NUM + MFP_INT_TIMER_D),
			routine, arg));
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - disconnect a routine from the auxiliary clock interrupt
*
* This routine disables the auxiliary clock interrupt and disconnects
* the routine currently connected to the auxiliary clock interrupt.
*
* SEE ALSO: sysAuxClkConnect (2)
*/

VOID sysAuxClkDisconnect ()

    {
    /* disable the auxiliary clock interrupt */

    sysAuxClkDisable ();

    /* connect the dummy routine, just in case */

    sysAuxClkConnect (logMsg, (int)"auxiliary clock interrupt\n");
    }
/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/

VOID sysAuxClkDisable ()

    {
    int temp;

    /* stop timer */

    temp = *MFP_TCDCR(MFP_BASE_ADR);
    temp &= 0x70;	/* timer D stopped */
    *MFP_TCDCR(MFP_BASE_ADR) = temp;

    /* disable interrupt */

    temp = *MFP_IERB(MFP_BASE_ADR);
    temp &= ~MFP_TIMER_D;
    *MFP_IERB(MFP_BASE_ADR) = temp;

    /* mask out interrupts */

    temp = *MFP_IMRB(MFP_BASE_ADR);
    temp &= ~MFP_TIMER_D;
    *MFP_IMRB(MFP_BASE_ADR) = temp;

    auxClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysAuxClkEnable - turn auxiliary clock interrupts on
*/

VOID sysAuxClkEnable ()

    {
    int temp;

    /* write the timer value (with prescale) */

    *MFP_TDDR(MFP_BASE_ADR) = HZ / (PRESCALE * auxClkTicksPerSecond);

    /* set mask */

    temp = *MFP_IMRB(MFP_BASE_ADR);
    temp |= MFP_TIMER_D;
    *MFP_IMRB(MFP_BASE_ADR) = temp;

    /* enable interrupt */

    temp = *MFP_IERB(MFP_BASE_ADR);
    temp |= MFP_TIMER_D;
    *MFP_IERB(MFP_BASE_ADR) = temp;

    /* start timer */

    temp = *MFP_TCDCR(MFP_BASE_ADR);
    temp |= 0x07;
    *MFP_TCDCR(MFP_BASE_ADR) = temp;

    auxClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysAuxClkGetRate - get rate of auxiliary clock
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkSetRate (2)
*/

int sysAuxClkGetRate ()
    
    {
    return (auxClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysAuxClkSetRate - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* SEE ALSO: sysAuxClkConnect (2), sysAuxClkGetRate (2)
*/

VOID sysAuxClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond >= 25)
	auxClkTicksPerSecond = ticksPerSecond;

    if (auxClkRunning)
	{
	sysAuxClkDisable ();
	sysAuxClkEnable ();
	}
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the bus
*
* SEE ALSO: sysBusToLocalAdrs (2)
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,     */
                        /* use address modifier codes as defined in vme.h, */
                        /* such as VME_AM_STD_SUP_DATA                     */
    char *localAdrs;    /* local address to convert                        */
    char **pBusAdrs;    /* where to return bus address                     */

    {
    if ((localAdrs < LOCAL_MEM_LOCAL_ADRS) || (localAdrs >= sysMemTop ()))
	{
	/* this is off-board memory - just return local address */

	*pBusAdrs = localAdrs;
	}
    else
	{
	/* this is on-board memory - map to bus address space;
	 *   the following memory mapping is established in sysSetProcNum():
	 *   - only processor 0 has memory on bus,
	 *   - the memory is placed in STD space at address LOCAL_MEM_BUS_ADRS.
	 */

	if ((sysProcNum != 0) ||
	    ((adrsSpace != VME_AM_STD_SUP_PGM) &&
	     (adrsSpace != VME_AM_STD_SUP_DATA) &&
	     (adrsSpace != VME_AM_STD_USR_PGM) &&
	     (adrsSpace != VME_AM_STD_USR_DATA)))
	    {
	    return (ERROR);
	    }

	*pBusAdrs = localAdrs + LOCAL_MEM_BUS_ADRS - LOCAL_MEM_LOCAL_ADRS;
	}

    return (OK);
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,     */
                        /* use address modifier codes as defined in vme.h, */
                        /* such as VME_AM_STD_SUP_DATA                     */
    char *busAdrs;      /* bus address to convert                          */
    char **pLocalAdrs;  /* where to return local address                   */

    {
    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    if (sysTarget == MV133)
		*pLocalAdrs = (char *) (0x00ff0000 | (int) busAdrs);
	    else
		*pLocalAdrs = (char *) (0xffff0000 | (int) busAdrs);
	    return (OK);

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:

	    if (sysTarget == MV133A)
		{
		if ((int) busAdrs < 0x100000 || (int) busAdrs > 0xf00000)
		    return (ERROR);
		}

	    *pLocalAdrs = (char *) ((int) busAdrs);
	    return (OK);

	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:

	    if (sysTarget == MV133 || sysTarget == MV133A)
		return (ERROR); /* no extended access from MVME-133 & A */

	    if ((int) busAdrs < 0xf00000 || (int) busAdrs > 0xfff00000)
		return (ERROR);

	    *pLocalAdrs = (char *) ((int) busAdrs);
	    return (OK);

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* sysIntDisable - disable interrupt level
*
* This routine disables the specified interrupt level.
*
* NOTE:
* Interrupt levels are jumperable but not software controllable.
* This routine has no effect.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    return (OK);
    }
/*******************************************************************************
*
* sysIntEnable - enable interrupt level
*
* This routine enables the specified interrupt level.
*
* NOTE:
* Interrupts are jumperable but not software controllable.
* This routine has no effect.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    return (OK);
    }
/*******************************************************************************
*
* sysIntAck - acknowledge interrupt
*
* This routine acknowledges the specified interrupt.
*
* NOTE:
* If jumpered correctly, VME interrupts will be acknowledged automatically.
* This routine has no effect.
* On some boards this routine returns the vector put on
* bus by interrupting device.
*
* RETURNS: NULL
*
* ARGSUSED
*/

int sysIntAck (intLevel)
    int intLevel;	/* interrupt level to acknowledge */

    {
    return (NULL);
    }
/*******************************************************************************
*
* sysIntGen - generate interrupt
*
* This routine generates a VME bus interrupt.
*
* NOTE:
* Not implemented.
* The MVME-133 and MVME-133A can generate requests on IRQ3.
* The MVME-133XT and MVME-134 can generate requests per jumpers.
* See section 4.3.9 of the Motorola User's Manual for complete details.
* This routine has no effect.
*
* RETURNS: ERROR unable to generate interrupt
*
* ARGSUSED
*/

STATUS sysIntGen (level, vector)
    int level;
    int vector;

    {
    return (ERROR);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
*
* RETURNS: ERROR, since there is no mailbox facility
*
* SEE ALSO: intConnect (2)
*
* ARGSUSED
*/

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine to be called at each mailbox interrupt */
    int arg;		/* argument with which to call routine */

    {
    return (ERROR);
    }
/******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables the mailbox interrupt.
*
* RETURNS: ERROR, since there is no mailbox facility
*
* ARGSUSED
*/

STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;		/* address of mailbox */

    {
    return (ERROR);
    }
/****************************************************************************
*
* sysGetProcNum - get processor number
*
* This routine returns the processor number previously set with 
* sysSetProcNum (2).
*
* RETURNS: processor number
*/

int sysGetProcNum ()

    {
    return (sysProcNum);
    }
/****************************************************************************
*
* sysSetProcNum - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*/

VOID sysSetProcNum (procNum)
    int procNum;		/* processor number */

    {
    sysProcNum = procNum;
    }
/*******************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* RETURNS: TRUE (successful set) or FALSE (failure)
*/

BOOL sysBusTas (addr)
    char *addr;		/* address to be tested and set */

    {
    return (vxTas (addr));
    }


/* miscellaneous support routines */

/*******************************************************************************
*
* sysNvRamGet - get contents out of non-volatile RAM
*
* Copies non-volatile memory into string.
* The string will be terminated with an EOS.
*
* This routine has no effect for the MVME133 and MVME133A.
*
* RETURNS: OK, or ERROR if MVME133 or MVME133A
*/

STATUS sysNvRamGet (string, strLen, offset)
    FAST char *string;	/* where to copy non-volatile RAM */
    int strLen;		/* maximum number of bytes to copy */ 
    int offset;		/* byte offset into non-volatile RAM */

    {
    FAST char *pNvRam = (char *)((int)RTC_RAM(RTC_BASE_ADR) + offset);
    FAST int ix;

    if (sysTarget == MV133 || sysTarget == MV133A)
	return (ERROR);

    strLen--;	/* leave room for terminating EOS */

    for (ix = 0; ix < RTC_RAM_SIZE && ix < strLen; ix++)
	*string++ = *pNvRam++;

    *string = EOS;

    return (OK);
    }
/*******************************************************************************
*
* sysNvRamSet - write to non-volatile RAM
*
* Copy string into non-volatile RAM.
*
* This routine has no effect for the MVME133 and MVME133A.
*/

VOID sysNvRamSet (string, strLen, offset)
    FAST char *string;	/* string to be copied into non-volatile RAM */
    int strLen;		/* maximum number of bytes to copy */
    int offset;		/* byte offset into non-volatile RAM */

    {
    FAST char *pNvRam = (char *)((int)RTC_RAM(RTC_BASE_ADR) + offset);
    FAST int ix;

    if (sysTarget == MV133 || sysTarget == MV133A)
	return;

    for (ix = 0; (ix < RTC_RAM_SIZE) && (ix < strLen); ix++)
	*pNvRam++ = *string++;
    }
