/* sysLib.c - Motorola MVME-131 system dependent library */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which UniWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "vme.h"
#include "memLib.h"
#include "sysLib.h"
#include "ioLib.h"
#include "mv131.h"
#include "config.h"
#include "iv68k.h"

IMPORT char end;		/* end of system, created automatically by ld */
IMPORT VOID logMsg ();

#define HZ	1000000		/* clock rate */

/* globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */


/* locals */

LOCAL VOID sysClkInt ();

LOCAL FUNCPTR sysClkRoutine    = NULL;
LOCAL FUNCPTR auxClkRoutine    = NULL;
LOCAL BOOL sysClkIsConnected   = FALSE;	/* hooked up to interrupt yet? */
LOCAL int clkTicksPerSecond    = 60;
LOCAL int auxClkTicksPerSecond = 60;
LOCAL int sysClkArg            = NULL;
LOCAL int auxClkArg            = NULL;
LOCAL int sysClkRunning        = FALSE;
LOCAL int auxClkRunning        = FALSE;


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "Motorola MVME-131"
*/

char *sysModel ()
    {
    return ("Motorola MVME-131");
    }
/*******************************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board
* It is normally called from usrInit (2) in usrConfig (1).
* This routine must be called in supervisor mode.
*
* Initialize control registers.
* Setup the CIO chip.
*/

VOID sysHwInit ()

    {
    /* init control registers */

    *MV131_CNT2 = 0xff;				/* disable all VME interrupts */
    *MV131_CNT3 = MV131_CNT3_INT_ENABLE;	/* enable interrupts */
    *MV131_CNT5 = MV131_CNT5_NOT_VMX_READ_ONLY | MV131_CNT5_VMX_DECODE_ENABLE |
		  MV131_CNT5_NOT_VMX_INT_REQ;

    /* reset CIO chip */
    *Z8036_INT (CIO_BASE) = Z8036_INT_RES;
    *Z8036_INT (CIO_BASE) = 0x0;
    
    /* preliminary initialization of CIO */
    *Z8036_CONF (CIO_BASE) = 0x0;		/* all interrupts disabled */
    *Z8036_INT (CIO_BASE) = Z8036_INT_RIGHT_JUST | Z8036_INT_NV | Z8036_INT_MIE;
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
*
* It starts probing at the end of bss, then tries to read a byte
* at every 4K boundary until it finds one that can't be read.
* This routine must be called in supervisor mode, the first time only.
* After that, it can be called in user mode as well.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    static char *memTop = NULL;		/* top of memory address */
    char bitBucket;

#define	PAGE_SIZE	0x1000

    if (memTop == NULL)
	{
	/* Look for the top of memory starting at the first even page
	 * boundary after _end. */

	memTop = (char *) ((((int) &end) + (PAGE_SIZE - 1)) &
			    (~ (PAGE_SIZE - 1)));

	/* search a possible 4 Mbytes */

	for (;	(vxMemProbe (memTop, READ, 1, &bitBucket) == OK) &&
		(memTop < (char *) 0x400000);
	     memTop += PAGE_SIZE)
	    ;

	/* memTop is now pointing to the first non-existent address */
	}

    return (memTop);
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-x, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor)
*
* INTERNAL
* Note that the "WARM" restart address is at (ROM_BASE_ADRS + 16) bytes.
*/

STATUS sysToMonitor (startType)
    int startType;   /* parameter is passed to ROM to tell it how to boot */
                     /* possible types are defined in sysLib.h            */
    {
    (* ((FUNCPTR) (ROM_BASE_ADRS + 16))) (startType);

    return (OK);	/* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect a routine to the system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* System clock interrupts are not enabled.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each system clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    if (!sysClkIsConnected &&
	intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), sysClkInt, 0) == ERROR)
	{
	return (ERROR);
	}

    sysClkRoutine = routine;
    sysClkArg     = arg;

    sysClkIsConnected = TRUE;

    return (OK);
    }
/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/

VOID sysClkDisable ()

    {
    /* stop interrupts */

    *Z8036_CONF (CIO_BASE) &= (~Z8036_CONF_1_ENA);

    /* clear interrupt enable mask */

    *Z8036_TIM1_CONT (CIO_BASE) = Z8036_CONT_INT_DIS;

    sysClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysClkEnable - turn system clock interrupts on
*/

VOID sysClkEnable ()

    {
    int tc;		/* time constant */

    /* Initialize timer A */

    tc = HZ / clkTicksPerSecond;

    if (tc < 1 || tc > 65535)		/* 16 bits */
	{
	printErr ("sysClkEnable: %d out of range.\n", clkTicksPerSecond);
	return;
	}

    /* set the mode */
    *Z8036_TIM1_MODE (CIO_BASE) = Z8036_TIM_MODE_SQUARE |
				  Z8036_TIM_MODE_RETRIG |
				  Z8036_TIM_MODE_CONT;

    /* set interrupt enable mask */
    *Z8036_TIM1_CONT (CIO_BASE) = Z8036_CONT_INT_EN;

    /* set time constant */
    *Z8036_TIM1_TC_MSB (CIO_BASE) = (tc & 0xff00) >> 8;
    *Z8036_TIM1_TC_LSB (CIO_BASE) = (tc & 0x00ff);

    /* enable interrupts */
    *Z8036_CONF (CIO_BASE) |= Z8036_CONF_1_ENA;

    /* start timer */
    *Z8036_TIM1_CONT (CIO_BASE) = Z8036_TIM_CONT_TRIG | Z8036_TIM_CONT_GATE;

    sysClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysClkInt - interrupt handler for CIO interrupts
*
* The timers all interrupt on the same level.
* This routine demultiplexes them and calls the routines that
* have been connected by sysClkConnect (2) and sysAuxClkConnect (2).
*/

LOCAL VOID sysClkInt ()

    {
    /* Timer 1 handler (sysClk) */

    if (*Z8036_TIM1_CONT (CIO_BASE) & Z8036_TIM_CONT_IP)
	{
	if (sysClkRoutine != NULL)
	    (* sysClkRoutine) (sysClkArg);

	/* acknowledge this interrupt - clear IP & IUS */

	*Z8036_TIM1_CONT (CIO_BASE) =
				Z8036_CONT_CLR_IP_IUS | Z8036_TIM_CONT_GATE;
	}

    /* Timer 2 handler (auxClk) */

    if (*Z8036_TIM2_CONT (CIO_BASE) & Z8036_TIM_CONT_IP)
	{
	if (auxClkRoutine != NULL)
	    (* auxClkRoutine) (auxClkArg);

	/* acknowledge this interrupt - clear IP & IUS */

	*Z8036_TIM2_CONT (CIO_BASE) =
				Z8036_CONT_CLR_IP_IUS | Z8036_TIM_CONT_GATE;
	}

    /* clear all other interrupt pending bits */

    *Z8036_TIM3_CONT (CIO_BASE) = Z8036_CONT_CLR_IP_IUS;
    *Z8036_A_CONT (CIO_BASE)    = Z8036_CONT_CLR_IP_IUS;
    *Z8036_B_CONT (CIO_BASE)    = Z8036_CONT_CLR_IP_IUS;
    }
/*******************************************************************************
*
* sysClkGetRate - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkSetRate (2)
*/

int sysClkGetRate ()
    
    {
    return (clkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkSetRate - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* System clock interrupts are not enabled.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkGetRate (2), sysClkEnable (2)
*/

VOID sysClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	clkTicksPerSecond = ticksPerSecond;

    if (sysClkRunning)
	{
	sysClkDisable ();
	sysClkEnable ();
	}
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    auxClkRoutine = routine;
    auxClkArg     = arg;

    return (OK);
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - disconnect a routine from the auxiliary clock interrupt
*
* This routine disables the auxiliary clock interrupt and disconnects
* the routine currently connected to the auxiliary clock interrupt.
*
* SEE ALSO: sysAuxClkConnect (2), sysAuxClkDisable (2)
*/

VOID sysAuxClkDisconnect ()

    {
    /* disable the auxiliary clock interrupt */

    sysAuxClkDisable ();

    /* connect dummy routine, just in case */

    sysAuxClkConnect (logMsg, (int) "auxiliary clock interrupt\n");
    }
/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/

VOID sysAuxClkDisable ()

    {
    /* disable the auxiliary clock interrupt */

    *Z8036_CONF (CIO_BASE) &= (~Z8036_CONF_2_ENA);

    /* clear interrupt enable mask */

    *Z8036_TIM2_CONT (CIO_BASE) = Z8036_CONT_INT_DIS;

    auxClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysAuxClkEnable - turn auxiliary clock interrupts on
*/

VOID sysAuxClkEnable ()

    {
    unsigned int tc;		/* time constant */

    /* Initialize timer B */
    /* the timer chip has been prepared in sysHwInit */

    tc = HZ / auxClkTicksPerSecond;

    if (tc < 1 || tc > 65535)	/* 16 bits */
	{
	printErr ("sysAuxClkEnable: %d out of range.\n", auxClkTicksPerSecond);
	return;
	}

    /* disable timer B interrupts */
    *Z8036_CONF (CIO_BASE) &= (~Z8036_CONF_2_ENA);

    /* Initialize timer B */
    *Z8036_TIM2_MODE (CIO_BASE) = Z8036_TIM_MODE_SQUARE |
				  Z8036_TIM_MODE_RETRIG |
				  Z8036_TIM_MODE_CONT;

    /* set interrupt enable mask */

    *Z8036_TIM2_CONT (CIO_BASE) = Z8036_CONT_INT_EN;

    /* set time constant */
    *Z8036_TIM2_TC_MSB (CIO_BASE) = (tc & 0xff00) >> 8;
    *Z8036_TIM2_TC_LSB (CIO_BASE) = (tc & 0x00ff);

    /* enable timer B interrupts */
    *Z8036_CONF (CIO_BASE) |= Z8036_CONF_2_ENA;

    /* Start timer B */
    *Z8036_TIM2_CONT (CIO_BASE) = Z8036_TIM_CONT_TRIG | Z8036_TIM_CONT_GATE;

    auxClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysAuxClkGetRate - get rate of auxiliary clock
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkSetRate (2)
*/

int sysAuxClkGetRate ()
    
    {
    return (auxClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysAuxClkSetRate - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* SEE ALSO: sysAuxClkConnect (2), sysAuxClkGetRate (2)
*/

VOID sysAuxClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	auxClkTicksPerSecond = ticksPerSecond;
	
    if (auxClkRunning)
	{
	sysAuxClkDisable ();
	sysAuxClkEnable ();
	}
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the bus
*
* SEE ALSO: sysBusToLocalAdrs (2)
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,     */
                        /* use address modifier codes as defined in vme.h, */
                        /* such as VME_AM_STD_SUP_DATA                     */
    char *localAdrs;    /* local address to convert                        */
    char **pBusAdrs;    /* where to return bus address                     */

    {
    if ((localAdrs < LOCAL_MEM_LOCAL_ADRS) || (localAdrs >= sysMemTop ()))
	{
	/* this is off-board memory - just return local address */

	*pBusAdrs = localAdrs;
	}
    else
	{
	/* this is on-board memory - map to bus address space;
	 *   the following memory mapping is established in sysSetProcNum():
	 *   - only processor 0 has memory on bus,
	 *   - the memory is placed in STD space at address 0
	 */

	if ((sysProcNum != 0) ||
	    ((adrsSpace != VME_AM_STD_SUP_PGM) &&
	     (adrsSpace != VME_AM_STD_SUP_DATA) &&
	     (adrsSpace != VME_AM_STD_USR_PGM) &&
	     (adrsSpace != VME_AM_STD_USR_DATA)))
	    return (ERROR);

	*pBusAdrs = localAdrs - LOCAL_MEM_LOCAL_ADRS + LOCAL_MEM_BUS_ADRS;
	}

    return (OK);
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,     */
                        /* use address modifier codes as defined in vme.h, */
                        /* such as VME_AM_STD_SUP_DATA                     */
    char *busAdrs;      /* bus address to convert                          */
    char **pLocalAdrs;  /* where to return local address                   */

    {
    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pLocalAdrs = (char *) (0xffff0000 | (int) busAdrs);
	    return (OK);

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	    *pLocalAdrs = busAdrs;
	    return (OK);

	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    if ((unsigned int) busAdrs < (unsigned int) 0xfff00000)
		return (ERROR);

	    *pLocalAdrs = busAdrs;
	    return (OK);

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* sysIntDisable - disable interrupt level
*
* This routine disables the specified interrupt level.
*
* RETURNS: OK
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    *MV131_CNT2 |= (1 << intLevel);
    return (OK);
    }
/*******************************************************************************
*
* sysIntEnable - enable interrupt level
*
* This routine enables the specified VME interrupt level.
*
* RETURNS: OK
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    *MV131_CNT2 &= ~(1 << intLevel);
    return (OK);
    }
/*******************************************************************************
*
* sysIntAck - acknowledge interrupt
*
* This routine acknowledges the specified interrupt.
*
* NOTE MVME131:
* If jumpered correctly, the VME interrupts will be acknowledged automatically.
* This routine has no effect.
*
* RETURNS: NULL
*
* ARGSUSED
*/

int sysIntAck (intLevel)
    int intLevel;	/* interrupt level to acknowledge */

    {
    return (NULL);
    }
/*******************************************************************************
*
* sysIntGen - generate VME bus interrupt
*
* This routine generates a VME bus interrupt.
*
* RETURNS: OK
*/

STATUS sysIntGen (level, vector)
    int level;		/* VME bus interrupt level to generate (1-7) */
    int vector;		/* interrupt vector (0-255) */

    {
    *MV131_CNT0 = vector;
    *MV131_CNT3 |= (level << 1);

    return (OK);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
*
* RETURNS: ERROR, since there is no mailbox facility
*
* SEE ALSO: intConnect (2)
*
* ARGSUSED
*/

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each mailbox interrupt */
    int arg;		/* argument with which to call routine */

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables the mailbox interrupt.
*
* RETURNS: ERROR, since there is no mailbox facility
*
* ARGSUSED
*/

STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;		/* address of mailbox */

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysGetProcNum - get processor number
*
* This routine returns the processor number previously set with 
* sysSetProcNum (2).
*/

int sysGetProcNum ()

    {
    return (sysProcNum);
    }
/*******************************************************************************
*
* sysSetProcNum - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*/

VOID sysSetProcNum (procNum)
    int procNum;		/* processor number */

    {
    sysProcNum = procNum;
    }
/*******************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* RETURNS: TRUE (successful set) or FALSE (failure)
*/

BOOL sysBusTas (addr)
    char *addr;		/* address to be tested and set */

    {
    return (vxTas (addr));
    }
