/* sysLib.c - Matrix MS-CPU100 system dependent library */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which UniWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
        - initialize hardware to known state
        - identify the system

    memory/address space functions:
        - get on-board memory limit
        - map from local to bus and bus to local address spaces
        - enable/disable cache memory
        - set/get non-volatile RAM

    bus interrupt functions:
        - enable/disable bus interrupt levels
        - generate bus interrupts

    serial channel functions (see tyCoDrv):
        - enable/disable serial channel interrupts
        - set serial channel baud rates
        - get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts

*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "vme.h"
#include "memLib.h"
#include "sysLib.h"
#include "config.h"
#include "iv68k.h"

IMPORT char end;        /* end of system, created automatically by ld */

IMPORT VOID sysAbortIntA ();


/* globals */

int   sysBus      = BUS;            /* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;            /* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS; /* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;   /* catastrophic message area */
int   sysProcNum;                   /* processor number of this cpu */
int   sysFlags;                     /* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];  /* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];  /* name of file from which we booted */


/* locals */

LOCAL int clkTicksPerSecond;
LOCAL int auxClkTicksPerSecond;
LOCAL FUNCPTR duartTxInt = NULL;
LOCAL FUNCPTR duartRxInt = NULL;

LOCAL FUNCPTR sysXpBrdRoutine = NULL;
LOCAL int sysXpBrdRoutineParm = NULL;

LOCAL FUNCPTR sysClkRoutine = NULL;
LOCAL int sysClkRoutineParm;
LOCAL int sysClkFreq;

LOCAL FUNCPTR sysTodcRoutine = NULL;
LOCAL int sysTodcRoutineParm;
LOCAL int sysTodcFreq;

LOCAL FUNCPTR auxClkRoutine = NULL;
LOCAL int auxClkRoutineParm;
LOCAL int auxClkFreq;

LOCAL FUNCPTR sysMailboxRtn = NULL;
LOCAL int sysMailboxArg;

LOCAL char  imr;        /* current value of duart imr register */
LOCAL char  acr;        /* current value of duart acr register */
LOCAL char  rop;        /* current value of duart ropbc register */
LOCAL char  sop;        /* current value of duart sopbc register */
LOCAL char  ipcr;       /* current value of duart ipcr register */


/* forward declarations */

LOCAL VOID sysBoardInt ();           /* The handler of autovector 6 ints. */


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "Matrix MS-CPU100".
*/

char *sysModel ()
    {
    return ("Matrix MS-CPU100");
    }
/*******************************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board including the
* board control & status register, the DUART and the vme map decoder.
* It is normally called from usrInit (2) in usrConfig (1).
*
*/

VOID sysHwInit ()

    {
    char setButNotUsed;         /* temp holding for duart input */ 

    *INTH_IRQ1_MKS = 0;         /* mask  interrupt 1 */
    *INTH_IRQ2_MKS = 0;         /* mask  interrupt 2 */
    *INTH_IRQ3_MKS = 0;         /* mask  interrupt 3 */
    *INTH_IRQ4_MKS = 0;         /* allow interrupt 4 for the cmc board */
    *INTH_IRQ5_MKS = 0;         /* mask  interrupt 5 */
    *INTH_IRQ6_MKS = 0;         /* allow interrupt 6 for board interrupts */
    *INTH_IRQ7_MKS = 0;         /* allow interrupt 7 for abort */
    sysImrSet (NULL, 0xff);     /* mask serial interrupts */
    *TODC_CR2  = 1;             /* mask time-of-day interrupt */

    /* handle bus clear interrupts */
    intSetVec (INUM_TO_IVEC (MS_AVEC_7), sysAbortIntA);

    /* handle all other board level interrupts */
    intSetVec (INUM_TO_IVEC (MS_AVEC_6), sysBoardInt);

    /* setup the dual ported ram for external access */
      sysSetDualPort (DUAL_PORT_SET_ADRS);  
  
    /* initialize the duart for baud rate generator set 2, set the          *
     * timer/counter for timer mode, and enable receive-sysfail and mailbox */

    sysAcrSet ((BRG_SELECT | TMR_EXT_CLK ) ,0);
    *DUART_IVR = MS_AVEC_6;         /* set the duart interrupt vector reg */
    setButNotUsed = *DUART_IPCR;    /* clear isr 7 and input port on duart */

    /* enable the mailbox interrupt */
    sysMailboxEnable ();

    /* enable the system-fail interrupt */
    sysSysFailEnable();  

    /* initialize nvram to accept commands */
    sysRopbcSet (OP5, NULL);                /* clear chip enable on duart */ 
    sysRopbcSet (C_T_OUTPUT, NULL);         /* clear shift clock on duart */ 
    sysSopbcSet (C_T_OUTPUT, NULL);         /* set shift clock on duart */ 
    sysSopbcSet (OP5, NULL);                /* set chip enable on duart */          

    /* enable the interrupt handler */
    *INTH_MAIN_MKS = 0x80;      /* allow  interrupts */
    *INTH_IRQ4_MKS = 0x80;      /* allow interrupt 4 for the cmc board */
    *INTH_IRQ6_MKS = 0x80;      /* allow interrupt 6 for board interrupts */
    *INTH_IRQ7_MKS = 0x80;      /* allow interrupt 7 for abort */

    /* turn off the boardfail light */
    sysRopbcSet (OP7, NULL);
    sysNvRamCmd(NV_RECALL_CMD, NULL);		/* get the nv ram loaded */
    sysAdjustDualPort ();			/* set the dual port ram */
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine finds the size of system RAM.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    return (MS_RAM_END_ADRS + 1);
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-x, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor).
*
* INTERNAL
* Note that the "WARM" restart address is at (ROM_BASE_ADRS + 16) bytes.
*/

STATUS sysToMonitor (startType)
    int startType;    /* parameter is passed to ROM to tell it how to boot.
               The possible type are defined in h/sysLib.h */

    {
    (* ((FUNCPTR) (ROM_BASE_ADRS + 16))) (startType);

    return (OK);    /* in case we ever continue from rom monitor */
    }
/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt. 
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

VOID sysClkConnect (routine, arg)
    FUNCPTR routine;    /* routine to be called at each system clock
                         * interrupt */
    int arg;            /* argument with which to call routine */

    {
    char setButNotUsed;        /* used to generate EOI on the timer */

    (void)intConnect (INUM_TO_IVEC (MS_AVEC_6), sysBoardInt, NULL);
    sysClkRoutine     = routine;
    sysClkRoutineParm = arg;
    sysImrSet (CTR_RDY_INT, 0);
    setButNotUsed = *DUART_CTROFF;       /* turn of the timer */
    }
/********************************************************************************
* sysClkDisable - turn off system clock interrupts
*
* NOTE
* Not implemented (always on).
*/
 
VOID sysClkDisable ()
 
    {
    }
/********************************************************************************
* sysClkEnable - turn system clock interrupts on
*
* NOTE
* Not implemented (always on).
*/   
 
VOID sysClkEnable ()
 
    {
    }
/*******************************************************************************
*
* sysClkGetRate - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkSetRate (2)
*/

int sysClkGetRate ()
    
    {
    return (sysClkFreq);
    }
/*******************************************************************************
*
* sysClkSetRate - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkGetRate (2)
*/

VOID sysClkSetRate (ticksPerSecond)
    int ticksPerSecond;        /* number of clock interrupts per second */
    
    {
    int ctr ;
    char setButNotUsed;        /* used to generate EOI on the timer */

    /* calculate the divide ratio, and write it to the timer chip.  If
       ticksPerSecond == 0, write a value of 0 to stop the clock. */

    ctr = (ticksPerSecond == 0) ? 0 : (3686400 / (2 * ticksPerSecond)); 

    *DUART_CTUR = (ctr & 0xff00) >> 8;
    *DUART_CTLR = (ctr & 0x00ff);

    setButNotUsed = *DUART_CTRON ;    /* start the timer */

    sysClkFreq = ticksPerSecond;
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt,
* and enables the auxiliary clock interrupt.
*
* RETURNS:
*  ERROR, as there is no auxiliary clock on this CPU board
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect (routine, parm)
    FUNCPTR routine;    /* routine called at each auxiliary clock interrupt */
    int parm;           /* argument with which to call routine */

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - clear the auxiliary clock routine
* 
* This routine disables the auxiliary clock interrupt, stops the timer,
* and disconnects the routine currently connected to the auxiliary clock
* interrupt.
*
* NOTE
* This routine has no effect.
*
* SEE ALSO: sysAuxClkConnect (2)
*/

VOID sysAuxClkDisconnect ()

    {
    }
/********************************************************************************
* sysAuxClkDisable - turn off auxiliary clock interrupts
*
* NOTE:
* There is no auxiliary clock on the CPU-100.
*/
 
VOID sysAuxClkDisable ()
 
    {
    }
/********************************************************************************
* sysAuxClkEnable - turn auxiliary clock interrupts on
*
* NOTE:
* There is no auxiliary clock on the CPU-100.
*/
 
VOID sysAuxClkEnable ()
    {
    }
/*******************************************************************************
*
* sysAuxClkSetRate - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
*
* NOTE
* This routine has no effect.
*
* SEE ALSO: sysAuxClkGetRate (2)
*/

VOID sysAuxClkSetRate (ticksPerSecond)
    int ticksPerSecond;        /* number of clock interrupts per second */
    
    {
    }
/*******************************************************************************
*
* sysAuxClkGetRate - get the auxiliary timer frequency
*
* This routine finds out the auxiliary clock speed.
*
* NOTE
* This routine has no effect.
*
* RETURNS: ERROR
*
* SEE ALSO: sysAuxClkSetRate (2)
*/

int sysAuxClkGetRate ()

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysTodcConnect - connect routine to time-of-day clock interrupt
*
* This routine connects the given function to the time of day clock  
* interrupt. It is normally called from usrRoot (2) in usrConfig (1) to 
* connect usrClock (2) to the system clock interrupt.
*
* SEE ALSO: intConnect (2)
*/

VOID sysTodcConnect (routine, arg)
    FUNCPTR routine;    /* routine to be called at each time-of-day 
                         * clock interrupt */
    int arg;            /* argument with which to call routine */

    {
    char setButNotUsed;        /* used to generate EOI on the timer */

    (void)intConnect (INUM_TO_IVEC (MS_AVEC_6), sysBoardInt, NULL);
    sysTodcRoutine     = routine;
    sysTodcRoutineParm = arg;
    }
/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the VME
*
* SEE ALSO: sysBusToLocalAdrs (2)
*
* ARGSUSED
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides;
                         * use address modifier codes as defined in vme.h,
                         * such as VME_AM_STD_SUP_DATA */
    char *localAdrs;    /* local address to convert */
    char **pBusAdrs;    /* where to return bus address */

    {
    if (localAdrs >= sysMemTop ())
        {
        /* this is off-board memory - just return local address */

        *pBusAdrs = localAdrs;
        return (OK);
        }
    else
        {
        return (ERROR);
        }
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*
* ARGSUSED
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,      */
                        /*  use address modifier codes as defined in vme.h, */
                        /*  vme.h, such as VME_AM_STD_SUP_DATA              */
    char *busAdrs;      /* bus address to convert                           */
    char **pLocalAdrs;  /* where to return local address                    */

    {
    switch (adrsSpace)
        {
        case VME_AM_SUP_SHORT_IO:
        case VME_AM_USR_SHORT_IO:
            *pLocalAdrs = (char *) (0xffff0000 | (int) busAdrs);
            return (OK);

        case VME_AM_STD_SUP_ASCENDING:
        case VME_AM_STD_SUP_PGM:
        case VME_AM_STD_SUP_DATA:
        case VME_AM_STD_USR_ASCENDING:
        case VME_AM_STD_USR_PGM:
        case VME_AM_STD_USR_DATA:
				    /* XXX ?? */
            *pLocalAdrs = (char *) (0x00000000 | (int) busAdrs);
            return (OK);

        case VME_AM_EXT_SUP_ASCENDING:
        case VME_AM_EXT_SUP_PGM:
        case VME_AM_EXT_SUP_DATA:
        case VME_AM_EXT_USR_ASCENDING:
        case VME_AM_EXT_USR_PGM:
        case VME_AM_EXT_USR_DATA:
				    /* XXX ?? */
            if (busAdrs < (char *) 0x0000000)
                return (ERROR);

            *pLocalAdrs = busAdrs;
            return (OK);

        default:
            return (ERROR);
        }
    }
/*******************************************************************************
*
* sysIntEnable - enable VME interrupt level
*
* This routine enables the specified VME interrupt level.
* Interrupts 1 - 3 are jumper enabled.
*
* RETURNS: OK, or ERROR if intLevel not in range 1-5
*/

STATUS sysIntEnable (intLevel)
    int intLevel;    /* interrupt level to enable */

    {
    if ((intLevel < 1) || (intLevel > 5))
        return (ERROR);
    else
        {
        switch (intLevel)
            {
            case 1:
                *INTH_IRQ1_MKS = 0x80;
                return (OK);
            case 2:
                *INTH_IRQ2_MKS = 0x80;
                return (OK);
            case 3:
                *INTH_IRQ3_MKS = 0x80;
                return (OK);
            case 4:
                *INTH_IRQ4_MKS = 0x80;
                return (OK);
            case 5:
                *INTH_IRQ5_MKS = 0x80;
                return (OK);
            default:
                return (ERROR);
            }
        }
    }
/*******************************************************************************
*
* sysIntDisable - disable VME interrupt level
*
* This routine alway returns error because VME interrupts are not under
* software control.
*
* RETURNS: ERROR.
*/

STATUS sysIntDisable (intLevel)
    int intLevel;    /* interrupt level to disable */

    {
    if ((intLevel < 1) || (intLevel > 5))
        return (ERROR);
    else
        {
        switch (intLevel)
            {
            case 1:
                *INTH_IRQ1_MKS = 0x00;
                return (OK);
            case 2:
                    *INTH_IRQ2_MKS = 0x00;
                return (OK);
            case 3:
                *INTH_IRQ3_MKS = 0x00;
                return (OK);
            case 4:
                *INTH_IRQ4_MKS = 0x00;
                return (OK);
            case 5:
                *INTH_IRQ5_MKS = 0x00;
                return (OK);
            default:
                return (ERROR);
            }
        }
    }
/*******************************************************************************
*
* sysIntAck - acknowledge VME interrupt
*
* This is a null routine because hardware acknowledges VME interrupts.
*
* RETURNS: OK
*/

STATUS sysIntAck (intLevel)
    int intLevel;    /* interrupt level to acknowledge */

    {
    return (OK);
    }
/*******************************************************************************
*
* sysIntGen - generate VME interrupt
*
* The MS-CPU100 is not a VMEbus interrupter.
*
* RETURNS: ERROR
*/

STATUS sysIntGen (intLevel)
    int intLevel;    /* interrupt level to generate */

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* The MS-CPU100 mailbox interrupt is handled through the DUART and Board 
* Control & Status Register. Therefore, the mailbox interrupt need only 
* be enabled.
*
* RETURNS: ERROR.
*
* SEE ALSO: intConnect (2), sysMailboxEnable(2)
*/

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;    /* routine called at each mailbox interrupt */
    int arg;            /* argument with which to call routine */

    {
    (void) intConnect (INUM_TO_IVEC (MS_AVEC_6), sysBoardInt, NULL);
    sysMailboxRtn = routine ;
    sysMailboxArg = arg;
    return (OK);
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
*  This routine enables the on-board mailbox interrupt. The MS-CPU100/110 has 
*  one long word memory location (0x100) ,when written to, that causes an 
*  interrupt to be propagated through the DUART and the Board Control and 
*  Status Register. This interrupt is handled by the  BCSR through auto-
*  vector 6.
*
* RETURNS: OK
*
* SEE ALSO: sysMailboxDisable(2) , sysMailboxConnect(2)
*
* ARGSUSED
*/

STATUS sysMailboxEnable (mailboxAdrs)
   char *mailboxAdrs;          /* mailbox address */

    {
    /* set opr[0] opposite ipcr[0] for change of state interrupt */

    if (ipcr = *DUART_IPCR & DUART_OPR_MBOX)
         sysRopbcSet (DUART_OPR_MBOX, NULL);
    else
        sysSopbcSet (DUART_OPR_MBOX, NULL);

    sysAcrSet (DUART_ACR_MBOX, 0);               /* unmask the interrupt */ 
    sysImrSet (INPUT_DELTA_INT, 0);              /* unmask DUART mask */

    return (OK);
    }
/*******************************************************************************
*
* sysMailboxDisable - disable mailbox interrupt
*
* This routine disables the on-board mailbox interrupt. This interrupt is 
* handled by the  BCSR through auto-vector 6.
*
* RETURNS: OK
*
* SEE ALSO: sysMailboxEnable(2), sysMailboxConnect(2)
*/

STATUS sysMailboxDisable ()

    {
    sysAcrSet (NULL, DUART_ACR_MBOX);               /* mask the interrupt */ 
    return (OK);
    }
/********************************************************************************
* sysGetProcNum - get processor number
*
* This routine returns the processor number previously set with
* sysSetProcNum (2).
*
* RETURNS: processor number
*/

int sysGetProcNum ()

    {
    return (sysProcNum);
    }
/*******************************************************************************
*
* sysSetProcNum - set processor number
*
* Set the processor number for the MS-CPU100.  Processor numbers should be
* unique on a single backplane.
*
*/

VOID sysSetProcNum (procNum)
    int procNum;        /* processor number */

    {
    sysProcNum = procNum;
    }
/*******************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* On the MS-CPU100 this is equivalent to the vxTas (2) routine.
*
* RETURNS: TRUE (successful set) or FALSE (failure)
*/

BOOL sysBusTas (addr)
    char *addr;        /* address to be tested and set */

    {
    return (vxTas (addr));
    }

/* miscellaneous support routines */

/*******************************************************************************
*
* sysSysFailEnable - enable recept of system fail interrupt
*
* This routine enables the on/off-board system fail interrupt.
* This interrupt is handled by the DUART and BCSR through auto-vector 6.
*
* RETURNS: OK
*/

STATUS sysSysFailEnable ()

    {
    sysAcrSet (DELTA_IP3_INT, 0);                /* unmask the interrupt */ 
    sysImrSet (INPUT_DELTA_INT, 0);              /* unmask DUART mask */
    return (OK);
    }
/*******************************************************************************
*
* sysSetDualPort - set the local memory address for vme bus access
*
* This routine enables the dual ported ram for access by other bus masters.
* This routine is valid immediately after power-up and reset only.              
*
* RETURNS: OK or ERROR if vme address is greater than 32
*
* SEE ALSO: sysAdjustDualPort(2)               
*
* NOMANUAL
*/

STATUS sysSetDualPort (vmeAddress)
    char vmeAddress;    /* value to set the vme address of local RAM  */
                        /* increments of .5 Mbyte upto 16.5 Mbytes accepted  */
                        /*  0 =  0 to .5 Mbytes  */
                        /*  1 = .5 to 1 Mbytes  */
                        /* 32 = 16 to 16.5 Mbytes */

    {
    char loopCnt;
    char vmdShiftClock = OP6;             /* DUART output register pin 6 */
    char dataIn = OP4;                    /* DUART output register pin 4 */
    
    /* check for poper value sent by requester */

    if (vmeAddress > 24)
        return(ERROR);

    /* put the address in the correct position */

    vmeAddress = (vmeAddress << 3) | ON_BOARD_RAM_SIZE;
    sysSopbcSet (vmdShiftClock, NULL);		/* enable the decoder chip */

	/* serially load in the address and  enable bits */
    for (loopCnt = 8; loopCnt > 0; loopCnt--)
        {
        if (vmeAddress  & 1)
            sysSopbcSet (dataIn, NULL);
        else
            sysRopbcSet (dataIn, NULL);
        sysRopbcSet (vmdShiftClock, NULL);
        sysSopbcSet (vmdShiftClock, NULL);
        vmeAddress >>= 1;
        }
    return (OK);
    }
/*******************************************************************************
*
* sysNvRamCmd - issue non-volatile RAM the passed command
*
* Issues one of seven available commands to the non-volatile RAM.
* The available commands are:
*   command   7 - - - - - - 0   operation 
*   WRDS      1 X X X X 0 0 0   Reset write-enable-latch
*   STO       1 X X X X 0 0 1   Store RAM data into EPROM
*   SLEEP     1 X X X X 0 1 0   Enter sleep mode (low power state)
*   WRITE     1 A A A A 0 1 1   Write data into RAM at address AAAA 
*   WREN      1 X X X X 1 0 0   Set write-enable-latch 
*   RCL       1 X X X X 1 0 1   Recall EEPROM data into RAM
*   READ      1 A A A A 1 1 0   Read data from RAM at address AAAA
*
* If the command sent is a read or a write the offset passed is addressing one
* of 16x16 bit RAM locations.
*
* RETURNS: OK (always)
*/

LOCAL STATUS sysNvRamCmd (command,offset)                
    char command;                      /* command to be issued to the NV RAM */
    char offset;                       /* address of NVRAM if read or write */
    {

    char shiftClock = C_T_OUTPUT;      /* bit value for the shift clock */
    char chipEnable = OP5;             /* bit value for chip enable */
    char dataIn = OP4;                 /* bit value for data in */
    int loopCnt;                       /* loop counter */
    char temp;
   

    sysSopbcSet (chipEnable, NULL);         /* prepare NVRAM for command */ 
    command <<= 5;
    for(loopCnt=3; loopCnt > 0 ; loopCnt--)
        {
        if ((command <<= loopCnt + 1) & 1)       /* check for bit zero set */
            sysSopbcSet (dataIn, NULL);         
        else
            sysRopbcSet (dataIn, NULL);
        sysRopbcSet (shiftClock, NULL);
        sysSopbcSet (shiftClock, NULL);
        }
    command <<= 1;

    /* if the command is a read or write add in the address of nvram */

    if (((temp = command & NV_CMD_MASK) == NV_WRITE_CMD) ||
        ( temp == NV_READ_CMD))
        for(loopCnt=5; loopCnt > 0 ; loopCnt--)
            {
            if ((command <<= loopCnt + 1) & 1)   /* check for bit zero set */
                sysSopbcSet (dataIn, NULL);         
            else
                sysRopbcSet (dataIn, NULL);
            sysRopbcSet (shiftClock, NULL);
            sysSopbcSet (shiftClock, NULL);
            }
    else
        {
        sysRopbcSet (chipEnable, NULL);
        sysRopbcSet (shiftClock, NULL);
        sysSopbcSet (shiftClock, NULL);
        sysSopbcSet (chipEnable, NULL);
        }
    return (OK); 
    }
/*******************************************************************************
*
* sysNvRamGet - get contents out of non-volatile RAM
*
* Copies non-volatile memory into string.
* The string will be terminated with an EOS.
*
* NOTE
* Does not work correctly!
*
* RETURNS: OK (always)
*/

STATUS sysNvRamGet (string, strLen, offset)
    FAST char *string;    /* where to copy non-volatile RAM */
    int strLen;           /* maximum number of bytes to copy */
    int offset;           /* (even) byte offset into non-volatile RAM */

    {
    FAST int ix;
    char shiftClock = C_T_OUTPUT;       /* bit value for the shift clock */
    char chipEnable = OP5;              /* bit value for chip enable */
    char dataIn;                        /* bit value for data in */
    char loopCnt;			/* loop counter */
        
    /* construct the character string from NV bits;
     * the bits are read from NVRAM lsb first and then reordered. */

    strLen = (strLen <= 2) ? 2 : strLen;
    for (loopCnt = strLen / 2; loopCnt > 0; loopCnt--)
        {

        /* get the first 2 data bits then do the loop */

        sysRopbcSet (shiftClock, NULL);     /* clear the shift clock */
        dataIn = (*DUART_OPCR & IPR4) >> 5; /* get the data bit from DUART */
        sysSopbcSet (shiftClock, NULL);     /* set the shift clock */
        dataIn |= (*DUART_OPCR & IPR4) >> 4; /* get the data bit from DUART */
        for (ix = NV_WIDTH - 2 ; ix > 0; ix--) 
            {
            dataIn >>= 1;                            /* get bit from DUART */
            sysRopbcSet (shiftClock, NULL);
            sysSopbcSet (shiftClock, NULL);
            dataIn |= (*DUART_OPCR & IPR4) >> 4;    /* get bit from DUART */
            }

        /* shift the bits into their correct positions then output */
        *(string++) = (dataIn & 0xff); 
        *(string++) = (dataIn & 0xff00) >> 8; 

        /* reset the chip for the next command */
        sysRopbcSet (chipEnable, NULL);     /* clear chip enable */
        sysRopbcSet (shiftClock, NULL);     /* clear the shift clock */
        sysSopbcSet (shiftClock, NULL);     /* set the shift clock */
        sysSopbcSet (chipEnable, NULL);     /* set chip enable */

        /* set the get command one more time */
        sysNvRamCmd(NV_READ_CMD, offset += 2);
        }
    *string = EOS;
/****************start***dmp******debug**************************/
    logMsg ("nvGet:    string=%s    or string=%x\n",string,string);    
/******************end***dmp******debug**************************/
    return (OK);  
    }
/*******************************************************************************
*
* sysNvRamSet - write to non-volatile RAM
*
* Copy string into non-volatile RAM.
*
* NOTE
* Does not work correctly!
*/

VOID sysNvRamSet (string, strLen, offset)
    FAST char *string;    /* string to be copied into non-volatile RAM */
    int strLen;        /* maximum number of bytes to copy */
    int offset;        /* (even) byte offset into non-volatile RAM */

    {
    FAST int ix;
    char shiftClock = C_T_OUTPUT;      /* bit value for the shift clock */
    char chipEnable = OP5;             /* bit value for chip enable */
    char dataOut = OP4;                /* bit value for data in */
    char temp = string[1];             /* get the first character */
    char loopCnt;                      /* loop counter */

    strLen--;    /* leave room for terminating EOS */
    strLen = (strLen < 2) ? 2 : strLen;

    /* set up the NVRAM for write */

    /* construct the character string from NV bits;
     * the bits are written to NVRAM lsb first. */

    for (loopCnt = strLen / 2; loopCnt > 0; loopCnt--) 
        {
        sysNvRamCmd (NV_WRITE_CMD,offset);
        dataOut = (((temp = *(string++)) << 8) | (temp = *(string++)));
        for (ix = 0; ix > NV_WIDTH; ix++)  
            {
            if ((dataOut >>= ix) & 1)              /* check for bit zero set */
                sysSopbcSet (dataOut, NULL);
            else
                sysRopbcSet (dataOut, NULL);
            sysRopbcSet (shiftClock, NULL);
            sysSopbcSet (shiftClock, NULL);
            }

        /* reset the chip for the next command */

        sysRopbcSet (chipEnable, NULL);     /* clear chip enable */
        sysRopbcSet (shiftClock, NULL);     /* clear the shift clock */
        sysSopbcSet (shiftClock, NULL);     /* set the shift clock */
        sysSopbcSet (chipEnable, NULL);     /* set chip enable */
        sysNvRamCmd(NV_WRITE_CMD, (offset += 2));   /* send another write cmd */
        }
/****************start***dmp******debug**************************/
    logMsg ("nvSet:    string=%s    or string=%x\n",string,string);    
/******************end***dmp******debug**************************/

    /* load from the shadow RAM to the NVRAM */

    sysNvRamCmd(NV_STO_CMD, NULL);
	return (OK);
    }
/*******************************************************************************
*
* sysImrSet - set and clear bits in the m68681's interrupt mask register
*
* This routine sets and clears bits in the duart's IMR.  It may be called
* either in user or supervisor state.
*
* This routine sets and clears bits in a local copy of the IMR, then
* writes that local copy to the duart.  This means that all changes to
* the IMR must go through this routine.  Otherwise, any direct changes
* to the IMR would be lost the next time this routine is called.
*
* NOMANUAL
*/

VOID sysImrSet (setBits, clearBits)
    char setBits;    /* which bits to set in the IMR */
    char clearBits;    /* which bits to clear in the IMR */

    {
    imr = (imr | setBits) & (~clearBits);
    *DUART_IMR = imr;
    }
/*******************************************************************************
*
* sysAcrSet - set and clear bits in the m68681's auxiliary control register
*
* This routine sets and clears bits in the duart's ACR.  It may be called
* either in user or supervisor state.
*
* This routine sets and clears bits in a local copy of the ACR, then
* writes that local copy to the duart.  This means that all changes to
* the ACR must go through this routine.  Otherwise, any direct changes
* to the ACR would be lost the next time this routine is called.
*
* NOMANUAL
*/

VOID sysAcrSet (setBits, clearBits)
    char setBits;    /* which bits to set in the ACR */
    char clearBits;    /* which bits to clear in the ACR */

    {
    acr = (acr | setBits) & (~clearBits);
    *DUART_ACR = acr;
    }
/*******************************************************************************
*
* sysRopbcSet - set and clear bits in the m68681's interrupt mask register
*
* This routine sets and clears bits in the duart's ROPBC.  It may be called
* either in user or supervisor state.
*
* This routine sets and clears bits in a local copy of the ROPBC, then
* writes that local copy to the duart.  This means that all changes to
* the ROPBC must go through this routine.  Otherwise, any direct changes
* to the ROPBC would be lost the next time this routine is called.
*
* NOMANUAL
*/

VOID sysRopbcSet (setBits, clearBits)
    char setBits;    /* which bits to set in the ROPBC */
    char clearBits;    /* which bits to clear in the ROPBC */

    {
    rop = (rop | setBits) & (~clearBits);
    *DUART_ROPBC = rop;
    }
/*******************************************************************************
*
* sysSopbcSet - set and clear bits in the m68681's interrupt mask register
*
* This routine sets and clears bits in the duart's SOPBC.  It may be called
* either in user or supervisor state.
*
* This routine sets and clears bits in a local copy of the SOPBC, then
* writes that local copy to the duart.  This means that all changes to
* the SOPBC must go through this routine.  Otherwise, any direct changes
* to the SOPBC would be lost the next time this routine is called.
*
* NOMANUAL
*/

sysSopbcSet (setBits, clearBits)
    char setBits;    /* which bits to set in the SOPBC */
    char clearBits;    /* which bits to clear in the SOPBC */

    {
    sop = (sop | setBits) & (~clearBits);
    *DUART_SOPBC = sop;
    }
/*******************************************************************************
*
* sysBoardInt - interrupt level processing for the MS-CPU100 
*
* This routine handles all on board interrupts. The interrupts are
* decoded by checking the board control & status registers and the duart's
* interrupt status reg, and the appropriate routine invoked.  
*/

LOCAL VOID sysBoardInt ()

    {
    LOCAL char intStatusReg;            /* duart int status reg */
    LOCAL char inputDeltaReg;           /* duart ipcr register */
    char  mailNot;                      /* mail flag */
    char setButNotUsed;                 /* used to generate EOI on the timer */
    char temp;

    intStatusReg = *DUART_ISR;              /* save a copy of duart isr */
    if (*BCSR_ACFAIL_IVR & 0x80)            /* ac fail interrupt */
        logMsg("ac fail");
    if (*BCSR_XPBRD_IVR & 0x80)
            {
            /* expansion board interrupt.  call expansion routine, if
             * there is one, else issue an EOI */

            if (sysXpBrdRoutine == NULL)
                *BCSR_XPBRD_IMR = 0;        /* EOI */
            else
                (*sysXpBrdRoutine)(0);
            }

    if (*BCSR_BUSMS_IVR & 0x80)             /* bus master interrupt */
        *BCSR_BUSMS_IMR = 0;
    if (*BCSR_DUART_ISR & 0x80)             /* duart interrupt */
        { 
        if (intStatusReg & CTR_RDY)            /* on-chip timer */
            {
            if (sysClkRoutine != NULL)
                (*sysClkRoutine) (sysClkRoutineParm);
            setButNotUsed = *DUART_CTROFF;        /* reset the interrupt */
            }

        /* check for system fail and mailbox interrupts */
        if (intStatusReg & INPUT_DELTA )
            {
            mailNot = 0;            /* turn on mail */

            /* check for system fail */
            if ( (temp = (setButNotUsed = *DUART_IP) & IPR3) == 0) 
                {
                mailNot = 1;            /* turn off mail */
                }    

            /* check for 1st mailbox interrupt */
            if ((inputDeltaReg = *DUART_IPCR) & DUART_OPR_MBOX & (mailNot == 0))
                {

                /* there will be two fetches on the mailbox unless
                 * IP3 is turned off */

                inputDeltaReg &= 0xf7;        /* turn off 2nd fetch */
                sysMailboxEnable();
                if (sysMailboxRtn != NULL)
                    (*sysMailboxRtn) (sysMailboxArg);
                }

            /* check for 2nd change-of-state on mailbox */    
            if (inputDeltaReg & IP3)
                {
                 sysMailboxEnable();  
                if (sysMailboxRtn != NULL)
                    (*sysMailboxRtn) (sysMailboxArg);
                }
            }
        if (*DUART_ISR & TX_RDY_A)
            {
            /* transmitter channel A interrupt.  call duart routine, if
             * there is one, else issue an EOI */

            if (duartTxInt == NULL)
                sysImrSet (0, TX_RDY_A);     /* EOI */
            else
                (*duartTxInt) (0);
            }

        if (*DUART_ISR & TX_RDY_B)
            {
            /* transmitter channel B interrupt.  call duart routine, if
             * there is one, else issue an EOI */

            if (duartTxInt == NULL)
                sysImrSet (0, TX_RDY_B);     /* EOI */
            else
                (*duartTxInt) (1);
            }

        if ((*DUART_ISR & RX_RDY_A) && (duartRxInt != NULL))
            (*duartRxInt) (0);            /* receiver channel A */

        if ((*DUART_ISR & RX_RDY_B) && (duartRxInt != NULL))
            (*duartRxInt) (1);            /* receiver channel B */

        }
    if (*BCSR_TODC_ISR & 0x80)
        {
        if (sysTodcRoutine != NULL)                   /* time-of-day */
            (*sysTodcRoutine) (sysTodcRoutineParm);
        }
    }
/*******************************************************************************
*
* sysDuartConnect - set interrupt routines for the MS-CPU100 
*
* This routine connects the serial driver interrupt routines to the
* interrupt routine.  It should be called once, from the serial driver.
*
* NOMANUAL
*/

VOID sysDuartConnect (recvRoutine, xmitRoutine)
    FUNCPTR recvRoutine;    /* receive routine */
    FUNCPTR xmitRoutine;    /* transmit routine */

    {
    (void)intConnect (INUM_TO_IVEC (MS_AVEC_6), sysBoardInt, NULL);
    duartTxInt = xmitRoutine;
    duartRxInt = recvRoutine;
    }
/*******************************************************************************
*
* sysXpBrdConnect - set interrupt routines for the MS-CPU100 expansion board
*
* This routine connects the expansion board interrupt routines to the
* interrupt.  
*
* RETURNS: ERROR (if expansion board is not present)
*/

LOCAL VOID sysXpBrdConnect (routine, arg)
    FUNCPTR routine;    /* routine to be called when the expansion board 
                         * interrupts */
    int arg;            /* argument with which to call routine */

    {
    /* check to make sure expansion board is present */
    if (*BCSR_XPBRD_OR & 0x80)
        {
        (void)intConnect (INUM_TO_IVEC (MS_AVEC_6), sysBoardInt, NULL);
        sysXpBrdRoutine     = routine;
        sysXpBrdRoutineParm = arg;
        *BCSR_XPBRD_IMR = 1;                /* unmask the interrupt */     
        }
    else
        return(ERROR);
    }
/*******************************************************************************
*
* sysFlash - flash the board-fail LEDs
*
* This routine turns on the board-fail LED for `flashTime' ticks
*  and then turns it off.
*
* NOMANUAL
*/

VOID sysFlash (flashTime)
    int flashTime;    /* ticks to delay */

    {
    sysSopbcSet (0x80, NULL);

    /* XXX shouldn't make kernel call -- could be used at interrupt level */
    taskDelay (flashTime);

    sysRopbcSet (0x80, NULL);
    }
