/* tyCoDrv.c - Ironics IV-1624A on-board SCC tty handler */

static char *copyright = "Copyright 1986-88, Wind River Systems, Inc.";

/*
modification history
--------------------
01i,25jul88,gae  changed name of header file scc8530.h to z8530.h
01h,22jul88,gae  fixed name to IV-1624A.  Cleaned up.
01g,20nov87,dnw  changed to use new INT_VEC_ definitions.
01f,24oct87,dnw  changed to not initialize channels until devices are created
		   for them (tyCoDrv() just resets all channels).
		 fixed tyCoInt() to be more robust about interrupts on channels
		   for which devices haven't been created.
01e,23oct87,ecs  delinted
01d,06jan87,dnw  changed to use sysClkFreq from sysLib instead of SYS_CLK_FREQ.
01c,21dec86,dnw  changed to not get include files from default directories.
01b,05sep86,jlf  fixed bug in tyCoStartup found by avant-garde.
01a,06jul86,jlf  written, by modifying 01c of the is20 version.
*/

/*
DESCRIPTION
This is the driver for the Ironics IV-1624A on-board serial ports.
It uses the SCCs only in asynchronous mode.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv () to
initialize the driver, and tyCoDevCreate to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:

    tyCoDrv ()

This routine should be called exactly once, before any reads, writes, or
tyCoDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.

.CS

    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device *
	int channel;	/* Physical channel for this device (0 or 1) *
	int rdBufSize;	/* Read buffer size, in bytes *
	int wrtBufSize;	/* Write buffer size, in bytes *

.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:

.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);

.CE

Each port to be used should have exactly one device associated with it,
by calling this routine.
There are eight channels on the IV-1624A board, channels 0-7.

IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.
Any baud rate between 0 and 38.4 kbaud is available.

*/

#include "UniWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "tyLib.h"
#include "iv1624.h"
#include "z8530.h"
#include "config.h"
#include "iv68k.h"

IMPORT int sysClkFreq;		/* external clock rate (10 or 12 MHz) */

#define DEFAULT_BAUD	9600

typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* true if this device has really been created */
    char *cr;		/* control reg I/O address */
    char *dr;		/* data port I/O address */
    char intVec;	/* interrupt vector address */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyCoDv [N_SIO_CHANNELS] =	/* device descriptors */
    {
    {{{{NULL}}}, FALSE, SCC0BASE, SCC0BASE+2, INT_VEC_SIO0},
    {{{{NULL}}}, FALSE, SCC1BASE, SCC1BASE+2, INT_VEC_SIO0},
    {{{{NULL}}}, FALSE, SCC2BASE, SCC2BASE+2, INT_VEC_SIO1},
    {{{{NULL}}}, FALSE, SCC3BASE, SCC3BASE+2, INT_VEC_SIO1},
    {{{{NULL}}}, FALSE, SCC4BASE, SCC4BASE+2, INT_VEC_SIO2},
    {{{{NULL}}}, FALSE, SCC5BASE, SCC5BASE+2, INT_VEC_SIO2},
    {{{{NULL}}}, FALSE, SCC6BASE, SCC6BASE+2, INT_VEC_SIO3},
    {{{{NULL}}}, FALSE, SCC7BASE, SCC7BASE+2, INT_VEC_SIO3},
    };

LOCAL int tyCoDrvNum;		/* driver number assigned to this driver */

/* forward declarations */

LOCAL VOID tyCoStartup ();
LOCAL int tyCoOpen ();
LOCAL int tyCoRead ();
LOCAL int tyCoWrite ();
LOCAL STATUS tyCoIoctl ();
LOCAL VOID tyCoInt ();


/*******************************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the serial driver, sets up interrupt vectors,
* and performs hardware initialization of the serial ports.
*/

STATUS tyCoDrv ()

    {
    /* check if driver already installed */
     
    if (tyCoDrvNum > 0)
	return (OK);
		    
    /* the IV-1624A board uses two interrupts levels, each shared between
     * two SCC chips. In order to eliminate confusion, we just let
     * the SCC chips generate their own vectors. Therefore, we need to
     * connect four vectors, one for each SCC chip. */

    (void) intConnect (INUM_TO_IVEC (INT_VEC_SIO0), tyCoInt, 0);
    (void) intConnect (INUM_TO_IVEC (INT_VEC_SIO1), tyCoInt, 1);
    (void) intConnect (INUM_TO_IVEC (INT_VEC_SIO2), tyCoInt, 2);
    (void) intConnect (INUM_TO_IVEC (INT_VEC_SIO3), tyCoInt, 3);

    tyCoHrdInit ();

    tyCoDrvNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen,
				(FUNCPTR) NULL, tyCoRead, tyCoWrite, tyCoIoctl);

    return (tyCoDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the serial ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*
* RETURNS: OK or ERROR
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0 - 7) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

    {
    if (tyCoDrvNum <= 0)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}
				     
    /* if this device already exists, don't create it */

    if (tyCoDv [channel].created)
	return (ERROR);

    /* initialize the ty descriptor */

    if (tyDevInit (&tyCoDv [channel].tyDev,
		    rdBufSize, wrtBufSize, tyCoStartup) != OK)
	{
	return (ERROR);
	}

    tyCoInitChannel (channel);	/* initialize the channel hardware */

    /* Mark the device as having been created, and add the device to
     * the I/O system.
     */

    tyCoDv [channel].created = TRUE;

    return (iosDevAdd (&tyCoDv [channel].tyDev.devHdr, name, tyCoDrvNum));
    }
/*******************************************************************************
*
* tyCoHrdInit - initialize the usart
*
* This routine initializes the usart for the UniWorks environment.
* This routine must be called in supervisor mode.
*/

LOCAL VOID tyCoHrdInit ()

    {
    char curMask;
    int i;
    int oldlevel;		/* current interrupt level mask */

    oldlevel = intLock ();	/* disable interrupts during init */

    for (i = 0; i < N_SIO_CHANNELS; i++)
	tyCoResetChannel (i);

    /* enable the interrupts for the serial chips, on the IVHAN chip. */

    curMask = *IVH_LRQM;
    curMask |= (1 << INT_SIOIRQ01);
    curMask |= (1 << INT_SIOIRQ23);
    *IVH_LRQM = curMask;

    intUnlock (oldlevel);
    } 
/*******************************************************************************
*
* tyCoResetChannel - reset a single channel
*/

LOCAL VOID tyCoResetChannel (channel)
    int channel;

    {
    FAST char *cr = tyCoDv [channel].cr;	/* SCC control reg adr */
    int delay;

    *cr = SCC_WR0_SEL_WR9;	/* write register 9 - master int ctrl */

    *cr = ((channel % 2) == 0) ?
	SCC_WR9_CH_A_RST :	/* reset channel A */
	SCC_WR9_CH_B_RST;	/* reset channel B */

    for (delay = 0; delay < 1000; delay++);	/* spin wheels for a moment */
    }
/*******************************************************************************
*
* tyCoInitChannel - initialize a single channel
*/

LOCAL VOID tyCoInitChannel (channel)
    int channel;

    {
    FAST char *cr = tyCoDv [channel].cr;	/* SCC control reg adr */
    int baudConstant;
    int oldlevel;		/* current interrupt level mask */

    oldlevel = intLock ();	/* disable interrupts during init */

    tyCoResetChannel (channel);	/* reset the channel */

    /* initialize registers */

    *cr = SCC_WR0_SEL_WR4;		/* write reg 4 - misc parms and modes */
    *cr = SCC_WR4_1_STOP | SCC_WR4_16_CLOCK;

    *cr = SCC_WR0_SEL_WR3;		/* write reg 3 - receive parms */
    *cr = SCC_WR3_RX_8_BITS;		/* disabled */

    *cr = SCC_WR0_SEL_WR5;		/* tx parms */
    *cr = SCC_WR5_TX_8_BITS;		/* disabled */

    *cr = SCC_WR0_SEL_WR6;
    *cr = 0;				/* clear sync data */

    *cr = SCC_WR0_SEL_WR7;
    *cr = 0;				/* transmit */

    *cr = SCC_WR0_SEL_WR9;		/* master interrupt control */
    *cr = SCC_WR9_MIE;			/* enable interrupts */

    *cr = SCC_WR0_SEL_WR10;		/* misc tx/rx control */
    *cr = 0;				/* clear sync, loop, poll */

    *cr = SCC_WR0_SEL_WR11;		/* clock mode */
    *cr = SCC_WR11_RX_BR_GEN | SCC_WR11_TX_BR_GEN | SCC_WR11_TRXC_OI |
	  SCC_WR11_OUT_BR_GEN;

    /* calculate the baud rate constant, from the input clock freq.
       assumes that the input clock is the system clock / 4, and
       the divide by 16 bit is set (done in tyCoHrdInit). */

    baudConstant = ((sysClkFreq / 4) / (DEFAULT_BAUD * 32)) - 2;

    *cr = SCC_WR0_SEL_WR12;		/* LSB of baud constant */
    *cr = baudConstant & 0xff;		/* write LSB */
    *cr = SCC_WR0_SEL_WR13;		/* MSB of baud constant */
    *cr = (baudConstant & 0xff00) >> 8;	/* write MSB */

    *cr = SCC_WR0_SEL_WR14;		/* misc control bits */
    *cr = SCC_WR14_BR_SRC;

    *cr = SCC_WR0_SEL_WR2;		/* interrupt vector */
    *cr = tyCoDv [channel].intVec;

    *cr = SCC_WR0_SEL_WR3;		/* write reg 3 - receive parms */
    *cr = SCC_WR3_RX_8_BITS | SCC_WR3_RX_EN;

    *cr = SCC_WR0_SEL_WR5;		/* tx parms */
    *cr = SCC_WR5_TX_EN | SCC_WR5_TX_8_BITS;

    *cr = SCC_WR0_RST_TX_CRC;		/* reset crc */

    *cr = SCC_WR0_SEL_WR14;		/* misc control bits */
    *cr = SCC_WR14_BR_SRC | SCC_WR14_BR_EN;

    *cr = SCC_WR0_SEL_WR15;		/* external/status interrupt cntrl */
    *cr = 0;

    *cr = SCC_WR0_RST_INT;		/* reset ext interrupts */
    *cr = SCC_WR0_RST_INT;		/* twice, because Ironics says so */

    *cr = SCC_WR0_SEL_WR1;		/* int and xfer mode */
    *cr = SCC_WR1_INT_ALL_RX | SCC_WR1_TX_INT_EN;

    intUnlock (oldlevel);
    }

/*******************************************************************************
*
* tyCoOpen - open file to usart
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char *name;
    int mode;

    {
    return ((int) pTyCoDv);
    }
/*******************************************************************************
*
* tyCoRead - task level read routine
*
* This routine fields all read calls to the IV-1624A usart.
*/

LOCAL int tyCoRead (pTyCoDv, buffer, maxbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int maxbytes;

    {
    return (tyRead ((TY_DEV_ID) pTyCoDv, buffer, maxbytes));
    }
/*******************************************************************************
*
* tyCoWrite - task level write routine
*
* This routine fields all write calls to the IV-1624A usart.
*/

LOCAL int tyCoWrite (pTyCoDv, buffer, nbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int nbytes;

    {
    return (tyWrite ((TY_DEV_ID) pTyCoDv, buffer, nbytes));
    }
/*******************************************************************************
*
* tyCoIoctl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to tyIoctl.
*/

LOCAL STATUS tyCoIoctl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

    {
    int baudConstant;
    STATUS status;
    char *cr;

    switch (request)
	{
	case FIOBAUDRATE:

	    /* calculate the baud rate constant, from the input clock freq.
	       assumes that the input clock is the system clock / 4, and
	       the divide by 16 bit is set (done in tyCoHrdInit). */

	    baudConstant = ((sysClkFreq / 4) / (arg * 32)) - 2;

	    cr = pTyCoDv->cr;
	    *cr = SCC_WR0_SEL_WR12;		/* LSB of baud constant */
	    *cr = baudConstant & 0xff;		/* write LSB */
	    *cr = SCC_WR0_SEL_WR13;		/* MSB of baud constant */
	    *cr = (baudConstant & 0xff00) >> 8;	/* write MSB */
	    status = OK;
	    break;

	default:
	    status = tyIoctl ((TY_DEV_ID) pTyCoDv, request, arg);
	    break;
	}
    return (status);
    }

/*******************************************************************************
*
* tyCoInt - interrupt level processing
*
* This routine handles an interrupt from the one of the SCCs.
* We determine from the paramter which SCC interrupted us, then look at
* the code to find out which channel, and what kind of interrupt.
*/

LOCAL VOID tyCoInt (sccNum)
    int sccNum;

    {
    FAST char *cr;
    FAST int channel;
    FAST char intStatus;
    FAST TY_CO_DEV *pTyCoDv;
    char outChar;

    /* We need to find out which channel interrupted.  We need to read
     * the B channel of the interrupting SCC to find out which channel
     * really interrupted.
     */

    channel = (sccNum * 2) + 1;
    pTyCoDv = &tyCoDv [channel];
    cr = pTyCoDv->cr;

    *cr = SCC_WR0_SEL_WR2;		/* read reg 2 */
    intStatus = *cr;

    if ((intStatus & 0x08) != 0)
	{
	--channel;	/* the A channel interrupted */
	--pTyCoDv;
	cr = pTyCoDv->cr;
	}

    switch (intStatus & 0x06)
	{
	case 0x00:		/* Tx Buffer Empty */
	    if (pTyCoDv->created && (tyITx (&pTyCoDv->tyDev, &outChar) == OK))
		*(pTyCoDv->dr) = outChar;
	    else
		{
		/* no more chars to xmit now.  reset the tx int,
		 * so the SCC doesn't keep interrupting us. */

		*cr = SCC_WR0_RST_TX_INT;
		}
	    break;

	case 0x04:		/* RxChar Avail */
	    if (pTyCoDv->created)
		(void) tyIRd (&pTyCoDv->tyDev, *(pTyCoDv->dr));
	    break;

	case 0x02:		/* External Status Change */
	case 0x06:		/* Special receive condition */
	    break;		/* ignore */

	}

    *cr = SCC_WR0_RST_HI_IUS;	/* reset the interrupt */
    }
/*******************************************************************************
*
* tyCoStartup - transmitter startup routine
*
* Call interrupt level character output routine.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    TY_CO_DEV *pTyCoDv;		/* ty device to start up */

    {
    char outChar;

    if (tyITx (&pTyCoDv->tyDev, &outChar) == OK)
	*(pTyCoDv->dr) = outChar;
    }
