/* vxALib.s - VxWorks/VRTX kernel assembly library */

	.data
	.asciz	"Copyright 1984-1988, Wind River Systems, Inc."

/*
modification history 
--------------------
*/

/*
DESCRIPTION
This library contains support for VRTX facilities.

C callable interface routines are provided for most VRTX system calls.  
Each interface routine gets the parameters passed to it in a normal 
C fashion (on the stack) and puts them in the proper registers where 
VRTX expects to find them, then traps to VRTX.

RETURNING VRTX STATUS CODES
If a VRTX system call returns an error status 
(non-zero in d0 on return from VRTX),
the interface routines set the task error status to the VRTX error code
returned, then return the value ERROR to the calling routine.  That routine
can find out the error code that VRTX returned by calling errnoGet ().
If all went well with VRTX, the interface routines return OK.

TASK CREATION
No direct access is provided to the VRTX task creation system calls.
Instead all VxWorks tasks should be created by the taskSpawn (2) or
taskCreate (2) calls in taskLib (1).

CHARACTER I/O
VxWorks does not use or support the character I/O facility
provided by the VRTX system calls SC_GETC, SC_PUTC, SC_WAITC,
UI_RXCHR, and UI_TXRDY.  Instead, VxWorks supplies a complete
device independent I/O system and a set of device drivers.

NOTE
Many routines in this module "link" and "unlk" the C frame pointer
(a6) although they don't use it in any way.  This is only for the benefit of
the stack trace facility, to allow it to properly trace tasks executing within
these routines.
*/

#define ASMLANGUAGE
#include "UniWorks.h"
#include "asm.h"

	.text
	.even

	/* internals */

	.globl _intEnt
	.globl _intExit
	.globl _vxAccept
	.globl _vxGblock
	.globl _vxGtime
	.globl _vxLock
	.globl _vxPcreate
	.globl _vxPend
	.globl _vxPextend
	.globl _vxPost
	.globl _vxQaccept
	.globl _vxQcreate
	.globl _vxQinquiry
	.globl _vxQpend
	.globl _vxQpost
	.globl _vxRblock
	.globl _vxStime
	.globl _vxTXcreate
	.globl _vxTaskEntry
	.globl _vxTcreateHook
	.globl _vxTdelay
	.globl _vxTdelete
	.globl _vxTimer
	.globl _vxTinquiry
	.globl _vxTpriority
	.globl _vxTresume
	.globl _vxTslice
	.globl _vxTsuspend
	.globl _vxTswitchHook
	.globl _vxUnlock
	.globl _vxVrtxGo
	.globl _vxVrtxInit

#if ((CPU == MC68020) || (CPU == MC68030))
	.globl _vxFclear
	.globl _vxFcreate
	.globl _vxFdelete
	.globl _vxFinquiry
	.globl _vxFpend
	.globl _vxFpost
	.globl _vxQEcreate
	.globl _vxQjam
	.globl _vxScreate
	.globl _vxSdelete
	.globl _vxSinquiry
	.globl _vxSpend
	.globl _vxSpost
#endif ((CPU == MC68020) || (CPU == MC68030))

	/* externals */

	.globl _taskSwitchTable		/* table of task switch routines */
	.globl _taskIdCurrent		/* for use with taskIdSelf () */

#if	FALSE
/* a6 offsets to arguments */

#define ARG1	8
#define ARG1W	10
#define ARG2	12
#define ARG2W	14
#define ARG3	16
#define ARG3W	18
#define ARG4	20
#define ARG5	24
#define ARG6	28
#define ARG7	32
#endif	FALSE

	.text
	.even

/*******************************************************************************
*
* vxVrtxInit - initialize VRTX
*
* This routine is a C interface to the VRTX system call VRTX_INIT.
*
* RETURNS: OK or VRTX error code
*
* SEE ALSO: VRTX User's Guide

* int vxVrtxInit ()

*/

_vxVrtxInit:
	moveq	#0x30,d0	/* VRTX_INIT */
	jsr	_sysKernelTrap	/* system trap */
	rts
/*******************************************************************************
*
* vxVrtxGo - start VRTX multi-tasking
*
* This routine is a C interface to the VRTX system call VRTX_GO.
*
* RETURNS: Doesn't!
*
* SEE ALSO: VRTX User's Guide

* int vxVrtxGo ()

*/

_vxVrtxGo:
	movew	#0x3000,sr	/* enable interrupts */
	moveq	#0x31,d0	/* VRTX_GO */
	jsr	_sysKernelTrap	/* system trap */

/*******************************************************************************
*
* vxTXcreate - task create with stack address and tcb extension
*
* This routine is the low-level access to the VRTX task creation facility.
* It should NOT be called directly.  
* Normal VxWorks tasks should be created by the "spawn" routines in vxLib.
*
* This routine provides a C callable interface to the VRTX SC_TCREATE
* system call to create a task, with the UniWorks task creation extensions 
* that allow task stack addresses and a tcb extension to be specified
* on task creation.
* These extensions are made possible by the added UniWorks task creation
* hook, vxTcreateHook, which MUST be specified in the VRTX configuration table
* in usrConfig (1).
*
* Note that this routine allows specification of both the bottom of
* stack and the initial stack pointer.  This allows the caller
* to push initial arguments on the stack for the new task 
* (as in the case of the "spawn" routines).
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet (2) will be called with one of the following values:
*	1 = id number already assigned
*	2 = no TCBs available
*
* NOMANUAL

* vxTXcreate (priority, id, mode, entryAddress,
*	      bottomSP, initialSp, tcbExtension)
*     TINY priority;		/* priority of the new task (0 - 255)  *
*     TINY id;			/* id number of the new task (0 - 255) *
*     int mode;			/* (0)user or (1)system mode           *
*     FUNCPTR entryAddress;	/* starting address of the new task    *
*     char *bottomSP;		/* bottom of the stack                 *
*     char *initialSP;		/* initial stack pointer               *
*     char *tcbExtension;	/* address of new task's tcb extension *

*/

_vxTXcreate:
	link	a6,#0
	moveml	d2-d3/a2-a3,a7@-	/* save regs */

	movel	a6@(ARG1),d1		/* get priority */
	movel	a6@(ARG2),d2		/* get id */
	movel	a6@(ARG3),d3		/* get mode */
	movel	a6@(ARG4),a0		/* get entry address */
	movel	a6@(ARG5),a1		/* Tcreate hook expects bot SP in a1 */
	movel	a6@(ARG6),a2		/* Tcreate hook expects init SP in a2 */
	movel	a6@(ARG7),a3		/* Tcreate hook expects tcbx in a3 */
	moveq	#0x0,d0			/* SC_TCREATE */
	jsr	_sysKernelTrap		/* system trap */

	moveml	a7@+,d2-d3/a2-a3	/* restore regs */
	bra	return

/*******************************************************************************
*
* vxTdelete - vx task delete
*
* This routine serves as a C interface to VRTX system call SC_TDELETE.
* THIS SHOULD NEVER BE CALLED.  USE taskDelete INSTEAD!
* The difference being, taskDelete reclaims the stack allocated for a task
* and this routine does not.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	1 = task id error (no task with specified id number)
*
* NOMANUAL

* int vxTdelete (id)
*     int id;		/* task id number *

*/

_vxTdelete:
	link	a6,#0
	movel	a6@(ARG1),d1		/* get id */
	moveq	#0x1,d0			/* SC_TDELETE */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vxTsuspend - suspend task
*
* This routine serves as a C interface to VRTX system call SC_TSUSPEND.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	1 = task id error (no task with specified id number)
*
* SEE ALSO: VRTX User's Guide

* int vxTsuspend (id)
*     int id;		/* task id number *

*/

_vxTsuspend:
	link	a6,#0
	movel	a6@(ARG1),d1		/* get id */
	moveq	#0x2,d0			/* SC_TSUSPEND */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vxTresume - resume task
*
* This routine serves as a C interface to VRTX system call SC_TRESUME.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	1 = task id error (no task with specified id number)
*
* SEE ALSO: VRTX User's Guide

* int vxTresume (id)
*     int id;		/* task id number *

*/

_vxTresume:
	link	a6,#0
	movel	a6@(ARG1),d1		/* get id */
	moveq	#0x3,d0			/* SC_TRESUME */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vxTpriority - task priority change
*
* This routine serves as a C interface to VRTX system call SC_TPRIORITY.
* It changes the priority of an existing task.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	1 = task id error (no task with specified id number)
*
* SEE ALSO: VRTX User's Guide

* int vxTpriority (id, priority)
*     int id;		/* task id number    *
*     int priority;	/* new task priority *

*/

_vxTpriority:
	link	a6,#0
	movel	d2,a7@-			/* save reg d2 */

	movel	a6@(ARG1),d1		/* get id */
	movel	a6@(ARG2),d2		/* get priority */
	moveq	#0x4,d0			/* SC_TPRIORITY */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg d2 */
	bra	return

/*******************************************************************************
*
* vxTinquiry - task inquiry
*
* This routine serves as a C interface to VRTX system call SC_TINQUIRY.
* It is used to obtain priority and status information about tasks.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	1 = task id error
*
* SEE ALSO: VRTX User's Guide

* int vxTinquiry (id, tid, priority, tstatus, tcbAddr)
*     int id;		/* task id, or 0 to inquire about calling task *
*     int *tid;		/* where to return the task's id number        *
*     int *priority;	/* where to return the task's priority         *
*     int *tstatus;	/* where to return the task's status           *
*     VX_TCB **tcbAddr;	/* where to return pointer to the task's TCB   *

*/

_vxTinquiry:
	link	a6,#0
	moveml	d2-d3,a7@-		/* save regs */

	movel	a6@(ARG1),d1		/* get id */
	moveq	#0x5,d0			/* SC_TINQIRY */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG2),a1		/* get pointer to id */
	movel	d1,a1@			/* return id */
	movel	a6@(ARG3),a1		/* get pointer to priority */
	movel	d2,a1@			/* return priority */
	movel	a6@(ARG4),a1		/* get pointer to tstatus */
	movel	d3,a1@			/* return tstatus */
	movel	a6@(ARG5),a1		/* get pointer to TCBaddress */
	movel	a0,a1@			/* return TCBaddress */

	moveml	a7@+,d2-d3		/* restore regs */
	bra	return

/*******************************************************************************
*
* vxLock - disable task rescheduling
*
* This routine serves as a C interface to VRTX system call SC_LOCK.
* It prevents task rescheduling until vxUnlock (2) is called.  Interrupts
* will continue to be serviced.
*
* SEE ALSO: intLock(2), vxUnlock(2), VRTX User's Guide

* VOID vxLock ()

*/

_vxLock:
	link	a6,#0
	moveq	#0x20,d0		/* SC_LOCK */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vxUnlock - enable task rescheduling
*
* This routine serves as a C interface to VRTX system call SC_UNLOCK.
*
* SEE ALSO: vxLock(2), VRTX User's Guide

* VOID vxUnlock ()

*/

_vxUnlock:
	link	a6,#0
	moveq	#0x21,d0		/* SC_UNLOCK */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vxGblock - get memory block
*
* This routine serves as a C interface to VRTX system call SC_GBLOCK.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	 3 = no memory blocks available
*	14 = no such partition.
*
* SEE ALSO: VRTX User's Guide

* int vxGblock (id, address)
*     short id;	        /* partition id number                    *
*     char *address;	/* receives the logical adrs of the block *

*/

_vxGblock:
	link	a6,#0
	movew	a6@(ARG1W),d1		/* get id */
	moveq	#0x6,d0			/* SC_GBLOCK */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG2),a1		/* get pointer to address */
	movel	a0,a1@			/* return address */
	bra	return

/*******************************************************************************
*
* vxRblock - release memory block
*
* This routine serves as a C interface to VRTX system call SC_RBLOCK.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	 4 = not a memory block
*	14 = no such partition
*
* SEE ALSO: VRTX User's Guide

* int vxRblock (id, address)
*     short id;		/* partition id number          *
*     char *address;	/* logical address of the block *

*/

_vxRblock:
	link	a6,#0
	movew	a6@(ARG1W),d1		/* get id */
	movel	a6@(ARG2),a0		/* get address */
	moveq	#0x7,d0			/* SC_RBLOCK */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vxPcreate - create memory partition
*
* This routine serves as a C interface to VRTX system call SC_PCREATE.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	 3 = no memory available
*	14 = partition id number already assigned
*
* SEE ALSO: VRTX User's Guide

* int vxPcreate (id, psize, bsize, paddress)
*     int id;		/* partition id number             *
*     int psize;	/* partition size in bytes         *
*     int bsize;	/* partition's block size in bytes *
*     char *paddress;	/* partition starting address      *

*/

_vxPcreate:
	link	a6,#0
	moveml	d2-d3,a7@-		/* save regs */

	movel	a6@(ARG1),d1		/* get id */
	movel	a6@(ARG2),d2		/* get partition size */
	movel	a6@(ARG3),d3		/* get block size */
	movel	a6@(ARG4),a0		/* get partition address */
	moveq	#0x22,d0		/* SC_PCREATE */
	jsr	_sysKernelTrap		/* system trap */

	moveml	a7@+,d2-d3		/* restore regs */
	bra	return

/*******************************************************************************
*
* vxPextend - extend memory partition
*
* This routine serves as a C interface to VRTX system call SC_PEXTEND.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	 3 = no memory available
*	14 = no such partition id number 
*
* SEE ALSO: VRTX User's Guide

* int vxPextend (id, esize, eaddress)
*     short id;		/* partition id number        *
*     int esize;	/* extension size in bytes    *
*     char *eaddress;	/* extension starting address *

*/

_vxPextend:
	link	a6,#0
	movel	d2,a7@-			/* save reg d2 */

	movew	a6@(ARG1W),d1		/* get id */
	movel	a6@(ARG2),d2		/* get extension size */
	movel	a6@(ARG3),a0		/* get extension address */
	moveq	#0x23,d0		/* SC_PEXTEND */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg d2 */
	bra	return

/*******************************************************************************
*
* vxPost - post a message
*
* This routine serves as a C interface to VRTX system call SC_POST.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with one of the following values:
*	5 = mailbox already in use
*	6 = zero message
*
* SEE ALSO: VRTX User's Guide

* int vxPost (pMailBox, message)
*     int *pMailBox;	/* pointer to mail box	*
*     int message;	/* message to post	*

*/

_vxPost:
	link	a6,#0
	movel	a6@(ARG1),a0		/* get mailbox address */
	movel	a6@(ARG2),d1		/* get message */
	moveq	#0x8,d0			/* SC_POST */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vxPend - pend for message
*
* This routine serves as a C interface to VRTX system call SC_PEND.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	10 = timeout
*
* SEE ALSO: VRTX User's Guide

* int vxPend (pMailBox, timeout, pMsg)
*     int *pMailBox;		/* pointer to mail box			*
*     unsigned timeout;		/* # of ticks before returning timeout	*
*     int *pMsg;		/* pointer where to return msg		*

*/

_vxPend:
	link	a6,#0
	movel	a6@(ARG1),a0		/* get mail box address */
	movel	a6@(ARG2),d1		/* get timeout */
	moveq	#0x9,d0			/* SC_PEND */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG3),a0		/* get pointer to msg */
	movel	d1,a0@			/* return message */
	bra	return

/*******************************************************************************
*
* vxAccept - accept a message
*
* This routine serves as a C interface to VRTX system call SC_ACCEPT.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	11 = no message present
*
* SEE ALSO: VRTX User's Guide

* int vxAccept (pMailBox, pMsg)
*     int *pMailBox;	/* pointer to mail box		*
*     int *pMsg;	/* pointer where to return msg	*

*/

_vxAccept:
	link	a6,#0
	movel	a6@(ARG1),a0		/* get mailbox address */
	moveq	#0x25,d0		/* SC_ACCEPT */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG2),a0		/* get pointer to message */
	movel	d1,a0@			/* return message */
	bra	return

/*******************************************************************************
*
* vxQpost - post message to queue
*
* This routine serves as a C interface to VRTX system call SC_QPOST.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with one of the following values:
*	12 = no such queue
*	13 = queue full
*
* SEE ALSO: VRTX User's Guide

* int vxQpost (qid, message)
*     short qid;	/* queue id number *
*     int message;	/* message to post *

*/

_vxQpost:
	link	a6,#0
	movel	d2,a7@-			/* save reg d2 */

	movew	a6@(ARG1W),d1		/* get qid */
	movel	a6@(ARG2),d2		/* get message */
	moveq	#0x26,d0		/* SC_QPOST */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg d2 */
	bra	return

#if ((CPU == MC68020) || (CPU == MC68030))
/*******************************************************************************
*
* vxQjam - jam message to beginning of queue
*
* This routine serves as a C interface to VRTX32 system call SC_QJAM.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with one of the following values:
*	12 = no such queue
*	13 = queue full
*
* SEE ALSO: VRTX32 User's Guide

* int vxQjam (qid, message)
*     short qid;	/* queue id number *
*     int message;	/* message to jam  *

*/

_vxQjam:
	link	a6,#0
	movel	d2,a7@-			/* save reg d2 */

	movew	a6@(ARG1W),d1		/* get qid */
	movel	a6@(ARG2),d2		/* get message */
	moveq	#0x1e,d0		/* SC_QJAM */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg d2 */
	bra	return
#endif ((CPU == MC68020) || (CPU == MC68030))
/*******************************************************************************
*
* vxQpend - pend for message from queue
*
* This routine serves as a C interface to VRTX system call SC_QPEND.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	10 = time-out
*	12 = no such queue
*
* SEE ALSO: VRTX User's Guide

* int vxQpend (qid, timeout, pMsg)
*     short qid;		/* queue id number			*
*     unsigned timeout;		/* # of ticks before returning timeout	*
*     int *pMsg;		/* pointer where to return msg		*

*/

_vxQpend:
	link	a6,#0
	movel	d2,a7@-			/* save reg d2 */

	movew	a6@(ARG1W),d1		/* get qid */
	movel	a6@(ARG2),d2		/* get timeout */
	moveq	#0x27,d0		/* SC_QPEND */
	jsr	_sysKernelTrap		/* system trap */

	cmpl	#0xa,d0			/* did it time out? */
	beq	timedout		/* then there's no message to return */
	movel	a6@(ARG3),a0		/* get pointer to msg */
	movel	d2,a0@			/* return message */
timedout:

	movel	a7@+,d2			/* restore reg d2 */
	bra	return

/*******************************************************************************
*
* vxQaccept - accept message from queue
*
* This routine serves as a C interface to VRTX system call SC_QACCEPT.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	11 = no message present
*	12 = no such queue
*
* SEE ALSO: VRTX User's Guide

* int vxQaccept (qid, pMsg)
*     short qid;	/* queue id number		*
*     int *pMsg;	/* pointer where to return msg	*

*/

_vxQaccept:
	link	a6,#0
	movel	d2,a7@-			/* save reg d2 */

	movew	a6@(ARG1W),d1		/* get qid */
	moveq	#0x28,d0		/* SC_QACCEPT */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG2),a0		/* get pointer to msg */
	movel	d2,a0@			/* return message */

	movel	a7@+,d2			/* restore reg d2 */
	bra	return

/*******************************************************************************
*
* vxQcreate - create message queue
*
* This routine serves as a C interface to VRTX system call SC_QCREATE.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	 3 = no memory available
*	12 = queue id number already assigned
*
* SEE ALSO: VRTX User's Guide

* int vxQcreate (qid, count)
*     short qid;	/* queue id number                *
*     short count;	/* number of entries in the queue *

*/

_vxQcreate:
	link	a6,#0
	movel	d2,a7@-			/* save reg d2 */

	movew	a6@(ARG1W),d1		/* get qid */
	movew	a6@(ARG2W),d2		/* get count */
	moveq	#0x29,d0		/* SC_QCREATE */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg d2 */
	bra	return

#if ((CPU == MC68020) || (CPU == MC68030))
/*******************************************************************************
*
* vxQEcreate - create FIFO message queue
*
* This routine serves as a C interface to VRTX32 system call SC_QECREATE.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	 3 = no memory available
*	12 = queue id number already assigned
*
* SEE ALSO: VRTX32 User's Guide

* int vxQEcreate (qid, count, pendOrder)
*     short qid;        /* queue id number                 *
*     short count;      /* number of entries in the queue  *
*     int pendOrder;    /* 0: pend tasks in priority order *
*                       /* 1: pend tasks in FIFO order     *

*/

_vxQEcreate:
	link	a6,#0
	moveml	d2-d3,a7@-		/* save regs */

	movew	a6@(ARG1W),d1		/* get qid */
	movew	a6@(ARG2W),d2		/* get count */
	movel	a6@(ARG3),d3		/* get count */
	moveq	#0x1f,d0		/* SC_QECREATE */
	jsr	_sysKernelTrap		/* system trap */

	moveml	a7@+,d2-d3		/* restore regs */
	bra	return
#endif ((CPU == MC68020) || (CPU == MC68030))
/*******************************************************************************
*
* vxQinquiry - message queue status inquiry
*
* This routine serves as a C interface to VRTX system call SC_QINQUIRY.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	12 = no such queue
*
* SEE ALSO: VRTX User's Guide

* int vxQinquiry (qid, pMsg, pNum)
*     short qid;	/* queue id number			*
*     int *pMsg;	/* ptr where to return msg		*
*     int *pNum;	/* ptr where to return count of msgs	*

*/

_vxQinquiry:
	link	a6,#0
	moveml	d2-d3,a7@-		/* save regs */

	movew	a6@(ARG1W),d1		/* get qid */
	moveq	#0x2a,d0		/* SC_QINQUIRY */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG2),a0		/* get pointer to msg */
	movel	d2,a0@			/* return message */

	movel	a6@(ARG3),a0		/* get pointer to count */
	andl	#0xffff,d3		/* make sure upper byte is clear */
	movel	d3,a0@			/* return count */

	moveml	a7@+,d2-d3		/* restore regs */
	bra	return

#if ((CPU == MC68020) || (CPU == MC68030))
/*******************************************************************************
*
* vxFcreate - create event flag group
*
* This routine serves as a C interface to VRTX32 system call SC_FCREATE.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	48 = no control blocks available
*
* SEE ALSO: VRTX32 User's Guide

* int vxFcreate (pGID)
*     UINT *pGID;	/* where to return event flag group ID number *

*/

_vxFcreate:
	link	a6,#0

	moveq	#0x17,d0		/* SC_FCREATE */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG1),a0		/* get pointer to group ID number */
	movel	d1,a0@			/* return group ID number */

	bra	return

/*******************************************************************************
*
* vxFdelete - delete event flag group
*
* This routine serves as a C interface to VRTX32 system call SC_FDELETE.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	49 = event flag group id error
*	50 = tasks pending on event flag group
*
* SEE ALSO: VRTX32 User's Guide

* int vxFdelete (efgid, forceDelete)
*     UINT efgid;	/* event flag group id number             *
*     int forceDelete;	/* 0: delete efg only if no tasks pending *
*               	/* 1: delete efg, ready all pending tasks *

*/

_vxFdelete:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get efgid */
	movel	a6@(ARG2),d2		/* get forceDelete */
	moveq	#0x18,d0		/* SC_FDELETE */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg */
	bra	return

/*******************************************************************************
*
* vxFpost - post event to event flag group
*
* This routine serves as a C interface to VRTX32 system call SC_FPOST.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with one of the following values:
*	49 = event flag group ID error
*	52 = event flag already set
*
* SEE ALSO: VRTX32 User's Guide

* int vxFpost (efgid, eventFlags)
*     UINT efgid;	/* event flag group ID number              *
*     UINT eventFlags;	/* mask of event flags in event flag group *

*/

_vxFpost:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get efgid */
	movel	a6@(ARG2),d2		/* get eventFlags */
	moveq	#0x1a,d0		/* SC_FPOST */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg */
	bra	return

/*******************************************************************************
*
* vxFpend - pend on event flag group
*
* This routine serves as a C interface to VRTX32 system call SC_FPEND.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	10 = timeout
*	49 = event flag group ID error
*	51 = event flag group is deleted
*
* SEE ALSO: VRTX32 User's Guide

* int vxFpend (efgid, timeout, eventFlags, maskOpt, pEfg)
*     UINT efgid;     /* event flag group ID number                      *
*     UINT timeout;   /* number of ticks before returning timeout        *
*     UINT eventFlags;/* mask of event flags in event flag group         *
*     int maskOpt;    /* 0: pend on OR mask, any spec'd flag readies task*
*                     /* 1: pend on AND mask, all spec'd flags are true  *
*     UINT *pEfg;     /* pointer to where to return event flag group     *

*/

_vxFpend:
	link	a6,#0
	moveml	d2-d4,a7@-		/* save regs */

	movel	a6@(ARG1),d1		/* get efgid */
	movel	a6@(ARG2),d2		/* get timeout */
	movel	a6@(ARG3),d3		/* get eventFlags */
	movel	a6@(ARG4),d4		/* get maskOpt */
	moveq	#0x19,d0		/* SC_FPEND */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG5),a0		/* get pEfg */
	movel	d2,a0@			/* return event flag group */

	moveml	a7@+,d2-d4		/* restore regs */
	bra	return

/*******************************************************************************
*
* vxFclear - clear events in event flag group
*
* This routine serves as a C interface to VRTX32 system call SC_FCLEAR.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	49 = event flag group ID error
*
* SEE ALSO: VRTX32 User's Guide

* int vxFclear (efgid, eventFlags, pEfg)
*     UINT efgid;	/* event flag group ID number              *
*     UINT eventFlags;	/* mask of event flags in event flag group *
*     UINT *pEfg;	/* return old event flag group here        *

*/

_vxFclear:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get efgid */
	movel	a6@(ARG2),d2		/* get eventFlags */
	moveq	#0x1b,d0		/* SC_FCLEAR */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG3),a0		/* get pEfg */
	movel	d2,a0@			/* return event flag group */

	movel	a7@+,d2			/* restore reg */
	bra	return

/*******************************************************************************
*
* vxFinquiry - event flag group inquiry
*
* This routine serves as a C interface to VRTX32 system call SC_FINQUIRY.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	49 = event flag group ID error
*
* SEE ALSO: VRTX32 User's Guide

* int vxFinquiry (efgid, pEfg)
*     UINT efgid;	/* event flag group ID number   *
*     UINT *pEfg;	/* return event flag group here *

*/

_vxFinquiry:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get efgid */
	moveq	#0x1c,d0		/* SC_FINQUIRY */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG2),a0		/* get pEfg */
	movel	d2,a0@			/* return event flag group */

	movel	a7@+,d2			/* restore reg */
	bra	return

/*******************************************************************************
*
* vxScreate - create semaphore
*
* This routine serves as a C interface to VRTX32 system call SC_SCREATE.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	48 = no control blocks available
*
* SEE ALSO: VRTX32 User's Guide

* int vxScreate (initVal, pendOrder, pSem)
*     UINT initVal;     /* initial value of semaphore, (0-65535) *
*     int pendOrder;    /* 0: pend tasks in priority order       *
*                       /* 1: pend tasks in FIFO order           *
*     UINT *pSem;       /* return semaphore ID number here       *

*/

_vxScreate:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get initVal */
	movel	a6@(ARG2),d2		/* get pendOrder */
	moveq	#0x2b,d0		/* SC_SCREATE */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG3),a0		/* get pSem */
	movel	d1,a0@			/* return semaphore ID number */

	movel	a7@+,d2			/* restore reg */
	bra	return

/*******************************************************************************
*
* vxSdelete - delete semaphore
*
* This routine serves as a C interface to VRTX32 system call SC_SDELETE.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	49 = semaphore id error
*	50 = tasks pending on semaphore (when forceDelete = 0)
*
* SEE ALSO: VRTX32 User's Guide

* int vxSdelete (semId, forceDelete)
*     UINT semId;       /* semaphore id number                          *
*     int forceDelete;  /* 0: delete semaphore only if no tasks pending *
*                       /* 1: delete semaphore, ready all pending tasks *

*/

_vxSdelete:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get semId */
	movel	a6@(ARG2),d2		/* get forceDelete */
	moveq	#0x2c,d0		/* SC_SDELETE */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg */
	bra	return

/*******************************************************************************
*
* vxSpost - post unit to semaphore
*
* This routine serves as a C interface to VRTX32 system call SC_SPOST.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with one of the following values:
*	49 = semaphore ID error
*	52 = semaphore overflow
*
* SEE ALSO: VRTX32 User's Guide

* int vxSpost (semId)
*     UINT semId;	/* semaphore id number *

*/

_vxSpost:
	link	a6,#0

	movel	a6@(ARG1),d1		/* get semId */
	moveq	#0x2e,d0		/* SC_SPOST */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vxSpend - pend on semaphore
*
* This routine serves as a C interface to VRTX32 system call SC_SPEND.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	10 = timeout
*	49 = semaphore ID error
*	51 = semaphore is deleted
*
* SEE ALSO: VRTX32 User's Guide

* int vxSpend (semId, timeout)
*     UINT semId;	/* semaphore id number                      *
*     UINT timeout;	/* number of ticks before returning timeout *

*/

_vxSpend:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get semId */
	movel	a6@(ARG2),d2		/* get timeout */
	moveq	#0x2d,d0		/* SC_SPEND */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg */
	bra	return

/*******************************************************************************
*
* vxSinquiry - semaphore inquiry
*
* This routine serves as a C interface to VRTX32 system call SC_SINQUIRY.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	49 = semaphore ID error
*
* SEE ALSO: VRTX32 User's Guide

* int vxSinquiry (semId, pSemVal)
*     UINT semId;	/* semaphore ID number         *
*     UINT *pSemVal;	/* return semaphore value here *

*/

_vxSinquiry:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get semId */
	moveq	#0x2f,d0		/* SC_SINQUIRY */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG2),a0		/* get pSemVal */
	movel	d2,a0@			/* return semaphore */

	movel	a7@+,d2			/* restore reg */
	bra	return

#endif ((CPU == MC68020) || (CPU == MC68030))
/*******************************************************************************
*
* vxGtime - get time
*
* This routine serves as a C interface to VRTX system call SC_GTIME.
* It returns the current number of "ticks" in the VRTX clock counter.
*
* RETURNS: Clock counter contents.
*
* SEE ALSO: VRTX User's Guide

* unsigned int vxGtime ()

*/

_vxGtime:
	link	a6,#0
	moveq	#0xa,d0			/* SC_GTIME */
	jsr	_sysKernelTrap		/* system trap */

	movel	d1,d0			/* time into d0 */
	unlk	a6
	rts

/*******************************************************************************
*
* vxStime - set time
*
* This routine serves as a C interface to VRTX system call SC_STIME.
* It sets the system clock to the specified time, in "ticks".
*
* SEE ALSO: VRTX User's Guide

* VOID vxStime (time)
*     unsigned time;	/* new clock counter value *

*/

_vxStime:
	link	a6,#0
	movel	a6@(ARG1),d1		/* get time */
	moveq	#0xb,d0			/* SC_STIME */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vxTdelay - delay task
*
* This routine serves as a C interface to VRTX system call SC_TDELAY.
* It suspends the calling task for the specified number of clock ticks.
*
*
* SEE ALSO: VRTX User's Guide

* VOID vxTdelay (delay)
*     int delay;	/* # of ticks for which task will be suspended *

*/

_vxTdelay:
	link	a6,#0
	movel	a6@(ARG1),d1		/* get delay */
	moveq	#0xc,d0			/* SC_TDELAY */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vxTslice - enable round-robin scheduling
*
* This routine serves as a C interface to VRTX system call SC_TSLICE.
* If interval == 0, round-robin scheduling is disabled.
*
* SEE ALSO: VRTX User's Guide

* VOID vxTslice (interval)
*     short interval;	/* number of ticks per task (0 to disable) *

*/

_vxTslice:
	link	a6,#0
	movew	a6@(ARG1W),d1		/* get interval */
	moveq	#0x15,d0		/* SC_TSLICE */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vxTimer - announce timer interrupt
*
* This routine serves as a C interface to VRTX system call UI_TIMER.
* It is called by the system clock interrupt handler
* to announce a system "tick" to VRTX.
*
* SEE ALSO: VRTX User's Guide

* VOID vxTimer ()

*/

_vxTimer:
	link	a6,#0
	moveq	#0x12,d0		/* UI_TIMER */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* return - universal exit
*
* All C callable routines in this library except vxGtime exit 
* via here. If the VRTX return code in d0 is not 0, the vx module number
* is or'd with d0 and errnoSet is called, and ERROR is returned in d0.
*
* NOMANUAL
*/

return:
	tstl	d0			/* OK? */
	beq	byebye			/* normal exit */

	orl	#0x10000,d0		/* or M_vx into errno */
	movel	d0,a7@-			/* push VRTX errno */
	jsr	_errnoSet		/* set task's status word */
	addql	#4,a7
	moveq	#-1,d0			/* return (ERROR) */

byebye:
	unlk	a6
	rts

/*******************************************************************************
*
* vxTaskEntry - task startup code following spawn
*
* This hunk of code is the initial entry point to every task created via
* the "spawn" routines.  Spawn has put the true entry point of the
* task into the tcb extension before creating the task,
* and then pushed exactly ten arguments (although the task may use
* fewer) onto the stack.  This code picks up the real entry point and calls it.
* Upon return, the 10 task args are popped, and the result of the main
* routine is passed to "exit" which terminates the task.
* This way of doing things has several purposes.  First a task is easily
* "restartable" (see "restart" routine in vxLib.c) since the real
* entry point is available in the tcb extension.  Second, the call to the main
* routine is a normal call including the usual stack clean-up afterwards,
* which means that debugging stack trace facilities will handle the call of
* the main routine properly.  
*
* NOMANUAL
*/

_vxTaskEntry:
	movel	#0,a6		/* make sure frame pointer is 0 */
#if ((CPU == MC68020) || (CPU == MC68030))
	movel	a0@(28),a0	/* get pointer to tcb extension */
#else ((CPU != MC68020) && (CPU != MC68030))
	movel	a0@(8),a0	/* get pointer to tcb extension */
#endif (CPU)
	movel	a0@,a0		/* entry adr is 1st member of tcb extension */
	jsr	a0@		/* call main routine */
	addl	#40,a7		/* pop args to main routine */
	movel	d0,a7@-		/* pass result to exit */
	jsr	_exit		/* gone for good */

/*******************************************************************************
*
* intEnt - enter interrupt level
*
* IntEnt should be jumped to, in place of the VRTX UI_ENTER call, when
* entering an interrupt.  This normally happens automatically, from
* the stub built by intConnect (2).
*
* IntEnt should NEVER be called from C.
*
* SEE ALSO: intConnect (2)

* VOID intEnt ()

*/

_intEnt:
	addql	#1,_intCnt	/* Bump the counter */

#if ((CPU != MC68020) && (CPU != MC68030))
	movel	d0,a7@-		/* save d0 */
	moveq	#0x16,d0	/* UI_ENTER */
	jsr	_sysKernelTrap	/* trap to VRTX */
	movel	a7@+,d0		/* restore d0 */
#endif (CPU)

	rts

/*******************************************************************************
*
* intExit - leave interrupt level
*
* This routine should be jumped to in place of the VRTX UI_EXIT call. This
* routine can NEVER be called from C.  It can only be jumped to from 
* assembly code.

* intExit ()

*/

_intExit:
	subql	#1,_intCnt	/* debump the counter */
	movl	d0,a7@-		/* store d0 on stack for VRTX */
	moveq	#0x11,d0	/* UI_EXIT */
	jmp	_sysKernelTrap	/* trap to VRTX (never to return here) */

/*******************************************************************************
*
* vxTcreateHook - task creation hook
*
* This routine should be specified in the VRTX configuration table,
* in usrConfig (1), as the task creation extension routine.
* This addition extends the VRTX SC_TCREATE system call to include 
* specification of the new task's stack address instead of using a 
* VRTX allocated stack, and allows specification of a TCB extension 
* address.  The following additional parameters are thus expected 
* in a trap to the VRTX SC_TCREATE system call:
*
*	a1 = bottom of stack
*	a2 = initial stack pointer
*	a3 = TCB extension pointer
*
* Note that the initial stack pointer is different from the bottom of
* stack if the caller has pushed initial arguments on the stack for
* the new task (as in the case of the "spawn" routines).
*
* If no bottom of stack is specified, i.e. TCREATE was called with a1 = zero, 
* then the new task's stack pointers are not modified.  This allows the
* normal VRTX stack allocation mechanism to be used if desired.
*
* On entry, a1 contains a pointer to the task's TCB.
* VRTX constructs the stack extension that contains the extra TCB stuff
* in its own workspace (this is pointed to by TCB.SP).  We must copy this
* over to the real stack that was allocated by the user (or by taskSpawn).
* Also, the TCB.USP must point to the initial stack location, while
* the TCB.SP must point to the extra TCB stuff on the stack (see VRTX
* documentation). THIS IS NOT A C CALLABLE ROUTINE.
*
* SEE ALSO: taskCreateHookAdd (2)

* vxTcreateHook ()

*/

#define TCB_EXT	0x08
#define TCB_A1	0x2c
#define TCB_A2	0x30
#define TCB_A3	0x34
#define TCB_SSP	0x38
#define TCB_USP	0x3c
#define TCB_BOS	0x40

_vxTcreateHook:
	movel	a1@(TCB_A3),a1@(TCB_EXT) /* copy TCB.a3 to TCB.tcb_extension */

	tstl	a1@(TCB_A2)		/* check for stack pointer specified */
	beq	tcend			/* no, leave vx stack pointers */

	moveml	d0-d2/a0-a3,a7@-	/* save regs */

	movel	a1@(TCB_SSP),a3		/* the SP of the new task points to */
					/* VRTX workspace; must be copied */
					/* to the task's stack. */
	movel	a1@(TCB_A2),a2		/* TCB.a2 should point to the */
					/* new stack */
#if ((CPU == MC68020) || (CPU == MC68030))
	subl	#28,a2	
	moveq	#13,d0			/* loop counter: copy 14 words */
#endif
#if (CPU == MC68010)
	subl	#28,a2	
	moveq	#13,d0			/* loop counter: copy 14 words */
#endif
#if (CPU == MC68000)
	subl	#26,a2	
	moveq	#12,d0			/* loop counter: copy 13 words */
#endif
	
copystk:
	movew	a3@+,a2@+		/* copy VRTX workspace -> task stack */
	dbf	d0,copystk

	movel	a1@(TCB_A2),d0		/* get sp out of TCB.a2 */
	movel	d0,a1@(TCB_USP)		/* put it in TCB.usp */
#if ((CPU == MC68020) || (CPU == MC68030))
	subl	#28,d0			/* move sp down past new stuff */
#endif
#if (CPU == MC68010)
	subl	#28,d0			/* move sp down past new stuff */
#endif
#if (CPU == MC68000)
	subl	#26,d0			/* move sp down past new stuff */
#endif
	movel	d0,a1@(TCB_SSP)		/* put it in TCB.sp */
	moveml	a7@+,d0-d2/a0-a3	/* restore regs */
tcend:
	rts

/*******************************************************************************
*
* vxTswitchHook - task switch hook
*
* This routine should be specified in the VRTX configuration table,
* as the task switch extension routine.
* It loops through a NULL terminated table of C task switch routines
* that have been specified by calls to taskSwitchHookAdd (2).
* Each routine in the table is called with pointers to
* the old and new tasks' TCB extensions on each task
* switch.  THIS IS NOT A C CALLABLE ROUTINE.
*
* SEE ALSO: taskSwitchHookAdd (2)

* vxTswitchHook ()

*/

_vxTswitchHook:
	moveml	d0-d2/a0-a2,a7@-	/* save regs */

	moveb	a2@(0xd),_taskIdCurrent + 3 /* update pointer to current task */

	movel	a2@(0x08),a7@-		/* push pointer to new tcbX */
	movel	a1@(0x08),a7@-		/* push pointer to old tcbX*/

	movel	#_taskSwitchTable,a2	/* get adrs of task switch rtn list */

tsloop:
	movel	a2@,d0			/* get next task switch rtn */
	beq	tsend			/* quit if end of list */

	movel	d0,a1
	jsr	a1@			/* call routine */

	addl	#4,a2			/* bump to next task switch routine */
	bra	tsloop			/* loop */

tsend:
	addl	#8,a7			/* clean up stack */

	moveml	a7@+,d0-d2/a0-a2	/* restore regs */
	rts
