/* compatVLib.c - compatibility routines for the vx sources */

/*
modification history
--------------------
*/

/*
DESCRIPTION
*/

#include "UniWorks.h"
#include "vrtx32.h"
#include "taskLib.h"
#include "sysSymTbl.h"

IMPORT          vxTaskEntry();

/*
 * DESCRIPTION The routines in this file are provided for compatibilty with
 * UniWorks 1.5. 
 */

compatVInit()
{
}
/*******************************************************************************
*
* taskStatus - turn numerical task status into a string.
*
* This routine cracks the VRTX task status word in the tcb and returns
* the appropriate string.
*/

VOID 
taskStatus(status, string)
	int             status;	/* tcb status */
	char           *string;	/* where to put string */

{
	switch (status & 0x5e) {
	case 0x00:
		switch (status & (VRTX_DELAY | VRTX_SUSPEND)) {
		case VRTX_READY:
			strcpy(string, "READY");
			break;	/* ready */
		case VRTX_SUSPEND:
			strcpy(string, "SUSPEND");
			break;	/* suspended */
		case VRTX_DELAY:
			strcpy(string, "DELAY");
			break;	/* task delay */
		case VRTX_DELAY | VRTX_SUSPEND:
			strcpy(string, "DELAY+S");
			break;	/* combination */
		}
		return;

	case VRTX_PEND:
		strcpy(string, "PEND");
		break;		/* suspended for msg */
	case VRTX_GETC:
		strcpy(string, "GETC");
		break;		/* suspended for input */
	case VRTX_PUTC:
		strcpy(string, "PUTC");
		break;		/* suspended for outpt */
	case VRTX_WAITC:
		strcpy(string, "WAITC");
		break;		/* awaiting spec char */
	case VRTX_QPEND:
		strcpy(string, "QPEND");
		break;		/* suspended on queue */

	default:		/* unanticipated combination */
		sprintf(string, "0x%02x", status);
		return;
	}

	if (status & VRTX_SUSPEND)
		strcat(string, "+S");	/* task also explicitly suspended */

	if (status & VRTX_DELAY)
		strcat(string, "+T");	/* timeout also in effect */
}
/*******************************************************************************
*
* printVrtxStatus - report VRTX status
*
* This routine translates a VRTX error status number into a string, and
* prints the string on standard out.  It is used primarily by other routines
* in usrLib (1) which need to report their status, but can be called
* independently from the shell, or by other application routines.
*/

VOID 
printVrtxStatus(status)
	int             status;	/* status returned by VRTX, or OK or ERROR */

{
	static char    *vx_return_text[] =	/* messages for each VRTX
						 * error code */
	{
	 "OK",			/* successful return */
	 "Task id error",
	 "No TCBs available",
	 "No memory available",
	 "Not a memory block",
	 "Memory block in use",
	 "Zero length message",
	 "Buffer full",
	 "WAITC in progress",
	 "Invalid system call",
	 "Time-out",
	 "No message present",
	 "Queue id error",
	 "Queue full",
	 "Partition id error",
	};

	if (status != OK)
		status = getStatus();

	printf("%s.\n", vx_return_text[status]);
}

/*******************************************************************************
*
* spawn - spawn a task
*
*/
int 
spawn(priority, id, options, entryAddress, stacksize, arg1,
      arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10)
	int             priority;	/* priority of new task */
	int             id;	/* task id of new task */
	int             options;/* options word for new task */
	FUNCPTR         entryAddress;	/* initial pc of new task */
	int             stacksize;	/* size (bytes) of stack needed */
	int             arg1, arg2, arg3;	/* arbitrary arguments to be */
	int             arg4, arg5, arg6;	/* passed to the routine */
	int             arg7, arg8, arg9, arg10;	/* started as a task */

{
	char           *topOfStack;
	char           *bottomOfStack;
	int             tid;


	/* allocate memory for stack and fill it for debugging */

	topOfStack = (char *) malloc((unsigned) (stacksize + sizeof(TCBX) +
						 MAX_SYS_SYM_LEN + 1));
	/* + 1 for EOS! */

	if (topOfStack == NULL)
		return (ERROR);

	bottomOfStack = topOfStack + stacksize;

	bfill(topOfStack, stacksize, 0xee);

	tid = spawnat(priority, id, options | VX_DEALLOC_STACK, entryAddress,
		      bottomOfStack, stacksize, (TCBX *) bottomOfStack,
	       	      arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, 
		      arg10);

	if ((tid == ERROR) || (taskActivate(tid) == ERROR)) {
		free(topOfStack);
		return (ERROR);
	}

	return (tid);
}
/*******************************************************************************
*
* spawnat - spawn a task at specified address
*/

int 
spawnat(pri, id, options, entryAdrs, botOfStack, stksize, pTcbx,
	arg1)
	int             pri;	/* priority of new task */
	int             id;	/* task id of new task */
	int             options;/* task option word */
	FUNCPTR         entryAdrs;	/* initial pc of new task */
	FAST char      *botOfStack;	/* bottom of new task's stack */
	int             stksize;/* size (bytes) of stack needed */
	FAST TCBX      *pTcbx;	/* address of new task's tcbx */
	int             arg1;

{
	char            name[MAX_SYS_SYM_LEN];
	char           *mallocName;
	int             val, type;
	FAST int       *argptr;
	FAST int       *sp;
	FAST int        ix;
	STATUS          status;

	/* set up task name */
	if (symValFind(sysSymTbl, entryAdrs, name, &val, &type) == ERROR)
		return (ERROR);
	mallocName = (char *) (((int) botOfStack) + sizeof(TCBX));
	strcpy(mallocName, name);

	options |= VX_SUPERVISOR_MODE;	/* TEMP! set SUP mode for all tasks */

	/* push args on the stack */

	argptr = &arg1 + MAX_TASK_ARGS;
	sp = (int *) botOfStack;

	for (ix = 0; ix < MAX_TASK_ARGS; ix++)
		*--sp = *--argptr;

	/* initialize tcb extension */

	bzero((char *) pTcbx, sizeof(TCBX));

	pTcbx->errorStatus	= OK;
	pTcbx->entry 		= entryAdrs;
	pTcbx->topOfStack 	= botOfStack - stksize;
	pTcbx->botOfStack 	= botOfStack;
	pTcbx->initialSP 	= (char *) sp;
	pTcbx->options 		= options;
	pTcbx->name 		= mallocName;

	for (ix = 0; ix < 3; ++ix)
		pTcbx->taskStd[ix] = ix;

	/*
	 * stop context switches so taskId won't get used by another task in
	 * between the time that the task id is fetched and used, and also,
	 * to keep the the new task from starting before we've had a chance
	 * to run the create hooks. 
	 */
	vxLock();		/* disable task switching */

	if (id == NONE)
		id = vxGetFreeTaskId();	/* next possible task id */

	pTcbx->taskId = id;

	status = vxTXcreate(pri, id, options & VX_SUPERVISOR_MODE,
			vxTaskEntry, botOfStack, pTcbx->initialSP, pTcbx);

	if (status != OK) {
		vxUnlock();
		return (ERROR);
	}
	/* the task has been created */

	vxTsuspend(id);	/* suspend the task so we can run create
				 * hooks */

	/*
	 * Task switching must be enabled so create hooks can take semaphores
	 * (ie. malloc) but this has a side effect that the task we have
	 * spawned could be resumed before all of the create hooks are run. 
	 */

	vxUnlock();

	return (id);
}
/*******************************************************************************
*
* restart - restart a task
*
*/
restart(tid)
{
	return (taskRestart(tid));
}
/*******************************************************************************
*
* vxDefaultTaskId - get default task id
*
*/
vxDefaultTaskId(tid)
{
	return(vxDefaultTaskId(tid));
}
/*******************************************************************************
*
* vxTaskTcb - get pointer to task's tcb
*
*/
VX_TCB *vxTaskTcb(tid)
{
	return((VX_TCB *) taskTcb(tid));
}
/*******************************************************************************
*
* vxMyTaskId - return calling task's id
*
*/
vxMyTaskId()
{
	return(taskMyId());
}
/*******************************************************************************
*
* vxFirstTcb - get first tcb on active list
*
*/
VX_TCB *vxFirstTcb()
{
	int idlist;

	if(taskIdListGet(idlist,1) == ERROR)
		return(ERROR);	
	return((VX_TCB *) taskTcb(idlist));
}
/*******************************************************************************
*
* vxPrintRegs - print contents of a task's registers
*
*/
VOID vxPrintRegs (pTcb)
    FAST VX_TCB *pTcb;		/* pointer to task's tcb */

    {
    int dregs[8];		/* task's data registers */
    int aregs[7];		/* task's address registers */
    char *sp;			/* task's stack pointer */
    USHORT sr;			/* task's status register */
    INSTR *pc;			/* task's pc */

    taskRegsGet (pTcb->tcb_id, dregs, aregs, &sp, &sr, &pc);

    printf ("\n");
    printf ("D0 = %8x   D4 = %8x   A0 = %8x   A4 = %8x   Task = %8d\n",
	    dregs[0], dregs[4], aregs[0], aregs[4], pTcb->tcb_id);
    printf ("D1 = %8x   D5 = %8x   A1 = %8x   A5 = %8x   SR   = %8x\n",
	    dregs[1], dregs[5], aregs[1], aregs[5], sr & 0xffff);
    printf ("D2 = %8x   D6 = %8x   A2 = %8x   A6 = %8x   PC   = %8x\n",
	    dregs[2], dregs[6], aregs[2], aregs[6], pc);
    printf ("D3 = %8x   D7 = %8x   A3 = %8x   US = %8x   SS   = %8x\n",
	    dregs[3], dregs[7], aregs[3], pTcb->tcb_usp, sp);
    }
/*******************************************************************************
*
* vxTaskRegs - get a task's registers from TCB and stack
*
*/
VOID vxTaskRegs (pTcb, dregs, aregs, sp, sr, pc)
    VX_TCB *pTcb;       /* TCB of task to get info about */
    int dregs[];        /* buffer for data registers 0 - 7 */
    int aregs[];        /* buffer for address registers 0 - 6 */
    char **sp;          /* buffer for stack pointer */
    USHORT *sr;         /* buffer for status register */
    INSTR **pc;         /* buffer for program counter */
{
	taskRegsGet(pTcb->tcb_id,dregs,aregs,sp,sr,pc);
}
/*******************************************************************************
*
* vxGetTaskRegs - get a task's registers from TCB and stack
*
*/
VOID vxGetTaskRegs (pTcb, dregs, aregs, sp, sr, pc)
    VX_TCB *pTcb;       /* TCB of task to get info about */
    int dregs[];        /* buffer for data registers 0 - 7 */
    int aregs[];        /* buffer for address registers 0 - 6 */
    char **sp;          /* buffer for stack pointer */
    USHORT *sr;         /* buffer for status register */
    INSTR **pc;         /* buffer for program counter */
{
	taskRegsGet(pTcb->tcb_id,dregs,aregs,sp,sr,pc);
}
/*******************************************************************************
*
* vxSetTaskRegs - set a task's registers from TCB and stack
*
*/
VOID vxSetTaskRegs (pTcb, dregs, aregs, sp, sr, pc)
    VX_TCB *pTcb;       /* TCB of task to set info */
    int dregs[];        /* data registers 0 - 7 */
    int aregs[];        /* address registers 0 - 6 */
    char *sp;           /* stack pointer */
    USHORT sr;          /* status register */
    INSTR *pc;          /* program counter */
{
	taskRegsSet(pTcb->tcb_id,dregs,aregs,sp,sr,pc);
}

/*******************************************************************************
*
* vxSetTaskSR - set task status register
*
*/
VOID vxSetTaskSR (pTcb, sr)
    VX_TCB *pTcb;       /* ptr to TCB of task */
    USHORT sr;          /* new SR */
{
	taskSRSet(pTcb->tcb_id,sr);
}
/*******************************************************************************
*
* vxSetTaskBreakable - set task breakable/unbreakable
*
*/
STATUS vxSetTaskBreakable (task, breakable)
    int task;           /* id of task to be set breakable/unbreakable */
    BOOL breakable;     /* TRUE = breakable, FALSE = unbreakable */
{
	return(taskOptionsSet(task,breakable,VX_UNBREAKABLE));
}
/*******************************************************************************
*
* vxAddTaskVar - add a task variable to a task
*
*/
STATUS vxAddTaskVar (pVar)
    int *pVar;  /* pointer to variable to be switched for task */
{
	return(taskVarAdd(taskTcb(0),pVar));
}

/*******************************************************************************
*
* vxDelTaskVar - delete a task variable from a task
*
*/
STATUS vxDelTaskVar (pVar)
    int *pVar;  /* pointer to task variable to be removed from task */
{
	return(taskVarDelete(taskTcb(0),pVar));
}
/*******************************************************************************
*
* vxGetTaskVar - get the value of a task variable
*
*/
STATUS vxGetTaskVar (tid, pVar)
    int tid;	
    int *pVar;  /* pointer to task variable to be removed from task */
{
	return(taskVarGet(tid,pVar));
}
/*******************************************************************************
*
* vxSetTaskVar - set the value of a task variable
*
*/
STATUS vxSetTaskVar (taskId, pVar, value)
    int taskId;         /* id of task whose task variable
                         * is to be set */
    int *pVar;          /* pointer to task variable to be
                         * set for this task */
    int value;          /* new value of task variable */
{
	return(taskVarSet(taskId,pVar,value));
}
/*******************************************************************************
*
* vxAddRebootRtn - add routine to be called every reboot
*
*/
STATUS vxAddRebootRtn (routine)
    FUNCPTR routine;    /* routine to be called when a task is created */
{
	return(rebootHookAdd(routine));
}
/*******************************************************************************
*
* vxAddTaskDeleteRtn - add routine to be called at every task delete
*
*/
STATUS vxAddTaskDeleteRtn (routine)
    FUNCPTR routine;    /* routine to be called when a task is deleted */
{
	return(taskDeleteHookAdd(routine));
}
/*******************************************************************************
*
* vxAddTaskCreateRtn - add routine to be called at every task create
*
*/
STATUS vxAddTaskCreateRtn (routine)
    FUNCPTR routine;    /* routine to be called when a task is created */
{
	return(taskCreateHookAdd(routine));
}
/*******************************************************************************
*
* vxAddTaskSwitchRtn - add routine to be called at every task switch
*
*/
STATUS vxAddTaskSwitchRtn (routine)
    FUNCPTR routine;    /* routine to be called at every task switch */
{
	return(taskSwitchHookAdd(routine));
}
/*******************************************************************************
*
* vxDelTaskCreateRtn - delete routine to be called at every task create
*
*/
STATUS vxDelTaskCreateRtn (routine)
    FUNCPTR routine;    /* routine to be called when a task is created */
{
	return(taskCreateHookDelete(routine));
}
/*******************************************************************************
*
* vxDelTaskDeleteRtn - delete routine to be called at every task delete
*
*/
STATUS vxDelTaskDeleteRtn (routine)
    FUNCPTR routine;    /* routine to be called when a task is deleted */
{
	return(taskDeleteHookDelete(routine));
}
/*******************************************************************************
*
* vxDelTaskSwitchRtn - delete routine to be called at every task switch
*
*/
STATUS vxDelTaskSwitchRtn (routine)
    FUNCPTR routine;    /* routine to be called when a task is switched */
{
	return(taskSwitchHookDelete(routine));
}
