/* tyLib.c - tty handler support library */

static char *copyright = "Copyright 1984-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
This library is meant to be used by ty drivers.  It handles all the
device independent functions of a normal serial device, such as XON-XOFF
processing, CR-LF conversion, and line editing functions.  Routines
contained in this library are only for use by drivers - there are no 
functions here that would normally be needed by a user.

For each device that uses this library, there must be a device, in the I/O
system's device list.  That device must contain, as part of it, a TY_DEV,
a pointer to which is a parameter to almost all the routines in this
library.  TyDevInit should be called exactly once to
initialize this TY_DEV.

Thereafter, various routines in this library may be called to service requests.

SEE ALSO: "I/O System", ioLib (1), iosLib (1)
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "ioLib.h"
#include "memLib.h"
#include "rngLib.h"
#include "wdLib.h"
#include "tyLib.h"
#include "sysLib.h"


IMPORT VOID reboot ();

/* special characters */

#define XON		0x11	/* ctrl-Q XON handshake */
#define XOFF		0x13	/* ctrl-S XOFF handshake */

LOCAL int tyXoffChars = 0;	/** TEMP **/
LOCAL int tyXoffMax   = 0;	/** TEMP **/

/* these are global only for the shell */

char tyBackspaceChar	= 0x7f;	/* default is DEL */
char tyDeleteLineChar	= 0x15;	/* default is control-U */
char tyEofChar		= 0x04;	/* default is control-D */

LOCAL char tyAbortChar	= 0x03;	/* default is control-C */
LOCAL char tyMonTrapChar= 0x18;	/* default is control-X */

LOCAL int tyXoffThreshold = 80;	/* max bytes free in input buffer before
				 * XOFF will be sent in OPT_TANDEM mode */
LOCAL int tyXonThreshold  = 100;/* min bytes free in input buffer before
				 * XON will be sent in OPT_TANDEM mode */
LOCAL int tyWrtThreshold  = 20;	/* min bytes free in output buffer before
				 * the next writer will be enabled */

LOCAL FUNCPTR abortFunc = NULL;	/* function to call when abort char received */

/*******************************************************************************
*
* tyDevInit - initialize ty device descriptor
*
* This routine initializes a ty device descriptor according to the
* specified parameters.  The initialization includes allocating read and
* write buffers of the specified sizes from the memory pool, 
* and initializing their respective buffer descriptors.
* The semaphores are initialized and the write semaphore 'given'
* to enable writers.  Also the transmitter startup routine pointer is set
* to the specified routine.  All other fields in the descriptor are zeroed.
*
* This routine should only be called by serial drivers.
*
* RETURNS
*  OK, or
*  ERROR if not enough memory to allocate data structures.
*/

STATUS tyDevInit (tydev, rd_buf_size, wt_buf_size, txStartup)
    FAST TY_DEV_ID tydev; /* pointer to ty device descriptor 
			   * to be initialized */
    int rd_buf_size;	  /* required read buffer size in bytes */
    int wt_buf_size;	  /* required write buffer size in bytes */
    FUNCPTR txStartup;	  /* device transmit startup routine */

    {
    /* clear out device */

    bzero ((char *) tydev, sizeof (TY_DEV));


    /* allocate read and write ring buffers */

    if ((tydev->wrtBuf = rngCreate (wt_buf_size)) == NULL)
	return (ERROR);

    if ((tydev->rdBuf = rngCreate (rd_buf_size)) == NULL)
	return (ERROR);


    /* initialize rest of device descriptor */

    tydev->txStartup = txStartup;

    tydev->rdSemId = semCreate ();
    tydev->wrtSemId = semCreate ();

    tyFlush (tydev);

    return (OK);
    }
/*******************************************************************************
*
* tyFlush - clear out a ty device descriptors buffers
*
* This routine resets a ty device's buffers to be empty and in the
* "initial" state.  Thus any characters input but not read by a task,
* and those written by a task but not yet output, are lost.
*/

LOCAL VOID tyFlush (tydev)
    FAST TY_DEV_ID tydev;	/* ptr to ty device descriptor to be cleared */

    {
    tyFlushRd (tydev);
    tyFlushWrt (tydev);
    }
/*******************************************************************************
*
* tyFlushRd - clear out a ty device descriptors read buffer
*/

LOCAL VOID tyFlushRd (tydev)
    FAST TY_DEV_ID tydev;	/* ptr to ty device descriptor to be cleared */

    {
    int level = intLock ();		/** disable interrupts **/

    rngFlush (tydev->rdBuf);
    semClear (tydev->rdSemId);
    tydev->rdState.semNeeded = TRUE;

    /* output an XON if necessary */

    if (tydev->rdState.xoff)
	{
	tydev->rdState.xoff = FALSE;
	tydev->rdState.pending = !tydev->rdState.pending;
	if (tydev->rdState.pending && !tydev->wrtState.busy)
	    (*tydev->txStartup) (tydev);
	}

    tydev->lnNBytes     = 0;
    tydev->lnBytesLeft  = 0;

    intUnlock (level);			/** re-enable interrupts **/
    }
/*******************************************************************************
*
* tyFlushWrt - clear out a ty device descriptors write buffer
*/

LOCAL VOID tyFlushWrt (tydev)
    FAST TY_DEV_ID tydev;	/* ptr to ty device descriptor to be cleared */

    {
    int level = intLock ();		/** disable interrupts **/

    rngFlush (tydev->wrtBuf);
    semClear (tydev->wrtSemId);
    semGive (tydev->wrtSemId);
    tydev->wrtState.semNeeded = FALSE;

    intUnlock (level);			/** re-enable interrupts **/
    }
/*******************************************************************************
*
* tyAbortFunc - set abort function
*
* This routine sets up a function that will be called when the abort
* character is received on a tty.  There is only one global abort function,
* used for any ty on which OPT_ABORT is enabled.
*
* When the abort character is received from a terminal with OPT-ABORT
* set, the function specified here will be called, with no parameters, from
* interrupt level.
*
* Setting an abort function of NULL effectively disables the abort altogether.
*
* SEE ALSO: tySetAbort (2)
*/

VOID tyAbortFunc (func)
    FUNCPTR func;	/* function to call when abort char is received */

    {
    abortFunc = func;
    }
/*******************************************************************************
*
* tySetAbort - change abort character
*
* This routine sets the abort character to the specified character.
* Typing the abort character to any device whose OPT_ABORT option is set
* will cause the shell task (task id 255) to be killed and restarted.
* Note that the character set by this routine applies to all devices
* whose handlers use the standard "tty" package "tyLib".
*
* The default abort character is ^C.
*
* SEE ALSO: tyAbortFunc (2)
*/

VOID tySetAbort (ch)
    char ch;		/* character to be made the abort character */

    {
    tyAbortChar = ch;
    }
/*******************************************************************************
*
* tySetBackspace - change backspace character
*
* This routine sets the backspace character to the specified character.
* Typing the backspace character to any device operating in "line protocol"
* mode (OPT_LINE option set) will cause the previous character typed to be
* deleted, up to the beginning of the current line.
* Note that the character set by this routine applies to all devices
* whose handlers use the standard "tty" package "tyLib".
*
* Default backspace character is ^H.
*/

VOID tySetBackspace (ch)
    char ch;		/* character to be made the backspace character */

    {
    tyBackspaceChar = ch;
    }
/*******************************************************************************
*
* tySetDeleteLine - change line-delete character
*
* This routine sets the delete character to the specified character.
* Typing the delete character to any device operating in "line protocol"
* mode (OPT_LINE option set) will cause all characters in the current
* line to be deleted.
* Note that the character set by this routine applies to all devices
* whose handlers use the standard "tty" package "tyLib".
*
* Default delete-line character is ^U.
*/

VOID tySetDeleteLine (ch)
    char ch;		/* character to be made the line-delete character */

    {
    tyDeleteLineChar = ch;
    }
/*******************************************************************************
*
* tySetEOF - change end-of-file character
*
* This routine sets the EOF character to the specified character.
* Typing the EOF character to any device operating in "line protocol"
* mode (OPT_LINE option set) will cause no character to be entered in
* the current line, but will cause the current line to be terminated
* (thus without a newline character).  The line is made available to
* reading tasks.  Thus if the EOF character is the first character input
* on a line, a line length of zero characters is returned to the reader.
* This is the standard end-of-file indication on a read call.
* Note that the character set by this routine applies to all devices
* whose handlers use the standard "tty" package "tyLib".
*
* Default EOF char is ^D.
*/

VOID tySetEOF (ch)
    char ch;		/* character to be made the eof character */

    {
    tyEofChar = ch;
    }
/*******************************************************************************
*
* tySetMonitorTrap - change monitor trap character
*
* This routine sets the trap-to-monitor character to the specified character.
* Typing the trap-to-monitor character to any device whose OPT_MON_TRAP
* option is set will cause the resident rom-monitor to be entered,
* if one exists in the system.  Once the rom-monitor is entered, 
* the normal UniWorks multitasking system is halted.  
*
* The default monitor trap character is ^X.
*
* Note that the character set by this routine applies to all devices
* whose handlers use the standard "tty" package "tyLib".  Also note that
* not all systems have a monitor trap available.
*/

VOID tySetMonitorTrap (ch)
    char ch;		/* character to be made the monitor trap character */

    {
    tyMonTrapChar = ch;
    }
/*******************************************************************************
*
* tyIoctl - special device control
*
* This routine handles device control requests for "tty" devices.
* It handles the following requests:
*
*   ioctl (fd, FIONREAD, &nBytes)
*	return in nBytes the number of characters 
*	available in the input buffer
*
*   ioctl (fd, FIONWRITE, &nBytes)
*	return in nBytes the number of characters 
*	available in the output buffer
*
*   ioctl (fd, FIOFLUSH)
*       discard all characters currently in input and 
*	output buffers
*
*   ioctl (fd, FIOSETOPTIONS, options)
*   ioctl (fd, FIOOPTIONS, options)
*       set device options. If the line-protocol (OPT_LINE) 
*	is changed, the input buffer is flushed.
*	The various options are described in tyLib.h
*
*   ioctl (fd, FIOGETOPTIONS)
*	returns the current device options
*
*   ioctl (fd, FIOCANCEL)
*       cancel read or write
*
*   ioctl (fd, FIOISATTY)
*       returns TRUE
*
* Any other request will return ERROR, and set status to
* S_ioLib_UNKNOWN_REQUEST.
*
* BUGS:
* In line-protocol mode (OPT_LINE option set) FIONREAD actually returns
* the number of characters available PLUS the number of lines in the buffer.
* Thus if five lines consisting of just newlines were in the input buffer,
* FIONREAD would return the value ten (five characters + five lines).
*
* VARARGS2 - not all requests include an arg.
*/

STATUS tyIoctl (tydev, request, arg)
    FAST TY_DEV_ID tydev;	/* pointer to device to control */
    int request;		/* request code */
    int arg;			/* some argument */

    {
    FAST int status = OK;
    int oldOptions;
    int oldlevel;		/* current interrupt level mask */

    switch (request)
	{
	case FIONREAD:
	    *((int *) arg) = rngNBytes (tydev->rdBuf);
	    break;

	case FIONWRITE:
	    *((int *) arg) = rngNBytes (tydev->wrtBuf);
	    break;

	case FIOFLUSH:
	    tyFlush (tydev);
	    break;

	case FIOFLUSHIN:
	    tyFlushRd (tydev);
	    break;

	case FIOFLUSHOUT:
	    tyFlushWrt (tydev);
	    break;

	case FIOGETOPTIONS:
	    return (tydev->options);

	case FIOSETOPTIONS:
	/*
	case FIOOPTIONS:
	*/
	    oldOptions = tydev->options;
	    tydev->options = arg;

	    if ((oldOptions & OPT_LINE) != (tydev->options & OPT_LINE))
		tyFlushRd (tydev);

	    if ((oldOptions & OPT_TANDEM) && !(tydev->options & OPT_TANDEM))
		{
		/* TANDEM option turned off: reset read and write states */

		oldlevel = intLock ();		/** disable interrupts **/

		/* output an XON if necessary */

		if (tydev->rdState.xoff)
		    {
		    tydev->rdState.xoff = FALSE;
		    tydev->rdState.pending = !tydev->rdState.pending;
		    if (tydev->rdState.pending && !tydev->wrtState.busy)
			    (*tydev->txStartup) (tydev);
		    }

		/* restart an XOFF'd transmitter */

		if (tydev->wrtState.xoff)
		    {
		    tydev->wrtState.xoff = FALSE;
		    if (!tydev->wrtState.busy)
			(*tydev->txStartup) (tydev);
		    }

		intUnlock (oldlevel);		/** re-enable interrupts **/
		}
	    break;

	case FIOCANCEL:
	    if (tydev->rdState.semNeeded)
		{
		tydev->rdState.canceled = TRUE;
/*		tydev->rdState.semNeeded = FALSE; */
		semGive (tydev->rdSemId);
		}

	    if (tydev->wrtState.semNeeded)
		{
		tydev->wrtState.canceled = TRUE;
/*		tydev->wrtState.semNeeded = FALSE; */
		semGive (tydev->wrtSemId);
		}
	    break;

	case FIOISATTY:
	    status = TRUE;
	    break;

	default:
	    errnoSet (S_ioLib_UNKNOWN_REQUEST);
	    status = ERROR;
	}

    return (status);
    }

/* raw mode i/o routines */

/*******************************************************************************
*
* tyWrite - task-level write routine for ttys
*
* This routine handles the task-level portion of the tty handler
* write function.
*
* RETURNS: number of bytes actually written to device.
*/

int tyWrite (tydev, buffer, nbytes)
    FAST TY_DEV_ID tydev;	/* Pointer to device structure */
    char *buffer;		/* buffer of data to write */
    FAST int nbytes;		/* Number of bytes in buffer */

    {
    FAST int bytesput;
    int oldlevel;		/* current interrupt level mask */
    int nbStart = nbytes;

    tydev->wrtState.canceled = FALSE;

    while (nbytes > 0)
	{
	semTake (tydev->wrtSemId); 

	if (tydev->wrtState.canceled)
	    {
	    errnoSet (S_ioLib_CANCELED);
	    return (nbStart - nbytes);
	    }

	bytesput = rngPutBuf (tydev->wrtBuf, buffer, nbytes);

	oldlevel = intLock ();			/** disable interrupts **/

	/* if xmitter not busy, start it */

	if (!tydev->wrtState.busy)
	    (*tydev->txStartup) (tydev);

	/* If more room in ringId, enable next writer. */

	if (rngFreeBytes (tydev->wrtBuf) > tyWrtThreshold)
	    semGive (tydev->wrtSemId);
	else
	    tydev->wrtState.semNeeded = TRUE;

	intUnlock (oldlevel);			/** re-enable interrupts **/

	nbytes -= bytesput;
	buffer += bytesput;
	}

    return (nbStart);
    }
/*******************************************************************************
*
* tyRead - task level read routine for ttys
*
* Handles the task level portion of the tty handler read function.
* Reads as many bytes as are available, but not more than maxbytes, into the
* buffer passed as an argument.
*
* This should only be called from serial device drivers.
*
* RETURNS: Number of bytes actually read into buffer.
*/

int tyRead (tydev, buffer, maxbytes)
    FAST TY_DEV_ID tydev;	/* device to read */
    char *buffer;		/* buffer to read into */
    int maxbytes;		/* maximum length of read */

    {
    FAST int nbytes;
    int oldlevel;		/* current interrupt level mask */
    FAST RING_ID ringId;
    FAST int n;
    int freeBytes;

    tydev->rdState.canceled = FALSE;

    semTake (tydev->rdSemId);

    if (tydev->rdState.canceled)
	{
	errnoSet (S_ioLib_CANCELED);
	return (0);
	}

    ringId = tydev->rdBuf;

    if (tydev->options & OPT_LINE)
	{
	if (tydev->lnBytesLeft == 0)
	    RNG_ELEM_GET (ringId, &tydev->lnBytesLeft, n);

	nbytes = rngGetBuf (ringId, buffer,
			    min ((int)tydev->lnBytesLeft, maxbytes));

	tydev->lnBytesLeft -= nbytes;
	}

    else
	nbytes = rngGetBuf (ringId, buffer, maxbytes);


    oldlevel = intLock ();		/** disable interrupts **/

    /* check if XON needs to be output */

    if ((tydev->options & OPT_TANDEM) && tydev->rdState.xoff)
	{
	freeBytes = rngFreeBytes (ringId);
	if (tydev->options & OPT_LINE)
	    freeBytes -= tydev->lnNBytes + 1;

	if (freeBytes > tyXonThreshold)
	    {
	    tydev->rdState.xoff = FALSE;
	    tydev->rdState.pending = !tydev->rdState.pending;
	    if (tydev->rdState.pending && !tydev->wrtState.busy)
		(*tydev->txStartup) (tydev);
	    }
	}

    /* If more characters in ringId, enable next reader. */

    if (rngEmpty (ringId))
	tydev->rdState.semNeeded = TRUE;
    else
	semGive (tydev->rdSemId);

    intUnlock (oldlevel);		/** re-enable interrupts **/

    return (nbytes);
    }
/*******************************************************************************
*
* tyITx - interrupt level output
*
* Get single character to be output to a device.  This routine looks
* at the ring buffer for the specified device and gives the caller the next
* available character, if there are any.
*
* RETURNS:
*	OK if more chars to send, or
*	ERROR if no more chars.
*	Location pointed at by charptr gets the character to be output.
*/

STATUS tyITx (tydev, charptr)
    FAST TY_DEV_ID tydev;	/* pointer to tty device descriptor */
    char *charptr;		/* ptr where to put character to be output */

    {
    FAST RING_ID ringId = tydev->wrtBuf;
    FAST int n;

    /* check if we need to output XON/XOFF for the read side */

    if (tydev->rdState.pending)
	{
	if (tydev->rdState.xoff)		/** TEMP **/
	    {					/** TEMP **/
	    if (tyXoffChars > tyXoffMax)	/** TEMP **/
		tyXoffMax = tyXoffChars;	/** TEMP **/
	    tyXoffChars = 0;			/** TEMP **/
	    }					/** TEMP **/
	*charptr = tydev->rdState.xoff ? XOFF : XON;
	tydev->rdState.pending = FALSE;
	tydev->wrtState.busy = TRUE;
	}

    /* check if output is in XOFF state before we return any char */

    else if (tydev->wrtState.xoff)
	tydev->wrtState.busy = FALSE;

    /* check for carriage return needed after linefeed */

    else if (tydev->wrtState.cr)
	{
	*charptr = '\r';
	tydev->wrtState.cr   = FALSE;
	tydev->wrtState.busy = TRUE;
	}

    /* check for no more characters to output */

    else if (RNG_ELEM_GET (ringId, charptr, n) == 0)	/* no more chars */
	tydev->wrtState.busy = FALSE;


    /* got a character to be output */

    else
	{
	tydev->wrtState.busy = TRUE;

	/* check for carriage return needs to be added after linefeed */

	if ((tydev->options & OPT_CRMOD) && (*charptr == '\n'))
	    tydev->wrtState.cr = TRUE;


	/* if task level needs semaphore, give it */

	if (tydev->wrtState.semNeeded && (rngFreeBytes(ringId) >tyWrtThreshold))
	    {
	    tydev->wrtState.semNeeded = FALSE;
	    semGive (tydev->wrtSemId);
	    }
	}

    return (tydev->wrtState.busy ? OK : ERROR);
    }
/*******************************************************************************
*
* tyIRd - interrupt level input
*
* Interrupt level character input processing routine for ttys.
* A device driver calls this routine when it has received a character.  This
* routine adds the character to the ring buffer for the specified device and, if
* task level is waiting for a semaphore, gives the semaphore.
*
* This routine also handles all the special characters, as specified in
* the option word for the device, such as X-ON, X-OFF, nextline, backspace,
* etc.
*
* RETURNS: OK, or ERROR if ring buffer full.
*/

STATUS tyIRd (tydev, inchar)
    FAST TY_DEV_ID tydev;	/* pointer to tty device descriptor */
    FAST char inchar;		/* character read */

    {
    FAST RING_ID ringId;
    FAST int n;
    int freeBytes;
    BOOL releaseTaskLevel;
    FAST int options = tydev->options;
    BOOL charEchoed = FALSE;
    STATUS status = OK;

    /* strip off parity bit if '7 bit' option set */

    if (options & OPT_7_BIT)
	inchar &= 0x7f;

    /* check for abort */

    if ((inchar == tyAbortChar) && (options & OPT_ABORT) && (abortFunc != NULL))
	(*abortFunc) ();

    /* check for trap to rom monitor */

    else if ((options & OPT_MON_TRAP) && (inchar == tyMonTrapChar))
	excToDoAdd (reboot, BOOT_WARM_AUTOBOOT);

    /* check for XON/XOFF received */

    else if ((options & OPT_TANDEM) && (inchar == XOFF))
	tydev->wrtState.xoff = TRUE;

    else if ((options & OPT_TANDEM) && (inchar == XON))
	{
	tydev->wrtState.xoff = FALSE;
	if (!tydev->wrtState.busy)
	    (*tydev->txStartup) (tydev);
	}


    else
	{
	if (tydev->rdState.xoff)	/** TEMP **/
	    tyXoffChars++;		/** TEMP **/

	/* check for carriage return needs to be turned into linefeed */

	if ((options & OPT_CRMOD) && (inchar == '\r'))
	    inchar = '\n';

	/* check for output echo required */

	if (options & OPT_ECHO)
	    {
	    ringId = tydev->wrtBuf;

	    /* echo the char.  some special chars are echoed differently */

	    if (options & OPT_LINE)
		{
		if (inchar == tyDeleteLineChar)
		    {
		    /* echo a newline */

		    RNG_ELEM_PUT (ringId, '\n', n);
		    charEchoed = TRUE;
		    }

		else if (inchar == tyBackspaceChar)
		    {
		    if (tydev->lnNBytes != 0)
			{
			/* echo BS-space-BS */

			rngPutBuf (ringId, " ", 3);
			charEchoed = TRUE;
			}
		    }

		else if ((inchar < 0x20) && (inchar != '\n'))
		    {
		    /* echo ^-char */

		    RNG_ELEM_PUT (ringId, '^', n);
		    RNG_ELEM_PUT (ringId, inchar + '@', n);
		    charEchoed = TRUE;
		    }

		else
		    {
		    RNG_ELEM_PUT (ringId, inchar, n);
		    charEchoed = TRUE;
		    }
		}
	    else
		/* just echo the char */
		{
		RNG_ELEM_PUT (ringId, inchar, n);
		charEchoed = TRUE;
		}

	    if ((!tydev->wrtState.busy) && charEchoed)
		(*tydev->txStartup) (tydev);
	    }


	/* put the char in the read buffer. */

	ringId = tydev->rdBuf;
	releaseTaskLevel = FALSE;

	if (!(options & OPT_LINE))
	    {
	    /* not line-mode; 
	     * just enter character and make immediately available to tasks */

	    if (RNG_ELEM_PUT (ringId, inchar, n) == 0)
		status = ERROR;			/* buffer full, not entered */

	    releaseTaskLevel = TRUE;
	    }
	else
	    {
	    /* line-mode;
	     * process backspace, line-delete, EOF, and newline chars special */

	    freeBytes = rngFreeBytes (ringId);

	    if (inchar == tyBackspaceChar)
		{
		if (tydev->lnNBytes != 0)
		    tydev->lnNBytes--;
		}
	    else if (inchar == tyDeleteLineChar)
		tydev->lnNBytes = 0;
	    else if (inchar == tyEofChar)
		{
		if (freeBytes > 0)
		    releaseTaskLevel = TRUE;
		}
	    else
		{
		if (freeBytes >= 2)
		    {
		    if (freeBytes >= (tydev->lnNBytes + 2))
			tydev->lnNBytes++;
		    else
			status = ERROR;	/* no room, overwriting last char */

		    rngPutAhead (ringId, inchar, (int)tydev->lnNBytes);

		    if ((inchar == '\n') || (tydev->lnNBytes == 255))
			releaseTaskLevel = TRUE;
		    }
		else
		    status = ERROR;	/* no room, not even for overwriting */
		}

	    /* if line termination indicated, put line byte count
	     * in 0th character of line and advance ring buffer pointers */

	    if (releaseTaskLevel)
		{
		rngPutAhead (ringId, (char) tydev->lnNBytes, 0);
		rngMoveAhead (ringId, (int) tydev->lnNBytes + 1);
		tydev->lnNBytes = 0;
		}
	    }


	/* check if XON/XOFF needs to be output */

	if (options & OPT_TANDEM)
	    {
	    freeBytes = rngFreeBytes (ringId);
	    if (tydev->options & OPT_LINE)
		freeBytes -= tydev->lnNBytes + 1;

	    if (!tydev->rdState.xoff)
		{
		/* if input buffer is close to full, send XOFF */

		if (freeBytes < tyXoffThreshold)
		    {
		    tydev->rdState.xoff = TRUE;
		    tydev->rdState.pending = !tydev->rdState.pending;
		    if (tydev->rdState.pending & !tydev->wrtState.busy)
			(*tydev->txStartup) (tydev);
		    }
		}
	    else
		{
		/* if input buffer has enough room now, send XON */

		if (freeBytes > tyXonThreshold)
		    {
		    tydev->rdState.xoff = FALSE;
		    tydev->rdState.pending = !tydev->rdState.pending;
		    if (tydev->rdState.pending & !tydev->wrtState.busy)
			(*tydev->txStartup) (tydev);
		    }
		}
	    }


	/* if task level needs semaphore, give it */

	if (releaseTaskLevel && tydev->rdState.semNeeded)
	    {
	    tydev->rdState.semNeeded = FALSE;
	    semGive (tydev->rdSemId);
	    }
	}

    return (status);
    }
