/* taskHookLib.c - task hook library */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
TASK SWITCH ROUTINES
The routines taskSwitchRtnAdd and taskSwitchRtnDel allow routines
to be dynamically added and deleted from a list of routines which are
called at each task switch.  This UniWorks extension to VRTX is accomplished
by the routine psosTswitchHook????? (2) which is installed in the task switch
hook of the VRTX configuration table in usrConfig.  This facility is used
by the debugger to provide task specific breakpoints and single stepping,
and by the "task variable" routines to switch task variable values.

SEE ALSO: "Architecture", vxALib (1)
*/

#include "UniWorks.h"
#include "taskLib.h"

/* globals */

FUNCPTR taskCreateTable [VX_MAX_TASK_CREATE_RTNS + 1];
FUNCPTR taskSwitchTable [VX_MAX_TASK_SWITCH_RTNS + 1];
FUNCPTR taskDeleteTable [VX_MAX_TASK_DELETE_RTNS + 1];


/*******************************************************************************
*
* taskCreateHookAdd - add routine to be called at every task create
*
* This routine adds the specified routine to a list of routines
* that get called whenever a task is created.  The routine should be
* declared as follows:
*
*	VOID createHook (pNewTcbX)
*	    TCBX *pNewTcbX;		/* pointer to new task's TCB *
*
* RETURNS: OK, or ERROR if table of task switch routines is full.
*
* SEE ALSO: taskCreateHookDelete(2)
*/

STATUS taskCreateHookAdd (createHook)
    FUNCPTR createHook;	/* routine to be called when a task is created */

    {
    return (taskHookAdd (createHook, taskCreateTable, VX_MAX_TASK_CREATE_RTNS));
    }
/*******************************************************************************
*
* taskCreateHookDelete - delete previously added task create routine
*
* This routine removes the specified routine from the list of
* routines to be called at each task create.
*
* RETURNS: OK, or ERROR if routine not in table.
*
* SEE ALSO: taskCreateHookAdd(2)
*/

STATUS taskCreateHookDelete (createHook)
    FUNCPTR createHook;		/* routine to be deleted from list */

    {
    return (taskHookDelete (createHook, taskCreateTable,
			    VX_MAX_TASK_CREATE_RTNS));
    }
/*******************************************************************************
*
* taskSwitchHookAdd - add routine to be called at every task switch
*
* This routine adds the specified routine to a list of routines
* that get called at every task switch.  The routine should be
* declared as follows:
* .CS
*
*	VOID switchHook (pOldTcbX, pNewTcbX)
*	    TCBX *pOldTcbX;	/* pointer to old task's TCBX *
*	    TCBX *pNewTcbX;	/* pointer to new task's TCBX *
*
* .CE
* RETURNS: OK, or ERROR if table of task switch routines is full.
*
* SEE ALSO: taskSwitchHookDelete(2)
*/

STATUS taskSwitchHookAdd (switchHook)
    FUNCPTR switchHook;	/* routine to be called at every task switch */

    {
    return (taskHookAdd (switchHook, taskSwitchTable, VX_MAX_TASK_SWITCH_RTNS));
    }
/*******************************************************************************
*
* taskSwitchHookDelete - delete previously added task switch routine
*
* This routine removes the specified routine from the list of
* routines to be called at each task switch.
*
* RETURNS: OK, or ERROR if routine not in table.
*
* SEE ALSO: taskSwitchHookAdd(2)
*/

STATUS taskSwitchHookDelete (switchHook)
    FUNCPTR switchHook;		/* routine to be deleted from list */

    {
    return (taskHookDelete (switchHook, taskSwitchTable,
			    VX_MAX_TASK_SWITCH_RTNS));
    }
/****************************************************************
*
* taskDeleteHookAdd - add routine to be called at every task delete
*
* This routine adds the specified routine to a list of routines
* that get called whenever a task is deleted.  The routine should be
* declared as follows:
*
*	VOID deleteHook (pTcbX)
*	    TCBX *pTcbX;	/* pointer to deleted task's TCBX *
*
* RETURNS: OK, or ERROR if table of task delete routines is full.
*
* SEE ALSO: taskDeleteHookDelete(2)
*/

STATUS taskDeleteHookAdd (deleteHook)
    FUNCPTR deleteHook;	/* routine to be called when a task is deleted */

    {
    return (taskHookAdd (deleteHook, taskDeleteTable, VX_MAX_TASK_DELETE_RTNS));
    }
/*******************************************************************************
*
* taskDeleteHookDelete - delete previously added task delete routine
*
* This routine removes the specified routine from the list of
* routines to be called at each task delete.
*
* RETURNS: OK, or ERROR if routine not in table.
*
* SEE ALSO: taskDeleteHookAdd(2)
*/

STATUS taskDeleteHookDelete (deleteHook)
    FUNCPTR deleteHook;		/* routine to be deleted from list */

    {
    return (taskHookDelete (deleteHook, taskDeleteTable,
			    VX_MAX_TASK_DELETE_RTNS));
    }

/***************************************************************************
*
* taskHookAdd - add hook to hook table
*/

LOCAL STATUS taskHookAdd (hook, table, maxEntries)
    FUNCPTR hook;	/* routine to be added to table */
    FUNCPTR table[];	/* table to which to add */
    int maxEntries;	/* max entries in table */

    {
    FAST int ix;

    taskLock ();			/* disable task switching */

    /* find slot after last hook in table */

    for (ix = 0; ix < maxEntries; ++ix)
	{
	if (table[ix] == NULL)
	    {
	    table[ix] = hook;
	    taskUnlock ();	/* re-enable task switching */
	    return (OK);
	    }
	}
    
    /* no free slot found */

    taskUnlock ();		/* re-enable task switching */

    errnoSet (S_taskLib_TASK_HOOK_TABLE_FULL);
    return (ERROR);
    }
/***************************************************************************
*
* taskHookDelete - delete hook from hook table
*/

LOCAL STATUS taskHookDelete (hook, table, maxEntries)
    FUNCPTR hook;	/* routine to be deleted from table */
    FUNCPTR table[];	/* table from which to delete */
    int maxEntries;	/* max entries in table */

    {
    FAST int ix;

    taskLock ();			/* disable task switching */

    /* find hook in hook table */

    for (ix = 0; ix < maxEntries; ++ix)
	{
	if (table [ix] == hook)
	    {
	    /* move all the remaining hooks up one slot in the table */

	    do
		table [ix] = table [ix + 1];
	    while (table [++ix] != NULL);

	    taskUnlock ();	/* re-enable task switching */
	    return (OK);
	    }
	}

    /* hook not found in table */

    taskUnlock ();		/* re-enable task switching */

    errnoSet (S_taskLib_TASK_HOOK_NOT_FOUND);
    return (ERROR);
    }
