/* errnoLib.c - error status library */

static char *copyright = "Copyright 1984-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This library contains routines for setting and examining the error status
values of tasks and interrupts.

Most UniWorks functions return a result of ERROR, or NULL in the case of
functions returning pointers, when they detect an error.
In addition they a set an "error status" or "errno" for the current
task or interrupt that elaborates the nature of the error.

This facility is analagous to the Unix error status mechanism in which error
status values are set in the global variable "errno".  However, in UniWorks
there are many contexts, both task and interrupt, that share the common
memory space and hence would conflict in their use of a global variable.
Instead, UniWorks supplies routines to set and get error status values,
and these routines maintain the value for each context separately.

This facility is used throughout UniWorks for error reporting,
but application modules are encouraged to use the same mechanism,
where appropriate.

ERROR STATUS VALUES
An error status is a 4 byte integer.
By convention, the most significant two bytes are the module number,
which indicates the module in which the error occurred,
and the lower two bytes indicate the specific error within that module.
Module number 0 is specifically reserved for Unix error numbers, so that
values from the Unix errno.h header can be set and tested without modification.
Module numbers 1-500 decimal are reserved for UniWorks modules.
These are defined in vwModNum.h.
All other module numbers are available to applications.

TASK ERROR STATUS
The error status of each task is kept in its TCB extension.
It is zeroed when the task is first spawned.
Thereafter, the error status is set whenever a UniWorks routine detects an error.
It is never cleared by UniWorks.

INTERRUPT LEVEL ERROR STATUS
It is possible to set and get error status from code executing
at interrupt level.
At interrupt level, error status is stored in an
array at an index corresponding to the level of interrupt nesting.
The routines to set and get error status explicitly check if they
are being executed at interrupt level, and if so, 
set the status at the appropriate interrupt nesting level.
Note that this is the interrupt nesting level, NOT the interrupt
mask level itself, since it is possible to have several nested interrupt
handlers active at the same mask level, if the interrupt mask level is
explicitly lowered by an interrupt handler.

The maximum interrupt nesting level that errnoLib is capable of handling
is MAX_INT_NEST (50).  Beyond that, errnoGet and errnoSet will return an error.

PRINTING ERROR STATUS VALUES
UniWorks can include a special symbol table called statSymTbl.
This is used by printErrno (2) to print human-readable error messages,
given the binary status value.

The table is created using the tool makeStatTbl (4) in vw/bin.  This tool
reads all the .h files in a directory and generates a C language file
which, when compiled, generates a symbol table.  Each symbol consists
of a value, which is an error status value, and a string, which is
the definition string of the value from the header file.

For example, suppose the header file vw/h/myFile.h contains the line:

    #define S_myFile_ERROR_TOO_MANY_COOKS	0x230003

StatSymTbl is created by running:

    makeStatTbl vw/h >statTbl.c

This creates a file, statTbl.c, that can be compiled into a special
symbol table called statSymTbl.  That file is then linked in with
UniWorks.  This is normally done automatically by the vw/config makefile.

If the user now types, to the UniWorks shell:

    -> printErrno 0x230003

printErrno would respond:

    
    S_myFile_ERROR_TOO_MANY_COOKS

There are several important things to note about this facility.
First, makeStatTbl (4) looks for error status lines of the form:

    #define S_xxx  <n>

where xxx is any string, and <n> is any number.
All UniWorks status's are of the form:

    #define S_thisFile_MEANINGFUL_ERROR_MESSAGE   0xnnnn

where thisFile is the name of the module.

This facility is available to the user.  Simply add header files with
status lines of the appropriate forms, and re-make UniWorks.

INTERNAL
It is possible to have more than 7 nested interrupts.
An interrupt handler may explicitly lower the interrupt mask,
then another interrupt at the same or possibly even lower level
can interrupt again.
Interrupt nesting would be bounded only by the size of the interrupt stack.

A safe implementation would be to have the errnoGet/Set routines use memory
locations that were allocated on the interrupt stack, one for each interrupt
entry.  This could be done by having a single global pointer to the current
interrupt status cell.  Interrupt entry code built by intConnect would
first save the value of that pointer on the interrupt stack and then leave
another 4 bytes on the interrupt stack for the new interrupt status and
set the global pointer to point to that location.  The interrupt exit
code would restore the global pointer to the previous value and then pop the
8 bytes off the interrupt stack.

This would take three instructions on interrupt entry:
	move.l	_stsIntStatusPtr,a7@-	# push old int status ptr
	subq	#4,a7			# make room for our status value on stk
	move.l	a7,_stsIntStatusPtr	# set new int status ptr
and two on exit:
	addq	#4,a7			# pop status value off stack
	move.l  a7@+,_stsIntStatusPtr	# restore old int status ptr

This is too expensive a solution and is probably overkill.
In this implementation the array of status values
indexed by intCount is MAX_INT_NEST deep, much larger than 7.
In errnoSet if intCount is more than MAX_INT_NEST the status is thrown away,
and errnoGet will return a special error S_errnoLib_INT_NEST_TOO_DEEP.
That way even if it does nest deeper than MAX_INT_NEST it won't crash,
just lose the status value.

INCLUDE FILES
The file vwModNum.h contains the module numbers for every UniWorks module.
Each module's include file contains the error numbers that that module
can generate.

SEE ALSO: "Architecture" (Subroutine Error Status), printErrno (2)
*/

#include "UniWorks.h"
#include "taskLib.h"
#include "symLib.h"
#include "errnoLib.h"


#define	MAX_INT_NEST	50		/* max. depth of interrupt nesting */

LOCAL int errnoIntLevel [MAX_INT_NEST];	/* statuses set at interrupt level */


/*******************************************************************************
*
* errnoGet - get error status value of calling task
*
* This routine gets the error status most recently set for the calling task.
* If at interrupt level, then the error status most recently set
* at the current interrupt nesting level is returned.
*
* RETURNS:
*    error status of the calling task (from task level) or
*    error status of the current interrupt nest level or
*    S_errnoLib_INT_NEST_TOO_DEEP if interrupt nest level too deep.
*
* SEE ALSO: errnoSet(2), errnoOfTaskGet(2)
*/

int errnoGet ()

    {
    int level = intCount ();

    /* if we are not in a task's context,
     * use array indexed at this interrupt level */

    if (level > 0)
	return ((level > MAX_INT_NEST) ?
		S_errnoLib_INT_NEST_TOO_DEEP : errnoIntLevel [level - 1]);
    else
	return (errnoOfTaskGet (0));
    }
/*******************************************************************************
*
* errnoOfTaskGet - get error status value of specified task
*
* This routine gets the error status most recently set for the specified task.
* If (task == 0), the calling task is assumed.
*
* This routine is provided primarily for debugging purposes.
* Normally, tasks use the routines errnoSet and errnoGet to set and get
* their own error status values.
*
* RETURNS: error status of the specified task, or ERROR if task doesn't exist.
*
* SEE ALSO: errnoSet(2), errnoGet(2)
*/

int errnoOfTaskGet (taskId)
    int taskId;			/* task's id, 0 means current task */

    {
    TCBX *pTcbX;		/* this gets pointer to current task's tcbX */

    /* get pointer to current task's tcbX, be sure it exists */

    if ((pTcbX = taskTcbX (taskId)) == NULL)
	return (ERROR);

    /* return the error status from the tcb extension */

    return (pTcbX->errorStatus);
    }
/*******************************************************************************
*
* errnoSet - set error status value of calling task
*
* This routine sets the error status for the calling task.
* If at interrupt level, then the error status at this nest level is set.
*
* RETURNS:
*	OK, or
*       ERROR if interrupt nest level too deep
*
* SEE ALSO: errnoGet(2), errnoOfTaskSet(2)
*/

STATUS errnoSet (errno)
    int errno;			/* error status value to set */

    {
    int level = intCount ();

    /* if we are not in a task's context,
     * use array indexed at this interrupt level */

    if (level > 0)
	{
	if (level > MAX_INT_NEST)
	    return (ERROR);

	errnoIntLevel [level - 1] = errno;
	return (OK);
	}
    else
	return (errnoOfTaskSet (0, errno));
    }
/*******************************************************************************
*
* errnoOfTaskSet - set error status value of specified task
*
* This routine sets the error status for the specified task.
* If (task == 0), the calling task is assumed.
*
* This routine is provided primarily for debugging purposes.
* Normally, tasks use the routines errnoSet and errnoGet to set and get
* their own error status values.
*
* RETURNS: OK, or ERROR if task doesn't exist.
*
* SEE ALSO: errnoSet(2), errnoOfTaskGet(2)
*/

STATUS errnoOfTaskSet (taskId, errno)
    int taskId;			/* task's id, 0 means current task */
    int errno;			/* error status value */

    {
    TCBX *pTcbX;		/* this gets pointer to current task's tcbX */

    /* get pointer to specified task's tcb */

    if ((pTcbX = taskTcbX (taskId)) == NULL)
	return (ERROR);

    /* stuff errno in tcb extension */

    pTcbX->errorStatus = errno;
    return (OK);
    }

