/* taLib.c - spy CPU activity library */

static char *copyright = "Copyright 1985-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/
  
/*
DESCRIPTION
This module provides a facility to monitor the CPU usage of all tasks.
The primary interface routine, spy (2), periodically prints a report
on CPU usage with taReport (2).  The report shows CPU usage taken up by
interrupt level, idle state, and running tasks.
The total usage since the beginning of spy (or the most recent taStart (2)),
and the change in usage (delta) since the last taReport was displayed.

The facility can also be used "manually", instead of using spy.
In that case, taStart is used to begin monitoring, and taReport provides
a one-time report of the same information as provided by spy.

This data is gathered by an interrupt level routine that is connected by
taStart to the auxiliary clock.  There is currently no way to use this
facility with CPU's that don't have an auxiliary clock.
Interrupts that are at a higher level than the auxiliary clock's
cannot be monitored.
*/

#include "UniWorks.h"
#include "taskLib.h"
#include "sysSymTbl.h"


#define MAX_SPY_TASKS	200		/* max tasks that can be spy'd */

/* spyTask parameters */

int spyTaskId		= ERROR;	/* ERROR = spy task not active */
int spyTaskOptions	= VX_UNBREAKABLE;
int spyTaskPriority	= 5;
int spyTaskStackSize	= 4000;

/* local variables */

LOCAL UINT spyTotalTicks;		/* all ticks since start */
LOCAL UINT spyIncTicks;			/* all ticks since last report */

LOCAL UINT spyInterruptTicks;		/* int ticks since start */
LOCAL UINT spyInterruptIncTicks;	/* int ticks since last report */

LOCAL UINT spyIdleTicks;		/* idle ticks since start */
LOCAL UINT spyIdleIncTicks;		/* idle ticks since last report*/

LOCAL BOOL spyClkRunning;		/* TRUE = taStart'ed */
LOCAL BOOL spyInitialized;		/* TRUE = hooks installed */
LOCAL int spyCreateCount;		/* num tasks created since report */
LOCAL int spyDeleteCount;		/* num tasks deleted since report */

/* display formats */

LOCAL char *spyFmt1 = "%-12.12s  %8x   %3d%% (%7d)  %3d%% (%7d)\n";
LOCAL char *spyFmt2 = "%-12.12s  %8s   %3d%% (%7d)  %3d%% (%7d)\n";

/*******************************************************************************
*
* spyCreateHook - initialize task tick counters
*
* This routine is installed as a task create hook so that the task
* tick counters can be initialized and taReport (2)
* can indicate when new tasks appear between reports.
*/

LOCAL VOID spyCreateHook (pTcbX)
    TCBX *pTcbX;	/* TCB extension of new task */

    {
    pTcbX->taskTicks    = 0;
    pTcbX->taskIncTicks = 0;
    spyCreateCount++;
    }
/*******************************************************************************
*
* spyDeleteHook - notify spyLib of task deletion
*
* This routine is installed as a task delete hook so that taReport (2)
* can indicate when tasks disappear between reports.
*
* ARGSUSED
*/

LOCAL VOID spyDeleteHook (pTcbX)
    TCBX *pTcbX;	/* TCB extension of new task */

    {
    spyDeleteCount++;
    }
/*******************************************************************************
*
* taInterrupt - spyLib interrupt service routine
*
* This routine is called at each tick of the auxiliary clock.
* When at interrupt level the interrupt tick count is incremented.
* If there is no active task then the idle tick count is incremented,
* otherwise increment the active task's tick counter.
*/

LOCAL VOID taInterrupt ()

    {
    FAST TCBX *pTcbX;

    if (intCount () > 1)
	spyInterruptIncTicks++;		/* we interrupted an interrupt */
    else
	{
	pTcbX = taskTcbX (0);		/* get our tcb extension */

	if (taskIsReady (pTcbX->taskId))
	    pTcbX->taskIncTicks++;
	else
	    spyIdleIncTicks++;
	}

    spyIncTicks++;
    }
/*******************************************************************************
*
* taStart - start collecting task activity data
*
* Begin data collection by enabling the auxilary clock interrupts.
* Data from previous collections is cleared.
*
* RETURNS:
*	OK, or
*	ERROR if CPU has no auxiliary clock, or
*             unable to install task create/delete hooks.
*
* SEE ALSO: sysAuxClkConnect (2)
*/

STATUS taStart (intsPerSec)
    int intsPerSec;	/* timer interrupt frequency, */
			/* 0 = use default of 100 */

    {
    FAST int ix;
    FAST int nTasks;
    FAST TCBX *pTcbX;
    int idList [MAX_SPY_TASKS];

    if (spyClkRunning)
	return (ERROR);		/* We're already going */

    if (!spyInitialized &&
	(taskCreateHookAdd (spyCreateHook) == ERROR ||
	 taskDeleteHookAdd (spyDeleteHook) == ERROR))
	{
	printf ("Unable to add create/delete hooks.\n");
	return (ERROR);
	}

    spyInitialized = TRUE;

    if (intsPerSec == 0)
	intsPerSec = 100;

    spyDeleteCount  = 0;
    spyCreateCount  = 0;
    spyTotalTicks = spyIdleTicks    = spyInterruptTicks    = 0;
    spyIncTicks   = spyIdleIncTicks = spyInterruptIncTicks = 0;


    /* initialize tick counters of tasks already running */

    nTasks = taskIdListGet (idList, NELEMENTS (idList));

    for (ix = 0; ix < nTasks; ++ix)
	{
	pTcbX = taskTcbX (idList [ix]);
	pTcbX->taskIncTicks = pTcbX->taskTicks = 0;
	}

    if (sysAuxClkConnect (taInterrupt, 0) != OK)
	{
	printf ("No auxiliary clock on CPU.\n");
	return (ERROR);
	}

    sysAuxClkSetRate (intsPerSec);

    if(sysAuxClkEnable () != OK)
	return (ERROR);

    spyClkRunning = TRUE;

    return (OK);
    }
/*******************************************************************************
*
* taStop - stop collecting task activity data
*
* This routine disables the auxiliary clock interrupts.
* Data collected remains valid until the next taStart (2).
*/

VOID taStop ()

    {
    sysAuxClkDisconnect ();
    spyClkRunning  = FALSE;
    }
/*******************************************************************************
*
* taReport - display task activity data
*
* This routine reports on data gathered at interrupt level.  It shows
* the number of ticks used by each task, and at interrupt level, since
* taStart (2), and also since the last taReport was made.  It
* also shows CPU utilization percentage, and percent idle.  Nothing
* is printed if no interrupts have happened since the previous taReport.
*/

VOID taReport ()

    {
    FAST TCBX *pTcbX;
    FAST int ix;
    FUNCPTR symbolAddress;
    char name [MAX_SYS_SYM_LEN + 1];
    TINY type;

    int idList [MAX_SPY_TASKS];		/* task specific statistics */
    int taskIncTicks [MAX_SPY_TASKS];
    int taskTotalTicks [MAX_SPY_TASKS];
    FAST int nTasks;

    int tmpIncTicks;			/* incremental snap shot */
    int tmpIdleIncTicks;
    int tmpInterruptIncTicks;

    int totalPerCent;
    int incPerCent;
    int sumTotalPerCent = 0;
    int sumIncPerCent   = 0;

    /* if there have been no ticks, there is nothing to report */

    if (spyIncTicks == 0)
	return;
	
    /* snap shot and clear task statistics */

    nTasks = taskIdListGet (idList, NELEMENTS (idList));

    for (ix = 0; ix < nTasks; ++ix)
	{
	pTcbX = taskTcbX (idList [ix]);

	/* order is important: save and clear incremental, then update total */

	taskIncTicks [ix]    = pTcbX->taskIncTicks;
	pTcbX->taskIncTicks  = 0;

	pTcbX->taskTicks    += taskIncTicks [ix];
	taskTotalTicks [ix]  = pTcbX->taskTicks;
	}


    /* save and clear incremental counts and accumulate totals */

    tmpIncTicks          = spyIncTicks;
    tmpIdleIncTicks      = spyIdleIncTicks;
    tmpInterruptIncTicks = spyInterruptIncTicks;

    spyIncTicks = spyIdleIncTicks = spyInterruptIncTicks = 0;

    spyTotalTicks       += tmpIncTicks;
    spyInterruptTicks   += tmpInterruptIncTicks;
    spyIdleTicks        += tmpIdleIncTicks;


    /* print info */

    printf ("\n");
    printf (
    "  ENTRY         TID     total %% (ticks)  delta %% (ticks)\n");
    printf (
    "------------  --------  ---------------  ---------------\n");

    for (ix = 0; ix < nTasks; ++ix)
	{
	/* find name in symbol table */

	pTcbX = taskTcbX (idList [ix]);

	symValFind (sysSymTbl, (int) pTcbX->entry, name, 
		        (int *) &symbolAddress, (UTINY *) &type);

	if (symbolAddress != pTcbX->entry)
	    name [0] = EOS;	/* no matching symbol */


	/* print line for this task */

	totalPerCent     = (taskTotalTicks [ix] * 100) / spyTotalTicks;
	incPerCent       = (taskIncTicks [ix] * 100) / tmpIncTicks;
	sumTotalPerCent += totalPerCent;
	sumIncPerCent   += incPerCent;

	printf (spyFmt1, name, idList [ix], totalPerCent, taskTotalTicks [ix],
					    incPerCent, taskIncTicks [ix]);
	}

    totalPerCent     = (spyInterruptTicks * 100) / spyTotalTicks;
    incPerCent       = (tmpInterruptIncTicks * 100) / tmpIncTicks;
    sumTotalPerCent += totalPerCent;
    sumIncPerCent   += incPerCent;

    printf (spyFmt2, "INTERRUPT", "", totalPerCent, spyInterruptTicks,
				      incPerCent, tmpInterruptIncTicks);

    totalPerCent     = (spyIdleTicks * 100) / spyTotalTicks;
    incPerCent       = (tmpIdleIncTicks * 100) / tmpIncTicks;
    sumTotalPerCent += totalPerCent;
    sumIncPerCent   += incPerCent;

    printf (spyFmt2, "IDLE", "", totalPerCent, spyIdleTicks,
				 incPerCent, tmpIdleIncTicks);

    printf (spyFmt2, "TOTAL", "", sumTotalPerCent, spyTotalTicks,
				  sumIncPerCent, tmpIncTicks);
    
    printf ("\n");
    
    if (spyCreateCount > 0)
	{
	printf ("%d task%s created.\n", spyCreateCount,
		spyCreateCount == 1 ? " was" : "s were");
	spyCreateCount = 0;
	}

    if (spyDeleteCount > 0)
	{
	printf ("%d task%s deleted.\n", spyDeleteCount,
		spyDeleteCount == 1 ? " was" : "s were");
	spyDeleteCount = 0;
	}
    }
/*******************************************************************************
*
* spyTask - run periodic task activity reports
*
* This routine is spawned as a task by spy (2), in order to provide the
* periodic task activity reports.  It simply loops printing a report,
* then delaying for the specified number of seconds.
*/

VOID spyTask (freq)
    int freq;		/* reporting frequency, in seconds */

    {
    int delay = freq * sysClkGetRate ();

    while (TRUE)
	{
	taReport ();
	taskDelay (delay);
	}
    }
/*******************************************************************************
*
* stopSpy - stop spying and reporting
*
* This routine calls taStop (2) and 
* if periodic reporting by spyTask (2) is active, it is terminated.
*/

int stopSpy ()

    {
    taStop ();
    if (spyTaskId != ERROR)
	{
	taskDelete (spyTaskId);
	spyTaskId = ERROR;
	return(OK);
	}
    return(ERROR);
    }
/*******************************************************************************
*
* spy - make regular task activity reports
*
* Collect task activity data, and periodically run taReport (2).
* Data will be gathered `ticksPerSec' times per second, and a report will
* be made every `freq' seconds.
* Spy spawns spyTask (2) to do the actual reporting.
* 
* It is not necessary to do a taStart (2) before running this!
*/

int spy (freq, ticksPerSec)
    int freq;			/* reporting frequency, in seconds */
				/* 0 = use default of 5 */
    int ticksPerSec;		/* interrupt clock frequency */
				/* 0 = use default of 100 */

    {
    if (freq == 0)
	freq = 5;	/* default frequency is 5 secs */

    if (taStart (ticksPerSec) == OK)
	{
	spyTaskId = taskSpawn ("spyTask", spyTaskPriority,
			       spyTaskOptions, spyTaskStackSize,
			       spyTask, freq, 0, 0, 0, 0, 0, 0, 0, 0, 0);
	if (spyTaskId == ERROR) {
	    printErr ("Unable to spawn spyTask.\n");
	    return(ERROR);
	}
	return(OK);
	}
    return(ERROR);
    }
/*******************************************************************************
*
* tahelp - print a helpful list of the task activity commands
*
* This routine prints the following list of task activity commands:
*
* .CS
*	tahelp                        Print this list
*	taStart [ticksPerSec]         Start task activity monitor running
*	                                at ticksPerSec ticks per second
*	taStop                        Stop collecting data
*	taReport                      Prints display of task activity
*	                                statistics
*	stopSpy                       Stop collecting data and stop reports
*	spy [freq[,ticksPerSec]]      Start taStart and do a report
*	                                every freq seconds
*	
*	ticksPerSec defaults to 100.  freq defaults to 5 seconds.
* .CE
*/

VOID tahelp ()

    {
    static char *spy_help[] = {
    "tahelp                        Print this list",
    "taStart [ticksPerSec]         Start task activity monitor running",
    "                                at ticksPerSec ticks per second",
    "taStop                        Stop collecting data",
    "taReport                      Prints display of task activity",
    "                                statistics",
    "stopSpy                       Stop collecting data and reports",
    "spy     [freq[,ticksPerSec]]  Start taStart and do a report",
    "                                every freq seconds",
    "",
    "ticksPerSec defaults to 100.  freq defaults to 5 seconds.",
    "",
    };
    FAST int ix;

    for (ix = 0; ix < NELEMENTS(spy_help); ix++)
	printf ("%s\n", spy_help [ix]);
    }
