/* symLib.c - symbol table subroutine library */

static char *copyright = "Copyright 1984-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
This is a facility for creating and manipulating symbol tables.
Each table entry consists of a symbol name, a value (normally used for 
an address) and a type (normally a UNIX type specification, as define 
in a.out.h).  Symbol names are variable length, up to a maximum specified 
in the table in which they are stored.  Tables are created with symCreate, 
symbols are added with symAdd, symbols may be looked up with symFind and
symValFind.  The value and type fields can be used for anything you
like, since they are treated here simply as data.

INCLUDE FILE: symLib.h

DEFICIENCIES
Since it is just an unordered table, it is slow.  There is currently no
way to delete symbols from the table, or to deallocate tables.
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "memLib.h"
#include "strLib.h"
#include "symLib.h"

/*******************************************************************************
*
* symAdd - add a symbol to a symbol table
*
* Add a symbol to the specified symbol table.  The symbol will be added at
* the end.
*
* RETURNS:
*	OK, or
*	ERROR if table full.
*/

STATUS symAdd (pTbl, name, adrs, type)
    FAST SYMTAB *pTbl;	/* pointer to symbol table to add to */
    char *name;		/* pointer to symbol name string */
    char *adrs;		/* symbol address */
    UTINY type;		/* symbol type */

    {
    FAST SYMBOL *pSte;
    FAST char *symName;
    FAST int length;

    /* check for symbol table already full */

    if (pTbl->nsymbols == pTbl->maxSymbols)
	{
	errnoSet (S_symLib_FULL_SYMBOL_TABLE);
	return (ERROR);
	}

    /* limit symbol length to maximum for the given table */

    length = strlen (name);
    if (length > pTbl->maxSymLen)
	length = pTbl->maxSymLen;

    /* allocate memory to hold string and copy string to it */

    symName = malloc ((unsigned) (length + 1));
    if (symName == NULL)
	return (ERROR);

    strncpy (symName, name, length);
    symName[length] = EOS;

    /* add symbol */

    pSte = &pTbl->symbols [pTbl->nsymbols++];
    pSte->name = symName;
    pSte->adrs = adrs;
    pSte->type = type;

    return (OK);
    }
/*******************************************************************************
*
* symCreate - create a symbol table
*
* Create and initialize symbol table of the specified size.
*
* RETURNS:
*	Symbol table ID, or
*	NULL if not enough room.
*/

SYMTAB_ID symCreate (maxSymbols, maxSymLen)
    int maxSymbols;	/* maximum number of symbols in table */
    int maxSymLen; 	/* maximum symbol length */

    {
    SYMTAB_ID symtab = (SYMTAB_ID) malloc ((unsigned) (sizeof (SYMTAB) +
				 	   (maxSymbols * sizeof (SYMBOL))));

    if (symtab != NULL)
	{
	symtab->symbols = (SYMBOL *) ((char *) symtab + sizeof (SYMTAB));
	symInit (symtab, maxSymbols, maxSymLen);
	}

    return (symtab);
    }
/*******************************************************************************
*
* symEach - call a routine for each entry in a symbol table
*
* This routine calls a user-supplied routine once for each entry in the
* symbol table.  The routine should be declared as follows:
*
* .CS
*  BOOL routine (name, val, type, arg)
*      char *name;   /* entry name *
*      int val;      /* value associated with the entry *
*      UTINY type;   /* entry type *
*      int arg;      /* arbitrary user-supplied argument *
* .CE
*
* The user-supplied routine should return TRUE if symEach is to continue
* calling it for each entry, or FALSE if it is done and symEach can exit.
*
* VARARGS2
*/

VOID symEach (pTbl, routine, routineArg)
    SYMTAB *pTbl;	/* pointer to symbol table */
    FUNCPTR routine;	/* the routine to call for each table entry */
    int routineArg;	/* arbitrary user-supplied argument */

    {
    FAST SYMBOL *pSte0 = &pTbl->symbols[0]; /* first symbol table entry */
    FAST SYMBOL *pSte = &pTbl->symbols [pTbl->nsymbols-1]; /* current entry */

    for (; pSte >= pSte0; --pSte)
	{
	if (!(*routine) (pSte->name, (int) pSte->adrs, pSte->type, routineArg))
	    break;
	}
    }
/*******************************************************************************
*
* symFind - find a symbol in a symbol table
*
* Find a symbol in the table, by name.  If we find the symbol, we fill
* in the type and the value of the symbol.
*
* RETURNS:
*	OK, or
*	ERROR if symbol not found.
*/

STATUS symFind (pTbl, name, pAdrs, pType)
    FAST SYMTAB *pTbl;	/* pointer to symbol table in which to look */
    FAST char *name;	/* symbol name to look for  */
    char **pAdrs;	/* pointer where to return symbol address */
    UTINY *pType;	/* pointer where to return symbol type */

    {
    FAST SYMBOL *pSte0 = &pTbl->symbols[0]; /* beginning of symbol table */
    FAST SYMBOL *pSte = &pTbl->symbols[pTbl->nsymbols-1]; /* current entry */
    int symLen = pTbl->maxSymLen;

    /* look at entries in symbol table, starting at the end */

    for (; pSte >= pSte0; --pSte)
	{
	if (strncmp (name, pSte->name, symLen) == 0)
	    {
	    *pAdrs = pSte->adrs;
	    *pType = pSte->type;

	    return (OK);
	    }
	}

    errnoSet (S_symLib_SYMBOL_NOT_FOUND);
    return (ERROR);
    }
/*******************************************************************************
*
* symFindType - find a symbol with specified type in a symbol table
*
* Find a symbol in the table, by name. Skip symbol table entries of non-matching
* type. If we find the symbol, we fill in the type and the value of the symbol.
*
* RETURNS:
*	OK, or
*	ERROR if symbol not found.
*/

STATUS symFindType (pTbl, name, pAdrs, pType, sType, mask)
    FAST SYMTAB *pTbl;	/* pointer to symbol table in which to look */
    FAST char *name;	/* symbol name to look for  */
    char **pAdrs;	/* pointer where to return symbol address */
    UTINY *pType;	/* pointer where to return symbol type */
    FAST UTINY sType;	/* symbol type to look for */
    FAST UTINY mask;	/* which bits in 'sType' to pay attention to */

    {
    FAST SYMBOL *pSte0 = &pTbl->symbols[0]; /* beginning of symbol table */
    FAST SYMBOL *pSte = &pTbl->symbols[pTbl->nsymbols - 1]; /* current entry */
    int symLen = pTbl->maxSymLen;

    sType &= mask;		/* strip out ignorable bits */

    /* look at entries in symbol table, starting at the end */

    for (; pSte >= pSte0; --pSte)
	{
	if (((pSte->type & mask) == sType) &&
	    (strncmp (name, pSte->name, symLen) == 0))
	    {
	    *pAdrs = pSte->adrs;
	    *pType = pSte->type;
	    return (OK);
	    }
	}

    errnoSet (S_symLib_SYMBOL_NOT_FOUND);
    return (ERROR);
    }
/***********************************************************************
*
* symInit - initialize a symbol table
*
* Initialize a symbol table.  The table must have already been created
* with symCreate (2).  Any symbols currently in the table will be lost.
*
* RETURNS: OK (always)
*/

STATUS symInit (pTbl, maxSymbols, maxSymLen)
    SYMTAB *pTbl;	/* pointer to symbol table to initialize */
    int maxSymbols;	/* maximum number of symbols in table */
    short maxSymLen;	/* maximum symbol length */

    {
    pTbl->nsymbols   = 0;		/* set number of active symbols to 0 */
    pTbl->maxSymbols = maxSymbols;
    pTbl->maxSymLen  = maxSymLen;

    return (OK);
    }
/***********************************************************************
*
* symValFind - find a symbol in a symbol table, given the value
*
* This routine finds a symbol table entry, given the value associated with
* that entry.  If there is no entry whose value is equal, it returns the
* entry whose value is just lower than the given value.
* The symbol name returned is up to pTbl->maxSymLen characters in length,
* plus a terminating EOS.  It also sets the type and the actual value
* of the symbol found.
*
* RETURNS:
*	OK, or
*	ERROR if val < lowest value in the table.
*/

STATUS symValFind (pTbl, val, name, pval, pType)
    FAST SYMTAB *pTbl;	/* pointer to symbol table */
    FAST int val;	/* value of symbol to find */
    char *name;		/* pointer where to return symbol name string */
    int *pval;		/* pointer where to return symbol address */
    UTINY *pType;	/* pointer where to return symbol type */

    {
    FAST SYMBOL *pSte0 = &pTbl->symbols[0]; /* beginning of symbol table */
    FAST SYMBOL *pSte = &pTbl->symbols[pTbl->nsymbols - 1]; /* current entry */
    FAST SYMBOL *pBest;
    FAST int bestval = 0;

    for (; pSte >= pSte0; --pSte)
	{
	if ((int) pSte->adrs == val)
	    {
	    /* We've found the entry.  Return it. */

	    strcpy (name, pSte->name);
	    *pval = (int) pSte->adrs;
	    *pType = pSte->type;

	    return (OK);
	    }
	else if (((int) pSte->adrs < val) && ((int) pSte->adrs > bestval))
	    {
	    /* We've found an entry closer than the best we had previously
	       found.  Keep track of it. */

	    pBest = pSte;
	    bestval = (int) pSte->adrs;
	    }
	}

    if (bestval > 0)
	{
	/* Return the info from the closest entry we found. */

	strcpy (name, pBest->name);
	*pval = (int) pBest->adrs;
	*pType = pBest->type;

	return (OK);
	}

    errnoSet (S_symLib_SYMBOL_NOT_FOUND);
    return (ERROR);	/* If the symbol table was empty */
    }
