/* rpcLib.c - rpc support library */

static char *copyright = "Copyright 1987,1988 Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/* 
UniWorks supports the SUN Microsystems' Remote Procedure Call library.  
RPC provides facilities for implementing client/server based architectures.
The underlying interprocess communication mechanism can be completely hidden,
permitting applications to be written without any reference to network sockets.
The package is structured such that a sophisticated user may access
lower level routines allowing greater control of the communication protocols.
The user is directed to SUN's "Remote Procedure Call Programming Guide" for
a complete discussion and tutorial.  An example of the use of RPC is provided
in the UniWorks demonstration directory; it is called "sprites".

The RPC facility is enabled by defining the constant "INCLUDE_RPC" in
the target CPU config.h file, e.g.:
.CS
    #define INCLUDE_RPC
.CE
UniWorks supports NFS (Network File System) and SUN's dbxtool,
both of which are built on top of RPC.
If either NFS or DBX are configured into the system, then RPC is as well.

IMPLEMENTATION
Every task which is going to make calls to the RPC library must first call
rpcTaskInit (2).  This routine installs a UniWorks task variable which contains
pointers to static data structures in the RPC package that must be switched
on each task context switch.

Be aware that the callrpc routine opens a socket that
will never be closed;  this is fine for a client task that remains alive
forever, however, in applications that spawn short lived tasks that
call callrpc, the I/O system will soon run out of file descriptors.  To
avoid this problem, instead of using callrpc,
use clntudp_create or clnttcp_create to create the RPC client object, and
then use clnt_call to make the actual RPC call.  The application is
then responsible for closing the associated socket when done with it.

RPC related objects (such as SVCXPRT's and CLIENT's) may not be 
shared among tasks;  objects created by one task may not be passed to 
another for use.  This is because the RPC package
contains static data structures that refer to these objects, and the
UniWorks implementation of RPC uses UniWorks task variables to switch
in a different copy of the static data structures for each task.

INCLUDE FILE: rpc.h

SEE ALSO: SUN RPC programmers guide, nfsLib (1), nfsDrv (3), dbxLib (1)
*/

#include "rpctypes.h"
#include "UniWorks.h"
#include "rpcGbl.h"
#include "taskLib.h"
#include "memLib.h"


IMPORT VOID portmapd ();


MODULE_LIST *taskModuleList;	/* task variable cell that points to rpc
				 * task structure */

int portmapdId;
int portmapdPriority  = 100;
int portmapdStackSize = 5000;
int portmapdOptions   = VX_SUPERVISOR_MODE | VX_UNBREAKABLE;

/* forward declarations */

LOCAL VOID rpcTaskDeleteHook ();


/*******************************************************************************
*
* rpcInit - initialize RPC package
*
* This routine must be called before using RPC; it spawns the portmap daemon.
* It is usually called by the root task, usrRoot(2), in usrConfig(1).
*
* RETURNS: OK or ERROR if unable to spawn portmapd
*/

STATUS rpcInit ()

    {
    /* spawn the portmap daemon */

    portmapdId = taskSpawn ("portmapd", portmapdPriority,
			    portmapdOptions, portmapdStackSize,
			    portmapd, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);

    return ((portmapdId == ERROR) ? ERROR : OK);
    }
/*******************************************************************************
*
* rpcTaskInit - initialize task's access to RPC package
*
* This routine must be called by a task before it makes any calls to 
* other routines in the RPC package.
* 
* RETURNS: OK or ERROR if out of memory, or unable to add task switch hooks
*/

STATUS rpcTaskInit ()

    {
    static BOOL rpcInstalled = FALSE;
    FAST MODULE_LIST *pModuleList;

    /* if rpcTaskInit has already been called for this task, 
       the task variable will no longer be null */

    if (taskModuleList != NULL)
	return (OK);

    /* allocate the block of pointers which becomes the task switch variable */

    pModuleList = (MODULE_LIST *) malloc (sizeof (MODULE_LIST));

    if (pModuleList == NULL)
	return (ERROR);

    bzero ((char *) pModuleList, sizeof (MODULE_LIST));

    /* each module contains an initialization routine which is responsible
       for allocating its own static variable space and filling in the 
       pointer to that space in the supplied MODULE_LIST */

    if ((auth_noneInit (pModuleList)   == ERROR) ||
        (clnt_rawInit (pModuleList)    == ERROR) ||
        (clnt_simpleInit (pModuleList) == ERROR) ||
        (svcInit (pModuleList)         == ERROR) ||
        (svc_rawInit (pModuleList)     == ERROR) ||
        (svc_simpleInit (pModuleList)  == ERROR))
	{
	return (ERROR);
	}

    /* NOTE: nfsClientCache may be set by nfs */

    /* now install taskModuleList as a task variable */

    if (taskVarAdd (0, (int *) &taskModuleList) == ERROR)
	return (ERROR);

    taskModuleList = pModuleList;

    if (!rpcInstalled)
	{
	if (taskDeleteHookAdd (rpcTaskDeleteHook) == ERROR)
	    return (ERROR);
	rpcInstalled = TRUE;
	}

    return (OK);
    }
/*******************************************************************************
*
* rpcTaskDeleteHook - deallocate RPC resources of exiting task
*
* This routine is the task delete hook for tasks using the RPC package.
* It is installed by rpcTaskInit.
*/

LOCAL VOID rpcTaskDeleteHook (pTcbX)
    FAST TCBX *pTcbX;	/* TCB extension of exiting task */

    {
    FAST MODULE_LIST *pModuleList;
    
    pModuleList = (MODULE_LIST *) taskVarGet (pTcbX->taskId,
					      (int *) &taskModuleList);

    if ((int) pModuleList == ERROR)
	return;	/* task didn't use RPC */

    /* free all the module list data structures */

    free ((char *)pModuleList->auth_none);
    free ((char *)pModuleList->clnt_raw);
    free ((char *)pModuleList->clnt_simple);
    free ((char *)pModuleList->svc);
    free ((char *)pModuleList->svc_raw);
    free ((char *)pModuleList->svc_simple);

    if ((char *)pModuleList->nfsClientCache != NULL)
	free ((char *)pModuleList->nfsClientCache);

    /* free the pModuleList itself */

    free ((char *)pModuleList);
    }
/*******************************************************************************
*
* abort - force a task to exit
*
* abort is provided for the handling of catastrophic RPC failures.
* It is referenced internally by the RPC library, and SHOULD NOT BE
* CALLED BY THE USER.
*
* NOMANUAL
*/

VOID abort ()

    {
    printErr ("task %d - aborted.\n", taskIdSelf ());
    exit (-1);
    }
/******************************************************************************
*
* rpcClntErrnoSet - set RPC client status
*
* rpcClntErrnoSet calls errnoSet with the given "rpc stat" or'd with the
* rpc status prefix.  
*
* NOMANUAL
*/

VOID rpcClntErrnoSet (status)
    int status;

    {
    errnoSet (M_rpcClntStat | (int) status);
    }
