/* routeLib.c - route library */

static char *copyright = "Copyright 1987,1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION

This library provides routines which deal with the network routing tables.
Routines are provided for adding and deleting routes that go through a
passive gateway.  routeShow() is provided for viewing the route tables.
*/

#include "UniWorks.h"
#include "mbuf.h"
#include "socket.h"
#include "route.h"
#include "in.h"
#include "ioctl.h"
#include "inetLib.h"

/* forward declarations */

STATUS routeAdd ();
STATUS routeDelete ();
VOID routeShow ();

/****************************************************************************
*
* routeAdd - add a route
*
* routeAdd updates the routing tables to include a gateway to the 
* given destination.  It is called from the UniWorks machine which needs
* access to the destination network (or machine) through a gateway.
* 
* The "destination" and "gateway" may be specified either by their internet
* addresses in standard internet address format (eg "90.0.0.2"), or by their
* names (eg "hostname") which have already been added to the remote host
* table (with remAddHost).
* 
* EXAMPLE
* The call:
* .CS
*	routeAdd ("90.0.0.0", "gate")
* .CE
* tells UniWorks that the machine with the host name "gate" is our gateway to
* the "90.0.0.0" net.  The host "gate" must already have been created by
* remAddHost(2).
*
* The call:
* .CS
*	routeAdd ("90.0.0.0", "91.0.0.3")
* .CE
* tells UniWorks that the machine with the Internet address "91.0.0.3" is our
* gateway to the "90.0.0.0" net.
*
* The call:
* .CS
*	routeAdd ("destination", "gate")
* .CE
* tells UniWorks that the machine with the host name "gate" is our gateway to
* the machine refered to as "destination".  The host names "gate" and
* "destination" must have already been created by remAddHost(2).
*
* RETURNS: OK | ERROR
*
*/

STATUS routeAdd (destination, gateway)
    char *destination;	/* inet adrs or name of route destination */
    char *gateway;	/* inet adrs or name of gateway to dest */

    {
    return (routeChange (destination, gateway, (int) SIOCADDRT));
    }

/****************************************************************************
*
* routeDelete - delete a route
*
* routeDelete deletes the given route from the system routing tables.
*
* SEE ALSO: routeAdd(2) for destination and gateway formats.
*
* RETURNS: OK | ERROR.
*
*/

STATUS routeDelete (destination, gateway)
    char *destination;	/* inet adrs or name of route destination */
    char *gateway;	/* inet adrs or name of gateway to dest */

    {
    return (routeChange (destination, gateway, (int) SIOCDELRT));
    }

/****************************************************************************
*
* routeChange - change the routing tables
*
* routeChange makes changes to the routing tables by adding or deleting
* a route with an ioctl call.
*
* This is a local routine which should not be used directly.
* Instead, routeAdd() and routeDelete() should be used as interfaces
* to this routine.
* 
* The "destination" and "gateway" may be specified either by their internet
* addresses in standard internet address format (eg "90.0.0.2"), or by their
* names (their host names) which have already been added to the remote host
* table (with remAddHost).
*
* RETURNS:  OK | ERROR.
*/

LOCAL STATUS routeChange (destination, gateway, ioctlCmd)
    char *destination;	/* internet adrs or name of route destination */
    char *gateway;	/* internet adrs or name of gateway to destination */
    int	ioctlCmd;	/* addition or deletion of a route */

    {
    int destInetAddr;		/* destination internet adrs */
    int gateInetAddr;		/* gateway internet adrs */
    struct rtentry route;	/* route entry */
    int s;
    int status;

    if ((((destInetAddr = remGetHostByName (destination)) == ERROR) &&
	 ((destInetAddr = (int) inet_addr (destination)) == ERROR)) ||
        (((gateInetAddr = remGetHostByName (gateway)) == ERROR) &&
	 ((gateInetAddr = (int) inet_addr (gateway)) == ERROR)))
	{
	return (ERROR);
	}

    routeFillEntry (&route, destInetAddr, gateInetAddr);

    /* add or delete the route */

    s = socket (AF_INET, SOCK_STREAM, 0);
    status = ioctl (s, ioctlCmd, (int) &route);
    close (s);
    return (status);
    }

/****************************************************************************
*
* routeFillEntry - fill in a route entry (struct rtentry)
*
* Fills in a route entry (struct rtentry) with destination and
* gateway information.
*
*/

LOCAL VOID routeFillEntry (pRoute, destInetAddr, gateInetAddr)
    struct rtentry *pRoute;	/* pointer to route entry */
    int destInetAddr;		/* destination internet address */
    int gateInetAddr;		/* gateway internet address */

    {
    struct sockaddr_in	*sin;

    /* zero out route entry */

    bzero ((caddr_t) pRoute, sizeof (struct rtentry));

    /* zero out sockaddr_in, fill in destination info */

    sin = (struct sockaddr_in *) &(pRoute->rt_dst);
    bzero ((caddr_t) sin, sizeof (struct sockaddr_in));
    sin->sin_family = AF_INET;
    sin->sin_addr.s_addr = destInetAddr;

    /* zero out sockaddr_in, fill in gateway info */

    sin = (struct sockaddr_in *) &(pRoute->rt_gateway);
    bzero ((caddr_t) sin, sizeof (struct sockaddr_in));
    sin->sin_family = AF_INET;
    sin->sin_addr.s_addr = gateInetAddr;

    pRoute->rt_flags = RTF_UP | RTF_GATEWAY;

    if (in_lnaof (destInetAddr) != INADDR_ANY)
	pRoute->rt_flags |= RTF_HOST;
    }
/*****************************************************************************
*
* routeShow - print route tables (host and net)
*
* This routine prints a nice table, showing the current routing information.
*/

VOID routeShow ()

    {
    register struct mbuf *m;
    register struct rtentry *rt;
    int i;

    printf ("\nROUTE NET TABLE\n");
    printf (" destination      gateway          flags\n");
    printf ("---------------------------------------\n");

    for (i = 0; i < RTHASHSIZ ; i++)
	{
	for (m = rtnet [i]; m; m = m->m_next)
	    {
	    rt = mtod(m, struct rtentry *);
	    routePrintRtentry (rt);
	    }
	}

    printf ("---------------------------------------\n");
    printf ("\nROUTE HOST TABLE\n");
    printf (" destination      gateway          flags\n");
    printf ("---------------------------------------\n");

    for (i = 0; i < RTHASHSIZ ; i++)
	{
	for (m = rthost [i]; m; m = m->m_next)
	    {
	    rt = mtod(m, struct rtentry *);
	    routePrintRtentry (rt);
	    }
	}
	printf ("---------------------------------------\n");
    }

/*****************************************************************************
*
* routePrintRtentry - print one route entry (rtentry)
*
*/

LOCAL VOID routePrintRtentry (rt)
    register struct rtentry *rt;

    {
    register struct sockaddr_in *sin;

    /* print destination internet address */

    sin = (struct sockaddr_in *) &(rt->rt_dst);
    printf ("%-16s ", (rt->rt_dst.sa_family == AF_INET) ?
		      inet_ntoa (sin->sin_addr) : "not AF_INET");
    
    /* print gateway internet address */

    sin = (struct sockaddr_in *) &(rt->rt_gateway);
    printf ("%-16s ", (rt->rt_gateway.sa_family == AF_INET) ?
		      inet_ntoa (sin->sin_addr) : "not AF_INET");
    
    printf ("%3d \n", rt->rt_flags);
    }
