/* netLib.c - network interface library */

static char *copyright = "Copyright 1987,1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
NetLib provides facilities related to the UniWorks network.  It includes
tools for manipulating the network routing tables, initialization
routines, the network daemon, and informational utilities for observing
the network.
*/

#include "param.h"
#include "types.h"
#include "UniWorks.h"
#include "inetLib.h"
#include "lstLib.h"
#include "rngLib.h"
#include "semLib.h"
#include "strLib.h"
#include "taskLib.h"
#include "systm.h"
#include "sysSymTbl.h"

#include "socket.h"
#include "in.h"
#include "route.h"

#include "if.h"
#include "ioctl.h"


IMPORT SEM_ID splSemId;
IMPORT ifreset ();

typedef struct
    {
    FUNCPTR routine;	/* routine to be called */
    int param1;		/* arg to routine */
    int param2;
    int param3;
    int param4;
    int param5;
    } TODO_NODE;

#define JOB_RING_SIZE (2 * 85 * sizeof (TODO_NODE))

/* netd parameters */

int netdId;
int netdPriority	= 5;	/* task priority at which network stuff runs */
int netdOptions	= VX_SUPERVISOR_MODE | VX_UNBREAKABLE;
int netdStackSize	= 4000;

/* local variables */

LOCAL SEM_ID netdSemId;
LOCAL RING_ID netJobRing;

/*****************************************************************************
*
* netd - the network daemon
*
* netd is spawned as a task by netStart (2).  It provides a facility for
* running network routines in a task context.
*
* INTERNAL
* netd reads messages from a ring buffer which is filled by calling
* netToDoAdd.
*/

VOID netd ()

    {
    TODO_NODE jobNode;
    int s;

    FOREVER
	{
	/* wait for somebody to wake us up */

	semTake (netdSemId);

	/* process requests in the toDo list */

	while (rngEmpty (netJobRing) == FALSE)
	    {
	    if (rngGetBuf (netJobRing, (char *) &jobNode, 
			   sizeof (jobNode)) != sizeof (jobNode))
		panic ("netd: netJobRing overflow!!!\n");

	    s = splnet ();

	    (*(jobNode.routine)) (jobNode.param1, jobNode.param2,
				  jobNode.param3, jobNode.param4,
				  jobNode.param5);
	    splx (s);
	    }
	}
    }
/*****************************************************************************
*
* netToDoAdd - add a routine to the netJobRing
*
* this routine provides an interface to the net daemon - tasks and interrupt
* level code may call this routine to have the net daemon do task level
* processing of network events.
*
* NOMANUAL
* VARARGS1
*/

STATUS netToDoAdd (routine, param1, param2, param3, param4, param5)
    FUNCPTR routine;
    int param1;
    int param2;
    int param3;
    int param4;
    int param5;

    {
    FAST int oldlevel;
    TODO_NODE newNode;
    BOOL ok;

    newNode.routine = routine;
    newNode.param1 = param1;
    newNode.param2 = param2;
    newNode.param3 = param3;
    newNode.param4 = param4;
    newNode.param5 = param5;

    oldlevel = intLock ();
    ok = rngPutBuf (netJobRing, (char *) &newNode, sizeof (newNode)) == 
							sizeof (newNode);
    intUnlock (oldlevel);

    if (!ok)
	panic ("netToDoAdd: ring buffer overflow\n");

    /* wake up the network daemon to process the request */

    semGive (netdSemId);

    return (OK);
    }
/*****************************************************************************
*
* netStart - initialize the network package
*
* netStart should be called once to initialize the network.
*
* RETURNS: OK or ERROR if error creating netd task
*/

STATUS netStart ()

    {
    static BOOL done = FALSE;

    if (done)
	return (netdId == ERROR ? ERROR : OK);

    done = TRUE;

    /* install socket driver */

    if (sockInit () == ERROR)
	panic ("netLibInit: couldn't install socket driver\n");

    if ((netJobRing = rngCreate (JOB_RING_SIZE)) == (RING_ID) NULL)
	panic ("netStart: couldn't create job ring\n");

    if (rebootHookAdd (ifreset) == ERROR)
	logMsg ("netLibInit: unable to add reset hook\n");

    netdSemId = semCreate ();
    splSemId = semCreate ();
    semGive (splSemId);

    mbinit ();

    ifinit ();
    domaininit ();

    netdId = taskSpawn ("netd", netdPriority,
		           netdOptions, netdStackSize,
			   netd, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
	
    return (netdId == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* nethelp - help command for the network
*
* Print a brief synopsis of network facilities that are typically called from
* the shell.
* .CS
*  remAddHost   "hostname","inetaddr" - add a host to the remote host
*                                       table. "inetaddr" must be in
*                                       standard internet address format
*                                       eg "90.0.0.4"
*  remShowHosts                       - prints current remote host table
*  netDevCreate "devname","hostname",protocol
*                                     - create an i/o device to access
*                                       files on the specified host
*                                       (protocol 0=rsh, 1=ftp)
*  routeAdd     "destaddr","gateaddr" - add route to route table
*  routeShow                          - prints current route table
*  iam          "usr"[,"passwd"]      - specify the user name by which
*                                       you will be known to remote
*                                       hosts (and optional password)
*  whoami                             - prints the current remote id
*  rlogin       "host"                - login to a remote host;
*                                       "host" can be inet address or
*                                       host name in remote host table
*
*  EXAMPLE:  -> remAddHost "wrs", "90.0.0.2"
*            -> netDevCreate "wrs:", "wrs", 0
*            -> iam "fred"
*            -> copy <wrs:/etc/passwd   /* copy file from host "wrs"
*            -> rlogin "wrs"            /* rlogin to host "wrs"
* .CE
*/

VOID nethelp ()

    {
 static char *help_msg [] =
 {
    /* page 1 */
 "remAddHost   \"hostname\",\"inetaddr\"    - add a host to the remote host table",
 "                                     \"inetaddr\" must be in standard",
 "                                     internet address format eg \"90.0.0.4\"",
 "remShowHosts                       - prints current remote host table",
 "netDevCreate \"devname\",\"hostname\",protocol",
 "                                   - create an i/o device to access files",
 "                                     on the specified host",
 "                                     (protocol 0=rsh, 1=ftp)",
 "routeAdd     \"destaddr\",\"gateaddr\" - add route to route table",
 "",
 "routeShow                          - prints current route table",
 "iam          \"usr\"[,\"passwd\"]      - specify the user name by which you",
 "                                     will be known to remote hosts",
 "                                     (and optional password)",
 "whoami                             - prints the current remote id",
 "rlogin       \"host\"                - login to a remote host;",
 "                                     \"host\" can be inet address or",
 "                                     host name in remote host table",
 "",
 "",
 /* page 2 */
 "EXAMPLE:   -> remAddHost \"wrs\", \"90.0.0.2\"",
 "           -> netDevCreate \"wrs:\", \"wrs\", 0",
 "           -> iam \"fred\"",
 "           -> copy <wrs:/etc/passwd   /* copy file from host \"wrs\" */",
 "           -> rlogin \"wrs\"            /* rlogin to host \"wrs\" */",
 NULL
 };
    FAST int i;
    char ch;

    printf ("\n\n");
    for (i = 0; help_msg [i] != NULL; i++)
	{
	printf ("%s\n", help_msg [i]);
	if ((i+1) % 20 == 0)
	    {
	    printf ("\nType <CR> to continue, Q<CR> to stop: ");
	    fioRdString (STD_IN, &ch, 1);
	    if (ch == 'q' || ch == 'Q')
		break;
	    else
		printf ("\n");
	    }
	}
    printf ("\n");
    }
/*******************************************************************************
*
* netErrnoSet - set network error status
*
* netErrnoSet calls errnoSet with the given "errno" or'ed with the
* network status prefix.  
*/

VOID netErrnoSet (status)
    int status;

    {
    errnoSet (M_errno | status);
    }

RING Prtring;
extern SYMTAB_ID sysSymTbl;

netPrtRing()
{
	int s,i;
	char buf[JOB_RING_SIZE];
	char str[20];
	int pval,ptype;
	TODO_NODE todonode;

	while(1) {
		s = intLock();
		bcopy(netJobRing, &Prtring, sizeof(RING));
		bcopy(netJobRing->buf, buf, netJobRing->bufSize);
		intUnlock(s);
		if ( rngEmpty(&Prtring) ) {
			taskDelay(1);
			continue;
		}
	
#ifdef VERBOSE
		printf("\nnetd netJobRing:  pToBuf=%d pFromBuf=%d bufSize=%d\n\n",
		    Prtring.pToBuf, Prtring.pFromBuf, Prtring.bufSize);
#endif VERBOSE
		printf("\n");
		i = 1;
		while ( ! rngEmpty(&Prtring) ) {
			rngGetBuf(&Prtring, &todonode, sizeof(todonode));
			symValFind(sysSymTbl,todonode.routine,str,&pval,&ptype);
			if((int)todonode.routine == pval)
				printf("%d %13s   0x%08x   0x%08x\n",
				    i,str,todonode.param1,
				    todonode.param2,todonode.param3,	
				    todonode.param4,todonode.param5);	
			else
				printf("%d 0x%08x   0x%08x   0x%08x\n",
				    i,todonode.routine,todonode.param1,
				    todonode.param2,todonode.param3,	
				    todonode.param4,todonode.param5);	
		i++;
		}
	}

}

