/* ifLib.c - network interface subroutine library */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
*/

#include "UniWorks.h"
#include "in.h"
#include "if.h"
#include "ioctl.h"
#include "inetLib.h"
#include "strLib.h"

/****************************************************************************
*
* ifconfig - configure a network interface
*
* ifconfig is used to assign an internet address to a particular interface.
* The named interface (supplied in the first parameter, eg "ex0" or "vb0")
* is assigned the internet address supplied in the second parameter, which
* may be a host name (which has been previously added to the host table
* with hostAdd (2)) or an internet address in standard internet address
* format (eg 90.0.0.4). 
*
* RETURNS:
*  OK, or
*  ERROR if address is already in table, or interface can't be configured.
*/

STATUS ifconfig (iname, addrfam, iaddr, argstr1, arg1, argstr2, arg2)
    char *iname;	/* Name of interface to configure */
    char *addrfam;		/* protocol - only inet is supported */
    char *iaddr;	/* Internet adrs to assign to interface */
    char *argstr1; 
    char *argstr2;
    int arg1; 
    int arg2;

    {
    char str[20];
    struct ifreq ifr;
    int so;
    STATUS status;
    int inetAddr;

    /* see if the address is in the host table */

    if (((inetAddr = remGetHostByName (iaddr)) == ERROR) &&
	((inetAddr = (int) inet_addr (iaddr)) == ERROR))
	return (ERROR);

    bzero ((caddr_t) &ifr.ifr_addr, sizeof (ifr.ifr_addr));
    ifr.ifr_addr.sa_family = AF_INET;
    ((struct sockaddr_in *) &ifr.ifr_addr)->sin_addr.s_addr = inetAddr;

    strncpy (ifr.ifr_name, iname, sizeof (ifr.ifr_name));

    /* we need a dummy socket to do the ioctl */    

    if ((so = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
	return (ERROR);

    status = ioctl (so, (int) SIOCSIFADDR, (int)&ifr);
    close (so);

    /* Check for netmask or broadcast params */

    if(argstr1 != NULL) {
	if(strcmp(argstr1,"netmask") == 0) {
		if(ifMaskSet(iname,arg1) == ERROR)
			return(ERROR);
    	} else if(strcmp(argstr1,"broadcast") == 0) {
		inet_ntoa_b(arg1,str);
		if(ifBroadcastSet(iname,str) == ERROR)
		return(ERROR);
    	}
    }

    if(argstr2 != NULL) {
    	if(strcmp(argstr2,"netmask") == 0) {
		if(ifMaskSet(iname,arg2) == ERROR)
			return(ERROR);
    	} else if(strcmp(argstr2,"netmask") == 0) {
		inet_ntoa_b(arg2,str);
		if(ifBroadcastSet(iname,str) == ERROR)
			return(ERROR);
    	}
    }

    return (status);
    }

/****************************************************************************
*
* ifBroadcastSet - set broadcast address for an interface
*
* ifBroadcastSet is used to change the broadcast address for a particular
* interface.
* The named interface (supplied in the first parameter, eg "ex0" or "bp0")
* is assigned the broadcast address supplied in the second parameter, which
* must be a string in standard internet address format (eg "90.0.0.0"). 
*
* The default broadcast address for an interface is the interface's inet
* address with a host part of all 1's (eg. "90.255.255.255").
* This conforms to current ARPA specifications.
* However, some older implementations of the Internet
* protocols use a host part of all 0's for the broadcast address.
* The UniWorks implementation will automatically receive a host part of 
* all 0's as a broadcast address, in addition to the default or specified
* broadcast address.  But if UniWorks is to broadcast to older systems
* using a host part of all 0's as the broadcast address, then this
* routine should be used to change the broadcast address of the interface.
*
* RETURNS: OK | ERROR
*/

STATUS ifBroadcastSet (interfaceName, broadcastAddress)
    char *interfaceName;	/* Name of interface to assign */
    char *broadcastAddress;	/* Broadcast adrs to assign to interface */

    {
    struct ifreq ifr;
    int so;
    STATUS status;
    int inetAddr;

    if ((inetAddr = (int) inet_addr (broadcastAddress)) == ERROR)
	return (ERROR);

    bzero ((caddr_t) &ifr.ifr_addr, sizeof (ifr.ifr_addr));
    ifr.ifr_addr.sa_family = AF_INET;
    ((struct sockaddr_in *) &ifr.ifr_addr)->sin_addr.s_addr = inetAddr;

    strncpy (ifr.ifr_name, interfaceName, sizeof (ifr.ifr_name));

    /* we need a dummy socket to do the ioctl */    

    if ((so = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
	return (ERROR);

    status = ioctl (so, (int) SIOCSIFBRDADDR, (int)&ifr);
    close (so);

    return (status);
    }

/****************************************************************************
*
* ifMaskSet - define a subnet for an interface.
*
* ifMaskSet may be used to allocate additional bits in an internet
* address to the network portion of the address.  The network portion  
* is specified with a mask that must contain ones in all positions
* that are to be interpreted as the network portion (this includes all
* the bits that are normally interpreted as the network portion for the
* given class of address, plus the additional bits that you wish to add.
* Note also that all bits must be contiguous.)
*
* NOTE: A subnet mask should be set for an interface prior to setting the
* interface's inet address with the routine ifconfig, so that the address
* will be interpreted correctly.
*
* SEE ALSO: UNIX man page for ifconfig.
*
* RETURNS: OK | ERROR
*/

STATUS ifMaskSet (ifName, netMask)
    char *ifName;	/* interface name */
    int netMask;	/* subnet mask */
    {
    struct ifreq ifr;
    int so;
    STATUS status;

    strncpy (ifr.ifr_name, ifName, sizeof (ifr.ifr_name));

    bzero ((caddr_t) &ifr.ifr_addr, sizeof (ifr.ifr_addr));
    ifr.ifr_addr.sa_family = AF_INET;
    ((struct sockaddr_in *) &ifr.ifr_addr)->sin_addr.s_addr = netMask;

    if ((so = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
	return (ERROR);

    status = ioctl (so, (int) SIOCSIFNETMASK, (caddr_t) &ifr); 
    close (so);

    return (status);
    }
