/* ftpLib.c - ARPA File Transfer Protocol library */

static char *copyright = "Copyright 1987-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION

This library provides facilities for transferring files to and from a
host via FTP (File Transfer Protocol).
This library provides only the "client" side of the FTP facilities.

FTP IN UNIWORKS
UniWorks supplies an I/O driver, netDrv (3), that allows transparent access of
remote files via standard I/O system calls.  In its FTP mode, netDrv
uses the FTP facilities provided by this library to access the remote files.
This is the primary use of FTP in UniWorks.  Thus for most purposes,
no familiarity with ftpLib itself is necessary.

HIGH LEVEL INTERFACE
The highest level of direct interface to FTP is provided by the routines
ftpXfer and ftpGetReply.
The routine ftpXfer connects to a specified remote FTP server,
logs in under a specified user name, and initiates a specified data transfer
command (see ftpXfer (2) for details).
The routine ftpGetReply is used to get control reply reply messages sent
by the remote FTP server in response to the commands sent.

LOW LEVEL INTERFACE
A lower level of interface is provided by the routines ftpHookup, 
ftpLogin, ftpInitDataConn, ftpGetDataConn, and ftpCommand.
These routines provide the primatives necessary to create and use both
control and data connections to remote FTP servers.
As an example of how to use these lower level routines, the following
code fragment implements roughly the same function as the ftpXfer routine
(see ftpXfer (2) for details):
.CS
    char *host, *user, *passwd, *acct, *dirname, *filename;
    int ctrlSock = ERROR;
    int dataSock = ERROR;

    if (((ctrlSock = ftpHookup (host)) == ERROR)		   ||
        (ftpLogin (ctrlSock, user, passwd, acct) == ERROR)	   ||
        (ftpCommand (ctrlSock, "TYPE I") != FTP_COMPLETE)	   ||
	(ftpCommand (ctrlSock, "CWD %s", dirname) != FTP_COMPLETE) ||
	((dataSock = ftpInitDataConn (ctrlSock)) == ERROR)	   ||
        (ftpCommand (ctrlSock, "RETR %s", filename) != FTP_PRELIM) ||
	((dataSock = ftpGetDataConn (dataSock)) == ERROR))
	{
	/* some error occured; close any open sockets and return *

	if (ctrlSock != ERROR)
	    close (ctrlSock);
	if (dataSock != ERROR)
	    close (dataSock);
	return (ERROR);
	}
.CE

INCLUDE FILE: ftpLib.h
*/

#include "ctype.h"
#include "UniWorks.h"
#include "socket.h"
#include "in.h"
#include "ftpLib.h"
#include "inetLib.h"

typedef struct sockaddr_in SOCKADDR_IN;

#define FTP_PORT	21

BOOL ftpVerbose = FALSE;	/* TRUE = print all incoming messages */
BOOL ftpDebug   = FALSE;	/* TRUE = print outgoing messages */

/************************************************************************
*
* ftpCommand - send FTP command and get reply
*
* This routine sends the specified command on the specified socket, which
* should be a control connection to a remote FTP server.
* The command is specified as a "printf" style format string with up
* to 6 arguments.
*
* This routine sends the command and then waits for the reply from the
* remote server.  The FTP reply code is returned just as in the call to
* routine ftpGetReply (2).
*
* RETURNS:
*   ERROR if read/write error or unexpected EOF;
*     or
*   FTP reply code most significant digit (reply class)
*   (see ftpGetReply (2) for details).
*
* EXAMPLE
* .CS
*    ftpCommand (ctrlSock, "TYPE I");		 /* image-type xfer *
*    ftpCommand (ctrlSock, "STOR %s", filename); /* init file write *
* .CE
*
* VARARGS2
*/

int ftpCommand (ctrlSock, fmt, arg1, arg2, arg3, arg4, arg5, arg6)
    int ctrlSock;	/* fd of control connection socket */
    char *fmt;		/* format string of command to send */
    int arg1;		/* arguments to format string */
    int arg2;
    int arg3;
    int arg4;
    int arg5;
    int arg6;

    {
    if (ftpDebug)
	{
	printErr ("---> ");
	printErr (fmt, arg1, arg2, arg3, arg4, arg5, arg6);
	printErr ("\n");
        }

    fdprintf (ctrlSock, fmt, arg1, arg2, arg3, arg4, arg5, arg6);
    fdprintf (ctrlSock, "\r\n");

    return (ftpGetReply (ctrlSock, !strcmp (fmt, "QUIT")));
    }
/**************************************************************************
*
* ftpXfer - initiate transfer via FTP
*
* This routine is used to initiate a transfer via a remote FTP server.
* It first establishes a connection to the FTP server on the specified host.
* Then it logs in with the specified user name, password, and account,
* as necessary for the particular host.
* Next the transfer type is set to "image" by sending the "TYPE I" command,
* and the directory is changed to the specified directory by sending
* the "CWD <dirname>" command.  Finally, the specified transfer command
* is sent with the specified filename as an argument, and a data connection
* is established.  The resulting control and data connection fds are returned
* via pCtrlSock and pDataSock respectively.
* Typical transfer commands would thus be "STOR %s" to write to a remote file,
* or "RETR %s" to read a remote file.
*
* After calling this routine, the data can be read or written to the remote
* server by reading or writing on the fd returned in pDataSock.
* When all incoming data has been read (as indicated by an EOF when reading
* the data socket) and/or all outgoing data has been written, the data
* socket fd should be closed.
* Then the routine ftpGetReply (2) should be called to receive the final reply
* on the control socket.
* Finally, the control socket should be closed.
*
* If the FTP command being invoked involves no data transfer (i.e. file
* delete or rename), pDataSock should be specified NULL, in which case
* no data connection will be established.
*
* EXAMPLE
* The following code fragment reads the file "/usr/fred/myfile" from the
* host "server", logged in as user "fred" with password "magic" and no
* account name.
*
* .CS
*   int ctrlSock;
*   int dataSock;
*   char buf [512];
*   int nBytes;
*
*   if (ftpXfer ("server", "fred", "magic", "",
*		 "RETR %s", "/usr/fred", "myfile",
*		 &ctrlSock, &dataSock) == ERROR)
*	return (ERROR);
*
*   while ((nBytes = read (dataSock, buf, sizeof (buf))) > 0)
*   	{
*	...
*   	}
*
*   close (dataSock);
*
*   if (nBytes < 0)		/* read error? *
*	status = ERROR;
*
*   if (ftpGetReply (ctrlSock) != FTP_COMPLETE)
*	status = ERROR;
*
*   close (ctrlSock);
* .CE
*
* RETURNS:
*  OK, or
*  ERROR, if any socket can't be created, or connection can't be made
*
* SEE ALSO: ftpGetReply (2)
*/

STATUS ftpXfer (host, user, passwd, acct, cmd, dirname, filename,
		pCtrlSock, pDataSock)
    char *host;		/* name of server host */
    char *user;		/* user name with which to login to host */
    char *passwd;	/* password with which to login to host */
    char *acct;		/* account with which to login to host */
    char *cmd;		/* command to send to host */
    char *dirname;	/* directory to 'cd' to before sending command */
    char *filename;	/* filename to send with command */
    int *pCtrlSock;	/* where to return control socket fd */
    int *pDataSock;	/* where to return data socket fd,
			 * (NULL == don't open data connection) */

    {
    FAST int ctrlSock = ERROR;
    FAST int dataSock = ERROR;

    if (((ctrlSock = ftpHookup (host)) == ERROR)			||
        (ftpLogin (ctrlSock, user, passwd, acct) != OK)			||
        (ftpCommand (ctrlSock, "TYPE I") != FTP_COMPLETE)		||
	((dirname[0] != EOS) &&
	    (ftpCommand (ctrlSock, "CWD %s", dirname) != FTP_COMPLETE))	||
	((pDataSock != NULL) &&
            ((dataSock = ftpInitDataConn (ctrlSock)) == ERROR))		||
        (ftpCommand (ctrlSock, cmd, filename) != FTP_PRELIM)		||
	((pDataSock != NULL) &&
	    ((dataSock = ftpGetDataConn (dataSock)) == ERROR)))
	{
	if (ctrlSock != ERROR)
	    close (ctrlSock);
	if (dataSock != ERROR)
	    close (dataSock);
	return (ERROR);
	}

    *pCtrlSock = ctrlSock;

    if (pDataSock != NULL)
	*pDataSock = dataSock;

    return (OK);
    }
/***********************************************************************
*
* ftpGetReply - get FTP command reply
*
* This routine gets a command reply on the specified control socket.
* All the lines of a single reply are read (multi-line replies are
* indicated by a continuation character '-' as the fourth character
* of all but the last line).
*
* The three digit reply code from the first line is saved and interpreted.
* The most significant digit of the reply code classifies the type of the code
* (see below).  This digit is returned as the result of this function.
* If the reply code indicates an error, the entire text of the reply
* is printed on standard error.
*
* The caller's task status is set to the complete three digit reply code
* (see errnoGet (2)).
*
* If an EOF is encountered on the specified control socket, but no EOF was
* expected (expecteof == FALSE) then ERROR is returned.
*
* RETURNS:
*   ERROR if read error or unexpected EOF;
*     or
*   FTP reply code most significant digit (reply class):
*      FTP_PRELIM	1 - positive preliminary 
*      FTP_COMPLETE	2 - positive completion
*      FTP_CONTINUE	3 - positive intermediate
*      FTP_TRANSIENT	4 - transient negative completion
*      FTP_ERROR	5 - permanent negative completion
*/

int ftpGetReply (ctrlSock, expecteof)
    int ctrlSock;	/* control socket fd of FTP connection */
    BOOL expecteof;	/* TRUE = EOF expected, FALSE = EOF is error */

    {
    char c;
    FAST int codeType;
    FAST int code;
    FAST int dig;
    int continuation;
    int origCode;
    BOOL eof;

    /* read all lines of a reply:
     *    do
     *	      while not eof and not eol
     *	          process char
     *    while not eof and not last line of reply
     */

    origCode = 0;
    codeType = 0;

    do
	{
	/* read all characters of a line */

	dig  = 0;
	code = 0;
	continuation = FALSE;

	while (!(eof = (read (ctrlSock, &c, 1) == 0)) && (c != '\n'))
	    {
	    dig++;

	    if (dig == 1)		/* char 1 is code type */
		codeType = c - '0';

	    if (dig <= 3)		/* chars 1-3 are code */
		{
		if (!isdigit (c))
		    code = -1;
		else
		    if (code != -1)
			code = code * 10 + (c - '0');
		}

	    if (dig == 4)		/* char 4 is continuation marker */
		continuation = (c == '-');

	    if ((c != '\r') &&
		(ftpVerbose || ((codeType == FTP_ERROR) && (dig > 4))))
		write (STD_ERR, &c, 1);
	    }
	
	/* print newline if we've been printing this reply */

	if (ftpVerbose || (codeType == FTP_ERROR))
	    printErr ("\n");

	/* save the original reply code */

	if (origCode == 0)
	    origCode = code;
	}
    /* while not eof and not last line of reply */
    while (!eof && !((dig >= 3) && (code == origCode) && !continuation));

    errnoSet (origCode);	/* set status to entire reply code */

    /* return error if unexpected eof encountered */

    if (eof & !expecteof)
	return (ERROR);
    else
	return (origCode / 100);	/* return most signif digit of reply */
    }
/**************************************************************************
*
* ftpHookup - get control connection to FTP server on specified host
*
* This routine establishes a control connection to the FTP server on the
* specified host.
* This is the first step in interacting with a remote FTP server at the
* lowest level.  (For a higher level interaction with a remote FTP server,
* see ftpXfer (2)).
*
* RETURNS:
*  fd of control socket, or
*  ERROR, if
*        invalid inet addr, or
*        couldn't find host name in table, or
*        couldn't create a socket, or
*        couldn't connect.
*
* SEE ALSO: ftpLogin (2), ftpXfer (2)
*/

int ftpHookup (host)
    char *host;		/* server host name or inet address */

    {
    FAST int ctrlSock;
    FAST int inetAddr;
    SOCKADDR_IN ctrlAddr;

    if (((inetAddr = (int) inet_addr (host)) == ERROR) &&
	((inetAddr = remGetHostByName (host)) == ERROR))
	return (ERROR);

    /* make our control socket */

    ctrlSock = socket (AF_INET, SOCK_STREAM, 0);
    if (ctrlSock < 0)
	return (ERROR);

    /* bind a name with no inet address and let system pick port;
     * this is just so we can find our socket address later */

    ctrlAddr.sin_family      = AF_INET;
    ctrlAddr.sin_addr.s_addr = INADDR_ANY;
    ctrlAddr.sin_port        = 0;
    if (bind (ctrlSock, (SOCKADDR *)&ctrlAddr, sizeof (ctrlAddr)) < 0)
	{
	close (ctrlSock);
	return (ERROR);
        }

    /* connect to other side */

    ctrlAddr.sin_addr.s_addr = inetAddr;
    ctrlAddr.sin_port        = FTP_PORT;
    if (connect (ctrlSock, (SOCKADDR *)&ctrlAddr, sizeof (ctrlAddr)) < 0)
	{
	close (ctrlSock);
	return (ERROR);
        }

    ftpGetReply (ctrlSock, FALSE);	/* read startup message from server */

    return (ctrlSock);
    }
/************************************************************************
*
* ftpLogin - login to remote FTP server
*
* This routine logs in to a remote server with the specified user name,
* password, and account name, as required by the specific remote host.
* This is typically the next step after calling ftpHookup (2) in interacting
* with a remote FTP server at the lowest level.
* (For a higher level interaction with a remote FTP server, see ftpXfer (2)).
*
* RETURN:
*  OK, or
*  ERROR if couldn't log in
*
* SEE ALSO: ftpHookup (2), ftpXfer (2)
*/

STATUS ftpLogin (ctrlSock, user, passwd, account)
    FAST int ctrlSock;	/* fd of control socket on which to login */
    char *user;		/* user name with which to login to host */
    char *passwd;	/* password with which to login to host */
    char *account;	/* account with which to login to host */

    {
    FAST int n;

    n = ftpCommand (ctrlSock, "USER %s", user);

    if (n == FTP_CONTINUE)
	n = ftpCommand (ctrlSock, "PASS %s", passwd);

    if (n == FTP_CONTINUE)
	n = ftpCommand (ctrlSock, "ACCT %s", account);

    if (n != FTP_COMPLETE)
	return (ERROR);

    return (OK);
    }
/*******************************************************************
*
* ftpInitDataConn - initialize FTP data connection
*
* This routine sets up the client side of a data connection for the
* specified control connection.  It creates the data port, informs the
* remote FTP server of the data port address, and initiates a 'listen'
* on that data port.  The server will then connect to this data port
* in response to a subsequent data-transfer command sent on the
* control connection (see ftpCommand (2)).
* This routine must be called BEFORE the data-transfer command is sent
* or the server's connect may fail.
*
* This routine is typically called after calling ftpHookup (2) and ftpLogin (2)
* to establish a connection with a remote FTP server at the lowest level.
* (For a higher level interaction with a remote FTP server, see ftpXfer (2)).
*
* RETURNS: fd of data socket created, or ERROR
*
* SEE ALSO: ftpHookup (2), ftpLogin (2), ftpCommand (2), ftpXfer (2)
*/

int ftpInitDataConn (ctrlSock)
    int ctrlSock;	/* fd of associated control socket */

    {
    FAST int dataSock;
    int result;
    int len;
    int optval;
    SOCKADDR_IN ctrlAddr;
    SOCKADDR_IN dataAddr;

    /* find out our inet address */

    len = sizeof (ctrlAddr);
    if (getsockname (ctrlSock, (SOCKADDR *)&ctrlAddr, &len) < 0)
	return (ERROR);

    /* first try - try to send port */

    dataSock = socket (AF_INET, SOCK_STREAM, 0);
    if (dataSock < 0)
	return (ERROR);

    dataAddr = ctrlAddr;	/* set our inet address */
    dataAddr.sin_port = 0;	/* let system pick port num */ 
    if (bind (dataSock, (SOCKADDR *)&dataAddr, sizeof (dataAddr)) < 0)
	{
	close (dataSock);
	return (ERROR);
	}

    if (listen (dataSock, 1) < 0)
	{
	close (dataSock);
	return (ERROR);
	}

    /* try to send socket address to other side */

    len = sizeof (dataAddr);
    if (getsockname (dataSock, (SOCKADDR *)&dataAddr, &len) < 0)
	{
	close (dataSock);
	return (ERROR);
	}

#define UCA(n)	(((int)(((char *)&dataAddr.sin_addr)[n])) & 0xff)
#define UCP(n)	(((int)(((char *)&dataAddr.sin_port)[n])) & 0xff)
    result = ftpCommand (ctrlSock, "PORT %d,%d,%d,%d,%d,%d",
		         UCA(0), UCA(1), UCA(2), UCA(3), UCP(0), UCP(1));

    if (result != FTP_ERROR)
	{
	if (result != FTP_COMPLETE)
	    {
	    close (dataSock);
	    return (ERROR);
	    }
	else
	    return (dataSock);
	}


    /* second try - try to get port # correct by default */

    close (dataSock);

    dataSock = socket (AF_INET, SOCK_STREAM, 0);
    if (dataSock < 0)
	return (ERROR);

    optval = 1;
    if ((setsockopt (dataSock, SOL_SOCKET, SO_REUSEADDR,
		     (caddr_t) &optval, sizeof (optval)) < 0) ||
        (bind (dataSock, (SOCKADDR *)&ctrlAddr, sizeof (ctrlAddr)) < 0))
	{
	close (dataSock);
	return (ERROR);
	}

    if (listen (dataSock, 1) < 0)
	{
	close (dataSock);
	return (ERROR);
	}

    return (dataSock);
    }
/************************************************************************
*
* ftpGetDataConn - get a completed FTP data connection
*
* This routine completes a data connection initiated by a call to
* ftpInitDataConn (2).  It waits for a connection on the specified socket
* from the remote FTP server.  The specified socket should be the one
* returned by ftpInitDataConn (2).
* The connection is established on a new socket, whose fd is returned
* as the result of this function.  The original socket, specified in the
* argument to this routine, is closed.
*
* Typically this routine is called after calling ftpInitDataConn (2) and
* ftpCommand (2) to initiate a data transfer from/to the remote FTP server.
*
* RETURNS:
*  fd of new data socket, or
*  ERROR if accept failed
*
* SEE ALSO: ftpInitDataConn (2), ftpCommand (2)
*/

int ftpGetDataConn (dataSock)
    int dataSock;	/* fd of data socket on which to await connection */

    {
    int newDataSock;
    SOCKADDR_IN from;
    int fromlen = sizeof (from);

    newDataSock = accept (dataSock, (SOCKADDR *) &from, &fromlen);

    close (dataSock);

    return (newDataSock);
    }
