/* etherLib.c - ethernet raw i/o routines and hooks */

static char *copyright = "Copyright 1987-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This module provides utilities that give direct access to ethernet packets.
Raw packets can be output directly to an interface using etherOutput (2).
Hooks are provided to examine and optionally process all incoming packets
(etherAddInputHook (2)) and all outgoing packets (etherAddOutputHook (2)).
The input hook can be used to receive raw packets that are not part
of any of the supported network protocols.
The input and output hooks can also be used to build network monitoring
and testing tools. 

Most UniWorks network drivers, even if they are not truly ethernet, still
are accessible via these routines.  This includes the backplane drivers,
Pronet driver, etc.

NOTE
Most normal network access should go through the higher level socket
interface.  These routines should rarely, if ever, be necessary for
applications.
*/

#include "UniWorks.h"
#include "mbuf.h"
#include "if.h"
#include "if_ether.h"
#include "param.h"
#include "systm.h"

FUNCPTR etherInputHookRtn  = NULL;	/* NULL = none */
FUNCPTR etherOutputHookRtn = NULL;	/* NULL = none */

/******************************************************************************
*
* etherOutput - send packet on ethernet interface
*
* This routine sends a packet on the specified ethernet interface by
* calling the interface's output routine directly.
*
* RETURNS: OK, or ERROR if out of mbufs.
*/

STATUS etherOutput (pIf, pEtherHeader, pData, dataLength)
    struct ifnet *pIf;			/* interface on which to send */
    struct ether_header *pEtherHeader;	/* ethernet header to send */
    FAST char *pData;			/* data to send */
    FAST int dataLength;		/* # of bytes of data to send */

    {
    FAST struct mbuf *m;	/* ptr to current mbuf piece */
    struct mbuf *top = 0;	/* ptr to 1st in mbuf chain */
    struct mbuf **mp = &top;	/* ptr to last mbuf link */
    struct sockaddr dst;	/* destination address */
    int oldLevel;

    /* construct dst sockaddr required by interface output routine;
     * all ethernet drivers interpret address family AF_UNSPEC as raw
     * ethernet header (this is used by arp) */

    dst.sa_family = AF_UNSPEC;
    *((struct ether_header *) dst.sa_data) = *pEtherHeader;


    /* put data in mbuf required by interface output routine */

    while (dataLength > 0)
	{
	/* get a free mbuf */

	MGET(m, M_DONTWAIT, MT_DATA);
	if (m == 0)
	    {
	    m_freem (top);	/* free chain constructed so far */
	    return (ERROR);
	    }

	/* copy in as much data as will fit */

	m->m_len = MIN (MLEN, dataLength);
	bcopy ((char *) pData, (char *) mtod(m, u_char *), m->m_len);

	pData      += m->m_len;		/* bump pointer */
	dataLength -= m->m_len;		/* decrement counter */

	/* link to mbuf chain */

	*mp = m;			/* stuff mbuf ptr in previous */
	mp = &m->m_next;		/* remember where to put next mbuf ptr*/
	}


    /* call interface's output routine */

    oldLevel = splnet ();
    (* pIf->if_output) (pIf, top, &dst);
    splx (oldLevel);

    return (OK);
    }
/******************************************************************************
*
* etherAddInputHook - add routine to receive all ethernet input packets
*
* This routine adds a hook routine that will be called for every ethernet
* packet that is received.
*
* Calling sequence of input hook routine:
* .CS
*     BOOL inputHook (pIf, buffer, length)
*         struct ifnet *pIf;    /* ptr to interface on which packet was recvd *
*         char *buffer;         /* ptr to received packet *
*         int length;           /* length of received packet *
* .CE
* The hook routine should return TRUE if it has handled the input packet and
* no further action should be taken with it.  It should return FALSE
* if it has not handled the input packet and normal (i.e. inet) processing
* should take place.
*
* The packet is in a temporary buffer when the hook routine is called.
* This buffer will be reused upon return from the hook.
* The hook routine should copy the input packet elsewhere if it needs
* to retain it.
*
* To effect the hook, a call out to etherInputHook should be included
* in the receive routine of every network driver that is to be included
* in this mechanism.
*
* RETURNS: OK.
*/

STATUS etherAddInputHook (inputHook)
    FUNCPTR inputHook;	/* routine to receive ethernet input */

    {
    etherInputHookRtn = inputHook;
    return (OK);
    }
/******************************************************************************
*
* etherAddOutputHook - add routine to receive all ethernet output packets
*
* This routine adds a hook routine that will be called for every ethernet
* packet that is to be transmitted.
*
* Calling sequence of output hook routine:
* .CS
*     BOOL inputHook (pIf, buffer, length)
*         struct ifnet *pIf;    /* ptr to interface on which packet will
*                                * be xmitted *
*         char *buffer;         /* ptr to packet to be transmitted *
*         int length;           /* length of packet to be transmitted *
* .CE
* The hook will be called immediately before transmission.
* The hook routine should return TRUE if it has handled the output packet and
* no further action should be taken with it.  It should return FALSE
* if it has not handled the output packet and normal transmission
* should take place.
*
* The ethernet packet data is in a temporary buffer when the hook routine
* is called.  This buffer will be reused upon return from the hook.
* The hook routine should copy the packet elsewhere if it needs
* to retain the output packet.
*
* To effect the hook, a call out to etherOutputHook should be included
* in the transmit routine of every ethernet driver that is to be included
* in this mechanism.
*
* RETURNS: OK.
*/

STATUS etherAddOutputHook (outputHook)
    FUNCPTR outputHook;	/* routine to receive ethernet output */

    {
    etherOutputHookRtn = outputHook;
    return (OK);
    }
