/* bootLib.c - boot rom subroutine library */

static char *copyright = "Copyright 1987,1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
When UniWorks is first booted, certain parameters are specified pertaining
to network addresses, boot device, host, file, etc.
This information is encoded into a single ascii string known as the boot line.
The boot line is placed at a known address (specified in config.h)
by the boot roms so that the system being booted can discover the parameters
that were used to boot the system.
The boot line is the ONLY means of communication from the boot roms to
the booted system.

This library contains routines for manipulating a boot line.
Routines are provided to interpret, construct, print, and prompt for
a boot line.
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "ctype.h"
#include "strLib.h"
#include "sysLib.h"

/* forward declarations */

LOCAL VOID skipSpace ();
LOCAL VOID promptParamString ();
LOCAL VOID promptParamNum ();

/*****************************************************************
*
* bootCrackParams - interpret the boot parameters from the boot line
*
* The boot line is of the form:
*
* bootdev(0,procnum)hostname:filename e=<n> b=<n> h=<n> g=<n> u=userid pw=passwd
*
* EXAMPLE
* .CS
* ex(0,0)host:/usr/vw/is20/config/UniWorks e=90.0.0.2 b=91.0.0.2
*       h=100.0.0.4 g=90.0.0.3 u=UniWorks pw=realtime f=1
*       
* where:
*   bootdev  = <ex> for ethernet | <bp> for backplane
*   procnum  = the processor id number if there are multiple
*              processors on the bus.
*   hostname = the name of the server to boot from.
*   e        = the internet address of the ethernet interface.
*   b        = the internet address of the backplane interface.
*   h        = the internet address of the server we are booting from.
*   g        = the internet address of the gateway between us and
*              the server (omit if we are on the same network).
*   u        = a valid userid on the server.
*   pw       = password for userid on the server
*              (only if "ftp" is to be used for file transfers).
*   f        = system dependent configuration flags in hexadecimal.
*
*  - all internet address are specified in standard "dot" notation
*    e.g. - 90.0.0.2
*  - the order of assigned values is arbitrary
* .CE
* RETURNS: Pointer following last character successfully cracked.
* This will point to EOS if the entire string was cracked successfully.
*/

char *bootCrackParams (bootParams, bootDev, hostName, bootFile, 
		          ead, bad, had, gad, usr, passwd, pProcNum, pFlags)
    char *bootParams;	/* boot line to be cracked */
    char *bootDev;	/* name of boot device */
    char *hostName;	/* name of host */
    char *bootFile;	/* name of file to boot */
    char *ead;		/* ethernet internet addr */
    char *bad;		/* backplane internet addr */
    char *had;		/* host internet addr */
    char *gad;		/* gateway internet addr */
    char *usr;		/* user id */
    char *passwd;	/* password */
    int *pProcNum;	/* where to return processor number */
    int *pFlags;	/* where to return configuration flags */

    {
    char *p1 = bootParams;
    char dummy [BOOT_FIELD_LEN];
    char procNum [BOOT_FIELD_LEN];
    char flags [BOOT_FIELD_LEN];

    /* initialize boot parameters */

    bootDev [0]  = EOS;
    hostName [0] = EOS;
    bootFile [0] = EOS;
    ead [0]      = EOS;
    bad [0]      = EOS;
    had [0]      = EOS;
    gad [0]      = EOS;
    usr [0]      = EOS;
    passwd [0]   = EOS;
    procNum [0]  = EOS;
    flags [0]    = EOS;

    *pProcNum    = NONE;
    *pFlags	 = 0;


    /* get boot device and proc num */

    if (!getWord  (&p1, bootDev, BOOT_FIELD_LEN, " \t(")	||
	!getConst (&p1, "(")					||
	!getNum   (&p1, (int *) dummy)				||
	!getConst (&p1, ",")					||
	!getNum   (&p1, pProcNum)				||
	!getConst (&p1, ")"))

	return (p1);


    /* skip over the ethernet address stuck in by ISI, if its there. */

    if (*p1 == '(')
	{
	getWord (&p1, dummy, BOOT_FIELD_LEN, ">");
	++p1;
	}


    /* get host name and boot file */

    if (!getWord  (&p1, hostName, BOOT_FIELD_LEN, " \t:")	||
	!getConst (&p1, ":")					||
	!getWord  (&p1, bootFile, BOOT_FIELD_LEN, " \t"))

	return (p1);


    /* get optional assignments */

    while (skipSpace (&p1), (*p1 != EOS))
	{
	if (!getAssign (&p1, "ead", ead, BOOT_FIELD_LEN) &&
	    !getAssign (&p1, "bad", bad, BOOT_FIELD_LEN) &&
	    !getAssign (&p1, "had", had, BOOT_FIELD_LEN) &&
	    !getAssign (&p1, "gad", gad, BOOT_FIELD_LEN) &&
	    !getAssign (&p1, "usr", usr, BOOT_FIELD_LEN) &&
	    !getAssign (&p1, "pw", passwd, BOOT_FIELD_LEN) &&
	    !getAssign (&p1, "vbnum", procNum, BOOT_FIELD_LEN) &&
	    !getAssign (&p1, "e", ead, BOOT_FIELD_LEN) &&
	    !getAssign (&p1, "b", bad, BOOT_FIELD_LEN) &&
	    !getAssign (&p1, "h", had, BOOT_FIELD_LEN) &&
	    !getAssign (&p1, "g", gad, BOOT_FIELD_LEN) &&
	    !getAssign (&p1, "u", usr, BOOT_FIELD_LEN) &&
	    !getAssign (&p1, "vb", procNum, BOOT_FIELD_LEN) &&
	    !getAssign (&p1, "n", procNum, BOOT_FIELD_LEN) &&
	    !getAssign (&p1, "f", flags, BOOT_FIELD_LEN))

	    break;
	}

    if (strlen (procNum) > 0)		/* interpret any procNum as dec int */
	sscanf (procNum, "%d", pProcNum);

    if (strlen (flags) > 0)		/* interpret any flags as hex int */
	sscanf (flags, "%x", pFlags);

    return (p1);
    }
/*****************************************************************
*
* bootEncodeParams - construct the boot command
*
* This routine constructs a boot line using the given parameters.
*/

STATUS bootEncodeParams (paramString, bootDev, hostName, bootFile,
			   ead, bad, had, gad, usr, passwd, procNum, flags)
    char *paramString;	/* pointer where to return the encoded boot line */
    char *bootDev;	/* name of boot device */
    char *hostName;	/* name of host */
    char *bootFile;	/* name of file to boot */
    char *ead;		/* ethernet internet addr */
    char *bad;		/* backplane internet addr */
    char *had;		/* host internet addr */
    char *gad;		/* gateway internet addr */
    char *usr;		/* user id */
    char *passwd;	/* password */
    int procNum;	/* processor number */
    int flags;		/* configuration flags */

    {
    sprintf (paramString, "%s(0,%d)%s:%s", bootDev, procNum, hostName,
	     bootFile);
    
    addAssignString (paramString, "e", ead);
    addAssignString (paramString, "b", bad);
    addAssignString (paramString, "h", had);
    addAssignString (paramString, "g", gad);
    addAssignString (paramString, "u", usr);
    addAssignString (paramString, "pw", passwd);

    addAssignNum (paramString, "f", "%x", flags);
    }
/*****************************************************************
*
* bootPrintParams - print boot line parameters
*
* This routine prints the boot parameters in the specified boot string.
*/

VOID bootPrintParams (paramString)
    char *paramString;		/* boot parameter string */

    {
    char bootDev [BOOT_FIELD_LEN];	/* boot device code */
    char hostName [BOOT_FIELD_LEN];	/* name of host */
    char bootFile [BOOT_FIELD_LEN];	/* file Name */
    char ead [BOOT_FIELD_LEN];		/* ethernet internet addr */
    char bad [BOOT_FIELD_LEN];		/* backplane internet addr */
    char had [BOOT_FIELD_LEN];		/* host internet addr */
    char gad [BOOT_FIELD_LEN];		/* gateway internet addr */
    char usr [BOOT_FIELD_LEN];		/* user name */
    char passwd [BOOT_FIELD_LEN];	/* password */
    int  procNum;			/* processor number */
    int  flags;				/* configuration flags */

    bootCrackParams (paramString, bootDev, hostName, bootFile, ead, bad,
		        had, gad, usr, passwd, &procNum, &flags);

    if (bootDev[0] != EOS)  printParamString ("boot device       = ", bootDev);
    if (hostName[0] != EOS) printParamString ("host name         = ", hostName);
    if (bootFile[0] != EOS) printParamString ("boot file         = ", bootFile);
    if (ead[0] != EOS)      printParamString ("inet on ethernet  = ", ead);
    if (bad[0] != EOS)      printParamString ("inet on backplane = ", bad);
    if (had[0] != EOS)      printParamString ("host inet         = ", had);
    if (gad[0] != EOS)      printParamString ("gateway inet      = ", gad);
    if (usr[0] != EOS)      printParamString ("user              = ", usr);
    if (passwd[0] != EOS)   printParamString ("ftp password      = ", passwd);
    if (procNum != NONE)    printf ("processor number  = %d\n", procNum);
    if (flags != 0)         printf ("flags             = 0x%x\n", flags);
    }
/************************************************************************
*
* bootPromptForParams - set up the net-boot configuration
*
* This routine prompts the user for each of the boot paramters.
* For each parameter, the current setting is displayed when
* a new value is prompted for.
* Typing a <CR> leaves the parameter unchanged.
* Typing a '.' clears the parameter.
*
* The string passed as a parameter is used for the default values.  A
* new boot line will be produced, changed as the user specifies.  That line
* will be copied over the string passed as a parameter.  If there are
* no defaults, string should be empty on entry.
*/

VOID bootPromptForParams (string)
    char *string;	/* default boot line.  may be changed */

    {
    char bootDev [BOOT_FIELD_LEN];	/* boot device code */
    char hostName [BOOT_FIELD_LEN];	/* name of host */
    char bootFile [BOOT_FIELD_LEN];	/* file Name */
    char ead [BOOT_FIELD_LEN];		/* ethernet internet addr */
    char bad [BOOT_FIELD_LEN];		/* backplane internet addr */
    char had [BOOT_FIELD_LEN];		/* host internet addr */
    char gad [BOOT_FIELD_LEN];		/* gateway internet addr */
    char usr [BOOT_FIELD_LEN];		/* user name */
    char passwd [BOOT_FIELD_LEN];	/* password */
    int  procNum;			/* processor number */
    int  flags;				/* configuration flags */

    /* interpret the boot parameters */

    bootCrackParams (string, bootDev, hostName, bootFile, ead, bad,
		        had, gad, usr, passwd, &procNum, &flags);

    /* prompt the user for each item */

    promptParamString ("boot device: ", bootDev, sizeof (bootDev));
    promptParamString ("host name: ", hostName, sizeof (hostName));
    promptParamString ("file name: ", bootFile, sizeof (bootFile));
    promptParamString ("inet on ethernet (e): ", ead, sizeof (ead));
    promptParamString ("inet on backplane (b): ", bad, sizeof (bad));
    promptParamString ("host inet (h): ", had, sizeof (had));
    promptParamString ("gateway inet (g): ", gad, sizeof (gad));
    promptParamString ("user (u): ", usr, sizeof (usr));
    promptParamString ("ftp password (pw) (blank = use rsh): ", passwd,
		       sizeof (passwd));
    promptParamNum ("processor number: ", &procNum, 6, "%d ");
    promptParamNum ("configuration flags (f): ", &flags, 8, "%x ");

    bootEncodeParams (string, bootDev, hostName, bootFile, ead, bad, had,
		        gad, usr, passwd, procNum, flags);
    }

/*******************************************************************************
*
* bootNetmaskExtract - extract netmask field from adrs:mask field
*
* This routine extracts the option subnet mask field from an internet address
* field.  Subnet masks can be specified for an internet interface by appending
* to the internet address a colon and the net mask in hex.  For example, the
* "inet on ethernet" field of the boot parameters could be specified as:
*
*    inet on ethernet: 90.1.0.1:ffff0000
*
* In this case, the network portion of the address (normally just 90)
* is extended by the subnet mask (to 90.1).  This routine picks off the
* optional trailing subnet mask by replacing the colon in the specified
* string with an EOS and then scanning the remainder as a hex number.
* This number, the net mask, is returned as the value of the function.
*
* RETURNS: value of subnet mask if any specified, otherwise 0.
*/

int bootNetmaskExtract (string)
    char *string;

    {
    FAST char *pColon;
    int netmask;

    /* find colon; return 0 if none */

    pColon = index (string, ':');
    if (pColon == NULL)
	return (0);


    /* terminate string at the colon and scan and return remainder as netmask */

    *pColon = EOS;

    if (sscanf (pColon + 1, "%x", &netmask) != 1)
	return (0);

    return (netmask);
    }

/*****************************************************************
*
* addAssignNum - add a numberic value assignment to a string
*/

LOCAL VOID addAssignNum (string, code, format, value)
    char *string;
    char *code;
    char *format;
    int value;

    {
    if (value != 0)
	{
	string += strlen (string);
	sprintf (string, " %s=", code);
	string += strlen (string);
	sprintf (string, format, value);
	}
    }
/*****************************************************************
*
* addAssignString - add a string assignment to a string
*/

LOCAL VOID addAssignString (string, code, value)
    char *string;
    char *code;
    char *value;

    {
    if (value[0] != EOS)
	{
	string += strlen (string);
	sprintf (string, " %s=%s", code, value);
	}
    }
/************************************************************************
*
* getWord - get a word out of a string
*
* Words longer than the specified max length are truncated.
*
* RETURNS: TRUE if word is successfully extracted from string, FALSE otherwise;
* Also updates ppString to point to next character following extracted word.
*/

LOCAL BOOL getWord  (ppString, pWord, length, delim)
    char **ppString;	/* ptr to ptr to string from which to get word */
    FAST char *pWord;	/* where to return word */
    int length;		/* max length of word to get including EOS */
    char *delim;	/* string of delimiters that can terminate word */

    {
    FAST char *pStr;

    skipSpace (ppString);

    /* copy up to any specified delimeter, EOS, or max length */

    pStr = *ppString;
    while ((--length > 0) && (*pStr != EOS) && (index (delim, *pStr) == 0))
	*(pWord++) = *(pStr++);

    *pWord = EOS;


    /* if we copied anything at all, update pointer and return TRUE */

    if (pStr != *ppString)
	{
	*ppString = pStr;
	return (TRUE);
	}


    /* no word to get */

    return (FALSE);
    }
/************************************************************************
*
* getConst - get a constant string out of a string
*
* case insensitive compare for identical strings
*/

LOCAL BOOL getConst (ppString, pConst)
    char **ppString;
    FAST char *pConst;

    {
    FAST int ch1;
    FAST int ch2;
    FAST char *pString;

    skipSpace (ppString);

    for (pString = *ppString; *pConst != EOS; ++pString, ++pConst)
	{
	ch1 = *pString;
	ch1 = (isascii (ch1) && isupper (ch1)) ? tolower (ch1) : ch1;
	ch2 = *pConst;
	ch2 = (isascii (ch2) && isupper (ch2)) ? tolower (ch2) : ch2;

	if (ch1 != ch2)
	    return (FALSE);
	}

    /* strings match */

    *ppString = pString;
    return (TRUE);
    }
/************************************************************************
*
* getNum - get a numeric value from a string
*/

LOCAL BOOL getNum (ppString, pProcNum)
    FAST char **ppString;
    int *pProcNum;

    {
    skipSpace (ppString);

    if (sscanf (*ppString, "%d", pProcNum) != 1)
	return (FALSE);

    /* skip over number */

    while (isdigit (**ppString))
	(*ppString)++;

    return (TRUE);
    }
/************************************************************************
*
* getAssign - get an assignment out of a string
*/

LOCAL BOOL getAssign (ppString, valName, pVal, length)
    FAST char **ppString;
    char *valName;
    char *pVal;
    int length;

    {
    skipSpace (ppString);
    if (!getConst (ppString, valName))
	return (FALSE);

    skipSpace (ppString);
    if (!getConst (ppString, "="))
	return (FALSE);

    return (getWord (ppString, pVal, length, " \t"));
    }
/**********************************************************************
*
* printClear - print string with '?' for unprintable characters
*/

LOCAL VOID printClear (param)
    FAST char *param;

    {
    FAST char ch;

    while ((ch = *(param++)) != EOS)
	printf ("%c", (isascii (ch) && isprint (ch)) ? ch : '?');
    }
/**********************************************************************
*
* printParamString - print string parameter
*/

LOCAL VOID printParamString (msg, param)
    char *msg;
    char *param;

    {
    printf (msg);
    printClear (param);
    printf ("\n");
    }
/**********************************************************************
*
* promptParamString - prompt the user for a string parameter
*
* - carriage return leaves the parameter unmodified;
* - "." clears the parameter (null string).
*/

LOCAL VOID promptParamString (msg, param, fieldWidth)
    char *msg;
    char *param;
    int fieldWidth;

    {
    int i;
    char buf [100];

    FOREVER 
	{
	printf ("%s ", msg);
	printClear (param);
	printf (" ");

	i = fioRdString (STD_IN, buf, sizeof (buf));
	if (i < fieldWidth)
	    break;
	printf ("too big - maximum field width = %d.\n", fieldWidth);
	}

    if (i == 1)
	return;			/* just CR; leave field unchanged */
    
    if (buf[0] == '.')
	{
	param [0] = EOS;	/* just '.'; make empty field */
	return;
	}

    strcpy (param, buf);	/* update parameter */
    }
/**********************************************************************
*
* promptParamNum - prompt the user for a parameter
*
* - carriage return leaves the parameter unmodified;
* - "." clears the parameter (0).
*/

LOCAL VOID promptParamNum (msg, pParam, fieldWidth, format)
    char *msg;
    int *pParam;
    int fieldWidth;
    char *format;
    {

    int i;
    char buf [100];

    FOREVER 
	{
	strcpy (buf, "%s ");
	strcat (buf, format);

	printf (buf, msg, *pParam);

	i = fioRdString (STD_IN, buf, sizeof (buf));
	if (i < fieldWidth)
	    break;
	printf ("too big - maximum field width = %d.\n", fieldWidth);
	}

    if (i == 1)
	return;			/* just CR; leave field unchanged */
    
    if (buf[0] == '.')
	{
	pParam = 0;		/* just '.'; make empty field */
	return;
	}

    sscanf (buf, format, pParam);	/* scan field */
    }
/*******************************************************************************
*
* skipSpace - advance pointer past white space
*
* Increments the string pointer passed as a parameter to the next
* non-white-space character in the string.
*/

LOCAL VOID skipSpace (strptr)
    char **strptr;	/* pointer to pointer to string */

    {
    while (isspace (**strptr))
	++*strptr;
    }
