/* relseg.c - relocate program segments to fixed addresses */

/*
modification history
--------------------
*/

/*
SYNOPSIS
relseg [-t text-address] [-d data-address] [-b bss-address] [-o outfile] file

DESCRIPTION
This program relocates the specified object module file, putting the
text, data, and bss segments at fixed addresses.
The segments are put at the addresses of the specified flags.
If no text address is specified, the text segment is put at 0.
If no data address is specified, the data segment is put immediately following
the text segment.
If no bss address is specified, the bss segment is put immediately following
the data segment.
The entry address is modified according to the text address.

The input object module should be fully linked but still relocatable.
The new object module is left in "rel.out"
and is NOT eligible for further linking or relocating.
The output file name may be specified with the "-o" option.

EXAMPLE
Suppose we want to create an object module in which the text and data 
segments are to be placed in a ROM whose address is 0xeee000, but whose
bss segment is to be placed in RAM at address 0x1000.
The commands to do this are:

	ld -o gronk -X -rd gronk.o ...
	relseg -t ee0000 -b 1000 gronk

The file "rel.out" is created containing the relocated object module.

SEE ALSO: UNIX BSD 4.3 a.out.h
*/

#ifdef	LINT
#include "UniWorks.h"
#include <stdioLib.h>
#else
#include <stdio.h>
#include "UniWorks.h"
#endif	LINT

#include "a_out.h"

#define N_DATAOFF(hdr)	(N_TXTOFF (hdr) + (hdr).a_text)
#define N_RTXTOFF(hdr)	(N_DATAOFF (hdr) + (hdr).a_data)

#define RTEXT	0
#define RDATA	1
#define RBSS	2
#define N_SEGMENTS	3

LOCAL char usage [] = 
"usage: relseg [-t text-addr] [-d data-addr] [-b bss-addr] [-o outfile] file";


/*******************************************************************************
*
* main - relocate program segments to fixed addresses
*/

VOID main (argc, argv)
    int argc;
    char *argv [];

    {
    static char fileout[100] = "rel.out";
    char *filename;
    FILE *infile1;
    FILE *infile2;
    FILE *outfile;
    struct exec hdr;
    int nSyms;
    struct nlist sym;
    int segDelta [N_SEGMENTS];
    int i;
    int segnum;

    /* get file name and segment offsets from args */

    for (i = 0; i < N_SEGMENTS; i++)
	segDelta [i] = NONE;

    if (argc < 2)
	error (usage);

    /* Crack the arguments */

    for (i = 1; i < (argc - 1); i++)
	{
	if (strcmp (argv [i], "-t") == 0)	/* text */
	    {
	    if (sscanf (argv [++i], "%x", &segDelta [RTEXT]) == 0)
		error (usage);
	    }
	else if (strcmp (argv [i], "-d") == 0)	/* data */
	    {
	    if (sscanf (argv [++i], "%x", &segDelta [RDATA]) == 0)
		error (usage);
	    }
	else if (strcmp (argv [i], "-b") == 0)	/* bss */
	    {
	    if (sscanf (argv [++i], "%x", &segDelta [RBSS]) == 0)
		error (usage);
	    }
	else if (strcmp (argv [i], "-o") == 0)	/* output file name */
	    {
	    if (sscanf (argv [++i], "%s", fileout) == 0)
		error (usage);
	    }
	else
	    error (usage);
	}

    /* Make sure there is exactly one argument left */

    if (i != (argc - 1))
	error (usage);

    filename = argv [i];

    /* open two input and one output stream to the object module */

    if (((infile1 = fopen (filename, "r")) == NULL) ||
        ((infile2 = fopen (filename, "r")) == NULL))
	{
	error ("relseg: error opening file %s", filename);
	}

    if ((outfile = fopen (fileout, "w")) == NULL)
	error ("relseg: error opening file %s", fileout);

    fcopy (infile1, outfile);
	
    /* reposition file to beginning */

    fseek (infile1, (long)0, 0);

    /* read the header of the object module */

    if (fread ((char *)&hdr, sizeof (hdr), 1, infile1) == 0)
	error ("relseg: error reading header");

    /* relocate the entry address and write the modified header back out */

    if (segDelta [RTEXT] == NONE)
	segDelta [RTEXT] = 0;

    hdr.a_entry += segDelta [RTEXT];

    if (fseek (outfile, (long)0, 0) == ERROR ||
        fwrite ((char *)&hdr, sizeof (hdr), 1, outfile) != 1)
	{
	error ("relseg: error writing modified header");
	}

    /* fill in defaults for unspecified addresses */

    if (segDelta [RDATA] == NONE)
	segDelta [RDATA] = segDelta [RTEXT] + hdr.a_text;

    if (segDelta [RBSS] == NONE)
	segDelta [RBSS] = segDelta [RDATA] + hdr.a_data;

    /* turn segment addresses into deltas from old addresses */

    segDelta [RDATA] -= hdr.a_text;
    segDelta [RBSS]  -= hdr.a_text + hdr.a_data;

    /* perform the relocations */

    if (fseek (infile1, (long)N_RTXTOFF (hdr), 0) == ERROR)
	error ("relseg: error reading relocation commands");

    relSegment (infile1, infile2, outfile, segDelta, N_TXTOFF (hdr),
		(int)hdr.a_trsize / sizeof (struct relocation_info));
    relSegment (infile1, infile2, outfile, segDelta, (int)N_DATAOFF (hdr),
		(int)hdr.a_drsize / sizeof (struct relocation_info));


    /* update the symbol table */

    nSyms = hdr.a_syms / sizeof (struct nlist);

    if (fseek (infile1, (long)N_SYMOFF (hdr), 0) == ERROR)
	error ("relseg: error reading symbol table");

    for (i = 0; i < nSyms; i++)
	{
	if (fread ((char *)&sym, sizeof (sym), 1, infile1) != 1)
	    error ("relseg: error reading symbol table");

	if ((sym.n_type & N_TYPE) != N_ABS)
	    {
	    segnum = segOfType ((int)sym.n_type);

	    if (segnum == ERROR)
		error ("relseg: unknown symbol type 0x%02x", sym.n_type);

	    sym.n_value = (unsigned) ((char *)sym.n_value + segDelta [segnum]);

	    if (fseek (outfile, (long)(N_SYMOFF (hdr) + (i * sizeof(sym))), 0)
		    == ERROR ||
		fwrite ((char *)&sym, sizeof (sym), 1, outfile) != 1)
		{
		error ("relseg: error writing symbol table");
		}
	    }
	}

    fclose (infile1);
    fclose (infile2);
    fclose (outfile);
    }
/*******************************************************************************
*
* segOfType - get segment number of specified type
*/

LOCAL int segOfType (type)
    int type;

    {
    switch (type & N_TYPE)
	{
	case N_TEXT: return (RTEXT);
	case N_DATA: return (RDATA);
	case N_BSS:  return (RBSS);

	case N_UNDF:
	case N_ABS:	/* XXX */
	case N_COMM:
	case N_FN:
	default:     return (ERROR);
	}
    }
/*******************************************************************************
*
* relSegment - perform relocations of a program segment
*/

LOCAL VOID relSegment (infile1, infile2, outfile, segDelta, segOffset, nRelCmds)
    FILE *infile1;
    FILE *infile2;
    FILE *outfile;
    int segDelta [N_SEGMENTS];
    int segOffset;
    int nRelCmds;

    {
    struct relocation_info relCmd;
    int i;
    int segnum;
    char *adrs;
    
    for (i = 0; i < nRelCmds; i++)
	{
	if (fread ((char *)&relCmd, sizeof (relCmd), 1, infile1) != 1)
	    error ("relseg: error reading relocation commands");

	if (!relCmd.r_pcrel && !relCmd.r_extern)
	    {
	    segnum = segOfType ((int)relCmd.r_symbolnum);
	    if (segnum == ERROR)
		error ("relseg: unknown segment 0x%02x", relCmd.r_symbolnum);

	    if (fseek (infile2, (long)(segOffset + relCmd.r_address), 0) 
		    == ERROR ||
		fread ((char *)&adrs, sizeof (adrs), 1, infile2) != 1)
		{
		error ("relseg: error reading text and data segments");
		}
	    
	    adrs += segDelta [segnum];

	    if (fseek (outfile, (long)(segOffset + relCmd.r_address), 0)
		    == ERROR ||
		fwrite ((char *)&adrs, sizeof (adrs), 1, outfile) != 1)
		{
		error ("relseg: error writing text and data segments");
		}
	    }
	}
    }
/*******************************************************************************
*
* fcopy - copy input to output
*/

LOCAL VOID fcopy (fpin, fpout)
    FILE *fpin;		/* file to be copied */
    FILE *fpout;	/* where to copy file */

    {
    char buffer [512];
    int nbytes;

    while ((nbytes = fread (buffer, sizeof (char), sizeof (buffer), fpin)) > 0)
	fwrite (buffer, sizeof (char), nbytes, fpout);
    }
/*******************************************************************************
*
* error - print error message and die
*
* This routine prints an error message on the standard error output and
* aborts the program with the error status ERROR.  The error message is
* output with the specified format with the single specified argument.
*
* VARARGS1
*/

LOCAL VOID error (format, arg)
    char *format;	/* format of error message */
    char *arg;		/* argument supplied to format (arbitrary type!) */

    {
    fprintf (stderr, format, arg);
    fprintf (stderr, "\n");
    exit (ERROR);
    }
