/*
 * $Header: drreg.h,v 1.2 87/12/10 14:43:36 root Exp $
 * $Source: /usr/src/sys/is68kdev/RCS/drreg.h,v $
 * $Log:	drreg.h,v $
 * Revision 1.2  87/12/10  14:43:36  root
 * additional commentary re driver use
 * 
 */ 

 /* IKON DR11W DRIVER FOR VME CONTROLLERS 10084/10089
  * COPYRIGHT 1987 Integrated Solutions Inc.
  * All rights reserved
 */


/*
 * The following is a  structure type that corresponds to
 * the 10084 board's register set.
 *
 * Note that some registers appear at 
 * different offsets, depending on whether
 * they are being written or read.
 *
 * The dma address register is 24 bits
 * wide - the low 16 bits are accessed as 
 * a ushort value.
 * The high 8 bits may be accesses as a char at one address 
 * or a as a ushort at the next lower work address.
 */
struct	drdevice	{
	ushort	dr_cstat;     	/* control and status sregister */
	ushort	dr_data;     	/* input and output data register */
	char	dr_addmod;     	/* address modifier for dm */
	char	dr_intvect;     /* interrupt vector */
	ushort	dr_pulse;     	/* pulse command register */
	ushort	dr_xx08;     	/* not used */
	ushort	dr_xx0A;     	/* not used */
	ushort	dr_xx0C;     	/* not used */
	ushort	dr_xx0E;     	/* not used */
	ushort	dr_xx10;     	/* not used */
	ushort	dr_walo;     	/* low dma add register--when written */
	ushort	dr_range;     	/* dma range count */
	ushort	dr_ralo;     	/* low dma add register--when read */
	ushort	dr_xx18;     	/* not used */
	ushort	dr_wahi;     	/* high dma add register--when written */
	ushort	dr_xx1C;     	/* not used */
	ushort	dr_rahi;     	/* high dma add register--when read */
};

		
/*
 * Local buffer headers.  The entries in this buffer header are filled
 * in by physio() which will call the local drstrategy() at which point,
 * the entries will have the same form as system buffers but will
 * actually point to the data in user space.
 */

/*
 * There are no minor devices here.  Each major device is another controller.
 */
#define DRUNIT(dev) (minor(dev))
#define DRADDR(dev) ((struct drdevice *)(drdinfo[DRUNIT(dev)])->qi_mi->qm_addr) 
#define	BLOCKSIZE	131068	/* max xfer in bytes */
#define DRADMD 0x3A    	        /* address modifier for dm */
#define DR_ZERO 0x0
#define DR_TICK 600		/* # clock ticks for drwatch calls */
#define DR_TOCK 2		/* default # drwatch calls until timeout */
#define DR_PADD 0x55		/* used as a PADD in various places */
#define PAD_NUM	1		/* used as a PADD in various places */

/*
 * dr_cstat bits when used as control register write
 */
#define RDMA 0x8000     	/* resets the dma end-of-range flag */
#define RATN 0x4000     	/* resets the attention flag */
#define RPER 0x2000     	/* resets the device parity error flag */
#define MCLR 0x1000     	/* master clear board and INIt device   */
#define CYCL 0x0100     	/* forces dma cycle if dma enabled */
#define IENB 0x0040     	/* enables interrupts */
#define FCN3 0x0008     	/* function bit 3 to de (FNCT3 H) */
#define FCN2 0x0004     	/* enables dma and pulses GO to device */
                                /* also asserts ACLO FCN2 H to device   */
#define FCN1 0x0002     	/* function bit 1 to device (FCN1 H)  */
#define GO   0x0001     	/* enables dma and pulses GO to device */

/*
 * dr_cstat bits when used as status register read
 */
#define DMAF 0x8000    		/* indicates dma end-of-range */
#define ATTF 0x4000     	/* indicates attention false-to-true */
#define ATTN 0x2000     	/* current state of ATTENTION H input */
#define PERR 0x1000     	/* set by external parity error */
#define STTA 0x0800     	/* STATUS A H input state */
#define STTB 0x0400     	/* STATUS B H input state */
#define STTC 0x0200     	/* STATUS C H input state */
#define REDY 0x0080     	/* board ready for command (dma not on) */
#define IENF 0x0040     	/* interrupts enabled if on */
#define BERR 0x0020     	/* set if bus error during dma */
#define FC3S 0x0008     	/* state of FCN3 latch */
#define FC2S 0x0004     	/* state of FCN2 latch */
#define FC1S 0x0002     	/* state of FCN1 latch */
#define OKMSK	0xFF80

/*
 * Define pulse command register bits -- which include RDMA,
 * RPER, MCLR, CYCL, FCN2, GO and two additional interrupt mask
 * control bits.
 */

#define SMSK 0x0040     	/* pulses interrupt mask on */
#define RMSK 0X0020     	/* pulses interrupt mask off */

struct	drop	{	/* DR operation */
	unsigned short	dr_cmd;		/* command bits */
	/*
	 * The next two args are IOC_OUT parameters (all ioctl calls)
	 */
	unsigned short	dr_timeout;	/* used if timeout cmd bit specified */
	unsigned short	dr_outdata;	/* used if DR_PIOW specified */
	/*
	 * The next three args are IOC_IN parameters (all ioctl calls)
	 */
	unsigned short	dr_indata;	/* the current data in input reg */
	unsigned short	dr_status;	/* the current status reg */
	unsigned short	dr_flags;	/* the current driver state flags */
};

#define	DR_IOC	1

/*
 * IOCTRL(2) call format (4.3bsd) is:
 *
 *	#include <sys/ioctl.h>
 *	#include <sys/drreg.h>
 *
 *	ioctrl( d, DR_IOC, drop )
 *
 *	int d;				// open descriptor
 *	unsigned long DR_IOC;		// DR_IOC is macro defined below
 *	struct drop *drop;		// pointer to DR operation structure
 *
 * dr_cmd is the logical or of one or more of:
 */
#define DR_WAIT 0x0010     	/* wait for attention interrupt  */
#define DR_RSET 0x1000     	/* issue master clear tos dev & board */
#define DR_PIOW 0x0020     	/* write to p-i/o register */
#define DR_PACL 0x0040     	/* send pulse to device */
#define DR_DACL 0x0080     	/* defer aclo pulse until go */
#define DR_PCYL 0x0100     	/* set cycle with next go */
#define DR_DFCN 0x0200		/* don't update function bits until next GO */
#define DR_RATN 0x4000		/* reset attention flag, USE W/ CAUTION */
#define DR_RDMA 0x8000		/* reset dma end-of-range flag, (DON'T USE) */
#define DR_SFCN 0x0001   	/* set function bits */
#define DR_STIM 0x0400   	/* set wait and dma time-out val */
#define DR_FMSK 0x000E   	/* function bit mask */
#define DR_RPER 0x2000   	/* clear parity flag */
#define DR_STGO 0xF000   	/* set GO bit  */
#define DR_PFN1 0x0002   	/* set FCN1  */
#define DR_PFN3 0x0008   	/* set FCN3  */
#define DR_STAT 0x00F0   	/* get last status back to user  */


/*
 * dr_status bits
 */
#define DR_DMAF 0x8000    	/* indicates dma end-of-range */
#define DR_ATTF 0x4000     	/* indicates attention false-to-true */
#define DR_ATTN 0x2000     	/* current state of ATTENTION H input */
#define DR_PERR 0x1000     	/* set by external parity error */
#define DR_STTA 0x0800     	/* STATUS A H input state */
#define DR_STTB 0x0400     	/* STATUS B H input state */
#define DR_STTC 0x0200     	/* STATUS C H input state */
#define DR_REDY 0x0080     	/* board ready for command (dma not on) */
#define DR_IENF 0x0040     	/* interrupts enabled if on */
#define DR_BERR 0x0020     	/* set if bus error during dma */
#define DR_FC3S 0x0008     	/* state of FCN3 latch */
#define DR_FC2S 0x0004     	/* state of FCN2 latch */
#define DR_FC1S 0x0002     	/* state of FCN1 latch */

/*
 * State bits for dr_flags.
 */
#define DR_OPEN 0x0001     	/* this dr11 has been opened*/
#define DR_PRES 0x0002     	/* this dr11 is present*/
#define DR_ACTV 0x0004     	/* waiting for end-of-range*/
#define DR_ATWT 0x0008     	/* waiting for attention int*/
#define DR_ATRX 0x0010     	/* attn received-resets when read*/
#define DR_TMDM 0x0020     	/* time-out waiting for e-o-r*/
#define DR_TMAT 0x0040     	/* time-out waiting for attention*/
#define DR_DMAX 0x0080     	/* end of range int received*/

/*
 * NOTA BENE:
 *
 * The DR driver assumes 1 logical device per physical device.
 * For example, suppose two DR controllers exist in cdevsw slots
 * 39 and 40.  Then, the following character special files should
 * created (note the minor device numbers -- they actually choose the
 * controller).
 *
 *	/etc/mknod /dev/dr0 c 39 0
 *	/etc/mknod /dev/dr1 c 40 1
 *
 * The actual direction of data transfer is not controlled by the call
 * to read() or write().  The direction of transfer is determined
 * by the "C1 CNTRL H" input signal.  If it is a 0 the Ikon hardware will
 * perform a read operation, if it is a 1, a write operation will be done.
 *
 * If multiple logical connections per physical connection is desired,
 * this driver must be modified (Chromatics graphics engines with dual
 * "heads" require this).
 *
 * The "C1 CNTRL H" signal is controlled by the external device, not by
 * the program.  In some applications -- particularly interprocessor link
 * mode (ala DEC) -- the "FNCT1 H" output is looped back into the DR11 (Ikon)
 * board and connected to the "C1 CNTL H" input.  In this case, the calling
 * program can (and must) set the direction of transfer prior to the
 * read or write call by calling ioctl().  In a driver that supported
 * ONLY link mode, this could be done within the read or write call.
 *
 * Because of the many ways to utilize a DR11-W, the driver does not
 * assume very much.  Consequently, the application level must engage
 * in machinations to set up DR11-W transfers.  The following is an
 * example of a typical DR11-W write process not operating in link mode
 * (one peer selects the transfer direction of the other).
 *
 * A typical device open is:
 *
 *	if( (fd = open( "/dev/dr0", O_RDWR )) < 0 ) {
 *		fprintf( stderr, "Cannot open /dev/dr0\n" );
 *		exit( 1 );
 *	}
 *
 * Now, suppose we want to do an output transfer to a cooperating peer.
 * First, we utilize the P-i/o output register as mechanism to signal
 * our peer of the desired transfer length.  Moreover, we indicate the
 * direction of "us to him" by setting the FCN bits appropriately.  Finally,
 * we issue an ATTN interrupt causing our peer to read the transfer length
 * and FCN bits.  He will then issue us an ATTN interrupt indicating he
 * has read the transfer length and FCN bits, thus we need to wait for
 * that event.
 *
 *	struct drop drop;
 *
 *	drop.dr_outdata = transfer_count;	// in even number of bytes
 *	dr.dr_cmd = DR_PIOW |		// indicate we want to load P-i/o reg
 *		DR_PACL |		// generate an ATTN interrupt
 *		DR_SFCN |		// us-to-him (DR_SFCN|DR_PFN1) would
 *					// be him-to-us
 *		DR_WAIT;		// wait for return ATTN interrupt
 *	if( ioctl( fd, DR_IOC, &drop ) < 0 )
 *		<issue error message>
 *
 * Next, we can start the DMA transfer.
 *
 * 	if( write( fd, buf, transfer_count ) < 0 )
 *		<issue error message>
 *
 * Similarly, the peer would be issuing ioctl() calls to implement the
 * converse machinations.
 *
 *
 * A few words about Ikon 10084 and CX1500 cabling are in order.
 * 
 * Often, bulkhead connectors engage a odd/even pin swap.  If two such swaps
 * occur, no problem.  For testing purposes, lets ignore the bulkhead
 * connections and discuss board cabling.
 * 
 * The Ikon 10084 DR11-W board's connectors correspond exactly to the
 * DEC DR11-W connectors.  The pin labelled on the connector as pin 1
 * (denoted by the etched arrow on the board connectors) corresponds to the
 * DEC pin designator VV.  The fact that DEC letters their pins in reverse
 * order when compared to the physical connectors used is CONFUSING.
 * 
 * A further confusing factor is that the cables are assembled with a
 * stripe on the pin 40 end---not the typical pin 1 end!!!!!!!!!!!
 * This is necessary since the cables's internal groud plane drain
 * wire is at the striped end of the cable and must be terminated to pin 40.
 *
 * DO NOT go by the etched triangle on the cable connectors.  It is
 * best to use a continuity checker between the cable ground plane (it
 * is usually accessible near the ends of the cable) and locate the
 * proper ground pin on the connector next to the striped end.  USE THIS
 * AS A GUIDE, since it is usually the case that one end of the cable will
 * have the connector installed in a reverse fashion.
 * 
 * Now, on each of the two Ikon connectors lets consider pins 1, 2, 39, and 40.
 * Of these four pins, only pin 40 is connected to GROUND.  By using this
 * fact, we can (via visual and ohm-meter check) derive the proper
 * cabling topology:
 * 
 * 
 *      IKON 10084
 * 	#
 * 	#
 * 	~
 * 	~ +-- Pin 40 is ground
 * 	# |
 * 	# V   J1
 * 	#-----------+
 * 	# *40   39* | 
 * 	# *38   37* |
 * 	~           ~
 * 	~           ~
 * 	# *4     3* |
 * 	# *2     1* |
 * 	#-----------+
 * 	#
 * 	#
 * 	~
 * 	~
 * 	# +-- Pin 40 is ground
 * 	# |
 * 	# V   J2
 * 	#-----------+
 * 	# *40   39* |
 * 	# *38   37* |
 * 	~           ~
 * 	~           ~
 * 	# *4     3* |
 * 	# *2     1* |
 * 	#-----------+
 * 	#
 * 	#
 * 	~
 * 	~     |
 * 	#     | Front
 * 	#     |  of
 * 	#     V ISI
 * 
 */ 

/*
 * The following is a loop-back test for the Ikon 10084 board.  Since this
 * device is actually a half-duplex device (one set of DMA registers), a
 * honest loop-back test cannot be performed.  However, a DMA transfer can
 * be issued in a loop-back configuration, then the P-i/o register read
 * which should contain the value of the last word of the DMA transfer.
 * Although short of a real loop-back test, this does provide some measure
 * of confidence in the operation of the hardware
 */

#ifdef NEVER_DEFINE_THIS_SYMBOL

#include <stdio.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/is68kdev/drreg.h>

char	*progname;
char	*dev = "/dev/dr0";
int	drfd;
short	verbose;
short	piodat[] = { 0xffff, 0xff00, 0x00ff, 0x5555, 0xaaaa, 0x0, 0x1234 };
short	outdat[] = { 0xffff, 0xff00, 0x00ff, 0x5555, 0xaaaa, 0x0, 0x4321 };
short	indat[ 7 ];

main( argc, argv )
	int	argc;
	char	**argv;
{
	struct	drop	drop;
	short		s, *sp;
	
	init( argc, argv );
	if( (drfd = open( dev, O_RDWR )) < 0 ) {
		fprintf( stderr, "FCannot open %s\n", dev );
		exit( 0 );
	}
	/*
	 * Check p-i/o loop by writing and reading the data registers.
	 */
	for( sp = piodat; sp < &piodat[ sizeof(piodat)/sizeof(short) ]; sp++ ) {
		drop.dr_outdata = *sp;
		drop.dr_cmd = DR_PIOW;
		if( ioctl( drfd, DR_IOC, &drop ) < 0 )
			goto pexit;

		if( verbose || (drop.dr_outdata != drop.dr_indata) ) {
			printf("P-i/o test:  out=0x%04x, in=0x%04x\n",
				drop.dr_outdata, drop.dr_indata );
		}
	}
	/*
	 * Since this is a half duplex device, we can't observe the
	 * entire transfer in loopback mode.  However, we can do a
	 * DMA transfer and then verify the last word.
	 */
	drop.dr_cmd = DR_PCYL | DR_DFCN | DR_SFCN;
	if( ioctl( drfd, DR_IOC, &drop ) < 0 )
		goto pexit;

	if( write( drfd, outdat, sizeof(outdat) ) < 0 )
		goto pexit;

	sleep( 1 );
	drop.dr_cmd = 0;
	if( ioctl( drfd, DR_IOC, &drop ) < 0 )
		goto pexit;

	s = outdat[ (sizeof(outdat)/sizeof(short)) - 1 ];
	if( verbose || (drop.dr_indata != s) ) {
		printf("Write DMA; expected 0x%04x, got 0x%04x\n",
			drop.dr_indata, s );
	}
	/*
	 * In similar manner, we can sort of check input by placing
	 * a known value in the output register and then forcing a DMA
	 * input.  The value should be repeated in all input locations.
	 */
	drop.dr_cmd = DR_PCYL | DR_DFCN | DR_SFCN | DR_PFN1 | DR_PIOW;
	drop.dr_outdata = 0x1423;
	if( ioctl( drfd, DR_IOC, &drop ) < 0 )
		goto pexit;

	if( read( drfd, indat, sizeof(indat) ) < 0 )
		goto pexit;

	for( sp=indat; sp < &indat[ sizeof(indat)/sizeof(short) ]; sp++ ) {
		if( verbose || (*sp != drop.dr_outdata) ) {
		  printf("Read DMA; expected 0x%04x, got 0x%04x\n",
			drop.dr_outdata, *sp );
		}
	}


	close( drfd );
	exit( 0 );
pexit:
	perror( progname );
	exit( 1 );
}


init( argc, argv )
	int	argc;
	char	**argv;
{
	register char	*ap;

	progname = argv[ 0 ];
	argv[ argc ] = 0;
	while( (ap = *++argv) != 0 ) {
		if( *ap == '-' ) {
			while( *++ap ) switch ( *ap ) {

			case 'v':
				verbose++;
				break;

			case 'f':
				dev = *++argv;
				break;

			default:
				usage();
				break;

			}
		} else {
			usage();
		}
	}
}


usage()
{
	fprintf( stderr, "Usage: %s [-v] [-f device]\n", progname );
	exit( 1 );
}

#endif

