/* sysALib.s - system dependent assembly language routines - ISI Liberator */

/*
DESCRIPTION
This module contains system-dependent routines which must be written
in assembly language.  They perform functions such as locking and
unlocking interrupts, trapping to the ROM monitor, etc.  

INTERNAL: Many routines in this module "link" and "unlk" the "c" frame pointer
a6@ although they don't use it in any way!  This only for the benefit of
the stacktrace facility to allow it to properly trace tasks executing within
these routines.

*/

#define ASMLANGUAGE
#include "UniWorks.h"
#include "sysLib.h"
#include "config.h"

	/* internals */

	.globl	_sysInit	/* start of system code */
	.globl  _sysKernelTrap
	.globl  _sysVwTrap
	.globl  _sysVwTrapRtn
	.globl	_sysCacheEnable

	/* externals */

	.globl	_usrInit	/* system initialization routine */

	.globl	_vme_short
	.globl	_vme_std
	.globl	_vme_ext
	.globl	_vme_window_std
	.globl	_vme_window_ext
	.set	_vme_short,VME_SHORTIO
	.set	_vme_std,  VME_STANDARD
	.set	_vme_ext,  VME_EXTENDED
	.set	_vme_window_std,RAM_LOCAL_ADRS - RAM_A24_ADRS
	.set	_vme_window_ext,RAM_LOCAL_ADRS - RAM_A32_ADRS

	.text
	.even

/***********************************************************************
*
* sysInit - start after boot
*
* This is the system start-up entry point for UniWorks in ram.
* It is the first code executed after booting.
* It disables the interrupts, sets up the stack,
* and jumps to the C routine usrInit in usrConfig.c.
*
* The initial stack is set to grow down from sysInit.
* Note that this initial stack is used only by usrInit,
* then is never used again.
* Memory for the stack needs to be accounted for when determining the load
* address of the system.
*
* THIS IS NOT A CALLABLE ROUTINE.

* sysInit ()	/* THIS IS NOT A CALLABLE ROUTINE *

*/

_sysInit:

/* disable interrupts, turn on M and S bits - this sets the system stack 
 * pointer to the master stack pointer which VRTX uses in VRTX_INIT. 
 */
	movew	#0x3700,sr		/**/
	movel	#_sysInit,a7		/* set stack to grow down from code */
	movel	#BOOT_WARM_AUTOBOOT,a7@- /* push start type arg = WARM_BOOT */
	jsr	_usrInit		/* never returns - starts up kernel */

/***********************************************************************
*
* sysKernelTrap - trap to kernel function
*
* This routine traps to the kernel. It can only be called from assembly
* language.
* The trap number for the kernel is defined in config.h.
*
* NOMANUAL

* sysKernelTrap ()

*/
_sysKernelTrap:
	trap	#TRAP_KERNEL
	rts

/***********************************************************************
*
* sysVwTrap - trap to UniWorks function
*
* This routine traps to UniWorks. It can only be called from assembly
* language.
* The trap number for UniWorks is defined in config.h.
*
* NOMANUAL

* sysVwTrap ()

*/
_sysVwTrap:
	trap	#TRAP_UNIWORKS
	rts

/*
 * This routine calls the routine whose address is in a0.  Since the system
 * is now in supervisor state (since we got here from a trap) this can be
 * used to call a routine in supervisor state.
 * THIS IS NOT A C CALLABLE ROUTINE!!
 */

_sysVwTrapRtn:
	jsr	a0@		/* vector to proper routine. */
	rte			/* return from trap */


/***********************************************************************
*
* sysCacheEnable - enable cache memory
*
* Enable or disable the 68020 on-chip cache.
*
* SEE ALSO: "MC68020 32-Bit Microprocessor User's Manual"

* VOID sysCacheEnable (enable)
*     BOOL enable;		/* enable if TRUE, otherwise disable *

*/
_sysCacheEnable:

	link	a6,#0
	movel	a6@(0x08),d1		/* put enable in d1 */

	cmpl	#1,d1			/* does enable == TRUE? */
	bne	sce0			/* if not, disable cache */
	movel	#0x09,d0		/* else, clear and enable cache */
	bra	sce1

sce0:	movel	#0,d0			/* disable cache */

	/* Enable or disable the cache by writing to the cache control
	   register.  Since the assemblers differ on the syntax for this one
	   (if they even have it), it's done with two .word's.  Yucko. */

sce1:
	.word	0x4e7b		/* movec ... 			*/
	.word	0x0002		/* .....	d0,cacr 	*/

	unlk	a6
	rts

	.globl	_sysBusTas
_sysBusTas:
	link	a6,#0
	movel	sp@(8),a0
	tstb	a0@
	bne	1f
	moveb	#1,a0@
	movel	#1,d0
	unlk	a6
	rts
1:	clrl	d0
	unlk	a6
	rts

	.globl	_ts_cmd_buf
	.globl	_ts_cmd_buf_end
_ts_cmd_buf:
	.space	64
_ts_cmd_buf_end:

	.data
	.globl	_cpuspeed
_cpuspeed:	.long	14



