/* nfsDrv.c - Network File System I/O driver */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
The driver nfsDrv provides facility for accessing files transparently
over the network via NFS.
By creating a network device with nfsMount, files on a remote
NFS machine, such as UNIX, may be accessed exactly like local files.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Three routines, however, must be called directly: nfsDrv () to
initialize the driver, and nfsMount () and nfsUnmount () to mount and unmount
file systems.

NFSDRV
Before using the driver, it must be initialized by calling the routine:
.CS
    nfsDrv ()
.CE
This routine should be called exactly once, before any reads, writes, or
other NFS calls.  Normally, it is called from usrRoot (2) in usrConfig (1).

CREATING NFS DEVICES
In order to access a remote file system, an NFS device must be created.
If there is a UNIX host "wrs" with an NFS mountable disk "/d0",
i.e. "/d0" is listed in the file /etc/exports on "wrs",
then files on "/d0" may be accessed from UniWorks.
The routine nfsMount can be used to create a UniWorks NFS device for "/d0/".
It is defined as follows:
.CS
STATUS nfsMount (host, fileSystem, localName)
    char *host;       /* name of remote host                    *
    char *fileSystem; /* name of remote directory to mount      *
    char *localName;  /* local device name for remote directory *
                      /* (NULL = use fileSystem name)           *
.CE
To create the device, make the call:
.CS
   nfsMount ("wrs", "/d0/", "/myd0/");
.CE
A file on the UNIX system "wrs" named /d0/dog may now be accessed
as "/myd0/dog".  The third parameter to nfsMount may be NULL, in
which case the device name will be the same as the second parameter.

The host ("wrs" in this case) must have already been specified with a
hostAdd (2) call.  This may have already been done if UniWorks was booted
from "wrs".  The routine nfsDevShow lists the mounted NFS devices.
.CS
    -> nfsDevShow                     
    device name          file system                                       
    -----------          -----------                                       
    /yuba1/              yuba:/yuba1/                                      
    /wrs1/               wrs:/wrs1                                          
    value = 0 = 0x0
    -> 
.CE

IOCTL
The NFS driver ioctl (2) call responds to the following codes:
.CS
 FIOSEEK  - seek to a byte in the open file, see fseek (2).
 FIOWHERE - return the current byte pointer into the open file.
 FIONREAD - return (in *arg) the number of unread bytes in the file.
.CE

SEE ALSO: nfsLib (1), hostAdd (2), "Network"
*/

#include "UniWorks.h"
#include "iosLib.h"
#include "ioLib.h"
#include "memLib.h"
#include "semLib.h"
#include "strLib.h"
#include "lstLib.h"
#include "nfsDrv.h"
#include "rpc.h"
#include "xdr_nfs.h"
#include "nfsLib.h"

/******** THIS NEEDS TO BE MOVED TO A GLOBAL PLACE (out of here and netDrv,
		doubly defined ********/

#define FD_MODE		3	/* mode mask for opened file */
				/* READ, WRITE or UPDATE */

/* XXX super semaphore - until semLib is more sophisticated */

typedef struct			/* SSEMAPHORE */
    {
    SEM_ID semId;
    int tid;
    } SSEMAPHORE;


typedef struct			/* NFS_DEV - nfs device structure */
    {
    DEV_HDR devHdr;
    char host[NFS_MAXNAMLEN];		/* host for this device	*/
    char fileSystem [NFS_MAXPATHLEN];	/* file system mounted on this device */
    nfs_fh fileHandle;			/* nfs file handle for mounted file
					   system */
    } NFS_DEV;

/* nfs file descriptor */

typedef struct			/* NFS_FD - nfs file descriptor */
    {
    NFS_DEV *nfsDev;		/* pointer to this file's nfs device */
    char *fileName;		/* name of file or directory */
    nfs_fh fileHandle;		/* file's file handle */
    nfs_fh dirHandle;		/* file's directory file handle */
    fattr fileAttr;		/* file's attributes */
    unsigned mode;		/* mode (READ, WRITE or UPDATE) */
    unsigned fileCurByte;	/* number of current byte in file */
    SSEMAPHORE nfsFdSem;	/* accessing semaphore */
    BOOL cacheValid;		/* TRUE if cache is valid, FALSE if cache is
				     empty or cacheCurByte not the same as 
				     the current byte in the file */

			/* cache fields are valid only if cacheValid = TRUE */
    char *cacheBuf;		/* pointer to file's read cache */
    char *cacheCurByte;		/* pointer to current byte in cache */
    unsigned cacheBytesLeft;	/* number of bytes left in cache between
				     cacheCurByte and the end of valid cache
				     material.  NOTE: This number may be
				     smaller than the amount of room left
				     in the cache for writing. */
    BOOL cacheDirty;		/* TRUE if cache has been written to, but not
				     flushed */
    } NFS_FD;

LOCAL int nfsDrvNum;		/* this particular driver number */

IMPORT unsigned nfsMaxMsgLen;	/* maximum length of an nfs read or write buf */
unsigned nfsCacheSize;		/* size of an nfs cache */

/* forward declarations */

LOCAL int nfsCreate ();
LOCAL int nfsDelete ();
LOCAL int nfsOpen ();
LOCAL STATUS nfsClose ();
LOCAL int nfsRead ();
LOCAL int nfsWrite ();
LOCAL int nfsIoctl ();


/*******************************************************************************
*
* nfsDrv - NFS driver installation
*
* Initializes and installs the NFS driver.
* Must be called before other NFS file functions are used.
*/

STATUS nfsDrv ()

    {
    if (nfsDrvNum == 0)
	{
	nfsCacheSize = nfsMaxMsgLen;

	nfsDrvNum = iosDrvInstall (nfsCreate, nfsDelete, nfsOpen, nfsClose, 
				   nfsRead, nfsWrite, nfsIoctl);
	}

    return (nfsDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* nfsMount - mount an NFS directory
*
* This routine creates a local name for a remote directory.  This is done
* by creating a device with name `localName', that refers to the
* remote directory `fileSystem' on the specified host.
* The host must have already been created with a hostAdd (2) call.
* If `localName' is NULL then the local device name will be the
* same as the `fileSystem' name.
*
* SEE ALSO: nfsUnmount (2), hostAdd (2)
*
* RETURNS: OK or ERROR if driver not installed, invalid host or out of memory
*/

STATUS nfsMount (host, fileSystem, localName)
    char *host;		/* name of remote host                    */
    char *fileSystem;	/* name of remote directory to mount      */
    char *localName;	/* local device name for remote directory */
			/* (NULL = use fileSystem name)           */

    {
    FAST NFS_DEV *pNfsDev;
    nfs_fh fileHandle;
    
    if (nfsDrvNum < 1)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}

    if ((pNfsDev = (NFS_DEV *) malloc (sizeof (NFS_DEV))) == NULL)
	return (ERROR);

    /* perform and nfs mount of the directory */

    if (nfsDirMount (host, fileSystem, &fileHandle) != OK)
	return (ERROR);

    /* fill in the nfs device descripter */

    strcpy (pNfsDev->host, host);
    strcpy (pNfsDev->fileSystem, fileSystem);
    bcopy ((char *) &fileHandle, (char *) &pNfsDev->fileHandle, sizeof(nfs_fh));

    if (localName == NULL)
	localName = fileSystem;		/* name the same as remote fs */

    /* add device to I/O system */

    return (iosDevAdd ((DEV_HDR *) pNfsDev, localName, nfsDrvNum));
    }
/*******************************************************************************
*
* nfsDevShow - display all mounted NFS devices
*
* This routine prints the device name and associated NFS file system
* on standard output.
*/

VOID nfsDevShow ()

    {
    char fileSysName [MAX_FILENAME_LENGTH];
    DEV_HDR *pDev0 = NULL;
    DEV_HDR *pDev1;

    printf ("%-20.20s %-50.50s\n", "device name", "file system");
    printf ("%-20.20s %-50.50s\n", "-----------", "-----------");

    /* get entries from the I/O system's device list */

    while ((pDev1 = iosNextDevGet (pDev0)) != NULL)
	{
	if (pDev1->drvNum == nfsDrvNum)
	    {
	    /* found an nfs device, print information */

	    strcpy (fileSysName, ((NFS_DEV *) pDev1)->host);
	    strcat (fileSysName, ":");
	    strcat (fileSysName, ((NFS_DEV *) pDev1)->fileSystem);
	    printf ("%-20.20s %-50.50s\n", pDev1->name, fileSysName);
	    }
	pDev0 = pDev1;
	}
    }
/*******************************************************************************
*
* nfsUnmount - unmount an NFS device
*
* RETURNS: OK or ERROR if `localName' not an NFS device, or unable to unmount
*
* SEE ALSO: nfsDirUnmount (2)
*/

STATUS nfsUnmount (localName)
    char *localName;	/* local of nfs device */

    {
    FAST NFS_DEV *pNfsDev;
    char *dummy;

    /* find the device in the I/O system */

    if ((pNfsDev = (NFS_DEV *) iosDevFind (localName, &dummy)) == NULL)
	return (ERROR);

    /* make sure device is an nfs driver */

    if (pNfsDev->devHdr.drvNum != nfsDrvNum)
	{
	errnoSet (S_nfsDrv_NOT_AN_NFS_DEVICE);
	return (ERROR);
	}

    /* perform an nfs unmount of the directory */

    if (nfsDirUnmount (pNfsDev->host, pNfsDev->fileSystem) == ERROR)
	return (ERROR);

    /* delete the device from the I/O system */

    iosDevDelete ((DEV_HDR *) pNfsDev);
    return (OK);
    }

/* routines supplied to the I/O system */

/*******************************************************************************
*
* nfsCreate - create a remote NFS file
*
* Returns an open nfs file descriptor.
* Files are created with O_CREAT or'd in.
*
* Used for creating files only, not directories.
* Called only by the I/O system.
*
* RETURNS: pointer to nfs file descriptor or ERROR
*/

LOCAL int nfsCreate (pNfsDev, fileName, mode)
    NFS_DEV *pNfsDev;	/* pointer to nfs device */
    char *fileName;	/* nfs file name (relative to mount point) */
    int mode;		/* always or'd with O_CREAT */

    {
    /* mode legality is checked for in nfsOpen */

    /* don't allow null filenames */

    if (fileName [0] == EOS)
	{
	errnoSet (S_ioLib_NO_FILENAME);
	return (ERROR);
	}

    /* open the file being created,
       give the file default UNIX file permissions */

    return (nfsOpen (pNfsDev, fileName , O_CREAT | mode, DEFAULT_FILE_PERM));
    }
/*******************************************************************************
*
* nfsDelete - delete a remote file
*
* Deletes a file on a remote system.
* Called only by the I/O system.
*
* RETURNS: OK | ERROR
*/

LOCAL int nfsDelete (pNfsDev, fileName)
    NFS_DEV *pNfsDev;	/* pointer to nfs device */
    char *fileName;	/* remote file name */

    {

    /* don't allow null filenames */

    if (fileName [0] == EOS)
	{
	errnoSet (S_ioLib_NO_FILENAME);
	return (ERROR);
	}

    return (nfsFileRemove (pNfsDev->host, &pNfsDev->fileHandle, fileName));
    }
/*******************************************************************************
*
* nfsOpen - open an NFS file
*
* nfsOpen opens the remote file.
* Called only by the I/O system.
*
* RETURNS: pointer to open network file descriptor || FOLLOW_LINK || ERROR
*/

LOCAL int nfsOpen (pNfsDev, fileName, flags, mode)
    NFS_DEV *pNfsDev;	/* pointer to nfs device */
    char *fileName;	/* remote file or directory name to open */
    int flags;		/* READ, WRITE, or UPDATE and O_CREAT */
    int mode;		/* UNIX chmod style */

    {
    int status = OK;
    char fullName [NFS_MAXPATHLEN];	/* full file or directory name */
    FAST NFS_FD *nfsFd;
    diropres dirResult;			/* directory info returned via nfs */
    nfs_fh dirHandle;			/* file's directory file handle */
    int openMode = flags & FD_MODE;	/* mode to open file with (READ, WRITE,
					   or UPDATE */

    if ((openMode != READ) && (openMode != WRITE) && (openMode != UPDATE))
	{
	errnoSet (S_nfsDrv_BAD_FLAG_MODE);
	return (ERROR);
	}

    if (fileName [0] == EOS)
	{
	if (flags & O_CREAT)
	    {
	    errnoSet (S_nfsDrv_CREATE_NO_FILE_NAME);
	    return (ERROR);
	    }
	strcpy (fullName, ".");
	}
    else
	(void) strcpy (fullName, fileName);

    if (flags & O_CREAT)	/* create file or directory */
	{
	status = nfsThingCreate (pNfsDev->host, fullName, &pNfsDev->fileHandle,
				 &dirResult, &dirHandle, (u_int) mode);
	}
    else			/* open existing file or directory */
	{
	status = nfsLookUpByName (pNfsDev->host, fullName, &pNfsDev->fileHandle,
				  &dirResult, &dirHandle);
	}

    if (status == FOLLOW_LINK)
	strcpy (fileName, fullName);

    if (status != OK)
	return (status);

    if ((nfsFd = (NFS_FD *) malloc (sizeof (NFS_FD))) == NULL)
	return (ERROR);

    /* file in file descriptor with newly retrieved information */

    bcopy ((char *) &dirResult.diropres_u.finfo.file,
	   (char *) &nfsFd->fileHandle, sizeof (nfs_fh));
    bcopy ((char *) &dirHandle,
	   (char *) &nfsFd->dirHandle, sizeof (nfs_fh));
    bcopy ((char *) &dirResult.diropres_u.finfo.attributes,
	   (char *) &nfsFd->fileAttr, sizeof (fattr));

    nfsFd->fileCurByte = 0;
    nfsFd->mode = openMode;
    nfsFd->cacheValid = FALSE;
    if ((nfsFd->cacheBuf = malloc (nfsCacheSize)) == NULL)
	{
	free ((char *) nfsFd);
	return (ERROR);
	}

    nfsFd->cacheCurByte = nfsFd->cacheBuf;
    nfsFd->cacheBytesLeft = 0;
    nfsFd->cacheDirty = FALSE;

    nfsFd->nfsDev = pNfsDev;

    if ((nfsFd->fileName = malloc ((unsigned) (strlen (fullName) + 1))) == NULL)
	{
	free ((char *) nfsFd->cacheBuf);
	free ((char *) nfsFd);
	return (ERROR);
	}

    strcpy (nfsFd->fileName, fullName);

    ssemInit (&nfsFd->nfsFdSem);
    ssemGive (&nfsFd->nfsFdSem);

    return ((int) nfsFd);
    }
/*******************************************************************************
*
* nfsClose - close a remote file
*
* Called only by the I/O system.
*
* RETURNS: OK or ERROR if file failed to flush
*/

LOCAL STATUS nfsClose (nfsFd)
    FAST NFS_FD *nfsFd;

    {
    int status = OK;

    ssemTake (&nfsFd->nfsFdSem);

    if (nfsFd->cacheDirty)
	status = nfsFlushCache (nfsFd);

    free (nfsFd->fileName);
    free (nfsFd->cacheBuf);
    semDelete (nfsFd->nfsFdSem.semId);		/* XXX delete super sem XXX */
    free ((char *) nfsFd);
    nfsClientClose ();

    return (status == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* nfsRead - read bytes from remote file
*
* nfsRead reads up to the specified number of bytes from the open NFS
* file and puts them into a buffer.  Bytes are read starting
* from the point marked by the file pointer.  The file pointer is then 
* updated to point immediately after the last character that was read.
* 
* A cache is used for keeping nfs network reads and writes down to a minimum.
* nfsRead returns as many bytes as it can read from the cache, up to the
* specified number of bytes.
*
* Called only by the I/O system.
*
* SIDE EFFECTS: moves file pointer
*
* RETURNS: number of bytes read or ERROR.
*/

LOCAL int nfsRead (nfsFd, buf, maxBytes)
    FAST NFS_FD *nfsFd;	/* pointer to open network file descriptor */
    char *buf;		/* pointer to buffer to receive bytes	*/
    FAST int maxBytes;	/* max number of bytes to read into buffer */

    {
    int nRead;			/* number of bytes actually read */
    int nCacheRead;		/* number of bytes read into cache */
    fattr newAttributes;	/* file attributes */
    STATUS status = OK;

    /* check for valid maxBytes */

    if (maxBytes < 0)
	{
	errnoSet (S_nfsDrv_INVALID_NUMBER_OF_BYTES);
	return (ERROR);
	}
 
    if (maxBytes == 0)
	return (0);

    /* if file opened for WRITE, don't read */

    if (nfsFd->mode == WRITE)
	{
	errnoSet (S_nfsDrv_WRITE_ONLY_CANNOT_READ);
	return (ERROR);
	}

    ssemTake (&nfsFd->nfsFdSem);

    /* if the read can't be done from the cache, freshen the cache by doing
     * an nfs read.
     */

    if (!nfsFd->cacheValid)
	{
	nCacheRead = nfsFileRead (nfsFd->nfsDev->host, &nfsFd->fileHandle,
			 nfsFd->fileCurByte, nfsCacheSize, nfsFd->cacheBuf,
			 &newAttributes);

	if (nCacheRead < 0)
	    {
	    ssemGive (&nfsFd->nfsFdSem);
	    return (ERROR);
	    }

	/* update file attributes */

	bcopy ((char *) &newAttributes, (char *) &nfsFd->fileAttr,
	       sizeof (fattr));

	nfsFd->cacheCurByte = nfsFd->cacheBuf;	/* update cache pointers */
	nfsFd->cacheBytesLeft = nCacheRead;

	nfsFd->cacheValid = TRUE;		/* cache is valid */
	}

    /* read as many bytes as possible from what's left in the cache,
     * up to the amount requested
     */

    if (maxBytes < nfsFd->cacheBytesLeft)
	nRead = maxBytes;
    else
	nRead = nfsFd->cacheBytesLeft;

    /* copy bytes into user's buffer */

    bcopy (nfsFd->cacheCurByte, buf, nRead);

    /* update file pointer */

    status = nfsSeek (nfsFd, (int) nfsFd->fileCurByte + nRead);

    ssemGive (&nfsFd->nfsFdSem);

    return (status == ERROR ? ERROR : nRead);
    }
/*******************************************************************************
*
* nfsWrite - write bytes to remote file
*
* nfsWrite copies up to the specified number of bytes from the buffer
* to the nfs file.  Bytes are written starting at the current file pointer.
* The file pointer is updated to point immediately after the last
* character that was written.
*
* A cache is used for keeping nfs network reads and writes down to a minimum.
* If all goes well, the entire buffer is written to the cache and/or actual
* file.
*
* Called only by the I/O system.
*
* SIDE EFFECTS: moves file pointer
*
* RETURNS:
*    Number of bytes written (error if != nBytes)
*    ERROR if nBytes < 0, or nfs write is not successful
*/

LOCAL int nfsWrite (nfsFd, buf, nBytes)
    FAST NFS_FD *nfsFd;	/* open file descriptor */
    char *buf;		/* buffer being written from */
    FAST int nBytes;	/* number of bytes to write to file */

    {
    int status = OK;
    FAST int nCacheWrite;	/* number of bytes written to cache */
    int nWritten = 0;		/* number of bytes written so far */
    char *pBuf = buf;		/* pointer within user's buffer */
    int cacheMargin;		/* margin left in cache for writing */
    FAST unsigned newPos;	/* new position of file pointer */
    char *pCacheWrite;		/* pointer in cache where write should start */

    /* check for valid number of bytes */

    if (nBytes < 0)
	{
	errnoSet (S_nfsDrv_INVALID_NUMBER_OF_BYTES);
	return (ERROR);
	}
 
    /* if file opened for READ, don't write */

    if (nfsFd->mode == READ)
	{
	errnoSet (S_nfsDrv_READ_ONLY_CANNOT_WRITE);
	return (ERROR);
	}

    ssemTake (&nfsFd->nfsFdSem);

    /* do the write, until entire buffer has been written out */

    while (nWritten < nBytes)
	{
	/* if the cache is valid, start writing from the current cache pointer
	 * to the end of the cache, otherwise, use the entire cache.
	 * Write as many bytes as will fit in the cache.
	 */

	if (nfsFd->cacheValid)
	    {
	    cacheMargin = nfsFd->cacheBuf + nfsCacheSize - nfsFd->cacheCurByte;
	    if (nBytes - nWritten > cacheMargin)
		nCacheWrite = cacheMargin;
	    else
		nCacheWrite = nBytes - nWritten;
	    pCacheWrite = nfsFd->cacheCurByte;
	    }
	else
	    {
	    if (nBytes - nWritten > nfsCacheSize)
		nCacheWrite = nfsCacheSize;
	    else
		nCacheWrite = nBytes - nWritten;
	    pCacheWrite = nfsFd->cacheBuf;
	    }

	/* copy the bytes into the cache */

	bcopy (pBuf, pCacheWrite, nCacheWrite);

	if (nCacheWrite > nfsFd->cacheBytesLeft)
	    nfsFd->cacheBytesLeft = nCacheWrite;

	/* cache has been written in, it is soiled */

	nfsFd->cacheDirty = TRUE;

	/* if the cache wasn't valid, update pointers and validate it */

	if (!nfsFd->cacheValid)
	    {
	    nfsFd->cacheCurByte = nfsFd->cacheBuf;
	    nfsFd->cacheBytesLeft = nCacheWrite;
	    nfsFd->cacheValid = TRUE;
	    }
	
	/* what's the new position of the file pointer? */

	newPos = nfsFd->fileCurByte + nCacheWrite;

	/* if the new file pointer reaches past the end of the file,
	 * the file has grown, update the size of the file
	 */

	if (newPos > nfsFd->fileAttr.size)
	    nfsFd->fileAttr.size = newPos;

	/* update the file pointer, any cache flushes will occur during
	 * the seek
	 */

	status = nfsSeek (nfsFd, (int) newPos);
	if (status == ERROR)
	    {
	    ssemGive (&nfsFd->nfsFdSem);
	    return (ERROR);
	    }
	nWritten += nCacheWrite;
	pBuf += nCacheWrite;
	}

    ssemGive (&nfsFd->nfsFdSem);

    return (nWritten);
    }
/*******************************************************************************
*
* nfsIoctl - do device specific control function
*
* Called only by the I/O system.
*
* RETURNS: whatever the called function returns
*/

LOCAL int nfsIoctl (nfsFd, function, arg)
    FAST NFS_FD *nfsFd;	/* open nfs file descriptor */
    FAST int function;	/* function code */
    FAST int arg;	/* argument for function */

    {
    switch (function)
	{
	case FIOSEEK:
	    return (nfsSeek (nfsFd, arg));

	case FIOWHERE:
    	    *((int *) arg) = nfsFd->fileCurByte;
    	    return (nfsFd->fileCurByte);

	case FIONREAD:
    	    *((int *) arg) = nfsFd->fileAttr.size - nfsFd->fileCurByte;
	    return (OK);

	case FIODIRENTRY:
	    /* this is a kludge for 'ls'.  Do the listing, then return
	       ERROR so that 'ls' doesn't try listing an rt-11 device */

	    (void) nfsLs (nfsFd->nfsDev->host, &nfsFd->fileHandle);
	    return (ERROR);

	default:
	    /* XXX should be _ioLib_ */
	    errnoSet (S_nfsDrv_UNKNOWN_REQUEST);
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* nfsSeek - change file's current character position
*
* This routine moves the file pointer by the offset to a new 
* position.  If the new position is past the end of the file,
* the pointer is moved to one byte past the end of file.
* If the new position moves the file pointer out of the range of
* what's in the cache and the cache is dirty (i.e. the cache has been
* written in), then the cache is written out to the nfs file and the
* cache is marked as invalid.
* 
* Called only by the I/O system.
*
* RETURNS: OK | ERROR
*/

LOCAL int nfsSeek (nfsFd, newPos)
    FAST NFS_FD *nfsFd;
    FAST int newPos;

    {
    FAST int interval;

    if (newPos < 0)
	return (ERROR);

    if (newPos == nfsFd->fileCurByte)
	return (OK);

    ssemTake (&nfsFd->nfsFdSem);

    if (newPos >= nfsFd->fileAttr.size)
	newPos = nfsFd->fileAttr.size;

    if (nfsFd->cacheValid)	/* if cache is valid, update cache pointer */
	{
	/* how far will file pointer be moved? */

	interval = newPos - nfsFd->fileCurByte;

	/* if the new pointer precedes what's in the cache,
	 * OR if the new ptr points past what's in the cache,
	 * OR if the new ptr points past the end of the valid cache bytes,
	 * THEN the cache is no longer valid.
	 *
	 * NOTE:  The cache is still valid if the new pointer points
	 * IMMEDIATELY after the valid bytes in the cache, but still
	 * hasn't reached the end of the cache buffer.  The cache can
	 * still be written to and should not be flushed until it is full
	 * or the file is closed (a similar technique is used in some states
	 * during drought conditions).
	 */

	if (((interval < 0) && (nfsFd->cacheCurByte + interval <
				nfsFd->cacheBuf))
	    || (interval > nfsFd->cacheBytesLeft)
	    || ((nfsFd->cacheBuf + nfsCacheSize) <= (nfsFd->cacheCurByte +
						    interval)))
	    {
	    /* if the new cache pointer goes out of range of the cache,
	     * mark the cache as invalid, and flush if necessary.
	     */

	    if (nfsFd->cacheDirty)
		if (nfsFlushCache (nfsFd) == ERROR)
		    {
		    ssemGive (&nfsFd->nfsFdSem);
		    return (ERROR);
		    }
	    nfsFd->cacheValid = FALSE;
	    nfsFd->cacheBytesLeft = 0;
	    nfsFd->cacheCurByte = nfsFd->cacheBuf;
	    nfsFd->cacheDirty = FALSE;
	    }
	else
	    {
	    /* if the new position stays within range of the cache,
	     * update the cache pointers.
	     */

	    nfsFd->cacheCurByte += interval;
	    nfsFd->cacheBytesLeft -= interval;
	    }
	}

    nfsFd->fileCurByte = newPos;
	
    ssemGive (&nfsFd->nfsFdSem);
    return (OK);
    }
/*******************************************************************************
*
* nfsFlushCache - flush the cache to the remote NFS file
*
* Called only by the I/O system.
*
* RETURNS: number of bytes written or ERROR
*/

LOCAL int nfsFlushCache (nfsFd)
    FAST NFS_FD *nfsFd;

    {
    fattr newAttributes;
    int nWrite;		/* how many bytes are written with nfs? */
    unsigned offset;	/* offset in the file where cache gets written */
    unsigned count;	/* number of bytes to write with nfs */
    unsigned dist;	/* distance between beginning of cache and current file
			     pointer (number of bytes) */

    if (!nfsFd->cacheValid)
	{
	errnoSet (S_nfsDrv_FATAL_ERR_FLUSH_INVALID_CACHE);
	return (ERROR);
	}

    dist = nfsFd->cacheCurByte - nfsFd->cacheBuf;
    offset = nfsFd->fileCurByte - dist;
    count = dist + nfsFd->cacheBytesLeft;

    nWrite = nfsFileWrite (nfsFd->nfsDev->host, &nfsFd->fileHandle,
			   offset, count, nfsFd->cacheBuf, &newAttributes);

    /* update file attributes */

    if (nWrite != ERROR)
	bcopy ((char *) &newAttributes, (char *) &nfsFd->fileAttr,
	       sizeof (fattr));

    return (nWrite);
    }

/* XXX super semaphore routines - until semLib is more sophisticated */

/*******************************************************************************
*
* ssemInit - super semInit
*
* This is just like the standard semCreate except that the semaphore
* includes a tid field.
*/

LOCAL VOID ssemInit (pSSemaphore)
    SSEMAPHORE *pSSemaphore;

    {
    pSSemaphore->tid = NONE;
    pSSemaphore->semId = semCreate ();
    }
/*******************************************************************************
*
* ssemGive - super semGive
*
* This is just like the standard semGive except that the semaphore
* includes a tid field.
*/

LOCAL VOID ssemGive (pSSemaphore)
    SSEMAPHORE *pSSemaphore;

    {
    pSSemaphore->tid = NONE;
    semGive (pSSemaphore->semId);
    }
/*******************************************************************************
*
* ssemTake - super semTake
*
* This is just like the standard semTake except if the semaphore is already
* taken by the requesting task, it won't deadlock trying to take it again.
*/

LOCAL VOID ssemTake (pSSemaphore)
    SSEMAPHORE *pSSemaphore;

    {
    int tid = taskIdSelf ();

    if (pSSemaphore->tid != tid)
	{
	semTake (pSSemaphore->semId);
	pSSemaphore->tid = tid;
	}
    }
