
/****************************************************************************
 File: xshape.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.11  $
   $Author:   gmills  $
     $Date:   24 Feb 1992 16:57:40  $

 This file contains the API for clsXShape, a skeletal class designed to be
 subclassed by particular shape recognition engines.  In particular,
 the GOWrite shape recognizer, clsCTShape, is a subclass of clsXShape.

 clsXShape inherits from clsOpenServiceObject.

****************************************************************************/
#ifndef XSHAPE_INCLUDED
#define XSHAPE_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif
#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif
#ifndef OSHEAP_INCLUDED
#include <osheap.h>
#endif
#ifndef OPENSERV_INCLUDED
#include <openserv.h>
#endif

/****  Terminology change  ****/

//      NEW NAME (use these)        OLD NAME (avoid using these, from uid.h)
#define theShapeEngines             theHWXEngines
#define theInstalledShapeProfiles   theInstalledHWXProtos
#define clsShapeEngineService       clsHWXEngineService
#define clsShapeProfileInstallMgr   clsHWXProtoInstallMgr
#define msgShapeSvcCurrentChanged   msgHWXSvcCurrentChanged
#define SHAPE_SVC_CURRENT_CHANGED   HWX_SVC_CURRENT_CHANGED
#define P_SHAPE_SVC_CURRENT_CHANGED P_HWX_SVC_CURRENT_CHANGED


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define xsMaxCharList   20  // largest allowable matchArraySize for msgXShapeRecognize
#define xsMaxPath        4  // most strokes allowable to send to msgXShapeRecognize
#define xsMinMatchScore minS16  // worst possible score for translation

#define xsDigitizerResolution 254 // temporary hack.  Eventually variable

/*
 The basic types of shape profile ("resource") data stored in 
 files.  This refers to the "alphabet" which the resource is able
 to recognize, not the use to which the recognized value will be
 put.  In particular a text resource may be used as part of the
 process of recognizing gestures, since some gestures are upper
 case letters.
 */
Enum16(XS_RESOURCE_TYPE) {
	xsResText		= 0,  // alphabetic (ascii)
	xsResReserved	= 1,  // reserved for use by GO
	xsResGesture	= 2   // gestures
};

/*
 The types of data structure used to return information from
 msgXShapeRecognize.
*/
Enum16(XS_MATCH_TYPE) {
	xsMatchAscii = 1,     // uses XS_ASCII_MATCH data structure
	xsMatchGesture = 2,   // uses XS_GESTURE_MATCH data structure
	xsMatchInternal = 3,  // uses subclass-specific data structure
	xsMatchInternal2 = 4, // uses alternate subclass-specific data structure
};

/*
 Eight principal compass directions for straight lines.
*/
Enum16(XS_DIRECTION) {
	xsRight = 0,
	xsUpRight = 1,
	xsUp = 2,
	xsUpLeft = 3,
	xsLeft = 4,
	xsDownLeft = 5,
	xsDown = 6,
	xsDownRight = 7,

	// Special indicators
	xsAllDirections = 8,  // used internally
	xsDirEndMark = 9      // marks end of array of directions
};

#define xsNumDirections  (8)

#define XSNextDirectionCCW(d) (((d) + 1) & 7)
#define XSNextDirectionCW(d) (((d) - 1) & 7)
#define XSOppositeDirection(d) ((d) ^ 4)
#define XSDeltaDirection(start, end) (((end) - (start)) & 7)
#define XSDeltaDirectionAdd(start, delta) (((start) + (delta)) & 7)


/*
 The following structures capture basic information about
 strokes (a stroke being a sequence of points passed through 
 by the pen).  See msgXShapeStrokePreview for further details.
*/
typedef struct XS_OCTAGON {
   S16 limit[xsNumDirections]; // max projection in each direction
} XS_OCTAGON, * P_XS_OCTAGON;

/*
 Data structure for returning information about recognition of an
 ascii character from msgXShapeRecognize.
*/
typedef struct XS_ASCII_MATCH {
	S16 score;       // "penalty" for the match
	U8 character;    // ascii code of proposed translation
	U8 segmentOffset;// reserved for GO.  msgXShapeRecognize should set to 0
} XS_ASCII_MATCH, *P_XS_ASCII_MATCH;

/*
 Data structure for returning information about recognition of a
 gesture from msgXShapeRecognize.
*/
typedef struct XS_GESTURE_MATCH {
	S16 score;       // "penalty" for the match
	U32 gestureId;   // proposed translation (id codes defined in xgesture.h)
	POINT hotPoint;  // coordinates of target point of the gesture
} XS_GESTURE_MATCH, *P_XS_GESTURE_MATCH;

/*
 Data structure for returning information about recognition of a 
 straight line or a dot.  Used by the GO context level processing
 to aid in segmentation.  These scores are calculated by the GO
 context engine;  they should not be calculated or used by 3rd party
 shape engine developers.
*/
typedef struct XS_LD_MATCH {
	S16 dotScore;    // Score for a dot.
	S16 lineScore02; // Score for horiz/vert line
	S16 lineScore13; // Score for forw/backw slanted line
} XS_LD_MATCH, *P_XS_LD_MATCH;

/*
 The XS_STROKE record holds information pertinent to a single stroke.
 PenPoint computes all fields of this structure except pData and
 numData.  The latter two are (optionally) computed by the shape matching
 engine.  They are intended to hold whatever information the shape
 matcher wishes to extract from a single individual stroke.
*/
typedef struct XS_STROKE { 
	struct XS_STROKE *pNextStroke;// pointer to next stroke
	struct XS_STROKE *pPrevStroke;// pointer to previous stroke

	U16 strokeId;         // a unique identifier of this stroke
	struct POINT *pPoint; // arr of digitizer points (pendown to penup)
	U16 numPoints;        // number of digitizer points (excl. end marker)
	XS_OCTAGON bound;        // bounds of this stroke

	P_UNKNOWN pData;      // subclass-specific data extracted from stroke
	U16 numData;          // subclass-specific counter for pData

	XS_ASCII_MATCH
		asciiMatch[xsMaxCharList];// cached results of single stroke recog.
	XS_LD_MATCH ldMatch;           // scores for line and dot matches
} XS_STROKE, *P_XS_STROKE;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Initialization Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNewDefaults: takes P_XSHAPE_NEW, returns STATUS
 category: class message
 Initializes the XSHAPE_NEW structure to default values.

 Zeros out pArgs->xshape and sets
//{
	pArgs->xshape.resType = xsResText;
	pArgs->xshape.resolution = xsDigitizerResolution;
//}
*/

/****************************************************************************
 msgNew: takes P_XSHAPE_NEW, returns STATUS
 category: class message
 Creates a new shape matching object.

 This message is sent to the xshape subclass by the service manager
 when  someone has requested a new shape matching engine.   The
 service manager has filled in all of the xshape fields.  In
 responding  to this message it is merely necessary to copy the
 fields of xshape_new data  into the new object's private instance data.
*/

typedef struct XSHAPE_NEW_ONLY {
	P_UNKNOWN pProfile;    // ptr to data in subclass specific format
	U16 numProfile;        // how many records (e.g. if pProfile pts to array)
	XS_RESOURCE_TYPE resType; // type of profile:  xsResText, resGesture
	OBJECT profDirHandle;  // handle to directory where profile resides
	S16 resolution;        // digitizer granularity (dots per inch)
	S16 charConstraints;   // flags to set restricted character sets
	S16 reserved16;        // pad for now
	U32 reserved[9];       // may be used in future
} XSHAPE_NEW_ONLY, *P_XSHAPE_NEW_ONLY;

typedef struct XSHAPE_NEW {
	openServiceObjectNewFields \
	XSHAPE_NEW_ONLY		xshape;
} XSHAPE_NEW, *P_XSHAPE_NEW;

/****************************************************************************
 msgFree:  takes pNull, returns STATUS
 Destroys the object, releasing any memory associated with the translation.

 If any heaps were created in response to msgNew, this is the time to
 destroy them.  NOTE:  This is NOT the place to free memory occupied by
 the data pointed to by pProfile.  That memory was allocated by your 
 service class in response to msgXShapeSvcCurrentChanged and should only
 be free in response to the next occurrence of the same message.
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Control Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgXShapeStrokePreview: takes P_XSHAPE_STROKE_PREVIEW, returns STATUS
 Computes and stores data relating to a single stroke

 This msg gives the class the opportunity to extract and store information
 that applies to an individual stroke, not to the combined set of strokes
 that form a character.  (The latter extraction should occur entirely
 within the method for msgXShapeRecognize.)

 This message is sent by the input system as part of its background
 processing of strokes as they are entered by the user.  Background
 processing allows the system to produce the final translation more
 quickly after the user taps the translate button.

 Furthermore, a single stroke may be submitted more than once to the
 shape engine for recognition, as the context engine tries out different
 combinations of strokes searching for the best segmentation.  Thus the
 stroke will be "previewed" only once, but may appear in several
 different combinations of strokes submitted for recognition.

 The subclass is responsible for defining the format and managing the
 memory that contains the information extracted in the preview
 process.  The pointer pData in the XS_STROKE record should be set to
 point to this data.  The field numData of the XS_STROKE record is available
 to record the number of records pointed to by pData (if it's an array).

 Memory for *pData should be allocated from a local heap whose heapId
 has been stored in the instance data for the object.  The heap should 
 be created in response to msgNew and destroyed in response to msgFree.

 The method for msgXShapeStrokePreview may assume that the following
 fields of the XS_STROKE record have already been calculated:
	strokeId
	bound
	pPoint
	numPoints
 All other fields should be ignored.

 The strokeId uniquely identifies the stroke (as far as this object
 is concerned).

 The bound implicitly defines the bounding octagon for the stroke by
 recording for each of the 8 directions the maximum of the projections
 of all points in the stroke in that direction.  Given a point P and a
 direction  d, the projection of P in direction d is defined to be
 the x-coordinate of  P  in a coordinate system which is rotated
 d*45 degrees counterclockwise from the base coordinate system.
 Computationally this works out to:
	    x			if d==0  (xsRight)
	  ( x+y)/r		if d==1  (xsUpRight)
	      y			if d==2  (xsUp)
	  (-x+y)/r		if d==3  (xsUpLeft)
	   -x			if d==4  (xsLeft)
	  (-x-y)/r		if d==5  (xsDownLeft)
	     -y			if d==6  (xsDown)
	  ( x-y)/r		if d==7  (xsDownRight)
 where  r  is sqrt(2).  Division by  r  is simulated in integer arithmetic
 as multiplication by 5 followed by (integer) division by 7.

 From the bound the method can calculate other quantities as needed
 using the following formulas:

//{
	baseline = - bound.limit[xsDown];
	// because  -max{-y} = min{y}
	height   = bound.limit[xsUp] + bound.limit[xsDown];
	// because max{y} + max{-y} = max{y} - min{y}
	width    = bound.limit[xsRight] + bound.limit[xsLeft];
	// because max{x} + max{-x} = max{x} - min{x}
//}

 pPoints points to an array of digitizer points, terminated with a record
 with coordinates (minS16, minS16).  numPoints tells how many points are
 in the array, EXCLUDING the terminating record.  (So numPoints can also be
 taken as the index of the terminating record.)  The 0th record corresponds
 to penDown, the (numPoints-1)th record to penUp.

*/
#define msgXShapeStrokePreview					MakeMsg(clsXShape, 3)

typedef struct XSHAPE_STROKE_PREVIEW {
	P_XS_STROKE pFirstStroke;     // IN: pointer to stroke record
} XSHAPE_STROKE_PREVIEW, *P_XSHAPE_STROKE_PREVIEW;


/****************************************************************************
 msgXShapeRecognize: takes P_XSHAPE_RECOGNIZE, returns STATUS
 Provide possible translations for a set of strokes.

 The set of strokes (given as a linked list) is a combination which the
 context level is testing to see if it represents a single character or
 gesture.  The job of the shape engine is to return an array of the
 most likely translations (or "matches") together with a weight (or
 "score") for each of them.  If the strokes do not match any of the
 forms which the shape engine is designed to recognize, it should return
 an empty array (i.e. the first record should be marked with score
 xsMinMatchScore).

 Scores are 0 or negative, with 0 representing the best possible match.
 Scores below 0 represent progressively worse matches.  The range is
 open ended below, but generally the scores for the most unlikely but
 still remotely possible translations should fall in the -80 to -120 
 range, or very occasionally below -120.  

 Different recognition technologies may have radically different
 approaches for arriving at scores and correspondingly different models
 of what the scores mean.  One technology may assign scores as a
 measure of the amount of deviation from an ideal form, a kind of 
 Euclidean distance function.  Another technology may arrive at scores
 through a process of statistical tests, so the score
 would represent the amount of statistical evidence	there is against
 a particular translation.  Yet another technology may compute 
 probabilities.

 In order to deal uniformly with a variety of different shape
 recognition technologies, the context level processor requires that
 the scores reported by the shape engine be scaled or calibrated
 according to the following guidelines:

 1.  "Reasonable" scores should fall roughly in the range 0 to -100.

 2.  "SCORES SHOULD BE SCALED LOGARITHMICALLY," with every 10 point drop
 in score representing roughly a 50% reduction in
 confidence/probability/proximity etc.  Thus for example a translation
 with a score of -50 is 1/8 as "good" (or 1/8 as "likely" or 8 times as
 "far" from being perfect) as a translation with a score of -20.

 3.  The score for each translation should reflect the confidence in
 that translation only.  It should NOT be influenced by the confidence in
 any other translation.  In particular, a high score for one translation
 does not preclude a high score for another translation.  For example
 'o' and 'O' may both score high (even perfect).  In this way, scores
 need not behave like probabilities: they do not represent slices from
 a fixed pie.

 4.  Similarly, there is no requirement that the scores "add up" to
 a fixed total.  For a particular sample, all of the scores may be
 poor, or the recognizer may even send back no translations.  The
 context engine is depending on this fact in order to be able to use
 the shape engine to help it choose the correct character segmentation.

 5.  Scores should not be "tainted" by knowledge of character frequency 
 in English or any other linguistic considerations.  It is the job of the
 context level processing to take linguistic information into account.
 The shape engine must consider all characters as a priori equally likely,
 otherwise the bias for common characters in text will be duplicated at
 both levels, resulting in unwanted effects.

*/
#define msgXShapeRecognize					MakeMsg(clsXShape, 5)

typedef struct XSHAPE_RECOGNIZE {
	P_XS_STROKE pFirstStroke;// IN: linked list of (at most xsMaxPath) strokes
	XS_MATCH_TYPE matchType;	 // IN: type of record in output array (matchAscii
									// for XS_ASCII_MATCH, xsMatchGesture for XS_GESTURE_MATCH)
	U16 matchArraySize;		 // IN: number of records in output array (at
									//     most xsMaxCharList)
	P_UNKNOWN pMatchResults;// IN: ptr to output array
} XSHAPE_RECOGNIZE, *P_XSHAPE_RECOGNIZE;


/****************************************************************************
 msgXShapeShapeCompatible: takes P_XSHAPE_COMPATIBLE, returns STATUS
 Checks the possibility of translating the strokes as the char
  
 Sees if there is anything about the strokes that absolutely rules out
 the letter as a translation.
 For example, some shape matchers may rule out certain translations based
 on the number of strokes in the list.

 This message is sent by the context level only when it has been instructed
 to allow the dictionary (spelling) or a template to propose characters when
 the shape level is stuck.  The context level makes this check just be sure
 that there is some remote possibility that the strokes do represent the
 proposed character before allowing the dictionary or template to propose
 it.
*/
#define msgXShapeShapeCompatible					MakeMsg(clsXShape, 6)

typedef struct XSHAPE_COMPATIBLE {
	P_XS_STROKE pFirstStroke;// IN: linked list of strokes
	U8 character;            // IN: desired translation for the strokes
	U8 strokeCount;          // IN: how many strokes in the linked list
	BOOLEAN compatible;      // OUT: is translation a priori possible
} XSHAPE_COMPATIBLE, *P_XSHAPE_COMPATIBLE;



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Training Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgXShapeShapeEvaluate: takes P_XTEACH_DATA, returns STATUS
 Checks how well the shape matcher translates the character.

 Reports back how well the current engine translates the strokes,
 knowing what the correct translation is.  Does NOT cause the engine
 to learn the new shape if it is translated poorly.
*/
#define msgXShapeShapeEvaluate				MakeMsg(clsXShape, 7)


/****************************************************************************
 msgXShapeShapeLearn: takes P_XTEACH_DATA, returns STATUS
 Forces shape matcher to learn new shape.

 Usually invoked based on the results from msgXShapeShapeEvaluate.
*/
#define msgXShapeShapeLearn					MakeMsg(clsXShape, 8)

#endif // XSHAPE_INCLUDE
