/****************************************************************************
 File: win.h

 (C) Copyright 1992 by Go Corporation, All Rights Reserved.

 $Revision:   1.27  $ 
   $Author:   cmeyer  $
     $Date:   13 Mar 1992 18:54:22  $

 This file provides the API's for clsWin, clsWinDev. Two abstract classes,
 clsDrwCtx and clsPixDev are also defined, but they are not used directly
 by application-level clients.

 clsDrwCtx inherits from clsObject.
 Defines the minimal behavior for a drawing context.

 clsPixDev inherits from clsObject.
 Defines the minimal behavior for a pixelmap graphics device.

 clsWinDev inherits from clsPixDev.
 Provides devices of clsPixDev that can have windows on them.

 clsImgDev inherits from clsWinDev.
 Provides window devices whose pixels are accessible in memory.

 clsWin inherits from clsObject.
 Provides windows onto clsWinDev objects.

 theScreen is a well-known instance of clsWinDev. It is the main
 display surface for PenPoint. 
 
 theRootWindow is a well-known instance of clsWin. It is the root of
 the window tree on theScreen.

 Terminology:

 DU4 -- Device Units, 4th Quadrant. A 4th quadrant coordinate system;
 device space, device units. This is used internally, but not seen by 
 application software.
                                                         
 LWC -- Logical Window Coordinates. A 1st quadrant coordinate system.
 The lower-left-hand corner of the window is 0,0. The units are device 
 pixels. These are the coordinates in which windowing operations are 
 specified and input is delivered.

 LUC -- Logical Unit Coordinates. The nature of the coordinate system 
 is determined by a drawing context. Such coordinates are always relative 
 to the window. Some drawing contexts will implement window messages that 
 takes LWC coordinates and transform them so that window operations can 
 occur in LUC space. See sysgraf.h for details.

****************************************************************************/

/**** Debugging Flags ****/
/*
 The clsWin debugging flag is 'W'.  Defined values are:

    flag0  (0x0001):    window layout
    flag1  (0x0002):    window layout
    flag2  (0x0004):    flash interesting regions during damage
    flag3  (0x0008):    bitmap caching
    flag4  (0x0010):    window filing
    flag5  (0x0020):    font cache char ops
    flag6  (0x0040):    font cache char ops
    flag7  (0x0080):    matrix/rectangle math
    flag8  (0x0100):    layout timing
    flag9  (0x0200):    font cache macro ops
    flag10 (0x0400):    msgWinDumpTree outputs input flags
    flag11 (0x0800):    Measure/Draw text
    flag12 (0x1000):    window printing/clipping
    flag13 (0x2000):    unused
    flag14 (0x4000):    unused
    flag15 (0x8000):    window bitblt coordinates
*/

#ifndef WIN_INCLUDED
#define WIN_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef GEO_INCLUDED
#include <geo.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Typedefs, #defines, and Status Values                                  *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define  stsWinConstraint        MakeStatus(clsWin,    1)
#define  stsWinHasParent         MakeStatus(clsWin,    3)
#define  stsWinParentBad         MakeStatus(clsWin,    4)
#define  stsWinBad               MakeStatus(clsWin,    5)
#define  stsWinInfiniteLayout    MakeStatus(clsWin,    6)
#define  stsWinNoEnv             MakeStatus(clsWin,    7)

#define  stsWinIsChild           MakeWarning(clsWin,    8)
#define  stsWinIsDescendant      MakeWarning(clsWin,    9)

#define  stsPixDevBad            MakeStatus(clsPixDev, 1)
#define  stsPixDevOutOfRegions   MakeStatus(clsPixDev, 2)

#define  stsWinDevBad            MakeStatus(clsWinDev, 1)
#define  stsWinDevFull           MakeStatus(clsWinDev, 2)
#define  stsWinDevCachedHit      MakeStatus(clsWinDev, 3)

typedef  OBJECT  WIN,        // an object of clsWin
                 DRW_CTX,    // an object of clsDrwCtx (abstract class)
                 PIX_DEV,    // an object of clsPixDev (abstract class)
                 WIN_DEV,    // an object of clsWinDev
                 IMG_DEV;    // an object of clsImgDev

typedef WIN      * P_WIN;
typedef DRW_CTX  * P_DRW_CTX;
typedef PIX_DEV  * P_PIX_DEV;
typedef WIN_DEV  * P_WIN_DEV;
typedef IMG_DEV  * P_IMG_DEV;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Window style flags                                                     *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define  wsClipChildren      ((U32)flag0)  // Don't draw on my children
#define  wsClipSiblings      ((U32)flag1)  // Don't draw on my siblings
#define  wsParentClip        ((U32)flag2)  // Borrow my parent's vis rgn
#define  wsSaveUnder         ((U32)flag3)  // Try to save pixels on insert
#define  wsGrowTop           ((U32)flag4)  // Pixels move to bottom on resize
#define  wsGrowBottom        ((U32)flag5)  // Pixels move to top    on resize
#define  wsGrowLeft          ((U32)flag6)  // Pixels move to right  on resize
#define  wsGrowRight         ((U32)flag7)  // Pixels move to left   on resize
#define  wsCaptureGeometry   ((U32)flag8)  // I capture m,s,i,e of children
#define  wsSendGeometry      ((U32)flag9)  // Send me delta,ins,ext advice
#define  wsSendOrphaned      ((U32)flag10) // Send msgOrphaned not msgFree

#define  wsSynchRepaint      ((U32)flag12) // ObjectCall to repaint
#define  wsTransparent       ((U32)flag13) // I am transparent
#define  wsVisible           ((U32)flag14) // I am visible
#define  wsPaintable         ((U32)flag15) // I can be painted
#define  wsSendFile          ((U32)flag16) // I should be filed

#define  wsShrinkWrapWidth   ((U32)flag17) // I shrink to fit children
#define  wsShrinkWrapHeight  ((U32)flag18) // I shrink to fit children
#define  wsLayoutDirty       ((U32)flag19) // My layout is dirty
#define  wsCaptureLayout     ((U32)flag20) // I'm dirty if children delta,
                                           // extract, insert, or
										   // child wsVisible changes
#define  wsSendLayout        ((U32)flag21) // I'm dirty if I change size or
                                           // wsShrinkWrapWidth/Height or
										   // wsMaskWrapWidth/Height changes

#define	 wsHeightFromWidth   ((U32)flag22) // height is computed from width
#define	 wsWidthFromHeight   ((U32)flag24) // width is computed from height

#define	 wsFileInline        ((U32)flag23) // file without object header


#define	 wsFileNoBounds      ((U32)flag26) // don't file the bounds
#define	 wsFileLayoutDirty   ((U32)flag27) // always dirty layout bit on restore
#define  wsMaskWrapWidth     ((U32)flag28) // mask out wsShrinkWrapWidth
#define  wsMaskWrapHeight    ((U32)flag29) // mask out wsShrinkWrapHeight

#define  wsDefault          (wsClipChildren  |    \
                             wsClipSiblings  |    \
                             wsPaintable     |    \
                             wsCaptureLayout |    \
                             wsSendLayout    |    \
                             wsLayoutDirty   |    \
                             wsVisible            \
                            )

typedef struct
{
  U32    input,                      // see input.h
         style;                      // see ws* flags above

} WIN_FLAGS, * P_WIN_FLAGS;       // part of WIN_METRICS

#define WinShrinkWrapWidth(style)     \
    (!((style) & wsMaskWrapWidth)  && ((style) & wsShrinkWrapWidth))

#define WinShrinkWrapHeight(style)    \
    (!((style) & wsMaskWrapHeight) && ((style) & wsShrinkWrapHeight))

#define WinShrinkWrap(style)          \
    (WinShrinkWrapWidth(style) || WinShrinkWrapHeight(style))

/*
 You can use these WinShrinkWrap macros are test if a window has
 shrink-wrap-width or shirnk-wrap-height enabled.  If wsMaskWrapWidth/Height
 is on, the shrink wrapping will be off in that dimension.  clsGrabBox will
 turn on wsMaskWrapWidth/Height if the user resizes a window and changes the
 width/height.  clsFrame will clear the wsMaskWrapWidth/Height bits and
 re-layout when the user tripple-taps on the title bar.
*/

Enum16(WIN_OPTIONS)
{ wsPosTop         = 0,
  wsPosBottom      = flag0,         // In:  to msgWinInsert...

  wsPosInFront     = wsPosTop,
  wsPosInBack      = wsPosBottom, 
                
  wsWinMoved       = flag9,         // Out: from msgWinDelta
  wsWinSized       = flag10,        // Out: from msgWinDelta
  wsParentMoved    = flag12,        // Out: from msgWinDelta
  wsParentSized    = flag13,        // Out: from msgWinDelta

  wsLayoutResize   = flag11,        // In:  to msgWinLayout...
  wsLayoutMinPaint = flag14,        // In:  to msgWinLayout...
  wsLayoutNoCache  = flag8,         // Out: from msgWinLayoutSelf
  wsLayoutDefault  = wsLayoutResize // In:  to msgWinLayout...
};

typedef struct
{
  WIN          parent,
               child;
  RECT32       bounds;
  WIN_DEV      device;
  WIN_FLAGS    flags;
  TAG          tag;
  WIN_OPTIONS  options;

} WIN_METRICS, * P_WIN_METRICS;
/*
 A P_WIN_METRICS is the argument to most of the messages defined by
 clsWin. However, for most of these messages, not all of the fields
 are used. In the discussion of each message below, fields which are
 not mentioned are not used; and they don't have to be initialized
 before sending the message. This is not to say that these "unused"
 fields are not modified during the call; they will be during the
 processing of some messages.
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Messages Sent to a Window                                              *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew takes P_WIN_METRICS, returns STATUS
    category: class message
    Creates a window.

 If pArgs->parent is not objNull, clsWin will create the window on the
 spceified parent's window device.  Note that the new window will not
 be inserted as a child of the specified parent.  You must send
 msgWinInsert to the new window after creating it to insert it into its
 parent.

 If pArgs->parent is objNull, the window will be created on pArgs->device.
 If pArgs->device is objNull, clsWin will create the window on OSThisWinDev().

 Returns
 	stsWinParentBad		if pArgs->parent is not objNull or a valid window
 	stsWinDevBad		if pArgs->device is not objNull or a valid window device
	stsWinDevFull		if the window device window array can't be grown

 See Also
 	msgWinInsert
*/
typedef	WIN_METRICS	WIN_NEW_ONLY, * P_WIN_NEW_ONLY;

#define	winNewFields  \
	objectNewFields  \
	WIN_NEW_ONLY	win;

typedef struct WIN_NEW
{
  winNewFields

} WIN_NEW, *P_WIN_NEW;

/****************************************************************************
 msgNewDefaults takes P_WIN_NEW, returns STATUS
    category: class message
    Initializes the WIN_NEW structure to default values.
//{
    object.cap         |= objCapCall;
    win.parent          = objNull;
    win.child           = objNull;
    win.device          = objNull;
    win.flags.style     = wsDefault;
    win.flags.input     = 0;
    win.tag             = 0;
    win.options         = wsPosTop;
    win.bounds.origin.x = 0;
    win.bounds.origin.y = 0;
    win.bounds.size.w   = 0;
    win.bounds.size.h   = 0;
//}

*/

/****************************************************************************
 msgWinInsert takes P_WIN_METRICS, returns STATUS
    Inserts or changes z-order of a window.

 You send this message to the child that you want to insert or change
 z-order.

 In parameters are:

//{
	pArgs->parent	= child's new parent or objNull;
 	pArgs->options	= either wsPosTop or wsPosBottom;
//}

 If pArgs->parent is not objNull or self's current parent, clsWin will insert
 self as a child of the specified parent.  If pArgs->options has wsPosTop on,
 self will be inserted as the top-most child; if wsPosBottom is on, self
 will be inserted as the bottom-most child.

 If pArgs->parent is objNull or self's current parent, clsWin will change the
 z-order of self according to pArgs->options.  If pArgs->options has wsPosTop
 on, self will be altered in z-space to be the top-most child; if wsPosBottom
 is on, self will be altered to be the bottom-most child.  If the z-order of
 self is changed, wsWinMoved will be or-ed into pArgs->options as an out
 parameter.

 If the receiver's parent has wsCaptureLayout on, wsLayoutDirty will be set
 on the receiver's parent.

 Returns
 	stsWinParentBad		if pArgs->parent is not objNull or a valid window
						on the same window device as self
	stsWinHasParent		if self already has a parent and pArgs->parent is
						not either objNull or self's current parent

 See Also
 	msgWinInsertSibling
	msgWinExtract
*/
#define msgWinInsert                      MakeMsg(clsWin,    1)  

/****************************************************************************
 msgWinInsertSibling takes P_WIN_METRICS, returns STATUS
    Inserts or changes z-order of a window (relative to a sibling).

 You send this message to the child that you want to insert or change
 z-order.  This message is similar to msgWinInsert, except pArgs->parent
 should be the intended sibling of the receiver.

 In parameters are:

//{
	pArgs->parent	= receiver's new sibling
 	pArgs->options	= either wsPosTop or wsPosBottom;
//}

 clsWin will insert self as a sibling of the specified sibling.
 If pArgs->options has wsPosTop on, self will be inserted as in front of
 pArgs->parent; if wsPosBottom is on, self will be inserted behind
 pArgs->parent.

 If pArgs->parent is already self's sibling, clsWin will change the
 z-order of self according to pArgs->options.  If pArgs->options has wsPosTop
 on, self will be altered in z-space to be in front of pArgs->parent; if
 wsPosBottom is on, self will be altered to be behind pArgs->parent.
 If the z-order of self is changed, wsWinMoved will be or-ed into
 pArgs->options as an out parameter.

 If the receiver's parent has wsCaptureLayout on, wsLayoutDirty will be set
 on the receiver's parent.

 Returns
 	stsWinParentBad		if pArgs->parent is not a valid window
						on the same window device as self
	stsWinHasParent		if self already has a parent and pArgs->parent is
						not a sibling of self

 See Also
 	msgWinInsert
	msgWinExtract
*/
#define msgWinInsertSibling               MakeMsg(clsWin,    2)  

/****************************************************************************
 msgWinExtract takes P_WIN_METRICS or pNull, returns STATUS
    Extracts a window from its parent. 
    
 If a P_WIN_METRICS is passed instead of pNull the same information
 returned by msgWinGetMetrics is returned in the WIN_METRICS structure.
 This will include the parent field BEFORE the extract is performed.
 If the window is already extracted, stsWinParentBad is returned,
 and any passed WIN_METRICS field is unmodified.

 If the receiver's parent has wsCaptureLayout on, wsLayoutDirty will be set
 on the receiver's parent.
*/
#define msgWinExtract                     MakeMsg(clsWin,    3)  

/****************************************************************************
 msgWinDelta takes P_WIN_METRICS, returns STATUS
    Moves and/or resizes a window. pArgs->bounds should be the newly
    desired bounds (size AND position).

 If the receiver is involved in a layout episode (msgWinLayout is being
 processed in the receiver's window tree), the new bounds will be
 remembered for use at the end of the layout episode.  If the new bounds
 has a new width or height, and a cached desired size is being remembered for
 the receiver, the desired size will be discarded if either of the following
 is true:

 	-: the new bounds has a new width and the receiver has wsHeightFromWidth
 	   on or does not have wsShrinkWrapWidth on

	-: the new bounds has a new height and the receiver has wsWidthFromHeight
	   on or does not have wsShrinkWrapHeight

 If the receiver is involved in a layout episode this is all that is done
 and stsOK is returned.

 If the receiver's parent has wsCaptureGeometry on, the parent will be sent
 msgWinDeltaOK.  If the parent responds with anything other than stsOK,
 that status will be returned and nothing else is done.  Otherwise,
 the (possibly modified) bounds returned by the parent will be used.
 If the parent modified the proposed child origin, wsParentMoved will
 be or-ed into pArgs->options as an out parameter. If the parent modified
 the proposed child size, wsParentSized will be or-ed into pArgs->options
 as an out parameter.

 If the receiver is visible and paintable (wsVisible and wsPaintable are
 on for the receiver and all of its ancestors), valid portions of the
 receiver's window may be copied to their new location to avoid damage and
 repaint of those portions.

 If the receiver has any of the grow bits on (wsGrowBottom/Top/Left/Right),
 the appropriate grow semantics will be applied to determine how to move
 the receiver's children and what portions of the receiver's window to
 damage for subsequent repaint.

 If pArgs->bounds is a new bounds and the receiver's parent has 
 wsCaptureLayout on, wsLayoutDirty will be set on the receiver's parent.

 If pArgs->bounds.size is a new size and the receiver has wsSendLayout on,
 wsLayoutDirty will be set on the receiver.

 Subclasses that want to know when their position or size has changed should
 not expect that msgWinDelta is the only way for this to happen.  If you need
 to know this information, you should turn on wsSendGeometry and catch
 msgWinMoved or msgWinSized.  clsWin may change a window's bounds without
 sending msgWinDelta to the window.
*/
#define msgWinDelta                       MakeMsg(clsWin,    4)  

/****************************************************************************
 msgWinLayout takes P_WIN_METRICS, returns STATUS
    Tells a window sub-tree to layout.

 You should send msgWinLayout to a window after you have altered the window
 in such a way that its bounds or its descendants bounds must be recomputed.

 For example, if you create an instance of clsTableLayout (a subclass that
 lays out its children in rows and columns) and insert children into it, you
 must send msgWinLayout to the table layout window to force it to "layout"
 itself and its children.
 
 After msgWinLayout has been sent, every window in the receiver's tree will
 be positioned and sized as required.  You can then use msgWinInsert to
 insert the root of the tree on the display and allow the windows to paint.

 In parameters:

//{
	bounds		= new final bounds for receiver if wsLayoutResize is not
				  on in pArgs->options

	options		= wsLayoutDefault, 0, or any combination of wsLayoutResize,
				  wsLayoutMinPaint
//}

 Subclasses must not catch msgWinLayout.  clsWin will respond by beginning a
 "layout episode" during which the windows in the receiver's tree will be
 layed out.

 The algorithm for a layout episode is as follows:
//{
 	for the receiver and each of its descendants
		If the window has wsLayoutDirty on
			If the bounds of the window have been fixed by a previous 
			msgWinDelta during the layout episode
				send the window	msgWinLayoutSelf with the following
				WIN_METRICS parameters:
					bounds.size	= current bounds.size;
					options		= 0;

			Otherwise,
				send the window msgWinLayoutSelf with the following
				WIN_METRICS parameters:
					options	= wsLayoutResize;

				copy back WIN_METRICS.bounds.size as the new size for the
				window.

			If the window's parent has wsLayoutDirty on, switch to the
			window's parent and continue down the tree from there.

	After the entire tree has been traversed, traverse the tree again and
	process wsCaptureGeometry and wsSendGeometry requests as follows:
	
	For each window
		If the origin or size has changed
			If the window's parent has wsCaptureGeometry on
				send msgWinDeltaOK to the window's parent;
				
			If the window has wsSendGeometry on
				send msgWinMoved and/or msgWinSized to the window.

	After the geometry notifications have been done, apply all of the new
	bounds for each window in the tree as in msgWinDelta.
//}

 If wsLayoutResize is NOT set in pArgs->options, then you must set
 pArgs->bounds to the new rectangle that the receiver must fit into -- 
 it will lay out accordingly; otherwise the receiver will lay out to its
 desired size.
 
 If wsLayoutMinPaint is not on, window damage will not be computed during the
 layout episode -- all of the windows in the window tree will be damaged and
 repaint after the layout episode.  This will result in faster layout, at the
 expense of some (possibly) unneccessary repaints.  If wsLayoutMinPaint is on,
 the true damaged area will be computed.  This may take longer, but will
 result in the minimal amount of repaint after the layout episode. 

 In general this message should not be handled by subclasses. However,
 it results in the sending of msgWinLayoutSelf, which does need to be
 handled by subclasses.

 Returns
 	stsWinInfiniteLayout	the layout episode does not appear to terminate

 See Also
 	msgWinLayoutSelf
*/
#define msgWinLayout                      MakeMsg(clsWin,   41)

/****************************************************************************
 msgWinLayoutSelf takes P_WIN_METRICS, returns STATUS
    Tells a window to layout its children (sent during layout).

 This message is sent by clsWin during a layout episode. It can be handled by
 knowledgeable window classes.
 
 When sent, pArgs->bounds.size contains the present size. If pArgs->options 
 is 0 then the window cannot change pArgs->bounds.size, it must lay out 
 its children, as best it can, within those bounds. If pArgs->option is
 wsLayoutResize then it may change pArgs->bounds.size to its desired
 size.

 After pArgs->bounds.size is determined, the window should msgWinDelta
 each child to its final position and size.

 In order to determine its desired size and layout, a window may need
 to send msgWinGetDesiredSize to some, or all, of its children first.

 clsWin responds to msgWinLayoutSelf by doing nothing and returning 
 the current window size in pArgs->bounds.size if wsLayoutResize is on in
 pArgs->options.

 See Also
 	msgWinLayout
	msgWinGetDesiredSize
	msgWinDelta
*/
#define msgWinLayoutSelf                  MakeMsg(clsWin,   42)

/****************************************************************************
 msgWinGetDesiredSize takes P_WIN_METRICS, returns STATUS
    Gets the desired size of a window (sent during layout).
 
 This message should not be handled by a subclass.

 If the receiver is not in a layout episode, clsWin responds by returning
 the receiver's current bounds.

 Otherwise, if the desired size has already been computed (cached) for the
 receiver, that value will be returned.

 Otherwise, msgWinLayoutSelf will be self-sent with the following
 WIN_METRICS parameters:
//{
	options = wsLayoutResize;
//}

 Subclasses should catch msgWinLayoutSelf, layout to their desired size
 and return the desired size in WIN_METRICS.bounds.size.  The computed
 desired size will be remembered in the window's cache for future use and
 will be passed back in pArgs->bounds.size.

 See Also
 	msgWinLayout
	msgWinLayoutSelf
*/
#define msgWinGetDesiredSize              MakeMsg(clsWin,   43)

/****************************************************************************
 msgWinGetBaseline takes P_WIN_METRICS, returns STATUS
    Gets the desired x,y alignment of a window.

 Subclasses can set pArgs->bounds.origin to reflect the window's desired
 baseline position. clsWin will set both x and y to 0,0.

 pArgs->bounds.size should contain the size of the window. This is useful
 for windows whose alignment is a function of window size (like centered).

 If the receiver does not have either an x or y baseline, wsNoXBaseline
 and/or wsNoYBaseline can be or-ed into pArgs->options as an out parameter.

 clsWin will always set pArgs->options to wsNoXBaseline | wsNoYBaseline (i.e.
 the default is the window has no x or y baseline).
*/
#define msgWinGetBaseline                 MakeMsg(clsWin,   46)

#define	wsNoXBaseline	((U16)flag0)
#define	wsNoYBaseline	((U16)flag1)

/****************************************************************************
 msgWinSetLayoutDirty takes BOOLEAN, returns BOOLEAN
    Turns wsLayoutDirty bit on or off, returns previous value.

 If the window has a cached desired size, and wsLayoutDirty comes on,
 the desired size will be discarded.
*/
#define msgWinSetLayoutDirty              MakeMsg(clsWin,   44)

/****************************************************************************
 msgWinSetLayoutDirtyRecursive takes BOOLEAN, returns nothing
    Turns wsLayoutDirty bit on for every window in subtree.

*/
#define msgWinSetLayoutDirtyRecursive     MakeMsg(clsWin,   45)

/****************************************************************************
 msgWinSend takes P_WIN_SEND, returns STATUS
    Sends a message up a window ancestry chain.
    
 The receiver may reply to the message or forward the message up 
 the window parent chain. clsWin will forward the message to the parent 
 using ObjectSendUpdate. If the message reaches the root window 
 stsMessageIgnored is returned. If the wsSendIntraProcess flag is true
 the message will not be propagated past a process transition (based
 on the owner of the window object); in this case stsMessageIgnored
 may also be returned.

 lenSend must be at least SizeOf(WIN_SEND) but may be larger to move
 more data to a window owned by another process. A single unit of data,
 data[0] is defined in WIN_SEND as a convenience. The message a window
 receives when msgWinSend is forwarded is msgWinSend, NOT msg. The
 field msg is provided so the receiving client can properly interpret
 the purpose of the msgWinSend.
*/
#define msgWinSend                        MakeMsg(clsWin,   36)   

Enum16(WIN_SEND_FLAGS)
{ wsSendDefault      = 0,
  wsSendIntraProcess = flag0,    // stop at process transition
};

typedef struct
{
  U32             lenSend;       // length of message, 
                                 // SizeOf(WIN_SEND) minimum
  WIN_SEND_FLAGS  flags;         //
  MESSAGE         msg;           // the "message"
  P_UNKNOWN       data[1];       // an argument to the message

  // clients can put 
  // more data here 
  // if needed

} WIN_SEND, * P_WIN_SEND;

/****************************************************************************
 msgWinGetMetrics takes P_WIN_METRICS, returns STATUS
    Gets full window metrics.

 pArgs->parent passes back the receiver's parent
 pArgs->child passes back self
 pArgs->bounds passes back size and parent relative position
 pArgs->device passes back self's device
 pArgs->flags passes back self's window and input flags
 pArgs->tag passes back self's tag
*/
#define msgWinGetMetrics                  MakeMsg(clsWin,    5)  

/****************************************************************************
 msgWinGetFlags takes P_WIN_METRICS, returns STATUS
    Like msgWinGetMetrics but passes back flags only.

 pArgs->flags passes back self's window and input flags.
*/
#define msgWinGetFlags                    MakeMsg(clsWin,    6)  

/****************************************************************************
 msgWinSetFlags takes P_WIN_METRICS, returns STATUS
    Sets the window flags.

 pArgs->flags should be set to the new window and input flags.

 If wsVisible is changed and the receiver's parent has wsCaptureLayout on,
 wsLayoutDirty will be set on the receiver's parent.

 If the new flags result in a new value for WinShrinkWrap() (e.g.
 wsShrinkWrapWidth changes) and the receiver has wsSendLayout on,
 wsLayoutDirty will be set on the receiver.
*/
#define msgWinSetFlags                    MakeMsg(clsWin,    7)  

/****************************************************************************
 msgWinGetTag takes P_WIN_METRICS, returns stsOK
    Like msgWinGetMetrics but passes back tag only.

 pArgs->tag passes back self's tag.
*/
#define msgWinGetTag                      MakeMsg(clsWin,   37)  

/****************************************************************************
 msgWinSetTag takes P_WIN_METRICS, returns STATUS
    Sets the window tag.

 pArgs->tag should be set to the new window tag.
*/
#define msgWinSetTag                      MakeMsg(clsWin,   38)  

/****************************************************************************
 msgWinIsVisible takes nothing, returns STATUS
    Returns stsOK if the receiver and all its ancestors have wsVisible on.

 clsWin will traverse the parent chain of the receiver until the
 parent is objNull or the root window of the receiver's device.  If the
 receiver or any of its ancestors have wsVisible off in their window flags,
 stsFailed is returned.  Otherwise, if the final ancestor is the root window
 on the receiver's device, stsOK is returned.
*/
#define msgWinIsVisible                   MakeMsg(clsWin,   40)  

/****************************************************************************
 msgWinIsDescendant takes P_WIN_METRICS, returns STATUS
    Checks if pArgs->child is a descendant of the receiver.

    In parameters:
		child:		child to look for
		options:	0 for direct children, wsEnumRecursive for recursive
		            or-in wsEnumSelf to include self in the search

 clsWin will check the receiver's children and return stsWinIsChild
 if pArgs->child is one of them.

 If pArgs->options has wsEnumRecursive on, the search will continue down
 the window tree until pArgs->child is found or all of the receiver's
 descendants have been examined.  If no match is found, stsNoMatch is
 returned.

 If pArgs->child is self and wsEnumSelf is on in pArgs->options, 
 stsWinIsChild is returned; otherwise stsNoMatch is returned.

 Returns Value
	stsWinIsChild:		if pArgs->child is self or a direct child
	stsWinIsDescendant:	if pArgs->child is a descendant
	stsNoMatch:			if pArgs->child is not a descendant
*/
#define msgWinIsDescendant                   MakeMsg(clsWin,   59)  

/****************************************************************************
 msgWinGetPopup takes P_WIN_METRICS, returns stsOK
    Gets the popup window.

 pArgs->child passes back self's popup window.

 The popup window is traversed during msgWinFindTag only. See
 msgWinFindTag for more details.
*/
#define msgWinGetPopup                    MakeMsg(clsWin,   53)  

/****************************************************************************
 msgWinSetPopup takes P_WIN_METRICS, returns STATUS
    Sets a popup window.

 pArgs->child should be set to the popup window.

 The popup window is traversed during msgWinFindTag only. See
 msgWinFindTag for more details.

 One example of popup window use is in clsMenuButton.  A menu button will
 set its popup window to be its menu.  This allows you to use msgWinFindTag
 on a menu bar and find a menu button in one of the popup menus.
*/
#define msgWinSetPopup                    MakeMsg(clsWin,   54)  

/****************************************************************************
 msgWinFindAncestorTag takes U32, returns OBJECT 
    Searches for a match on argument tag. Returns match or objNull.

 The search is up the ancestor chain; the first match found is returned. 
 If no match is found, objNull is returned.
*/
#define msgWinFindAncestorTag             MakeMsg(clsWin,   49)
	
/****************************************************************************
 msgWinFindTag takes U32, returns OBJECT 
    Searches for a match on argument tag. Returns match or objNull.

 The search is breadth first; but, it starts with the first child of
 the window, not the window itself.  The first match found is returned. 
 If no match is found, objNull is returned. Trees rooted at popup
 windows (set with msgWinSetPopup) are traversed too. The traversal
 order is siblings first, then children, then popups.
*/
#define msgWinFindTag                     MakeMsg(clsWin,   39)

/****************************************************************************
 msgWinSetVisible takes BOOLEAN, returns BOOLEAN
    Turns window visibility bit on or off, returns previous value.

 If visibility is changed and the receiver's parent has wsCaptureLayout on,
 wsLayoutDirty will be set on the receiver's parent.
*/
#define msgWinSetVisible                  MakeMsg(clsWin,    8)  

/****************************************************************************
 msgWinSetPaintable takes BOOLEAN, returns BOOLEAN
    Turns window paintability bit on or off, returns previous value.
*/
#define msgWinSetPaintable                MakeMsg(clsWin,    9)

/****************************************************************************
 msgWinBeginRepaint takes P_RECT32 or pNull, returns STATUS
    Sets up window for painting on "dirty" region. 
    
 A BeginRepaint/EndRepaint pair bracket an update episode for a
 window. They should be sent ONLY in response to the receipt of 
 msgWinRepaint. If pArgs is not pNull a rectangle describing the 
 bounds of the dirty region is passed back.
*/
#define msgWinBeginRepaint                MakeMsg(clsWin,   10)  

/****************************************************************************
 msgWinEndRepaint	takes nothing, returns STATUS
    Tells window system that repainting has ended for this window.
*/
#define msgWinEndRepaint                  MakeMsg(clsWin,   11)  

/****************************************************************************
 msgWinBeginPaint takes P_RECT32 or pNull, returns STATUS
    Sets up window for painting on its visible region.

 A BeginPaint/EndPaint pair can be used to paint on a window at
 anytime, even if it is not dirty. If pArgs is not pNull a
 rectangle describing the bounds of the visible region is passed back.
*/
#define msgWinBeginPaint                  MakeMsg(clsWin,   12)  

/****************************************************************************
 msgWinEndPaint	takes nothing, returns STATUS
    Tells window system that painting has ended for this window.
*/
#define msgWinEndPaint                    MakeMsg(clsWin,   13)  

/****************************************************************************
 msgWinDirtyRect takes P_RECT32 or pNull, returns STATUS
    Marks all or part of a window dirty.

 If pNull is passed the entire window is marked dirty. If the dirty
 part is visible, the window will eventually receive msgWinRepaint
 as a side effect of this message.
*/
#define msgWinDirtyRect                   MakeMsg(clsWin,   14)  

/****************************************************************************
 msgWinUpdate takes nothing, returns STATUS
    Forces a window to repaint now, provided that it needs repainting.
   
 The window and all its descendants that need painting are sent
 msgWinRepaint. However, only windows owned by the current subtask 
 are processed.
*/
#define msgWinUpdate                      MakeMsg(clsWin,   35)  

/****************************************************************************
 msgWinCleanRect takes P_RECT32 or pNull, returns STATUS
    Marks all or part of a window clean.

 If pNull is passed the entire window is marked clean. In general
 it is not a good idea to mark a window clean. Window activity is
 asynchronous and application software has no way of knowing if 
 the window is really clean.
*/
#define msgWinCleanRect                   MakeMsg(clsWin,   15)  

/****************************************************************************
 msgWinCopyRect takes P_WIN_COPY_RECT, returns STATUS
    Copies pixels within a window.

 In general, pixels which are dirty, invisble, or just off the edge of
 the window, are not copied. Rather, at the destination it is recognized
 that they did not get copied, and they are marked dirty instead. Also,
 it is assumed that pixels at the source need to be repainted. (This
 behavior is controlled by the two flags wsSrcNotDirty and wsDstNotDirty).

 The intent of this message is that it be used as an accelerator; to move
 potentially good pixels to a new location. It should be sent OUTSIDE of
 an update episode. Then, areas that require repainting will be 
 marked dirty and handled by the next update episode.

 If, by mistake, this message is sent inside an update episode it will
 probably not copy any pixels, because it will assume that all the
 pixels that are currently being updated are dirty.

 The use of wsCopyNormal is recommended to copy the normal planes
 and skip the pen plane(s). More precise control over which planes are 
 copied is available with the use of flags wsCopyNormal, wsPlanePen 
 and wsPlaneMask (in conjunction with the planeMask field).

 If wsChildrenStay is not in in pArgs->flags and the receiver has children
 in the area being copied, the children will be moved also.  Note that
 even if the receiver has wsCaptureGeometry on, the receiver will not be
 sent msgWinDeltaOK when the children are moved.  However, each child that
 has wsSendGeometry on and is moved will be sent msgWinMoved.
*/
#define msgWinCopyRect                    MakeMsg(clsWin,   16)

Enum16(WIN_COPY_FLAGS)
{ wsCopyNormal   = 0,          // normal copy of normal planes
  wsPlanePen     = flag0,      // do pen plane(s) too
  wsPlaneMask    = flag1,      // use planeMask
  wsSrcNotDirty  = flag2,      // don't mark source dirty
  wsDstNotDirty  = flag3,      // don't mark dirty dst pixels dirty
  wsChildrenStay = flag4,
  wsCopyRelative = flag5,      // xy is a delta on srcRect.origin
};

typedef struct
{
  RECT32          srcRect;		 // rectangle    in LWC
  XY32            xy;            // new location in LWC
  WIN_COPY_FLAGS  flags;
  U16             planeMask;

} WIN_COPY_RECT, * P_WIN_COPY_RECT;

/****************************************************************************
 msgWinTransformBounds takes P_WIN_METRICS, returns STATUS
    Transforms bounds from receiver's to another window's LWC.

 Set the pArgs->parent to a window or use objNull for the 
 receiver's actual parent. pArgs->bounds in the receiver's LWC 
 are transformed into the equivalent bounds in the parent's LWC.
*/
#define msgWinTransformBounds             MakeMsg(clsWin,   18)  

/****************************************************************************
 msgWinEnum takes P_WIN_ENUM, returns STATUS
    Enumerate a window's children.

 Here is some sample code for enumerating the direct children of a window:
//{
	WIN_ENUM e;                                
	WIN      w[10];                            
	U16      i;                                
                                            
	e.max    = 10;				// e.pWin is an array of 10 WINs
	e.count  = maxU16;			// allocate as much storage as needed
	e.pWin   = w;				// put windows in w array
	e.flags  = wsEnumChildren;	// return only direct children
	e.next   = 0;				// start from the first child
                                            
	s = ObjectCall(msgWinEnum,parent,&e);      

	// stsEndOfData means we got them all
	if (s == stsEndOfData)                      
	  	s = stsOK;                                

	// e.count is the actual number of children
	for (i = 0; i < e.count; i++) {                                           
	  	child = e.pWin[i];

  		// put code that does something with
  		// child here
	}

	// free any allocated storage
	if (e.pWin != w)
		StsWarn(OSHeapBlockFree(e.pWin));
//}

 If you want to retrieve all of the window metrics for each window,
 turn on wsEnumMetrics in pArgs->flags and set pArgs->pWin to
 an array of WIN_METRICS structs.

 Returns
 	stsEndOfData	if all of the descendants have been returned

 See Also
 	WinEachChild
*/
#define msgWinEnum                        MakeMsg(clsWin,   33)  

Enum16(WIN_ENUM_FLAGS)
{ wsEnumChildren     = 0,      // enum children only
  wsEnumSelf         = flag0,  // enum self too
  wsEnumRecursive    = flag1,  // enum children of children...
  wsEnumFlags        = flag2,  // return flags too
  wsEnumBreadthFirst = flag3,  // 
  wsEnumSendFile     = flag4,  // enum only windows with
                               // wsSendFile == TRUE
  wsEnumMetrics      = flag5   // return WIN_METRICS
};

typedef struct
{
  U16              max,         // in = size of pWin[] and pFlags[] arrays
                                
                   count;       // in = # to return in arrays
                                // if count > max then memory may be allocated
                                // out = # of valid entries in arrays
                                
  P_WIN            pWin;         
  P_WIN_FLAGS      pFlags;      // in = ptr to arrays
                                // out = if memory was allocated
                                // client should free the memory
                                
  U16              next;        // in  = 0 to start at beginning 
                                //       OR previous out value to pick up
                                //       where we left off
  WIN_ENUM_FLAGS   flags;       // in  = see options

} WIN_ENUM, * P_WIN_ENUM;

/****************************************************************************
 WinEachChild returns nothing
    Helper macro for enumerating the direct children of a window

 You can use WinEachChild to retrieve the direct children of a window.

 See Also
 	WinEndEachChild
	msgWinEnum
*/
#define WinEachChild(parent, child, s)        \
{                                             \
  WIN_ENUM _e;                                \
  WIN      _w[10];                            \
  U16      _i;                                \
                                              \
  _e.max    = 10;                             \
  _e.count  = maxU16;                         \
  _e.pWin   = _w;                             \
  _e.flags  = wsEnumChildren;                 \
  _e.next   = 0;                              \
                                              \
  s = ObjectCall(msgWinEnum,parent,&_e);      \
                                              \
  if (s == stsEndOfData)                      \
    s = stsOK;                                \
                                              \
  for (_i = 0; _i < _e.count; _i++)           \
  {                                           \
    child = _e.pWin[_i];

    // put code that does something with
    // child here

/****************************************************************************
 WinEndEachChild returns nothing
    Ending helper macro for most common window enumeration idiom.
*/
#define WinEndEachChild                       \
  } /* for */                                 \
  if (_e.pWin != (P_WIN)_w)                   \
    OSHeapBlockFree(_e.pWin);                 \
} // end scope

/*
 WinEachChild and WinEndEachChild

 Use WinEachChild(parent,child,status) to start a for loop enumeration
 of the children of parent. The variable child will be set for each
 child. Close the enumeration with WinEndEachChild. Here is an example;
 notice that semicolons are NOT used.

    WinEachChild(p,c,s)
        // send a message to c
        // break if necessary
    WinEndEachChild
    // s is set here

 The code placed between these macros becomes the body of a for loop.
 If it is necessary to exit the loop early, use a break statement, not 
 a return or goto, so that WinEndEachChild is reached. If an error in 
 the enum occurs, the for loop will not be executed, and the status 
 value will be set.
*/

/****************************************************************************
 msgWinRepaint takes nothing, returns STATUS
    category: descendant responsibility
    Tells a window to repaint itself.

 Windows only receive this if the wsPaintable flag is true.
 This message is sent by the window system during an update episode. It
 should NOT be sent by the application. 

 If you want a window to be updated immediately (synchronously), 
 use msgWinUpdate.

 Upon receipt of this message, applications should NOT perform other 
 windowing operations that are visually significant (msgWinDelta, 
 msgWinInsert, msgWinExtract, etc.). When this message is received; 
 it is too late. The only thing that should happen is repainting.

 See Also
    msgWinBeginRepaint
    msgWinEndRepaint
*/
#define msgWinRepaint                     MakeMsg(clsWin,   21)

/****************************************************************************
 msgWinOrphaned takes nothing, returns STATUS
    category: advisory message
    Tells a window its parent has been freed.

 Windows only receive this if the wsSendOrphaned flag is true.
*/
#define msgWinOrphaned                    MakeMsg(clsWin,   22)

/****************************************************************************
 msgWinInsertOK takes P_WIN_METRICS, returns STATUS
    category: advisory message
    Informs a potential parent of a pending child insertion.

 pArgs->child is the window that is being inserted; pArgs->bounds is
 its bounds, which the parent can modify. If receiver does not return
 stsOK the insertion will be denied.

 Windows only receive this if wsCaptureGeometry in flags is true.
*/
#define msgWinInsertOK                    MakeMsg(clsWin,   23)

/****************************************************************************
 msgWinExtractOK takes P_WIN_METRICS, returns STATUS
    category: advisory message
    Informs parent of a pending child extraction.

 pArgs->child is the window that is being extracted. If receiver does 
 not return stsOK the extraction will be denied.

 Windows only receive this if wsCaptureGeometry in flags is true.
*/
#define msgWinExtractOK                   MakeMsg(clsWin,   24)

/****************************************************************************
 msgWinDeltaOK takes P_WIN_METRICS, returns STATUS
    category: advisory message
    Informs parent of a pending change in a child window's size or position.

 pArgs are the arguments to msgWinDelta. If receiver does not return 
 stsOK the delta will be denied.

 Windows only receive this if wsCaptureGeometry in flags is true.
*/
#define msgWinDeltaOK                     MakeMsg(clsWin,   25)

/****************************************************************************
 msgWinFreeOK takes WIN, returns STATUS
    category: advisory message
    Informs parent of the pending destruction of a child window.

 Windows only receive this if wsCaptureGeometry in flags is true.
*/
#define msgWinFreeOK                      MakeMsg(clsWin,   26)

/****************************************************************************
 msgWinInserted takes WIN, returns STATUS
    category: advisory message
    Advises window that it has been inserted. 
    
 pArgs is the window that actually was inserted, it may be self or 
 an ancestor. If it is an ancestor, the window is being inserted
 indirectly, as part of a sub-tree insertion.

 Windows only receive this if wsSendGeometry in flags is true.
*/
#define msgWinInserted                    MakeMsg(clsWin,   27)

/****************************************************************************
 msgWinExtracted takes WIN, returns STATUS
    category: advisory message
    Advises window that it has been extracted. 
    
 pArgs is the window that actually was extracted, it may be self 
 or an ancestor. If it is an ancestor, the window is being extracted
 indirectly, as part of a sub-tree extraction.

 Windows only receive this if wsSendGeometry in flags is true.
*/
#define msgWinExtracted                   MakeMsg(clsWin,   28)

/****************************************************************************
 msgWinVisibilityChanged takes WIN, returns STATUS
    category: advisory message
    Advises window that its visibility may have changed.
    
 pArgs is the window that actually was changed, it may be self or 
 an ancestor. If it is an ancestor, the window is being made
 visible or invisible indirectly, as part of a sub-tree insertion or
 extraction.

 Note that if pArgs is an ancestor, the ancestor's visibility change may
 not have changed self's visibility.  Use msgWinIsVisible to determine
 self's current visibility.

 Windows only receive this if wsSendGeometry in flags is true.

 See Also
    msgWinIsVisible
*/
#define msgWinVisibilityChanged           MakeMsg(clsWin,   60)

/****************************************************************************
 msgWinMoved takes P_WIN_METRICS, returns STATUS
    category: advisory message
    Advises window that it, or an ancestor, has moved.

 Windows only receive this if wsSendGeometry in flags is true.
 pArgs->bounds.origin is the previous position. pArgs->child is the
 window that actually moved, it may be self or an ancestor. If it 
 is an ancestor, the window is being moved indirectly, as part of a 
 sub-tree move.
*/
#define msgWinMoved                       MakeMsg(clsWin,   29)

/****************************************************************************
 msgWinSized takes P_WIN_METRICS, returns STATUS
    category: advisory message
    Advises window that it, or an ancestor, has changed size.

 Windows only receive this if wsSendGeometry in flags is true.
 pArgs->bounds.size is the previous size. pArgs->child is the
 window that actually changed size, it may be self or an ancestor.
 If it is an ancestor, the window did not actually change size,
 the ancestor did.
*/
#define msgWinSized                       MakeMsg(clsWin,   30)

/****************************************************************************
 msgWinStartPage takes pNull, returns STATUS
    category: advisory message
    Advises window that it is on a printer, and printing is about
    to commence.

 clsWin does nothing and returns stsOK in response to this message.

 This message is sent before a page is about to be printed. The 
 window may want to set a state variable used to change the way
 the window paints on a printer.
*/
#define msgWinStartPage                   MakeMsg(clsWin,   48)

/****************************************************************************
 msgWinSort takes P_WIN_SORT, returns STATUS
    Sorts a window's children into a back to front order determined
    by a client supplied comparison function.

 The client must create a function of the profile P_WIN_SORT_PROC
 that takes two windows (A,B) and returns -1 if A < B, 0 if A == B, and
 +1 if A > B. The comparison will normally be based on information
 retrieved from the windows (for instance, msgLabelGetString).
*/
#define msgWinSort                        MakeMsg(clsWin,   52)

typedef S16 FunctionPtr(P_WIN_SORT_PROC)(WIN        winA,
                                         WIN        winB,
                                         P_UNKNOWN  pClientData
                                        );
typedef struct 
{
  P_WIN_SORT_PROC   pSortProc;     // In:  comparison callback
  P_UNKNOWN         pClientData;   // In:  parameter to callback
  BOOLEAN           changed;       // Out: did sort cause change in order

} WIN_SORT, *P_WIN_SORT;

/****************************************************************************
 msgWinGetEnv takes P_WIN_ENV, returns STATUS
    Gets the current window environment.
 
 The window environment is information filed with the root of 
 each filed tree of windows.

 This message would not normally be used by application software.
*/
#define msgWinGetEnv                      MakeMsg(clsWin,   47)

typedef struct WIN_ENV 
{
  U8       scale;              // system font scale
  U16      sysFontId,          // system font
           userFontId;         // user font
  SIZE32   ppm;                // device pixels per meter 

} WIN_ENV, *P_WIN_ENV;

typedef struct WIN_SAVE_ENV 
{ 
  WIN_ENV  env;                // environment being saved
  U32      spare1;
  U32      spare2;

} WIN_SAVE_ENV, *P_WIN_SAVE_ENV;

typedef struct WIN_RESTORE_ENV 
{
  WIN_ENV  env;                // the saved environment
  BOOLEAN  scaleChanged,
           sysFontIdChanged,   // these are true if the current
           userFontIdChanged,  // environment has changed from
           ppmWChanged,        // the saved environment.
           ppmHChanged;
  U32      spare1;
  U32      spare2;

} WIN_RESTORE_ENV, *P_WIN_RESTORE_ENV;

/****************************************************************************
 msgWinDumpTree takes pNull, returns STATUS
    In lieu of msgDump. Dumps a dense subset of information for the
    window and all it's children recursively.

 Debug /DW 2 causes the input flags to be printed, otherwise
 the window flags are printed.

 This function may not work unless the debugging version of win.dll
 is being used.
*/
#define msgWinDumpTree                    MakeMsg(clsWin,   51)

/****************************************************************************
 msgWinHitDetect takes P_WIN_METRICS, returns STATUS
    Locates the window "under" a point.

 pArgs->bounds.origin is a point relative to the receiver. The window 
 tree, starting with the root window, is searched for a window underneath 
 this point. The result is returned in pArgs->child. 

 If the search is NOT successful, pArgs->child will be objNull.
*/
#define msgWinHitDetect                   MakeMsg(clsWin,   58)    

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from other classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Causes an object to file itself in an object file.

 clsWin will save its instance data and file each direct child that
 has wsSendFile on.

 If pArgs->root is self, clsWin will file the window environment along
 with its instance data.  The window environment is retrieved by self-sending
 msgWinGetEnv.  If pArgs->pEnv is not pNull, the current environment info
 (WIN_SAVE_ENV) will be copied to the storage provided (pArgs->pEnv should
 either be pNull or a P_WIN_SAVE_ENV).  Subclasses of clsWin can make use of
 pArgs->pEnv to look at the environment under which the window is being saved.
 The filed window environment will be used during msgRestore to adjust the
 window bounds and/or dirty the window layout if the restore environment is
 not the same as the saved environment.
 
 If wsFileNoBounds is on in self's window style flags, the current bounds
 will not be filed.  This will save space in the filed window.

 If self's desired size has been computed (via msgWinGetDesiredSize during
 msgWinLayout processing), the desired size will be filed.

 For each child of self that has wsSendFile on, clsWin will do the following:
//{
 	If wsFileInline is on in the child's window style flags, the class of the
	child window will be filed, and	then the child will be sent msgSave with
	the following OBJ_SAVE parameters:

		all fields as in *pArgs,
		objSave = pointer to current save environment

	This will file the child "inline" without the usual resource file object
	header.  This will save storage, but the child will not have its own
	resId and can only be restored by restoring its parent.

	If wsFileInline is not on in the child's window style flags, the child
	window will be filed by sending msgResPutObject to pArgs->file with the
	child's uid as pArgs.
//}

 Returns
 	stsWinNoEnv		if pArgs->root != self and pArgs->pEnv is pNull

 See Also
	msgRestore
 	msgWinGetEnv
*/

/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.
	
 clsWin will restore its instance data from pArgs->file.  Each filed child
 window will also be restored.  The window will be created on the window
 device returned from OSThisWinDev().

 If the window environment was filed when the window was saved, the window
 environment will be restored and copied to pArgs->pEnv if it is not pNull
 (pArgs->pEnv must be either pNull or P_WIN_RESTORE_ENV).  The current
 window environment will be retrieved using msgWinGetEnv and compared to
 the filed window environment.

 If wsFileNoBounds is on in self's window style flags, the bounds will be
 set to (0, 0, 0, 0) and the window will be marked as layout-dirty
 (wsLayoutDirty will be or-ed into the window's style flags).  Otherwise,
 the filed bounds will be restored and adjusted to compensate for differences
 in the save/restore-time device resolution and orientation.
 
 clsWin will or-in wsLayoutDirty into the window's style flags if any of the
 following are true (in this context "changed" means that the current window
 environment values do not match the window environment filed with the window
 tree):

 	wsFileLayoutDirty is on in the window's style flags
	The system font or system font scale has changed
	The user font has changed
	The pixels-per-meter in x or y have changed

 Each child that was filed will be restored as follows:
//{
 	If wsFileInline was on in the child's window style flags, the child's
	class will be read in from pArgs->file and msgRestoreInstance will be
	sent to the classs with the following OBJ_RESTORE parameters:

		all fields as in *pArgs
		object		= msgNewDefaults to clsObject
		object.key	= pArgs->object.key;
		object.cap	= pArgs->object.cap;
		object.heap	= pArgs->object.heap;

	 If wsFileInline was not on in the child's window style flags, the child's
	 resId will be read in from pArgs->file and the child will be restored
	 by sending msgResReadObject to pArgs->file with the following
	 RES_READ_OBJECT parameters:

	 	mode		= resReadObjectOnce;
		objectNew	= same as object in wsFileInline case above
//}
 After all of the children have been restored, they will be inserted into
 the restored parent.  Note that the wsCaptureGeometry and wsSendGeometry
 protocol is not used for these inserts (e.g. the parent will not be sent
 msgWinInsertOK, even if the parent has wsCaptureGeometry on).

 See Also
	msgSave
 	msgWinGetEnv
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Messages Sent to a Window Device                                       *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 As a rule applications should not send these messages to theScreen. 
 They would be used if the application creates image devices.
*/

/****************************************************************************
 msgNew takes P_WIN_DEV_NEW, returns STATUS
    category: class message
    Creates a windowing device.
*/
typedef struct
{
  U16	initialWindows;		// default window slots to allocate

} WIN_DEV_NEW_ONLY, * P_WIN_DEV_NEW_ONLY;

typedef struct
{
  OBJECT_NEW        object;
  WIN_DEV_NEW_ONLY  winDev;

} WIN_DEV_NEW, * P_WIN_DEV_NEW,
  IMG_DEV_NEW, * P_IMG_DEV_NEW;

/****************************************************************************
 msgNewDefaults takes P_WIN_DEV_NEW, returns STATUS
    category: class message
    Initializes the WIN_DEV_NEW structure to default values.
//{
 win.Dev.initialWindows = 100;
//}
*/

/****************************************************************************
 msgWinDevGetRootWindow takes P_OBJECT, returns STATUS
    Passes back root window for receiver. 
*/
#define msgWinDevGetRootWindow            MakeMsg(clsWinDev,10)    

/****************************************************************************
 msgWinDevBindScreen takes P_CHAR, returns STATUS
    Binds window device to a screen.
*/
#define msgWinDevBindScreen               MakeMsg(clsWinDev, 6)    

/****************************************************************************
 msgWinDevBindPrinter takes OBJECT, returns STATUS
    Binds window device to an object of clsPrn.
*/
#define msgWinDevBindPrinter              MakeMsg(clsWinDev, 7)    

/****************************************************************************
 msgWinDevBindPixelmap takes P_WIN_DEV_PIXELMAP, returns STATUS
    Binds window device to a pixelmap.

 Note that you should not file the memory allocated by msgWinDevBindPixelmap,
 since the memory is device-dependant and you may be restored on a different
 screen device or system processor.
*/
#define msgWinDevBindPixelmap             MakeMsg(clsWinDev,11)

/****************************************************************************
 msgWinDevSizePixelmap takes P_WIN_DEV_PIXELMAP, returns STATUS
    Computes the amount of memory needed for a single plane.
*/
#define msgWinDevSizePixelmap             MakeMsg(clsWinDev,12)

typedef struct
{
  OBJECT      device;         // in  = device to be "compatible" with
  SIZE32      size;           // in  = w,h of device to allocate
  U16         planeCount;     // in  = # planes to allocate
  SIZEOF      planeSize;      // out = amount of memory for one plane
  PP_UNKNOWN  pPlanes;        // in  = plane memory

} WIN_DEV_PIXELMAP, * P_WIN_DEV_PIXELMAP;

/****************************************************************************
 msgWinDevSetOrientation takes PIX_DEV_ORIENT, returns STATUS
    Changes orientation of a window device.
*/
#define msgWinDevSetOrientation           MakeMsg(clsWinDev, 8)

Enum16(PIX_DEV_ORIENT)
{
  pdUL                   	= 0,
  pdUR                   	= 1,
  pdLR                   	= 2,
  pdLL                   	= 3,
  pdOrientLandscapeNormal  	= pdLL,
  pdOrientPortraitNormal   	= pdUL,
  pdOrientLandscapeReverse	= pdUR,		// not supported on printers
  pdOrientPortraitReverse  	= pdLR    	// not supported on printers

};

/****************************************************************************
 msgPixDevGetMetrics takes P_PIX_DEV_METRICS, returns nothing
    Gets metrics of a pixelmap device.
*/
#define msgPixDevGetMetrics               MakeMsg(clsPixDev, 1)

typedef struct
{
  SIZE32          size,                 // size of device in DU4
                  ppm;                  // pixel per meter in DU4
  U16             planes;               // # of planes total
  U16             planeMask;            // mask representing all planes
  U16             planeNormalCount;     // # of normal (not pen) planes
  U16             planeNormalMask;      // mask for the normal planes
  U16             planePenCount;        // # of pen planes
  U16             planePenMask;         // mask for the pen planes
  PIX_DEV_ORIENT  orient;               // pdUL, pdLR, etc.

  P_UNKNOWN       devPhysical,          // handles to physical and
                  devLogical;           // logical device drivers

  U16             mode;                 // private flags, see pix...

  OBJECT          prn;                  // printer (or objNull)
  PP_UNKNOWN      ppDryRunRgn;
                                    
} PIX_DEV_METRICS, * P_PIX_DEV_METRICS;

/****************************************************************************
 msgWinDevPrintPage takes nothing, returns STATUS
    Repaints and outputs a page.
*/
#define msgWinDevPrintPage                MakeMsg(clsWinDev, 9)    

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Messages sent to a drawing context                                     *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDrwCtxSetWindow takes WIN, returns WIN
    Binds a drawing context to a window, returns old window.
*/
#define msgDrwCtxSetWindow                MakeMsg(clsDrwCtx, 3)

/****************************************************************************
 msgDrwCtxGetWindow takes nothing, returns WIN
    Returns the window to which a drawing context is bound.
*/
#define msgDrwCtxGetWindow                MakeMsg(clsDrwCtx, 4)

#endif // WIN_INCLUDED

//REFGEN BEGINIGNORE

#ifdef  WIN_PRIVATE
#ifndef WIN_PRIVATE_INCLUDED
#define WIN_PRIVATE_INCLUDED

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Messages and Functions used internally by the Window Manager           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
#include <image.h>

#define  pixDraw                  0
#define  pixGrafOnly              flag0
#define  pixTextOnly              flag1
#define  pixDryRun                flag2
#define  winEnumRecursionLimit    50
#define  initWinSynchVer          ((SYNCH_VER)1)

typedef U16  SYNCH_VER;

typedef struct
{
  WIN_METRICS  metrics;          // out=just like msgWinGetMetrics
  RECT32       devBounds;        // out=bounds, root relative
  P_UNKNOWN    pDev;             // out=ptr to WIN_DEV instance data
  U16          ord;              // out=ord of win
  WIN_FLAGS    parentFlags;      // out=
  SYNCH_VER    devSig;           // out=device signature, unique

} WIN_INFO, * P_WIN_INFO;

STATUS  PASCAL Win0Init(void);
BOOLEAN EXPORTED0  WinGetInfo(WIN self, P_WIN_INFO pArgs);
H_CLIP  EXPORTED0  WinGetCachedVisible(P_UNKNOWN p, U16 ord);
void    EXPORTED0  WinPenPaint(WIN self);
void    EXPORTED0  WinDirtyRegion(WIN self, H_CLIP rgn);
void    EXPORTED0  WinDevCopyDirty(P_UNKNOWN  srcDev, 
                                   U16        sw,
                                   DEV_COORD  sul,
                                   DEV_COORD  slr,
                                   P_UNKNOWN  dstDev,
                                   U16        dw,
                                   DEV_COORD  dul
                                  );

Enum16(PIX_DEV_CURSOR)
{ pdCursorTest,          // test for intersection
  pdCursorOff,           // force the cursor off
  pdCursorIgnore         // don't worry about the cursor
};

typedef struct
{
  // NOTE TO DRAWING CONTEXT IMPLEMENTORS
  // Parameters ul,lr are in DU4 coordinates.
  // Rectangles in visRgn are DU4.
  
  WIN             win;             // in=window to draw on
  SYNCH_VER       winSynch,        // out=1 or greater
                  devSynch;        // out=1 or greater
  U16             ord;             // in/out=ord of win on device

  SIZE32          size;            // out=in LWC, size of window

  XY32            ul,              // out=in 32 bit DU4, draw here
                  lr;

  H_CLIP          updRgn;          // out=in DU4, clip to this

  PIX_DEV_CURSOR  curTest;         // in=what to do with the cursor

} PIX_DEV_DRAW, * P_PIX_DEV_DRAW;

/*
 PixDevBeginDraw takes win as a parameter, locks the device and
 passes back winSynch and devSynch; it also fills in ord for use by
 PixDevGetDraw which returns size, ul, lr, and updRgn
 PixDevEndDraw unlocks the device
*/

STATUS EXPORTED0 PixDevBeginDraw (P_UNKNOWN p, P_PIX_DEV_DRAW pArgs);
STATUS EXPORTED0 PixDevGetDraw   (P_UNKNOWN p, P_PIX_DEV_DRAW pArgs);
STATUS EXPORTED0 PixDevGetDrawForDrawing(P_UNKNOWN p, P_PIX_DEV_DRAW pArgs);
STATUS EXPORTED0 PixDevEndDraw   (P_UNKNOWN p, P_PIX_DEV_DRAW pArgs);
STATUS EXPORTED0 PixDevGetMetrics(P_UNKNOWN p, P_PIX_DEV_METRICS pArgs);
PP_UNKNOWN EXPORTED0 PixDevGetDirtyTextRgn(P_UNKNOWN p);

/****************************************************************************
 msgWinDevHitDetect takes P_WIN_DEV_XY, returns STATUS
    Determines windows "under" a point.

 Return Value
    stsOK if new hit info is returned
    stsWinDevCachedHit if hit was on previous hit
    stsOK with count == 0 if no hit was made at all
*/
#define msgWinDevHitDetect                MakeMsg(clsWinDev, 1)    

typedef struct
{
  XY32       xy;      // point to hit-detect in LDC coordinates
  U16        count;   // In-max entries in arrays; Out-# returned
  P_WIN      pWin;    // In-ptr to array of windows
  P_U32      pFlags;  // In-ptr to array of window flags.input

} WIN_DEV_XY, * P_WIN_DEV_XY;

/****************************************************************************
 msgWinDevLocalXY takes P_WIN_DEV_XY, returns STATUS
    Converts window coordinate obtained with msgWinDevHitDetect.
*/
#define msgWinDevLocalXY                  MakeMsg(clsWinDev, 2)

/****************************************************************************
 msgWinDevRepaint takes nothing, returns STATUS
    Repaints all dirty windows on receiver that are owned by OSThisTask().
*/
#define msgWinDevRepaint                  MakeMsg(clsWinDev, 3)

/****************************************************************************
 msgWinDevWarmStart takes nothing, returns nothing
    Restore state of graphics hardware and force repaint of all windows.

 Should only be sent by kernel, and only to theScreen.
*/
#define msgWinDevWarmStart                MakeMsg(clsWinDev, 4)  

/****************************************************************************
 msgWinDevHideDrawing takes nothing, returns STATUS
    An off screen buffer is created and all drawing is redirected to it.
*/
#define msgWinDevHideDrawing              MakeMsg(clsWinDev, 13)

/****************************************************************************
 msgWinDevShowDrawing takes nothing, returns STATUS
    Restores mode started by msgWinDevHideDrawing.
*/
#define msgWinDevShowDrawing              MakeMsg(clsWinDev, 14)

/****************************************************************************
 msgWinDevMixRGB takes P_SYSDC_MIX_RGB, returns STATUS
    Programs a palette slot to a specific RGB value.
*/
#define msgWinDevMixRGB                   MakeMsg(clsWinDev, 5)

/****************************************************************************
 msgWinPrivate1 takes P_RECT32 or pNull, returns STATUS
    Used only by the UI toolkit.
*/
#define msgWinPrivate1                    MakeMsg(clsWin,   50)

/****************************************************************************
 msgWinPrivate2 takes P_WIN_METRICS, returns STATUS
    Used only by the UI toolkit.
*/
#define msgWinPrivate2                    MakeMsg(clsWin,   57)

/****************************************************************************
 msgWinPaintPowerUp takes pNull, returns STATUS
    Used only by theRootWindow.
*/
#define msgWinPaintPowerUp                MakeMsg(clsWin,   56)

/****************************************************************************
 msgWinInsertChildren takes P_WIN, returns STATUS
    Inserts each child as a child of the receiver. Used only by the 
    UI toolkit.

 Args is a null-terminated array of child windows.
 This does not obey the capture/send geometry protocol.
 The children are inserted bottom-to-top.
 The parent must not be visible.
 The parent must not be in a layout episode.
 The parent's layout is not dirtied.
 No check is made for parent already a child of each child.
*/
#define msgWinInsertChildren              MakeMsg(clsWin,   55)

/****************************************************************************
 msgWinAtDesiredSize takes pNull, returns STATUS
    Returns stsOK if the current size is at least as big as the cached
	desired size.

    Used only by the UI toolkit.
	Returns stsFailed if there is no cached desired size or the width or
	height of the current size are smaller than the width or height of the
	desired size.
*/
#define msgWinAtDesiredSize               MakeMsg(clsWin,   61)

/*
 WinGetTransforms and WinInset are to be used only by the UI toolkit.
*/

typedef struct
{
  OBJECT        win;               // in  = for this window's device
  SCALE         layouts,           // out = get layout=>device
                twips;             // out = get twips=>device

} WIN_TRANSFORMS, * P_WIN_TRANSFORMS;

typedef struct
{
  OBJECT        win;               // in =
  WIN_METRICS   metrics;           // in/out = .bounds is rect to inset
  BOOLEAN       prefetch;          // in = get window metrics first
                               
  RECT16        twips,             // in = inset amounts in twips
                layouts;           // in = inset amounts in layout units

} WIN_INSET, * P_WIN_INSET;

STATUS EXPORTED0 WinGetTransforms(P_WIN_TRANSFORMS);
STATUS EXPORTED0 WinInset(P_WIN_INSET);
S32    EXPORTED0 LayoutUPM(void);
S32    EXPORTED0 RulesPM(void);

#endif // WIN_PRIVATE_INCLUDED
#endif // WIN_PRIVATE

//REFGEN ENDIGNORE
