/****************************************************************************
 File: sysgraf.h

 (C) Copyright 1992 by Go Corporation, All Rights Reserved.

 $Revision:   1.23  $
   $Author:   sbartlet  $
     $Date:   06 Mar 1992 15:13:10  $

 This file provides the API for clsSysDrwCtx.

 clsSysDrwCtx inherits from clsDrwCtx, an abstract class.
 Defines the fundamental drawing services. An instance of clsSysDrwCtx,
 often called a "DC", is an object that is used to draw onto windows.
 After a DC is created, it is bound to a window (see msgDcSetWindow).
 After this step, drawing messages sent to the DC will result in 
 drawing onto the bound window. While a DC may remain bound to a 
 window forever, such drawing messages are only effective inside an
 "update episode" bracketed by msgWinBeginRepaint and msgWinEndRepaint.

 There are a number of other DC messages that do not have to be sent
 inside an "update episode"; for instance msgDcLWCtoLUC_XY32. However,
 many of these messages implicitly require device or window metrics
 to produce the correct results. Thus, as a rule, a DC should be 
 bound to a window before it is used.

 Terminology:
                                                          
 DU4 -- Device Units, 4th Quadrant. A 4th quadrant coordinate system;
 device space, device units. This is used internally, but not seen by 
 application software.

 LWC -- Logical Window Coordinates. A 1st quadrant coordinate system.
 The lower-left-hand corner of the window is 0,0. The units are device 
 pixels. 

 LUC -- Logical Unit Coordinates. A 1st quadrant coordinate system
 provided by the DC. The default units can be a real-world measure like 
 points or mils; and they can be translated, rotated and scaled.

 A number of font-related data structures are defined in sysfont.h.

****************************************************************************/
#ifndef SYSGRAF_INCLUDED
#define SYSGRAF_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef GEO_INCLUDED
#include <geo.h>
#endif

#ifndef WIN_INCLUDED
#include <win.h>
#endif

#ifndef SYSFONT_INCLUDED
#include <sysfont.h>
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Overview			                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Sysgraf (aka clsSysDrawCtx aka ImagePoint) is the lowest level drawing
 interface PenPoint provides above the bit level.  The division of labor here
 is that Windows worry about parceling out screen real-estate while Sysgraf
 worries about drawing on the screen.  If you want to draw things in a
 window, you create a drawing context (an instance of clsSysDrawCtx), bind it
 to the window you want to draw in (by sending msgDcSetWindow to the
 drawing context), and send messages to the drawing context.  

 If you plan to use a drawing context to render text, you should
 understand the use of msgDcMeasureText, which lets you determine how
 large a piece of text will be before you actually draw it.  It is
 also important to know that although sysgraf allows you to set many
 different parameters, including font, rotation, line thickness, etc. you
 may only change these between drawing calls.  That is, if you want to
 render plain text, a word in italics, and more plain text, you need
 to send three separate msgDcDrawText messages, changing to italics after
 the first one and back to normal after the second.

 If you plan to use sysGraf at all, it will be well worth your while to
 browse all the messages below.
*/

// Message numbers available: 7, 8, 9, 34, 35, 36, 37, 38; next up: 110

typedef  DRW_CTX      SYSDC,         // an object of clsSysDrwCtx
                    * P_SYSDC;       // and a ptr to one

#define  stsDcHitOn              MakeStatus(clsSysDrwCtx,    2)           
#define  stsDcHitIn              MakeStatus(clsSysDrwCtx,    3)
#define  stsDcHitOut             MakeStatus(clsSysDrwCtx,    4)

/****************************************************************************
 msgNew takes P_SYSDC_NEW, returns STATUS
    category: class message
    Creates a system drawing context.
*/
typedef struct SYSDC_NEW_ONLY {

	U32		reserved;

} SYSDC_NEW_ONLY, *P_SYSDC_NEW_ONLY;

#define	sysdcNewFields	\
	objectNewFields		\
	SYSDC_NEW_ONLY  	sysDc;

typedef struct
{
  sysdcNewFields

} SYSDC_NEW, * P_SYSDC_NEW;

/****************************************************************************
 msgNewDefaults takes P_SYSDC_NEW, returns STATUS
    category: class message
    Initializes the SYSDC_NEW structure to default values.
//{
    sysDc.reserved = 0;
//}
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Binding to a Window                                                    *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDcSetWindow takes WIN, returns WIN
    Binds a window to the receiver and returns the previously bound window.

 All output through the DC will now appear on this window.
 A DC must be bound to a window before most messages will work.
*/
#define msgDcSetWindow             msgDrwCtxSetWindow

/****************************************************************************
 msgDcGetWindow takes pNull, returns WIN
    Gets the window to which the drawing context is bound.
*/
#define msgDcGetWindow             msgDrwCtxGetWindow          


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Graphic State Control                                                  *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDcInitialize takes pNull, returns stsOK
    Sets graphics state to initial values.

 The initial values are:
//{
	units in (LUC)	 = msgDcUnitsPoints
	units out		 = msgDcUnitsDevice
	matrix			 = identity, 1st quadrant
	premultiply		 = FALSE
	clipping		 = none, except to window
	raster op		 = sysDcRopCopy
	drawing mode	 = sysDcDrawNormal | sysDcHoldDetail
	plane mask		 = see msgDcPlaneNormal
	line.cap		 = sysDcCapButt
	line.join		 = sysDcJoinMiter
	line.thickness	 = 1 unit (point)
	line.miterLimit	 = 10
	line.radius		 = 0
	foreground color = sysDcRGBBlack
	background color = sysDcRGBWhite
	fill pattern	 = sysDcPatBackground
	fill mode		 = even/odd (see sysDcWindingFill)
	line pattern	 = sysDcPatForeground
	logical font	 = default font, size is 1 unit (point)
//}
*/
#define msgDcInitialize            MakeMsg(clsSysDrwCtx, 50)

/****************************************************************************
 msgDcPush takes P_SYSDC_STATE, returns stsOK
    Gets the graphics state and stores it.

 While the names msgDcPush/msgDcPop imply a stack-like use for these
 messages (as is their intended application); this is not a requirement.
 There is no stack internal to the DC. State is copied in and out
 of the argument buffer.

 One application is to pre-stage frequently needed combinations of
 state (fonts, colors, etc.) in an array of these buffers; and
 then pop them into a single DC as needed. This is more memory 
 efficient than having several DC's, and nearly as fast.

 SYSDC_STATE is an opaque data type. There is no value in examining
 the bytes therein. It can be stored temporarily; but, it should not be 
 filed, as it may change from release to release of the software.
*/
#define msgDcPush                  MakeMsg(clsSysDrwCtx, 31)

typedef  struct
{
  U8  state[448];  

} SYSDC_STATE, * P_SYSDC_STATE;

/****************************************************************************
 msgDcPop takes P_SYSDC_STATE, returns stsOK
    Sets the graphics state from one saved by msgDcPush.
*/
#define msgDcPop                   MakeMsg(clsSysDrwCtx, 32)

/****************************************************************************
 msgDcPushFont takes P_SYSDC_FONT_STATE, returns stsOK
    Gets the font state and stores it.

 The same comments made under msgDcPush apply to msgDcPushFont.
*/
#define msgDcPushFont              MakeMsg(clsSysDrwCtx, 51)

typedef  struct
{
  U8  state[256];

} SYSDC_FONT_STATE, * P_SYSDC_FONT_STATE;

/****************************************************************************
 msgDcPopFont takes P_SYSDC_FONT_STATE, returns stsOK
    Sets the font state from one saved by msgDcPushFont.
*/
#define msgDcPopFont               MakeMsg(clsSysDrwCtx, 52)

/****************************************************************************
 msgDcSetMode takes SYSDC_MODE, returns SYSDC_MODE
    Sets the drawing mode and returns the old SYSDC_MODE.
*/
#define msgDcSetMode               MakeMsg(clsSysDrwCtx,  2)

Enum16(SYSDC_MODE)
{
  sysDcDrawNormal     = 0,
  sysDcDrawFast       = flag0,  // draw faster with gross loss of fidelity
  sysDcDrawDynamic    = flag1,  // sets up XOR style drawing
  sysDcHoldDetail     = flag2,  // keeps lines from vanishing
  sysDcWindingFill    = flag3,

  sysDcHitTest        = flag4,  // must set with msgDcHitTest
  sysDcAccumulate     = flag7,  // must set with msgDcAccumulateBounds
  sysDcHoldLine       = flag5,  // must set with msgDcHoldLine
  sysDcPreMultiply    = flag6   // can  set with msgDcSetPreMultiply

};

/****************************************************************************
 msgDcGetMode takes pNull, returns SYSDC_MODE
    Gets the drawing mode.
*/
#define msgDcGetMode               MakeMsg(clsSysDrwCtx, 65)

/****************************************************************************
 msgDcSetPreMultiply takes BOOLEAN, returns BOOLEAN
    Sets the pre-multiply state and returns the old state.

 This affects the matrix arithmetic implicit in msgDcScale, msgDcRotate
 and msgDcTranslate. The default mode is post-multiply. The default for 
 PostScript is pre-multiply; so when borrowing algorithms from PostScript
 sources this could be useful.
*/
#define msgDcSetPreMultiply        MakeMsg(clsSysDrwCtx, 96)

/****************************************************************************
 msgDcSetRop takes SYSDC_ROP, returns SYSDC_ROP
    Sets the raster op and returns the old rop.

 Note that there are not many good reasons to be using this message;
 the results are rather device dependent. If you need to draw with an 
 XOR raster op, use msgDcSetMode to set the sysDcDrawDynamic flag instead.
*/
#define msgDcSetRop                MakeMsg(clsSysDrwCtx,  1)   

Enum16(SYSDC_ROP)
{
  sysDcRopCOPY,
  sysDcRopAND,
  sysDcRopOR,
  sysDcRopXOR,
  sysDcRopNCOPY,
  sysDcRopNAND,
  sysDcRopNOR,
  sysDcRopNXOR

};

/****************************************************************************
 msgDcPlaneNormal takes nothing, returns SYSDC_PLANE_MASK
    Sets the plane mask to the normal plane(s), returning the old mask.
*/
#define msgDcPlaneNormal           MakeMsg(clsSysDrwCtx, 41)   

typedef U16 SYSDC_PLANE_MASK;

/****************************************************************************
 msgDcPlanePen takes nothing, returns SYSDC_PLANE_MASK
    Sets the plane mask to the plane(s) for pen ink, returning the old mask.

 In most situations it is better to use clsTrack to draw on the pen plane(s).
 See track.h.
*/
#define msgDcPlanePen              MakeMsg(clsSysDrwCtx, 42)   

/****************************************************************************
 msgDcPlaneMask takes SYSDC_PLANE_MASK, returns SYSDC_PLANE_MASK
    Sets an arbitrary plane mask, returning the old mask.

 This interface is NOT RECOMMENDED for application software.
 It is inherently non-portable.
*/
#define msgDcPlaneMask             MakeMsg(clsSysDrwCtx, 43)   

/****************************************************************************
 msgDcGetLine takes P_SYSDC_LINE, returns COORD16
    Gets all line attributes if pArgs is P_SYSDC_LINE. Returns line thickness.

 If P_SYSDC_LINE is pNull then only line thickness is returned.
*/
#define msgDcGetLine               MakeMsg(clsSysDrwCtx, 62)

/****************************************************************************
 msgDcSetLine takes P_SYSDC_LINE, returns COORD16
    Sets all line attributes. Returns old line thickness.

 Both line thickness and the radius value for creating rounded corner 
 rectangles are in LUC.
*/
#define msgDcSetLine               MakeMsg(clsSysDrwCtx,  4)   

Enum16(SYSDC_CAP)
{ sysDcCapSquare   = 0,
  sysDcCapButt     = 1,
  sysDcCapRound    = 2,
};

Enum16(SYSDC_JOIN)
{ sysDcJoinMiter   = 0,
  sysDcJoinBevel   = 1,
  sysDcJoinRound   = 2,
};

typedef struct
{
  SYSDC_CAP          cap;
  SYSDC_JOIN         join;
  COORD16            thickness;
  U16                miterLimit;   // Choose + number, 10 recommended.
  S16                radius;       // For rounded corner rectangles
                                   //   use + number or sysDcRadiusAuto.
                                   // For square corner rectangles use 0.
                                   // This number is in LUC.

} SYSDC_LINE, * P_SYSDC_LINE;

#define sysDcRadiusAuto         (-1)

/****************************************************************************
 msgDcSetLineThickness takes COORD16, returns COORD16
    Sets line thickness to new value; returns old line thickness.

 This is the best message for quickly changing line thickness and
 restoring it back.
*/
#define msgDcSetLineThickness      MakeMsg(clsSysDrwCtx, 79)   

/****************************************************************************
 msgDcHoldLine takes BOOLEAN, returns BOOLEAN
    Turns hold line thickness mode on/off; returns old hold mode.

 msgDcHoldLine(TRUE) causes the current line thickness to be made 
 immune from the effects of scaling (msgDcScale, msgDcUnitsXXXX). 
 msgDcHoldLine(FALSE) will cancel hold mode.

 msgDcSetLine/Thickness messages will cause the line thickness to
 change, but having changed, it will still be immune from the
 effects of scaling until hold mode is canceled.

 The DC must be bound to a window when this message is sent.
*/
#define msgDcHoldLine              MakeMsg(clsSysDrwCtx, 63)   

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Device Independent Color                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define sysDcRGBTransparent ((U32)0)
#define sysDcRGBBlack       (SysDcGrayRGB(0))
#define sysDcRGBGray66      (SysDcGrayRGB(85))
#define sysDcRGBGray33      (SysDcGrayRGB(170))
#define sysDcRGBWhite       (SysDcGrayRGB(255))

typedef union 
{
  U32    all;
  struct
  {
    U8   red,
         green,
         blue,
         transparency;
  }      rgb;

} SYSDC_RGB, * P_SYSDC_RGB;

#define  SysDcGrayRGB(v)     MakeU32(MakeU16(v,v),MakeU16(v,255))

/*
 These messages set and get the foreground and background colors by
 RGB specification. The "set" messages take an RGB specification
 (cast to a U32) and return stsOK.

 The "get" messages store the current value into a U32 (or SYSDC_RGB)
 pointed to by pArgs.

 The structure SYSDC_RGB is a union of the four r-g-b-t fields and a
 U32. This allows RGB values to be compared easily as U32 values.
 The transparency byte should always be 255 for opaque color. It can be
 0 when setting the background color to transparent (in which case the
 red, green, blue values are not examined). Intermediate transparency
 values are not supported.

 The macro SysDcGrayRGB takes a value between 0..255 and returns a
 U32 with the r-g-b bytes set to the value, and the transparency byte 
 set to 255. The value 0 can be used for a pure transparent RGB.

 The set messages find the closest matching color to the RGB specification;
 they do not create new colors. To create new colors see msgDcMixRGB (which
 is not implemented yet).

 Unlike the palette oriented messages (msgDcSetForegroundColor, 
 msgDcSetBackgroundColor) colors set using these RGB messages 
 are portable across a variety of devices and are automatically 
 retranslated when the DC is connected to a different device.
*/

/****************************************************************************
 msgDcSetForegroundRGB takes U32, returns stsOK
    Sets foreground color using an RGB specification.

 When using this interface, see the constants sysDcRGB... for
 the standard colors.
*/
#define msgDcSetForegroundRGB      MakeMsg(clsSysDrwCtx, 75)

/****************************************************************************
 msgDcSetBackgroundRGB takes U32, returns stsOK
    Sets background color using an RGB specification.

 When using this interface, see the constants sysDcRGB... for
 the standard colors.
*/
#define msgDcSetBackgroundRGB      MakeMsg(clsSysDrwCtx, 76)

/****************************************************************************
 msgDcInvertColors takes pNull, returns stsOK
    Swaps foreground and background colors.
*/
#define msgDcInvertColors          MakeMsg(clsSysDrwCtx, 64)

/****************************************************************************
 msgDcGetForegroundRGB takes P_U32 or P_SYSDC_RGB, returns stsOK
    Returns foreground RGB value.
*/
#define msgDcGetForegroundRGB      MakeMsg(clsSysDrwCtx, 77)

/****************************************************************************
 msgDcGetBackgroundRGB takes P_U32 or P_SYSDC_RGB, returns stsOK
    Returns background RGB value.
*/
#define msgDcGetBackgroundRGB      MakeMsg(clsSysDrwCtx, 78)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Device Dependent Color                                                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef U16 SYSDC_COLOR;

#define sysDcInkTransparent ((SYSDC_COLOR)0x8000)
#define sysDcInkBlack       ((SYSDC_COLOR)0x0000)
#define sysDcInkGray66      ((SYSDC_COLOR)0x0001)
#define sysDcInkGray33      ((SYSDC_COLOR)0x0002)
#define sysDcInkWhite       ((SYSDC_COLOR)0x0003)

/****************************************************************************
 msgDcMatchRGB takes U32, returns SYSDC_COLOR
    Returns palette entry that best matches an RGB.

 This interface is NOT RECOMMENDED for application software.
 Set colors directly using the msgDcSetForegroundRGB and 
 msgDcSetBackgroundRGB messages.
*/
#define msgDcMatchRGB              MakeMsg(clsSysDrwCtx, 10)

/****************************************************************************
 msgDcSetForegroundColor takes SYSDC_COLOR, returns SYSDC_COLOR
    Sets foreground color using a hardware palette index, returning old color.

 This interface is NOT RECOMMENDED for application software.
 Use msgDcSetForegroundRGB instead of this message.

 When using this interface, see the constants sysDcInk... for
 predefined palette index values.
*/
#define msgDcSetForegroundColor    MakeMsg(clsSysDrwCtx,  5)

/****************************************************************************
 msgDcSetBackgroundColor takes SYSDC_COLOR, returns SYSDC_COLOR
    Sets background color using a hardware palette index, returning old color.

 This interface is NOT RECOMMENDED for application software.
 Use msgDcSetBackgroundRGB instead of this message.

 When using this interface, see the constants sysDcInk... for
 predefined palette index values.
*/
#define msgDcSetBackgroundColor    MakeMsg(clsSysDrwCtx,  6)

/****************************************************************************
 msgDcMixRGB takes P_SYSDC_MIX_RGB, returns STATUS
    Programs a palette slot to a specific RGB.

 *** NOT IMPLEMENTED YET ***

 This interface is NOT RECOMMENDED for application software.
 The type SYSDC_MIX_RGB is defined now to support msgWinDevMixRGB.
*/
#define msgDcMixRGB                MakeMsg(clsSysDrwCtx, 80)

typedef struct
{
  SYSDC_COLOR  slot;
  SYSDC_RGB    spec;

} SYSDC_MIX_RGB, * P_SYSDC_MIX_RGB;

/****************************************************************************
 msgDcSetLinePat takes SYSDC_PATTERN, returns SYSDC_PATTERN
    Sets the line pattern; returns old value.

 The line pattern is used to draw lines around the edge of
 geometric figures when the line thickness is > 0.

 When using this interface, see the constants sysDcPat... for
 predefined patterns.
*/
#define msgDcSetLinePat            MakeMsg(clsSysDrwCtx, 11)   

typedef U16 SYSDC_PATTERN;

#define sysDcPat75     ((SYSDC_PATTERN)1)      //  75% fgnd  25% bgnd
#define sysDcPat50     ((SYSDC_PATTERN)2)      //  50% fgnd  50% bgnd
#define sysDcPat25     ((SYSDC_PATTERN)3)      //  25% fgnd  75% bgnd
#define sysDcPat12     ((SYSDC_PATTERN)4)      //  12% fgnd  88% bgnd
#define sysDcPat6      ((SYSDC_PATTERN)5)      //   6% fgnd  94% bgnd
#define sysDcPat3      ((SYSDC_PATTERN)6)      //   3% fgnd  97% bgnd
#define sysDcPat2      ((SYSDC_PATTERN)7)      //   2% fgnd  98% bgnd

#define sysDcPatLD50   ((SYSDC_PATTERN)8)      // darkest  left  diagonal
#define sysDcPatLD37   ((SYSDC_PATTERN)9)      //          left  diagonal
#define sysDcPatLD25   ((SYSDC_PATTERN)10)     //          left  diagonal
#define sysDcPatLD12   ((SYSDC_PATTERN)11)     // lightest left  diagonal

#define sysDcPatRD50   ((SYSDC_PATTERN)12)     // darkest  right diagonal
#define sysDcPatRD37   ((SYSDC_PATTERN)13)     //          right diagonal
#define sysDcPatRD25   ((SYSDC_PATTERN)14)     //          right diagonal
#define sysDcPatRD12   ((SYSDC_PATTERN)15)     // lightest right diagonal

#define sysDcPatBackground ((SYSDC_PATTERN)0)      //   0% fgnd 100% bgnd
#define sysDcPatForeground ((SYSDC_PATTERN)0xFFF1) // 100% fgnd   0% bgnd

#define sysDcPatNil        ((SYSDC_PATTERN)0xFFF0) //   0% fgnd   0% bgnd
#define sysDcPatRandom     ((SYSDC_PATTERN)0xFFF2) //  debugging aid

/****************************************************************************
 msgDcSetFillPat takes SYSDC_PATTERN, returns SYSDC_PATTERN
    Sets the fill pattern; returns old value.

 The fill pattern is used to draw the interior of closed
 geometric figures.

 When using this interface, see the constants sysDcPat... for
 predefined patterns. sysDcPatRandom is unique for each window.
*/
#define msgDcSetFillPat            MakeMsg(clsSysDrwCtx, 12)

/****************************************************************************
 msgDcGetLinePat takes pNull, returns SYSDC_PATTERN
    Gets the line pattern.

 *** NOT IMPLEMENTED YET ***
*/
#define msgDcGetLinePat            MakeMsg(clsSysDrwCtx, 13)

/****************************************************************************
 msgDcGetFillPat takes pNull, returns SYSDC_PATTERN
    Gets the fill pattern.

 *** NOT IMPLEMENTED YET ***
*/
#define msgDcGetFillPat            MakeMsg(clsSysDrwCtx, 14)

/****************************************************************************
 msgDcMixPattern takes P_SYSDC_MIX_PAT, returns STATUS
    Mixes a custom pattern.

 *** NOT IMPLEMENTED YET ***
*/
#define msgDcMixPattern            MakeMsg(clsSysDrwCtx, 15)

typedef struct
{
  SYSDC_PATTERN  slot;
  U8             pattern[8];

} SYSDC_MIX_PAT, * P_SYSDC_MIX_PAT;
 
/****************************************************************************
 msgDcAlignPattern takes P_XY32, returns STATUS
    Sets the pattern alignment in LUC.

 Can be used to keep pattern tiling aligned to a particular point 
 in LUC when pixels are moved (msgWinCopyRect or wsGrow* flags).
 This is most commonly used to preserve pattern alignment during 
 "scrolling" when parts of an image are copied pixels, and parts
 are newly painted pixels.

 The default alignment is 0,0 in LUC. If the image is scrolled by
 msgDcTranslate then this message may not be necessary, as the
 alignment point will move in device space too.

 If P_XY32 is pNull, default alignment is set to 0,0.
*/
#define msgDcAlignPattern          MakeMsg(clsSysDrwCtx, 16)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	LUC Space Transformations                                              *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDcUnitsMetric takes pNull, returns stsOK
    Sets input units to 0.01 mm.
*/
#define msgDcUnitsMetric           MakeMsg(clsSysDrwCtx, 17)   

/****************************************************************************
 msgDcUnitsMil takes pNull, returns stsOK
    Sets input units to 0.001 inch.
*/
#define msgDcUnitsMil              MakeMsg(clsSysDrwCtx, 18)   

/****************************************************************************
 msgDcUnitsPoints takes pNull, returns stsOK
    Sets input units to points (1/72 of an inch).
*/
#define msgDcUnitsPoints           MakeMsg(clsSysDrwCtx, 19)   

/****************************************************************************
 msgDcUnitsTwips takes pNull, returns stsOK
    Sets input units to 1/20 of a point.
*/
#define msgDcUnitsTwips            MakeMsg(clsSysDrwCtx, 20)   

/****************************************************************************
 msgDcUnitsPen takes pNull, returns stsOK
    Sets input units to pen sample units.
*/
#define msgDcUnitsPen              MakeMsg(clsSysDrwCtx, 71)   

/****************************************************************************
 msgDcUnitsLayout takes pNull, returns stsOK
    Sets input units to UI toolkit layout units.

 Note that the scale this implicitly computes is a function of the
 current system font size. However, if the system font size changes
 after this message is sent, the scale is not "reliably" reevaluated
 (because of caching it may or may not be reevaluated). Thus, you
 may need to observe theSystemPreferences. For a small performance
 cost you can just send this message prior to each operation that
 is affected by unit scaling.
*/
#define msgDcUnitsLayout           MakeMsg(clsSysDrwCtx, 85)

/****************************************************************************
 msgDcUnitsRules takes pNull, returns stsOK
    Sets input units to the 'rules' associated with the system font.

 A 'rule' is 1/20 of the thickness of a line that aesthetically matches
 the weight of the system font, as specified by the font designer.
 Typically this will be the thickness of a single underline, and so
 a rule would be 1/20 of an underline.

 Note that the scale this implicitly computes is a function of the
 current system font size. However, if the system font size changes
 after this message is sent, the scale is not "reliably" reevaluated
 (because of caching it may or may not be reevaluated). Thus, you
 may need to observe theSystemPreferences. For a small performance
 cost you can just send this message prior to each operation that
 is affected by unit scaling.
*/
#define msgDcUnitsRules            MakeMsg(clsSysDrwCtx, 3)

/****************************************************************************
 msgDcUnitsDevice takes pNull, returns stsOK
    Sets input units to device pixels.
*/
#define msgDcUnitsDevice           MakeMsg(clsSysDrwCtx, 21)   

/****************************************************************************
 msgDcUnitsWorld takes pNull, returns stsOK
    Sets input units to an arbitrary number of device pixels.

 See Also
    msgDcScaleWorld
*/
#define msgDcUnitsWorld            MakeMsg(clsSysDrwCtx, 25)   

/****************************************************************************
 msgDcUnitsOut takes MESSAGE, returns stsOK
    Sets output units produced by transformation of input units.

 Takes one of: 

    msgDcUnitsMetric
    msgDcUnitsMil   
    msgDcUnitsPoints
    msgDcUnitsTwips 
    msgDcUnitsPen
    msgDcUnitsLayout
    msgDcUnitsDevice

 In general, this message should not be used. Reverse transformations,
 from device units to other units can be made by using the 
 msgDcLUCtoLWC... messages. 

 This interface can be used to change from one logical unit system to another. 
 Since most such transformations are known in advance this is generally 
 useless; however, transformation to and from pen units to a known unit 
 system is the real purpose of this interface. For instance, pen units to 
 mils can be used to store pen units in a device independent form. Pen units 
 can thus remain device dependent.

 This interface cannot change a graphic device unit into a device independent
 unit. To do this, units IN must be the chosen target unit (e.g. points), 
 units OUT must be device, and the reverse transformation, msgDcLUCtoLWC 
 must be used.
*/
#define msgDcUnitsOut              MakeMsg(clsSysDrwCtx, 70)   

/****************************************************************************
 msgDcIdentity takes pNull, returns stsOK
    Sets LUC matrix to identity.
        
 See Also
    msgDcIdentityFont
*/
#define msgDcIdentity              MakeMsg(clsSysDrwCtx, 22)

/****************************************************************************
 msgDcRotate takes ANGLE, returns stsOK
    Rotates LUC matrix.
*/
#define msgDcRotate                MakeMsg(clsSysDrwCtx, 23)   

/****************************************************************************
 msgDcScale takes P_SCALE, returns stsOK
    Scales LUC matrix.
        
 If P_SCALE is pNull then operation is same as msgDcIdentity.
*/
#define msgDcScale                 MakeMsg(clsSysDrwCtx, 26)   

/****************************************************************************
 msgDcScaleWorld takes P_SIZE32, returns stsOK
    Creates a world scale of window width/height.

 The window width/height is divided into SIZE32 width/height units.
 If the window is not physically square on the graphic device then
 the scale will not be uniform in x and y. 
 
 This message scales the LUC matrix. Typically, this matrix must be reset
 to identity (msgDcIdentity), and this message must be resent, whenever 
 the window changes size (see msgWinSized for help).

 The DC must be bound to a window when it receives this message.
*/
#define msgDcScaleWorld            MakeMsg(clsSysDrwCtx, 61)   

/****************************************************************************
 msgDcTranslate takes P_XY32, returns stsOK
    Translates LUC matrix.
*/
#define msgDcTranslate             MakeMsg(clsSysDrwCtx, 24)   

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Coordinate Conversion                                                  *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 These messages convert coordinates from LUC to LWC, or LWC to LUC.
 The DC must be bound to a window before it receives these messages.
*/

/****************************************************************************
 msgDcLWCtoLUC_XY32 takes P_XY32, returns stsOK
    Transforms a point from window (device) space to logical space.

 The DC transforms by:
	LWC --> fractional LUC --> round to nearest integer LUC
*/
#define msgDcLWCtoLUC_XY32         MakeMsg(clsSysDrwCtx, 27)   

/****************************************************************************
 msgDcLUCtoLWC_XY32 takes P_XY32, returns stsOK
    Transforms a point from logical space to window (device) space.

 The DC transforms by:
	LUC --> fractional LWC --> round to nearest integer LWC
*/
#define msgDcLUCtoLWC_XY32         MakeMsg(clsSysDrwCtx, 39)   

/****************************************************************************
 msgDcLWCtoLUC_SIZE32 takes P_SIZE32, returns stsOK
    Transforms a size from window (device) space to logical space.

 The DC transforms by:
	LWC --> fractional LUC --> round to nearest integer LUC --> Abs()
*/
#define msgDcLWCtoLUC_SIZE32       MakeMsg(clsSysDrwCtx, 44)   

/****************************************************************************
 msgDcLUCtoLWC_SIZE32 takes P_SIZE32, returns stsOK
    Transforms a size from logical space to window (device) space.

 The DC transforms by:
	LUC --> fractional LWC --> round to nearest integer LWC --> Abs()
*/
#define msgDcLUCtoLWC_SIZE32       MakeMsg(clsSysDrwCtx, 45)   

/****************************************************************************
 msgDcLWCtoLUC_RECT32 takes P_RECT32, returns stsOK
    Transforms a rectangle from window (device) space to logical space.

 The DC transforms by:
	1) converting the rectangle's origin and opposite corner (x+w, y+h)
	into fractional LUC,
	2) rounding each point to the nearest integer coordinate, and
	3) using those coordinates to determine a rectangle (whose width and
	height may be positive or negative).
*/
#define msgDcLWCtoLUC_RECT32       MakeMsg(clsSysDrwCtx, 46)   

/****************************************************************************
 msgDcLUCtoLWC_RECT32 takes P_RECT32, returns stsOK
    Transforms a rectangle from logical space to window (device) space.

 The DC transforms by:
	1) converting the rectangle's origin and opposite corner (x+w, y+h)
	into fractional LWC,
	2) rounding each point to the nearest integer coordinate, and
	3) using those coordinates to determine a rectangle (whose width and
	height may be positive or negative).
*/
#define msgDcLUCtoLWC_RECT32       MakeMsg(clsSysDrwCtx, 47)   

/****************************************************************************
 msgDcGetMatrix takes P_MAT, returns stsOK
    Returns the LWC matrix.
        
 The DC must be bound to a window when this message is sent.
 This matrix transforms LUC to LWC (first quadrant, but device 
 dependent units) coordinates. These coordinates are suitable
 for positioning windows.
*/
#define msgDcGetMatrix             MakeMsg(clsSysDrwCtx, 40)   

/****************************************************************************
 msgDcGetMatrixLUC takes P_MAT, returns stsOK
    Returns the LUC matrix.

 This matrix combines transformations to LUC space. It is
 identity unless msgDcScale, msgDcRotate, msgDcTranslate,
 msgDcScaleWorld, or msgDcSetMatrixLUC have been previously sent.

 The default for these combinations is post-multiplication. See message
 msgDcSetPreMultiply for more on this subject.
*/
#define msgDcGetMatrixLUC          MakeMsg(clsSysDrwCtx, 87)

/****************************************************************************
 msgDcSetMatrixLUC takes P_MAT, returns stsOK
    Replaces the LUC matrix.
*/
#define msgDcSetMatrixLUC          MakeMsg(clsSysDrwCtx, 88)   

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Clipping                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDcClipRect takes P_RECT32 or pNull, returns stsOK
    Sets or clears clip rectangle.
        
 If P_RECT32 is pNull then operation is same as msgDcClipClear.
*/
#define msgDcClipRect              MakeMsg(clsSysDrwCtx, 28)

/****************************************************************************
 msgDcClipClear takes pNull, returns stsOK
    Returns clipping to entire window.
*/
#define msgDcClipClear             MakeMsg(clsSysDrwCtx, 29)   

/****************************************************************************
 msgDcClipNull takes pNull, returns stsOK
    Suspends all clipping (except to raw device).
        
 The pen handler uses this to dribble ink anywhere on-screen (in
 the pen plane(s) only).

 This interface is NOT RECOMMENDED for application software.
 It will protection fault if the caller does not have hardware privilege.
*/
#define msgDcClipNull              MakeMsg(clsSysDrwCtx, 30)   

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Hit Detection                                                          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDcHitTest takes P_RECT32 or pNull, returns stsOK
    Turns hit testing on/off.
        
 To turn hit testing on supply a rectangle to test against. To turn hit 
 testing off send pNull.

 In general, drawing messages (msgDcDraw...) will return one of the
 status values stsDcHit... if hit testing is on:

    stsDcHitOn if the line intersects the hit rectangle 
    stsDcHitIn if the rectangle is inside a closed figure
    stsDcHitOut if there was no hit

 The following drawing messages implement hit testing:

    msgDcDrawPolyline
    msgDcDrawPolygon
    msgDcDrawBezier
    msgDcDrawRectangle
    msgDcDrawEllipse
    msgDcDrawImage
*/
#define msgDcHitTest               MakeMsg(clsSysDrwCtx, 66)   

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Bounds Accumulation                                                    *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 A region is available to accumulate the bounding rectangles of drawing
 operations. msgDcAccumulateBounds(pNull) clears this region to empty and
 turns on accumulation. At this point, as in hit testing, drawing 
 operations will not be output; rather, their bounding rectangles will
 be added to the accumulation. At any time the accumulation can be 
 retrieved by using msgDcGetBounds. It can be retrieved with 
 another call to msgDcAccumulateBounds(P_RECT32); which will both retrieve 
 it, and turn off accumulation so normal drawing can resume. 
 
 Normally, bounds are accumulated for the purpose of repainting part of a 
 window. msgDcDirtyAccumulation can be used to add the accumulation 
 directly to the dirty region of the current window. This is more efficient 
 than getting the bounds rectangle and then sending msgWinDirtyRect. 

 Bounds accumulation occurs in DU4 space; while the bounds rectangle
 is returned in LUC, it always represents a rectangle in DU4. Thus,
 drawing which is clipped because of windowing, or because it falls off
 the edges of the device, is not accumulated.

 The bounds accumulation region itself is not part of the logical state,
 although the flag that determines whether drawing operations accumulate
 or draw is part of the logical state. Thus, while calls to push and pop
 the state may turn accumulation on or off, there are not separate copies
 of the accumulation region itself in the state.

 Bounds accumulation and hit testing cannot be performed at the same time. 
 If, through program error, both modes are enabled, bounds accumulation 
 will take priority.

 Neither bounds accumulation or hit testing should be used during repainting
 initiated by the window manager sending msgWinRepaint. Rather, they should 
 be used within a msgWinBeginPaint msgWinEndPaint bracket.
*/

/****************************************************************************
 msgDcAccumulateBounds takes P_RECT or pNull, returns stsOK
    Starts or stops bounds accumulation; retrieve bounds.

 If pArgs is pNull, clears current accumulation and turns accumulation 
 on. If pArgs is P_RECT32, returns accumulated bounds, and turns bounds 
 accumulation off.

 The DC computes the LUC rectangle by:
	1) transforming the DU4 origin and opposite corner to fractional LUC,
	2) rounding each point to the nearest integer coordinate, and
	3) using those coordinates to determine a rectangle (whose width and
	height are non-negative).
*/
#define msgDcAccumulateBounds      MakeMsg(clsSysDrwCtx, 81)

/****************************************************************************
 msgDcDirtyAccumulation takes P_RECT32 or pNull, returns stsOK
    Marks accumulation dirty; turns accumulation off; retrieves bounds.

 Adds current bounds accumulation directly to the dirty region of 
 the current window; then clears current bounds accumulation and 
 turns accumulation off. If pArgs is P_RECT32, returns accumulated 
 bounds as in msgDcAccumulateBounds.
*/
#define msgDcDirtyAccumulation     MakeMsg(clsSysDrwCtx, 82)

/****************************************************************************
 msgDcGetBounds takes P_RECT32, returns stsOK
    Retrieves current accumulation bounds rectangle.
    
 Does not clear accumulation or turn accumulation off.  The DC computes
 the LUC rectangle as in msgDcAccumulateBounds.
*/
#define msgDcGetBounds             MakeMsg(clsSysDrwCtx, 83)

typedef struct
{
  U16     count;    // number of points in points array
  P_XY32  points;   // pointer to array of at least 2 points

} SYSDC_POLYGON , * P_SYSDC_POLYGON ,
  SYSDC_POLYLINE, * P_SYSDC_POLYLINE;

typedef struct
{
  RECT32  bounds;
  XY32    rays[2];

} SYSDC_ARC_RAYS, * P_SYSDC_ARC_RAYS;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Open Figures                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDcDrawPolyline takes P_SYSDC_POLYLINE, returns STATUS
    Draws a line; needs at least 2 points.  Returns either hit test or stsOK.

 Return Value
	stsDcHitOn
	stsDcHitIn
	stsDcHitOut
*/
#define msgDcDrawPolyline          MakeMsg(clsSysDrwCtx,100)   

/****************************************************************************
 msgDcDrawBezier takes P_XY32 (array of 4), returns STATUS
    Draws a Bezier curve; needs exactly 4 points.
    
 Returns either hit test or stsOK.

 Return Value
	stsDcHitOn
	stsDcHitIn
	stsDcHitOut
*/
#define msgDcDrawBezier            MakeMsg(clsSysDrwCtx,104)   

/****************************************************************************
 msgDcDrawArcRays takes P_SYSDC_ARC_RAYS, returns STATUS
    Draws an arc using the two rays method.  Returns either hit test or stsOK.

 Return Value
	stsDcHitOn
	stsDcHitIn
	stsDcHitOut
*/
#define msgDcDrawArcRays           MakeMsg(clsSysDrwCtx,105)   

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Closed Figures                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDcSetPixel takes P_SYSDC_PIXEL, returns STATUS
    Sets a pixel with a value.

 If rgb is true, the color is interpreted as an RGB value; if not,
 color.all will be interpreted as a SYSDC_COLOR (a hardware palette 
 index). If rgb is used then the transparency byte must be 255 (opaque)
 or the drawing will not take place.
*/
#define msgDcSetPixel              MakeMsg(clsSysDrwCtx,108)   

/****************************************************************************
 msgDcGetPixel takes P_SYSDC_PIXEL, returns STATUS
    Gets a pixel value.

 If rgb is TRUE the color is returned as an RGB value; if not
 color.all will be a small number which should be interpreted as
 a SYSDC_COLOR (a hardware palette index). If rgb is used the
 transparency byte will always be returned as 255 (opaque).
*/
#define msgDcGetPixel              MakeMsg(clsSysDrwCtx,109)   

typedef struct
{
  BOOLEAN    rgb;
  SYSDC_RGB  color;
  XY32       xy;

} SYSDC_PIXEL, * P_SYSDC_PIXEL;

/****************************************************************************
 msgDcDrawRectangle takes P_RECT32, returns STATUS
    Draws a rectangle.  Returns either hit test or stsOK.

 Return Value
	stsDcHitOn
	stsDcHitIn
	stsDcHitOut
*/
#define msgDcDrawRectangle         MakeMsg(clsSysDrwCtx,101)   

/****************************************************************************
 msgDcDrawEllipse takes P_RECT32, returns STATUS
    Draws an ellipse.  Returns either hit test or stsOK.

 Return Value
	stsDcHitOn
	stsDcHitIn
	stsDcHitOut
*/
#define msgDcDrawEllipse           MakeMsg(clsSysDrwCtx,102)   

/****************************************************************************
 msgDcDrawPolygon takes P_SYSDC_POLYGON, returns STATUS
    Draws a polygon.  Returns either hit test or stsOK.

 Return Value
	stsDcHitOn
	stsDcHitIn
	stsDcHitOut
*/
#define msgDcDrawPolygon           MakeMsg(clsSysDrwCtx,103)   

/****************************************************************************
 msgDcDrawSectorRays takes P_SYSDC_ARC_RAYS, returns STATUS
    Draws a sector (pie wedge) using the two rays method.

 Returns either hit test or stsOK.

 Return Value
	stsDcHitOn
	stsDcHitIn
	stsDcHitOut
*/
#define msgDcDrawSectorRays        MakeMsg(clsSysDrwCtx,106)   

/****************************************************************************
 msgDcDrawChordRays takes P_SYSDC_ARC_RAYS, returns STATUS
    Draws a chord using the two rays method.  Returns either hit test or stsOK.

 Return Value
	stsDcHitOn
	stsDcHitIn
	stsDcHitOut
*/
#define msgDcDrawChordRays         MakeMsg(clsSysDrwCtx,107)   

/****************************************************************************
 msgDcFillWindow takes pNull, returns stsOK
    Frames window with a line and fills the window.

 Draws a rectangle exactly the size of the window. All line, fill
 and color attributes apply.

 When drawing a rectangle, the first pixel of line thickness is painted
 "inside" the rectangle, the second "outside", and it alternates from
 there. Therefore, lines > 1 pixel thick will have 1/2 their thickness
 fall outside the window when using this message. If that drawing is
 clipped (as it normally is) the line will appear 1/2 as thick as one
 would expect.
*/
#define msgDcFillWindow            MakeMsg(clsSysDrwCtx, 33)   

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Sampled Image Processing                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDcDrawImage takes P_SYSDC_IMAGE_INFO, returns STATUS
    Draws an image from sampled image data. The image will be scaled,
    rotated, translated, according to the current state.

 This message is similar to the PostScript image operator. Sample data, 
 in the form of numbers ranging from 0..max are interpreted as grey 
 values. 0 is black and max is white. The value of max is determined 
 by the size of the input numbers, which can be 1, 2, 4 or 8 bits.

 Because the sample data may be large, in a file, or incrementally 
 decompressed, this message can work with a callback strategy. The
 callback can be either a function (flag sysDcImageCallBack), or an object 
 (flag sysDcImageCallObject) to which msgDcGetSrcRow is sent. In both 
 cases the argument is the same pointer to a SYSDC_IMAGE_INFO that is the 
 argument to msgDcDrawImage itself. To support client context during 
 the callback, the field pClientData is provided, for the callback to
 use as necessary.

 The source sample data width and height is described by srcSize. The
 rectangle at the destination, which will be filled by the image, is
 dstRect; or optionally, the flag sysDcImageFillWindow can be used
 to fill the entire window.

 During the callback, pBuffer will point to a buffer that needs to be 
 filled with srcSize.w samples. If pBuffer is pNull, it means the operator 
 is skipping a row (because of clipping perhaps). Thus, no samples need 
 to be provided, but the context must be "advanced" to skip the row. 
 If anything goes wrong, the callback can return FALSE (for a function),
 or a bad status code (for an object) to terminate the drawing.
 
 If callback is not used at all, then pBuffer should be set by the caller
 to point to all of the sample data at the outset.

 The result of the drawing is that dstRect is filled with an image.
 Since dstRect is in LUC space, its size and location is the same as
 if it were drawn with msgDcDrawRectangle.

 Before using this interface to display "tiff" images, investigate
 clsTiff (tiff.h) for a much higher level service.

 Return Value
	stsDcHitOn
	stsDcHitIn
	stsDcHitOut
*/
#define msgDcDrawImage             MakeMsg(clsSysDrwCtx, 48)   
#define msgDcGetSrcRow             MakeMsg(clsSysDrwCtx, 49)   

Enum16(SYSDC_IMAGE_FLAGS)
{
  sysDcImageNoFilter      =  0,       // fast but poor fidelity
  sysDcImageLoFilter      =  flag0,   // use this for most image data
  sysDcImageHiFilter      =  flag1,   // use this for color image data

  sysDcImageRunLength     =  flag2,          // run length encoded
  sysDcImage1BPS          =  flag3,          // 1 bit  per sample
  sysDcImage2BPS          = (flag2|flag3),   // 2 bits per sample 
  sysDcImage4BPS          =  flag4,          // 4 bits per sample
  sysDcImage8BPS          =  0,              // 8 bits per sample

  sysDcImageCallBack      =  flag8,   // callBack is a P_SYSDC_GETROW function
  sysDcImageCallObject    =  flag9,   // callBack is a OBJECT

  sysDcImageFillWindow    =  flag10,  // dstRect not provided

  sysDcImagePolarityFalse =  flag11   //      paint '0' w/ background color
                                      // else paint '1' w/ foreground color
};                                    

typedef struct SYSDC_IMAGE_INFO * P_SYSDC_IMAGE_INFO;
typedef BOOLEAN FunctionPtr(P_SYSDC_GETROW)(P_SYSDC_IMAGE_INFO pCtx);

typedef struct SYSDC_IMAGE_INFO
{
  RECT32             dstRect;      // destination size and position
  SIZE16             srcSize;      // # of source samples
  SYSDC_IMAGE_FLAGS  flags;
  union
  {
    P_SYSDC_GETROW   function;
    OBJECT           object;
  }                  callBack;

  P_UNKNOWN          pBuffer;
  P_UNKNOWN          pClientData;

  P_UNKNOWN          reserved[3];

} SYSDC_IMAGE_INFO;

/****************************************************************************
 msgDcDrawImageMask takes P_SYSDC_IMAGE_INFO, returns STATUS
    Draws a mask from sampled image data. Similar to msgDcDrawImage.

 This message is similar to the PostScript imagemask operator and 
 msgDcDrawImage. The input parameters are the same as for msgDcDrawImage
 with the addition of one flag, sysDcImagePolarityFalse, which would
 normally not be set (TRUE). However, the results of this message are
 visually different than msgDcDrawImage.

 msgDcDrawImage reduces the input data to grey values and paints an 
 opaque parallelogram. The values of the current foreground and
 background colors have no effect on the behavior of msgDcDrawImage.

 msgDcDrawImageMask reduces the input data to the values '0' and '1'. 
 The default behavior is for the '1' values to be painted with the 
 current foreground color; the '0' values are not painted at all.

 This behavior can be reversed by setting the sysDcImagePolarityFalse
 flag. In this case the '0' values are painted with the current
 background color and the '1' values are not painted.

 Return Value
	stsDcHitOn
	stsDcHitIn
	stsDcHitOut
*/
#define msgDcDrawImageMask         MakeMsg(clsSysDrwCtx, 97)   

/****************************************************************************
 msgDcCacheImage takes P_SYSDC_CACHE_IMAGE, returns STATUS
    Passes back a cached image in pCache, given a sampled image and
    an optional mask.

 A "cached image" is a segment of memory (pCache) that contains the
 device-dependent (pixelmap) representation of a sampled image (see
 msgDcDrawImage), and optionally a mask. 
 
 This operator is intended to be used for cursors and icons. 
 It currently does not work on printer devices.

 Once cached, the image can be drawn (with hotspot adjustment) using
 msgDcCopyImage.

 Because of its device dependent representation, a cached image becomes
 obsolete when the device rotation changes (landscape vs. portrait). Thus, 
 you may need to observe theSystemPreferences and rebuild the cache
 when appropriate. When you are finished with the cached image you should
 free it with OSHeapBlockFree.
*/
#define msgDcCacheImage            MakeMsg(clsSysDrwCtx, 91)   

typedef struct
{
  SYSDC_IMAGE_INFO   image[2];       // in  = [0] is image, [1] is mask
  BOOLEAN            hasMask;        // in  = if this is true
  XY16               hotSpot;        // in
  P_UNKNOWN          pCache;         // out = cache (segment)

} SYSDC_CACHE_IMAGE, * P_SYSDC_CACHE_IMAGE;

/****************************************************************************
 msgDcCopyImage takes P_SYSDC_COPY_IMAGE, returns STATUS
    Copies a cached image to the bound window.

 The image is copied, such that the hotspot aligns on xy.
*/
#define msgDcCopyImage             MakeMsg(clsSysDrwCtx, 92)

typedef struct
{
  XY32               xy;             // in = destination location
  P_UNKNOWN          pCache;         // in

} SYSDC_COPY_IMAGE, * P_SYSDC_COPY_IMAGE;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Fonts                                                                  *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Some of the data structures used in the font interface are declared in
 sysfont.h.

 All font metric information is currently computed in LUC space. However,
 because all the relevant numbers, except for scaling, are integers,
 significant round-off error can occur. For instance, at 10 or 12 points,
 a small feature, like x-height, will be a very small number. If LUC is
 relatively coarse, the error may be significant. The same holds true for the
 quality of inter-character spacing. Each character within a string of text
 output is positioned in LUC space, and the positioning will be no more
 accurate than the granularity of LUC. In general, the use of TWIPS units, or
 even finer units, is recommended if high quality text at small point
 sizes is required. Note that this may change in the future--read on.

 While this approach produces less than perfect results on screen, it
 does have the benefit of maintaining very close correspondence between
 screen and printer; such that the same code can be used for both with
 no significant variance. In general, each character will be positioned
 to within one LUC unit or one device pixel (whichever is larger), of
 accuracy.

 In future versions, the text measurement messages may change so that they
 advance character by character in an internal coordinate space that doesn't
 match LUC.  This would allow accurate intercharacter spacing regardless
 of the granularity of LUC.
*/

/****************************************************************************
 SysDcFontId returns U16
    Takes a 4 byte string font description and returns a 16-bit font id
    number.
*/
U16 EXPORTED SysDcFontId(P_CHAR     // In: a string like "HE55"
                        );

/****************************************************************************
 SysDcFontString returns void
    Takes a 16-bit font id number and passes back a 4 char string.

 The string buffer should be at least 5 bytes long.
*/
void EXPORTED SysDcFontString(U16,      // In: a font id number
                              P_CHAR  // Out: a string like "HE55"
                             );

/****************************************************************************
 msgDcOpenFont takes P_SYSDC_FONT_SPEC or pNull, returns stsOK
    Opens a font.
        
 Specifying pNull will open default font.
*/
#define msgDcOpenFont              MakeMsg(clsSysDrwCtx, 53)

/****************************************************************************
 msgDcScaleFont takes P_SCALE or pNull, returns stsOK
    Scales font matrix.

 If argument is pNull then behavior is same as msgDcIdentityFont.
 The default size of a newly opened font is 1 unit (LUC).
 Use this message to scale to the desired size.

 Note that this scaling is cumulative (multiplicative). A scale 
 of 10,10 followed by a scale of 12,12 will result in a scale of 
 120,120. When "switching to absolute sizes" a msgDcIdentityFont 
 will usually be needed.

 Note also that font scale is affected by the overall scale
 established by the msgDcUnits... messages, and msgDcScale.
*/
#define msgDcScaleFont             MakeMsg(clsSysDrwCtx, 54)

/****************************************************************************
 msgDcIdentityFont takes pNull, returns stsOK
    Sets font matrix scale to default of 1 unit (LUC).
*/
#define msgDcIdentityFont          MakeMsg(clsSysDrwCtx, 72)

/****************************************************************************
 msgDcDrawText takes P_SYSDC_TEXT_OUTPUT, returns stsOK
    Draws text in the current font.
*/
#define msgDcDrawText              MakeMsg(clsSysDrwCtx, 55)

/****************************************************************************
 msgDcMeasureText takes P_SYSDC_TEXT_OUTPUT, returns stsOK
    Computes size of text and advances pArgs->cp accordingly.

 Measuring stops when stop is exceeded. Stop will normally be a 
 "right margin". This is used to measure out lines of text from a 
 large buffer.  Upon return lenText will be the number of characters
 that "fit". This information can then be used to break and justify lines.

 To simply measure an entire string, set cp.x = 0 and stop = maxS32; 
 upon return cp.x will be the length of the string in LUC. 
 
 During measuring, other parameters, like spaceExtra and otherExtra are 
 significant.
*/
#define msgDcMeasureText           MakeMsg(clsSysDrwCtx, 57)

/****************************************************************************
 msgDcDrawTextRun takes P_SYSDC_TEXT_OUTPUT, returns stsOK
    Like msgDcDrawText, except run spacing applies.

 Run spacing is important when spaceExtra and otherExtra contain
 non-zero values; especially when underlining. For instance, if
 otherExtra is 10, then 10 units will be added after the last
 character when using run spacing. It would not be added when using
 the normal DrawText message. This affects the cp.x value returned,
 and is visually significant when underlining or strikethrough
 are performed (the 10 units would have the lines or not). It will
 also affect the result when centering or right-justifying text.
*/
#define msgDcDrawTextRun           MakeMsg(clsSysDrwCtx, 73)

/****************************************************************************
 msgDcMeasureTextRun takes P_SYSDC_TEXT_OUTPUT, returns stsOK
    Like msgDcMeasureText, except run spacing applies.

 See comments about "run spacing" under msgDcDrawTextRun.
*/
#define msgDcMeasureTextRun        MakeMsg(clsSysDrwCtx, 74)

/****************************************************************************
 msgDcDrawTextDebug takes P_SYSDC_TEXT_OUTPUT, returns stsOK
    Like msgDcDrawText, except text is drawn with debugging 
    lines around each char.

 This function may not work unless the debugging version of win.dll
 is being used.
*/
#define msgDcDrawTextDebug         MakeMsg(clsSysDrwCtx, 56)

/****************************************************************************
 msgDcPreloadText takes P_SYSDC_TEXT_OUTPUT, returns stsOK
    Preloads pText into cache.

 If pArgs is pNull or pArgs->pText is pNull a default set 
 is preloaded.

 This message causes the characters to be rasterized into the
 font cache so that during a subsequent msgDcDrawText there are
 no hesitations during a cache miss. This is not normally 
 necessary, but might be useful in a "slide show" application.
*/
#define msgDcPreloadText           MakeMsg(clsSysDrwCtx, 58)

/****************************************************************************
 msgDcGetCharMetrics takes P_SYSDC_CHAR_METRICS, returns stsOK
    Gets char metrics information for a string.

 These character metrics are more precise in some ways than those
 returned by msgDcGetFontMetrics. For instance, the width of a character 
 is a purely logical value. The character image may extend past its 
 width to the right, and may extend to the left past its "left edge". 
 Similarly, some characters will extend above the "ascender" line or 
 below the "descender" lines (which are just imaginary lines that guide 
 the letterforms in general).

 For each character in the string, the information returned is the
 minimum and maximum x and y coordinates found in that glyph, as if
 the glyph were drawn at 0,0. There are no "string semantics" to
 the "string" (x is not accumulating left to right); rather, this 
 is similar to a "width table" except values for a specific string 
 only are returned.

 See the caveat below for msgDcGetFontWidths.
*/
#define msgDcGetCharMetrics        MakeMsg(clsSysDrwCtx, 84)

/****************************************************************************
 msgDcGetFontMetrics takes P_SYSDC_FONT_METRICS, returns stsOK
    Gets the font metrics for the current font.
*/
#define msgDcGetFontMetrics        MakeMsg(clsSysDrwCtx, 59)

/****************************************************************************
 msgDcGetFontWidths takes P_SYSDC_FONT_WIDTHS, returns stsOK
    Gets the font width table of the current font.

 This width table is an array of 255 COORD16 values. Try to use the 
 msgDcMeasureText interface instead; as width tables become less 
 practical as character sets get larger and larger (e.g., Kanji).

 Another important reason to use msgDcMeasureText instead is that the
 measureText/drawText interfaces may change in the future to advance character
 by character in an internal coordinate space that doesn't match LUC.  This
 would allow accurate intercharacter spacing regardless of the granularity
 of LUC.  In short, we do not guarantee that merely adding up widths obtained
 by msgDcGetFontWidths would match the results of using msgDcMeasureText.
 msgDcMeasureText always represents the correct behavior, while
 msgDcGetFontWidths should be thought of as an approximation.
*/
#define msgDcGetFontWidths         MakeMsg(clsSysDrwCtx, 60)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Special Messages                                                       *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDcDrawPageTurn takes P_SYSDC_PAGE_TURN, returns stsOK
    Draws a page turn effect over the bound window.

*/
#define msgDcDrawPageTurn          MakeMsg(clsSysDrwCtx, 86) 

typedef struct
{
  P_RECT32  pBounds;      // may be pNull for entire window
  U16       fxNo,         // must be 0
            iterations;
  BOOLEAN   fxFwd,
            landscape;

} SYSDC_PAGE_TURN, * P_SYSDC_PAGE_TURN;

/****************************************************************************
 msgDcCopyPixels takes P_SYSDC_PIXELS, returns stsOK
    Copies pixels from srcWindow to the bound window.

 The rectangle pBounds on srcWindow is copied to the destination
 (bound) window at location xy. If dstDirty is TRUE, "dirty" pixels 
 from the srcWindow cause the corresponding pixels on the destination
 window to be marked dirty (however, the dirty pixels ARE copied anyway).

 The srcWindow must be on an "image device". See clsImgDev.

 Return Value
    stsTruncatedData:   source rectangle not entirely on the window device;
                        some dest pixels not affected.
*/
#define msgDcCopyPixels            MakeMsg(clsSysDrwCtx, 89)

typedef struct
{
  OBJECT    srcWindow;    // in=on a clsImgDev
  P_RECT32  pBounds;      // in=may be pNull for entire window
  XY32      xy;           // in=destination location
  BOOLEAN   dstDirty;     // in=

} SYSDC_PIXELS, * P_SYSDC_PIXELS;

/****************************************************************************
 msgDcDrawPixels takes P_SYSDC_PIXELS, returns stsOK
    Draws foreground and background colors in the bound window's pixels
    using srcWindow's pixel values as a stencil.

 Like msgDcCopyPixels except the source clsImgDev window must be
 only 1 plane and the dstDirty processing is not performed. 
 
 '1' pixels from the source are drawn with the foreground color, 
 and '0' pixels with the background color.

 Return Value
    stsTruncatedData:   source rectangle not entirely on the window device;
                        some dest pixels not affected.
*/
#define msgDcDrawPixels            MakeMsg(clsSysDrwCtx, 90)

/****************************************************************************
 msgDcScreenShot takes P_SYSDC_SCREEN_SHOT, returns stsOK
    Captures a screen image to a "tiff" file.

 pBounds can be a rectangle that is off the window and those pixels
 will be captured too (actually, wraparound will occur); no clipping is 
 implied by the window, only the relative positioning of pBounds. If 
 pBounds is pNull the whole window will be captured. If LUC are rotated 
 non-modulo 90 degrees an upright rectangle bounding pBounds will be 
 captured.

 If you are just capturing screen shots for documentation, try using 
 the SShot utility application first.
*/
#define msgDcScreenShot            MakeMsg(clsSysDrwCtx, 67) 

typedef struct
{
  P_RECT32  pBounds;     
  P_CHAR	pFileName;

} SYSDC_SCREEN_SHOT, * P_SYSDC_SCREEN_SHOT;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from other classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDrwCtxSetWindow takes WIN, returns WIN
    Binds a window to the receiver and returns the previously bound window.

 All output through the DC will now appear on this window.
 A DC must be bound to a window before most messages will work.
*/


/****************************************************************************
 msgDrwCtxGetWindow takes pNull, returns WIN
    Gets the window to which the drawing context is bound.
*/


/****************************************************************************
 msgWinDirtyRect takes P_RECT32 or pNull, returns STATUS
    Marks all or part of a window dirty.

 If P_ARGS is not null, the DC will transform the rectangle into LWC and pass
 the message on to the DC's bound window.  The DC transforms the rectangle by:
	1) converting the rectangle's origin and opposite corner (x+w, y+h)
	into fractional LWC,
	2) rounding each point to the nearest integer coordinate, and
	3) using those coordinates to determine a rectangle whose width and
	height are non-negative.

 If the P_ARGS is null, the DC will just pass the message on to the DC's
 bound window.
*/


/****************************************************************************
 msgWinBeginPaint takes P_RECT32 or pNull, returns STATUS
    Sets up window for painting on its visible region.

 The P_ARGS is handled the same as in msgWinDirtyRect.
*/


/****************************************************************************
 msgWinBeginRepaint takes P_RECT32 or pNull, returns STATUS
    Sets up window for painting on "dirty" region. 
    
 The DC will pass the message on to the DC's bound window and then, if P_ARGS
 is not null, transform the out parameter rectangle from LWC to LUC.  The DC
 transforms the rectangle by:
	1) converting the rectangle's origin and opposite corner (x+w, y+h)
	into fractional LUC,
	2) rounding each point to the nearest integer coordinate,
	3) using those coordinates to determine a rectangle whose width and
	height are non-negative, and finally
	4) ensuring that the resulting width and height are at least 1.
*/


/****************************************************************************
 msgWinBeginPaint takes P_RECT32 or pNull, returns STATUS
    Sets up window for painting on its visible region.

 The P_ARGS is handled the same as in msgWinBeginRepaint.
*/


/****************************************************************************
 msgWinDelta takes P_WIN_METRICS, returns STATUS
    Moves and/or resizes a window. pArgs->bounds should be the newly
    desired bounds (size AND position).

 The DC will transform pArgs->bounds into LWC and pass the message on to
 the DC's bound window.  The DC transforms:
	the  in parameter bounds in the same manner as msgDcLUCtoLWC_RECT32, and
	the out parameter bounds in the same manner as msgDcLWCtoLUC_RECT32.
*/


/****************************************************************************
 msgWinTransformBounds takes P_WIN_METRICS, returns STATUS
    Transforms bounds from receiver's to another window's LWC.

 The P_ARGS is handled the same as in msgWinDelta.
*/


/****************************************************************************
 msgWinHitDetect takes P_WIN_METRICS, returns STATUS
    Locates the window "under" a point.

 The DC will pass the message on to the DC's bound window.  The bounds passed
 along to the window will be a copy of pArgs->bounds that the DC has
 transformed into LWC as in msgDcLUCtoLWC_RECT32.
*/

/****************************************************************************
 msgWinGetMetrics takes P_WIN_METRICS, returns STATUS
    Gets full window metrics.

 The DC will pass the message on to the DC's bound window, and then return
 a pArgs->bounds that is transformed as in msgDcLWCtoLUC_RECT32.
*/


/****************************************************************************
 msgWinCopyRect takes P_WIN_COPY_RECT, returns STATUS
    Copies pixels within a window.

 The DC will first transform the pArgs->srcRect from LUC to LWC as in
 msgWinDelta.  The DC will then transform pArgs->xy:
	if wsCopyRelative is set, as in msgDcLUCtoLWC_SIZE32
	if wsCopyRelative is cleared, as in msgDcLUCtoLWC_XY32.

 The DC will then pass the message on to the DC's bound window.
*/


/*
 Drawing contexts respond to every other clsWin message by just forwarding
 the message on to its bound window.  The P_ARGS are not touched by the DC.
*/


/****************************************************************************
 msgWinDevBindPixelmap takes P_WIN_DEV_PIXELMAP, returns STATUS
    Binds window device to a pixelmap.

 The DC will pass the message on to pArgs->device.  The pArgs->size
 passed along to the device will be a copy of pArgs->size that the DC has
 transformed into LWC by:
	1) setting up a local rectangle of x=0, y=0, w=pArgs->size.w,
	h=pArgs->size.h
	2) transforming this rectangle into LWC as in msgWinDirtyRect
	(using the transformation matrix of the DC), and
	3) setting the copied size to the resulting rectangle's size.

 The DC will also change the pArgs->device passed along to be the device
 on which the DC's bound window was created.
*/


/****************************************************************************
 msgWinDevSizePixelmap takes P_WIN_DEV_PIXELMAP, returns STATUS
    Computes the amount of memory needed for a single plane.

 The P_ARGS is handled the same as in msgWinDevBindPixelmap.
*/


/*
 Drawing contexts respond to every other clsWinDev message by just forwarding
 the message on to its bound window.  Note that clsWin's response to
 clsWinDev messages is to just call ancestor (except for messages sent to
 the root window on the device, which get passed on to the device).
 
 The P_ARGS are not touched by the DC.
*/


#endif // SYSGRAF_INCLUDED

//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Private or Privileged Interfaces                                       *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifdef  SYSGRAF_PRIVATE
#ifndef SYSGRAF_PRIVATE_INCLUDED
#define SYSGRAF_PRIVATE_INCLUDED

/****************************************************************************
 msgDcSetPenWindow takes WIN, returns stsOK
    Similar to msgDcSetWindow.
        
 Does a SetWindow and synchronizes state too. Used only to dribble 
 pen ink on the pen plane(s).

 This interface is NOT RECOMMENDED for application software.
 It will protection fault if the caller does not have hardware privilege.
*/
#define msgDcSetPenWindow          MakeMsg(clsSysDrwCtx, 68) 

/****************************************************************************
 msgDcDrawPenStroke takes P_SYSDC_POLYLINE, returns stsOK
    Similar to msgDcDrawPolyline.
        
 Draws a polyline without locking the display. Used only to dribble 
 pen ink on the pen plane(s).

 This interface is NOT RECOMMENDED for application software.
 It will protection fault if the caller does not have hardware privilege.
*/
#define msgDcDrawPenStroke         MakeMsg(clsSysDrwCtx, 69) 

/****************************************************************************
 msgDcSetCursor takes P_UNKNOWN or pNull, returns stsOK
    Sets the cursor to be the specified cached image; if pNull, the cursor
    is turned off.
	
 The DC must be bound to a window on theScreen.
 Does not affect the cursor position.

 This interface is NOT RECOMMENDED for application software.
 It will protection fault if the caller does not have hardware privilege.
*/
#define msgDcSetCursor             MakeMsg(clsSysDrwCtx, 93) 

/****************************************************************************
 msgDcSetCursorIsBusyUI takes BOOLEAN, returns stsOK
    Indicates that the cursor represents the busy UI.
  
 When the cursor is displayed using msgDcSetCursor, and the cursor is
 the busy UI, the first attempt to paint under the cursor will result
 in msgBusyDisplay(false) to theBusyManager.
 
 This interface is NOT RECOMMENDED for application software.
 It will protection fault if the caller does not have hardware privilege.
*/
#define msgDcCursorIsBusyUI             MakeMsg(clsSysDrwCtx, 98) 

/****************************************************************************
 msgDcReplaceCursor takes P_UNKNOWN, returns stsOK
    Like msgDcSetCursor except the current cursor is replaced with the 
    specified cached image. 
    
 The cursor is not turned off and then back on. The replacement must
 have an identical size, mask and hotspot for this operation to work 
 properly. Also, the DC must NOT be in the process of updating the 
 bound window.

 This interface is NOT RECOMMENDED for application software.
 It will protection fault if the caller does not have hardware privilege.
*/
#define msgDcReplaceCursor         MakeMsg(clsSysDrwCtx, 95) 

/****************************************************************************
 msgDcSetCursorPosition takes P_XY32 or pNull, returns stsOK
    Moves the cursor to x,y (in LUC); if pNull the cursor is turned off.

 This interface is NOT RECOMMENDED for application software.
 It will protection fault if the caller does not have hardware privilege.
*/
#define msgDcSetCursorPosition     MakeMsg(clsSysDrwCtx, 94) 

/****************************************************************************
 msgDcMeasureTextX takes P_SYSDC_TEXT_OUTPUTX, returns stsOK
    Computes size of text and advances pArgs->cp accordingly.

 Similar to msgDcMeasureText, with more out parameters.
 This message is temporary, and will be removed in PenPoint 2.0.
*/
#define msgDcMeasureTextX          MakeMsg(clsSysDrwCtx, 99)

/*
 WARNING: This struct is temporary, and will be removed in PenPoint 2.0.
*/
typedef struct
{
  U16        alignChr;    // use sysDcAlignChr...                
  U16        underline;   // use 0,1, or 2                
  U16        strikeout;   // use 0 or 1                
  P_CHAR     pText;                   
  U16        lenText;     // in (and out for measure)
  XY32       cp;          // in and out, where to place string
  COORD32    stop;        // used by msgDcMeasureText
  U16        spaceChar;   // code for space, usually 32                
  COORD16    spaceExtra,  // added to width of space                
             otherExtra;  // added to width of every char                
  XY32       minExtent,   // Out: minimum visual coord
             maxExtent;   // Out: maximum visual coord

} SYSDC_TEXT_OUTPUTX, * P_SYSDC_TEXT_OUTPUTX;

typedef struct 
{
  RES_ID          resId;   // In: resId for bitmap
  PIX_DEV_ORIENT  orient;  // In: old, Out: new
  P_UNKNOWN       pCache;  // In: old, Out: new
  SIZE32          size;    // Out: size of image (device units)

} SYSDC_IMAGE_RESOURCE, * P_SYSDC_IMAGE_RESOURCE;

STATUS     EXPORTED    SysDcReleaseImageResource(P_SYSDC_IMAGE_RESOURCE);
STATUS     EXPORTED    SysDcAcquireImageResource(WIN,P_SYSDC_IMAGE_RESOURCE);
STATUS     EXPORTED0   SysDcFlushFontCache(U16);
STATUS     EXPORTED0   SysDcPruneFontList(U16,OBJECT);
STATUS     EXPORTED0   SysDcGetFontInfo(U16,P_SYSDC_FONT_METRICS);
BOOLEAN    EXPORTED0   SysDcLineIntersectRect(P_XY32,P_RECT32);
void       EXPORTED0   SysDcCursorOff(P_UNKNOWN);
BOOLEAN    EXPORTED0   SysDcCursorOn(void);
P_UNKNOWN  RINGCHELPER SysDcGetCursor(void);

#endif // SYSGRAF_PRIVATE_INCLUDED
#endif // SYSGRAF_PRIVATE

//REFGEN ENDIGNORE
