/****************************************************************************
 File: qhelp.h

 Copyright 1992, GO Corporation, All Rights Reserved

 $Revision:   1.8  $
     $Date:   25 Mar 1992 11:40:12  $

 This file contains the API definition for clsQuickHelp.
 clsQuickHelp inherits from clsFrame.
 clsQuickHelp provides an interface to the Quick Help Server.

 theQuickHelp is a well-known instance of clsQuickHelp.

 theQuickHelp provides system wide quick help, and is the only
 instance of clsQuickHelp in the system, built at boot time.
 Clients should not create instances of this object, nor
 should they subclass this object. This file defines an interface to
 display quick help text in the standard quick help window. Programmers
 should rarely have to call ANY of the functions in this file, as default
 calling of quick help is provided by default in clsGWin (see gwin.h).
 However, some applications may need to invoke quick help, or change the
 quick help text, hence the public message to open quick help,
 and to show a quick help screen.

 A quick help resource consists of a string array resource with each
 array item mapping to a single quick help panel.  This resource is
 identified by creating a List resource ID  from the administered
 portion of the quick help ID (MakeListResId(helpID, resGrpQhelp, 0))
 and the quick help group.  The TAG portion of the quick help ID is
 used to index into the string array.  Each quick help strings will
 have two "parts".  The first part will be the title and the second
 part will be the text.  The title will be separated from the text by
 including two vertical line characters (||) following the title which
 will NOT be printed.

 These resources, which are defined below, are put into the application
 resource files and displayed using msgQuickHelpShow, which takes the
 resource ID.  As mentioned, gWin defines a default behavior for
 calling the object with this message.  All application typically need
 to do is provide their gWin objects (or subclasses) with helpId
 resources.

 Quick help for an object is generally displayed in one of two ways.
 The first is when an object decides to display quick help for itself.
 An example is gWin's response to the '?' gesture. gWin posts theQuickHelp
 msgQuickHelpShow, which opens the quick help window and displays quick
 help for the object. The second is when theQuickHelp window is open, and
 the system is in quick help mode. When the user taps on objects on the
 screen, the object is sent msgQuickHelpHelpShow. The object will respond
 by posting msgQuickHelpShow back to theQuickHelp. When the quick help
 window is dismissed, by hitting closed or envoke help notebook, the
 object that received msgQuickHelpHelpShow will receive msgQuickHelpHelpDone.
 This message will also be sent when tapping on successive objects
 while in quick help mode. It will not be sent when quick help was initially
 brought up directly from the object when it posted msgQuickHelpShow (such as
 the gWin response to the '?' gesture.
****************************************************************************/
#ifndef QHELP_INCLUDED
#define QHELP_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef RESFILE_INCLUDED
#include <resfile.h>
#endif

/**** Debugging Flags ****/
/*
 Quick Help uses the debugging flag set 'q'.  Defined flags are:
 	0001:	General quick help debugging information
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Types and Constants						               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * These tags are used for defining three quick help screens: 1) the
 * quick help intro screen that gives directions on quick help,  2) the
 * "No help available" screen, and 3) the help not found screen.
 */
#define hlpQuickHelpSignOn				MakeTag(clsQuickHelp, 1)
#define hlpQuickHelpNoHelp				MakeTag(clsQuickHelp, 2)
#define hlpQuickHelpNotFound			MakeTag(clsQuickHelp, 3)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages											   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgQuickHelpShow			takes P_QUICK_DATA, returns STATUS
	Displays the Quick Help associated with the resource ID.

 Gets the quick help resource from either the system resource files or the
 application specific resource files.  If the quick help resource can't
 be found, will display the "Quick help not found" message in the
 quick help screen.  Typically called from gWin in order to display
 the help screen for a help gesture. Would take the gWin helpId and
 the application uid.  Needs the application object in order to reference
 the resource files of the application to find application specific help IDs.
 Typically not called directly by applications, but called indirectly
 through gWin inheritence.  Will call msgQuickHelpOpen to open the
 quick help window as necessary.

 Typically called by objects in response to a ? gesture, or in response to
 msgQuickHelpHelpShow.

 See Also
	gwin.h
	msgQuickHelpOpen
	msgQuickHelpHelpShow
*/
#define msgQuickHelpShow			MakeMsg(clsQuickHelp, 1)
typedef struct QUICK_DATA {
	U32 helpId;			// Help ID of the screen to show
	OBJECT appUID;		// UID of the application.  Used to find resources
						// of application specific help IDs.
	U32 reserved;		// Reserved for future use
} QUICK_DATA, *P_QUICK_DATA;

/****************************************************************************
 msgQuickHelpHelpShow	takes P_XY32, returns STATUS
	Sent to a window to display a quick help request.

 Sent from theQuickHelp to a window when it is required to display its
 quick help. Typically the window will respond by posting
 msgQuickHelpShow. Sent as the user taps on various windows while quick
 help is being displayed.

 See Also
	msgQuickHelpHelpDone
*/
#define msgQuickHelpHelpShow		MakeMsg(clsQuickHelp, 7)

/****************************************************************************
 msgQuickHelpHelpDone	takes OBJECT, returns STATUS
	Sent to a window when quick help is no longer displayed.

 Sent to the last object asked to display quick help via
 msgQuickHelpHelpShow when help is no longer needed on said object.
 Can be sent because the user tapped somewhere else and a new object
 is about to be sent msgQuickHelpHelpShow, quick help has been terminated
 by the user, or the help notebook has been entered. Takes the new object
 receiving a msgQuickHelpHelpShow if because the user tapped elsewhere, or
 null if quick help is being terminated or going to the help notebook.
 Note that this message is only sent to object which previously received
 msgQuickHelpHelpShow, and not those objects generating a help request
 by posting msgQuickHelpShow directly.

 See Also
	msgQuickHelpHelpShow
*/
#define msgQuickHelpHelpDone		MakeMsg(clsQuickHelp, 8)

/****************************************************************************
 msgQuickHelpOpen			takes nothing, returns STATUS
	Forces the Quick Help window to appear.

 Opens the quick help window on the screen.  If the quick help window
 is already on the screen, will simply return stsOK. The quick help window
 is a modal filter that will grab all input till closed via msgQuickHelpClose.
 Self sent to when msgQuickHelpShow is posted.  Also sent from the
 help notebook icon to invoke quick help.
*/
#define msgQuickHelpOpen			MakeMsg(clsQuickHelp, 2)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Notification Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgQuickHelpOpened			takes nothing, returns STATUS
	category: observer notification
	Indicates that the quick help window has been opened.

 Sent to observers of the quick help that the quick help window
 has been opened.
*/
#define msgQuickHelpOpened			MakeMsg(clsQuickHelp, 128)

/****************************************************************************
 msgQuickHelpClosed			takes nothing, returns STATUS
	category: observer notification
	Indicates that the quick help window has been closed.

 Sent to observers of theQuickHelp to indicate that the quick help
 window has been closed.
*/
#define msgQuickHelpClosed			MakeMsg(clsQuickHelp, 129)

/****************************************************************************
 msgQuickHelpInvokedNB		takes nothing, returns STATUS
	category: observer notification
	Indicates that the notebook associated with quick help should be open.

 Sent to observers when msgQuickHelpInvokeNB is recieved.  The help note
 book is an observer, and will bring itself up when this message is
 recieved.
*/
#define msgQuickHelpInvokedNB		MakeMsg(clsQuickHelp, 130)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Private Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * clsWin notifies observers when it receives msgQuickHelpHelpDone.
 * Observers can then return stsQuickHelpRemoveObserver to have clsWin
 * remove them from the window's observer list.  theQuickHelp will ignore
 * this status.
 */
#define	 stsQuickHelpRemoveObserver			MakeWarning(clsQuickHelp, 1)

/****************************************************************************
 msgQuickHelpClose			takes nothing, returns STATUS
	Forces the Quick Help window to disappear.

 Sent from the "Done" button to close the quick help window.  Will
 cause the window to disappear.
*/
#define msgQuickHelpClose			MakeMsg(clsQuickHelp, 3)

/****************************************************************************
 msgQuickHelpSetMode		takes QHELP_MODE, returns STATUS
	Selects the Help panel to display either gestures or summary.

 Currently, only the summary screen is used.  Do not call this function
 to change the mode.
*/
#define msgQuickHelpSetMode			MakeMsg(clsQuickHelp, 4)
Enum16(QHELP_MODE) {
	qHelpSummary,
	qHelpGesture
};

/****************************************************************************
 msgQuickHelpGetMode		takes P_QHELP_MODE, returns STATUS
	Passes back the Quick Help display mode.

 Currently, only the summary data screen is used.  Do not call this
 function to get a different mode.
*/
#define msgQuickHelpGetMode			MakeMsg(clsQuickHelp, 5)

/****************************************************************************
 msgQuickHelpInvokeNB		takes nothing, returns STATUS
	Brings up the help notebook.

 Sent when the "Help Notebook" button is tapped to bring up the
 help notebook from the quick help.  Sends all observers the
 message msgQuickHelpInvokedNB, which causes the help notebook
 to be displayed.  Posts msgQuickHelpClose to self.
*/
#define msgQuickHelpInvokeNB		MakeMsg(clsQuickHelp, 6)

#endif

