/****************************************************************************
 print.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.21  $
   $Author:   tbartlet  $
     $Date:   21 Feb 1992 16:30:16  $

 This file contains the API for clsPrint.

 clsPrint inherits from clsApp.
 Provides a wrapper to guide PenPoint documents through the printing process.

 To print a document, the Application Framework creates a wrapper 
 document (an instance of clsPrint) that embeds the document to be 
 printed in itself. To print the document, the wrapper first opens the 
 document to the printer (rather than to the screen). The wrapper then 
 uses and instance of clsPrLayout to send printing-related messages to 
 the document and any of its embedded documents.

 Developers: You should not subclass clsPrint. However, to support 
 printing, your application needs to handle many of the messages defined
 by clsPrint.
****************************************************************************/

/****  Pagination  ****/
/*
 There are two basic styles of pagination: flow and nonflow. The
 printing wrapper sends msgPrintGetProtocols to a document to ask it
 what style of pagination it supports. 

 For more information on pagination, please refer to the chapter on Printing
 in the PenPoint Architectural Reference.
*/
/****  Option Cards for Printing  ****/
/*
 The Application Framework provides a Print Setup option sheet, which
 allows the user to change margins and the running headers and footers
 that are printed with a document.

 If your application has other printing options that you want the user
 to change, you should add your option cards to the Print Setup sheet.
 To do so, your application should handle msgAppAddCards and should
 add your cards when the tag passed in is tagAppPrintSetupOptSheet.
*/

#ifndef PRINT_INCLUDED
#define PRINT_INCLUDED

#ifndef UUID_INCLUDED
#include <uuid.h>
#endif
#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif
#ifndef GEO_INCLUDED
#include <geo.h>
#endif
#ifndef SYSFONT_INCLUDED
#include <sysfont.h>
#endif
#ifndef WIN_INCLUDED
#include <win.h>
#endif
#ifndef EMBEDWIN_INCLUDED
#include <embedwin.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****  Status Codes  ****/
#define stsPrintErrorCheckPrinter	MakeStatus(clsPrint, 1)

/****  Print Option Sheet Tags  ****/
#define tagPrJobDialog				MakeTag(clsPrint, 255)
#define tagPrOption					MakeTag(clsPrint, 254)
#define tagPrPrinterLabel	  		MakeTag(clsPrint, 253)
#define	tagPrEnabledLabel			MakeTag(clsPrint, 252)
#define	tagPrPaperSizeLabel			MakeTag(clsPrint, 251)
#define tagPrPagesLabel				MakeTag(clsPrint, 250)
#define tagPrStartingPageLabel		MakeTag(clsPrint, 249)
#define tagPrPrinter				MakeTag(clsPrint, 128)
#define tagPrStatus					MakeTag(clsPrint, 140)
#define tagPrEnabledOn				MakeTag(clsPrint, 141)
#define tagPrEnabledOff				MakeTag(clsPrint, 142)
#define tagPrPages					MakeTag(clsPrint, 129)
#define tagPrPagesAll				MakeTag(clsPrint, 160)
#define tagPrPagesRange				MakeTag(clsPrint, 161)
#define	tagPrPagesFrom				MakeTag(clsPrint, 162)
#define	tagPrPagesTo				MakeTag(clsPrint, 163)
#define tagPrPaperSize				MakeTag(clsPrint, 131)
#define	tagPrPaperWidth				MakeTag(clsPrint, 174)
#define	tagPrPaperHeight			MakeTag(clsPrint, 175)
#define tagPrStartingPage			MakeTag(clsPrint, 132)


/****  Print Margins  ****/
/*
 This structure contains the margin offsets (in Mils) measured
 from the top, bottom, left, and right edges of the paper.
*/
typedef struct PRINT_MARGINS {
	S32 	top;						// offset for top margin
	S32 	bottom;						// offset for bottom margin
	S32 	left;						// offset for left margin
	S32 	right;						// offset for right margin
} PRINT_MARGINS, *P_PRINT_MARGINS;


/****  Header and Footer Strings  ****/
/*
 This structure contains the strings for either a header or a footer.
*/
typedef struct PRINT_HFDATA {  			
	U8 		reserved;				   	// reserved - must be 0
	char 	leftData[nameBufLength];	// string on left side
	char 	centerData[nameBufLength];	// string in center
	char 	rightData[nameBufLength];	// string on right side
} PRINT_HFDATA,  *P_PRINT_HFDATA;


/****  Print Setup  ****/
/*
 This structure contains setup information for printing.
*/
typedef struct PRINT_SETUP {
	OBJECT 			frame;				// reserved
	PRINT_MARGINS 	mainMargins;		// print margins for the document
	PRINT_MARGINS 	headerMargins;		// print margins for the header
	PRINT_MARGINS 	footerMargins;		// print margins for the footer
	PRINT_HFDATA 	headerInfo;			// strings to display in the header
	PRINT_HFDATA 	footerInfo;			// strings to display in the footer
	SYSDC_FONT_SPEC	fontSpec;			// header/footer font data
	U16 			fontSize;			// header/footer font size, in points
} PRINT_SETUP, *P_PRINT_SETUP;


/****  Embeddee Print Info  ****/
/*
 Users can decide:
	-:	to not print an embedded document;
	-:	to print the visible portion of an embedded document in the
		place in the parent document where it is embedded;
	-:	to print the entire embedded document at the end of the parent.

 This structure contains information for printing embedded documents.
 Note: expandInPlace and printBorders are not currently implemented.
*/
typedef struct EMBEDDEE_PRINT_INFO {
	U16 expandInPlace	: 1;	// TRUE to print entire contents in place
	U16 expandAtEnd		: 1;	// TRUE to print entire contents at end
								// FALSE to print visible portion in place
	U16 invisible		: 1;	// TRUE to not print
	U16 printBorders	: 1;	// TRUE to show borders around the window
	U16 reserved		: 12;	// reserved
	U16 reserved2		: 16;	// reserved
} EMBEDDEE_PRINT_INFO, *P_EMBEDDEE_PRINT_INFO;


/****  Spool mode values  ****/
/* 
 Note: Spooling is not implemented.
 */
#define prModeCopy			0		// to copy the doc for spooling
#define prModeLock			1		// to lock the doc for spooling


/****  Print Metrics  ****/
/*
 This structure defines the public instance data that clsPrint maintains 
 for a document. You get a copy of this structure when you send 
 msgPrintGetMetrics to a document.
*/
typedef struct PRINT_METRICS {
	U32		reserved1;		 	// reserved
	U16 	pageRangeStart;	 	// start page # (not used if pageAll is TRUE)
	U16 	pageRangeEnd;	 	// end page # (not used if pageAll is TRUE)
	U16 	startingPage;	 	// starting page # (to be printed on pages)
	U16 	copies;			 	// not used
	U16 	collating:		2; 	// not used
	U16 	orientation:	2; 	// either prOrientPortraitNormal or
							   	// pdOrientLandscapeNormal (see win.h)
	U16 	pageAll:		1; 	// TRUE to print all pages
	U16 	spoolMode:		2; 	// see spool mode values
	U16 	firstPageHeader:1; 	// TRUE to enable first page headers
	U16 	reserved2:		8; 	// reserved
	U8 		paperSizeType;	   	// Popular paper type (see clsPrn.h)
	SIZE32 	paperSize;		   	// Size of paper in Mils
	PRINT_SETUP firstPageSetup;			// not used
	PRINT_SETUP pageSetup;				// page setup information
	char  		printer[nameBufLength];	// name of printer to use
	EMBEDDEE_PRINT_INFO	embedding;		// how to print embedded documents
	U32 		reservedData[6];		// reserved
} PRINT_METRICS, *P_PRINT_METRICS;


/****  Print Embeddee Action  ****/
/*
 This structure is used by msgPrintEmbeddeeAction and msgPrintExamineEmbeddee
 to pass information about the child being processed.
*/
typedef struct PRINT_EMBEDDEE_ACTION {
    WIN  				embeddedWin;		// embedded win to act on
	U16	 				action;				// proposed embeddee action	flag
	EMBEDDEE_PRINT_INFO	embedPrintInfo;		// embeddee print properties	  	
	U32	 				reserved[3];		// reserved
} PRINT_EMBEDDEE_ACTION, *P_PRINT_EMBEDDEE_ACTION;


/****  Embeddee Action Flags  ****/
#define prEmbedActionAsIs			0	// visible part printed in place (default)
#define prEmbedActionExpandInPlace 	1	// not supported
#define prEmbedActionExtract 		2	// invisible or moved to end; either
										// way, child removed from parent

/****  Print Page  ****/
/*
 This structure is used by msgPrintStartPage and msgPrintLayoutPage to 
 pass information about what page needs to be printed next.
*/
typedef struct PRINT_PAGE {
	U16 	pageNumber;			// In:	#pages printed when this one is done
	U16 	displayPageNumber;	// In:	page number to display on page 
	U16 	logicalPageNumber;	// In: 	#times msgPrintStartPage has been sent
	OBJECT	jobUID;				// In:	print layout driver object
	OBJECT	appLayoutUID;		// Out: obj to receive msgPrintEmbeddeeAction
	U32 	reserved[3];		// reserved
} PRINT_PAGE, *P_PRINT_PAGE;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages											   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgPrintStartPage		takes P_PRINT_PAGE,  returns STATUS
	Advance the document to its next logical page.	
 
 This message is sent to a document as a signal to initialize its internal 
 pagination data to a new page.  When the document has no more pages to print
 it should return stsEndOfData in response to this message. Note: the 
 document does not return stsEndOfData when it paginates its last page; it
 waits until the next time this message is sent (when it has no data left
 to paginate). If the document does have more pages to print, 
 the following happens:
	-:	the document receives msgPrintGetProtocols
	-:	the mainWin of document receives msgWinLayout at least once
	-:	the document receives msgPrintLayoutPage
 If appLayoutUID is objNull, the print layout driver will send any 
 messages regarding embeddee actions (msgPrintEmbeddeeAction) to the document;
 otherwise it will send them to the appLayoutUID object set by the document
 in this structure.

 Developers: Your application should handle this message to support
 pagination.
*/
#define msgPrintStartPage			MakeMsg(clsPrint, 1)

/****************************************************************************
 msgPrintLayoutPage		takes P_PRINT_PAGE,  returns STATUS
 	Document lays out its logical page.

 The wrapper sends this message to the document after it sends 
 msgPrintStartPage and msgPrintGetProtocols. This message can be thought 
 of as a substitute for msgWinLayout. However, unlike msgWinLayout, it 
 is sent only once per page.

 Developers: Your application should handle this message to support
 pagination.
*/
#define msgPrintLayoutPage			MakeMsg(clsPrint, 12)

/****************************************************************************
 msgPrintGetMetrics		takes P_PRINT_METRICS, returns STATUS
	Gets the application's print metrics.

 You can send this message to OSThisApp() to get the current application's 
 print metrics.	During printing you can send this message to the jobUID 
 (given in the pArgs of msgPrintStartPage) to get EFFECTIVE print metrics.
 EFFECTIVE print metrics are those from the original top-level document
 in this print job. Deferred embedded documents print with 
 effective margins, headers and footers, and orientation; the values in 
 their own print metrics are ignored.


 Developers: Your application does not need to handle this message.
*/
#define msgPrintGetMetrics			MakeMsg(clsPrint, 2)


/****************************************************************************
 msgPrintSetMetrics		takes P_PRINT_METRICS, returns STATUS
	Sets the application's print metrics.

 You can send this message to OSThisApp() to set the current application's
 print metrics.

 Developers: Your application does not need to handle this message.
*/
#define msgPrintSetMetrics			MakeMsg(clsPrint, 3)


/****************************************************************************
 msgPrintApp	takes P_PRINT_DATA, returns STATUS.
	Prints a document.

 This message prints the document. If you want to invoke printing, you
 send this message to thePrintManager, using ObjectSend or ObjectPost.

 Developers: Your application does not need to handle this message.
*/
#define msgPrintApp					MakeMsg(clsPrint, 4)

typedef struct PRINT_DATA {
	OBJECT 	appUID;					// In: UID if this is the active app
	UUID 	appUUID;			   	// In: application UUID
	U32 	reserved[2];			// reserved
} PRINT_DATA, *P_PRINT_DATA;


/****************************************************************************
 msgPrintPaperArea		takes P_PRINT_AREA, returns STATUS.
	Passes back the width and height of the printing area on the paper.

 thePrintManager returns the size of the printing area on a sheet of paper, 
 adjusted to take into account margin values and interpreted relative to the 
 orientation. Thus, thePrintManager swaps the computed width and height 
 values if the page orientation is landscape vs portrait.
 
 The size of the printing area is in Mils. It does not account for 
 printer hardware limitations, i.e., the "unprintable area" on a page.
 
 You can send this message to thePrintManager at any time to get the 
 the current document's printing area. You can either pass in the metrics
 from which to compute the area or set pMetrics = pNull. 
 If pMetrics is pNull, thePrintManager will obtain the print metrics
 from theProcessResList.

 Developers: Your application does not need to handle this message.
*/
#define msgPrintPaperArea			MakeMsg(clsPrint, 7)

typedef struct PRINT_AREA {
	P_PRINT_METRICS	pMetrics;		// In: pNull or metrics for computation
	SIZE32 			area;			// Out: size of print area, in Mils.
} PRINT_AREA, *P_PRINT_AREA;



/****************************************************************************
 msgPrintGetProtocols		takes P_PRINT_PROTOCOLS,  returns STATUS
	Gets the pagination and embeddee printing protocols for the document.	

 The wrapper sends this message to the document after each msgPrintStartPage.

 Developers: Your application needs to handle this message and pass back
 the pagination method (see "paginationMethod Values" below) and the 
 embeddee searching method (see "embeddeeSearch Values").
*/
#define msgPrintGetProtocols			MakeMsg(clsPrint, 9)

typedef struct PRINT_PROTOCOLS {
	U16 paginationMethod; 		//	Out: paginationMethod value
	U16 embeddeeSearch;	  		//  Out: embeddeeSearch value
} PRINT_PROTOCOLS, *P_PRINT_PROTOCOLS;	

/****  paginationMethod Values  ****/
#define prPaginationTile 	1		// tile pagination style
#define prPaginationFlow 	2		// flow pagination style
#define prPaginationScale	3		// scale pagination style

/****  embeddeeSearch Values  ****/
#define prEmbeddeeSearchByPrintJob  1	// print layout driver finds children
										// for the application
#define prEmbeddeeSearchByApp		2	// app finds children while paginating

/****************************************************************************
 msgPrintEmbeddeeAction	takes P_PRINT_EMBEDDEE_ACTION,  returns STATUS
 	Asks the document for permission to perform an action on an embeddee.

 The wrapper sends this message to the (top-level) document being printed; 
 it requests permission to perform an action on an embeddee.

 Developers: You should handle this message and return stsOK for yes, 
 stsRequestDenied for no.

 In parameters:
 	embeddedWin:   	embedded win to act on
 	action:		   	proposed embeddee action
 	embedPrintInfo:	embeddee print properties
*/
#define msgPrintEmbeddeeAction	MakeMsg(clsPrint, 10)

/****************************************************************************
 msgPrintExamineEmbeddee	takes P_PRINT_EMBEDDEE_ACTION,  returns STATUS
 	Sent to the print layout driver to interpret an embedded window's 
	print properties.
 
 Documents that are being printed (or their layout objects) can send 
 this message to the wrapper. It tells the print layout driver to 
 interpret the embedded win's print properties and propose an action 
 via msgPrintEmbeddeeAction. msgPrintEmbeddeeAction is sent subsequently 
 even if no action is necessary.

 In parameters:
 	embeddedWin:   	embedded win to examine 

 Out parameters:
 	action:		   	proposed embeddee action
 	embedPrintInfo:	embeddee print properties

 Developers: You do not need to handle this message.
*/
#define msgPrintExamineEmbeddee	MakeMsg(clsPrint, 11)


/****************************************************************************
 msgPrintSetPrintableArea	takes PRINTABLE_AREA,  returns STATUS
 	Sent to the printJob to adjust margins for the "unprintable area".

 A (top-level) document can send this to the printJob during printing 
 as a request to adjust margins to account for printer hardware limitations 
 (i.e., an unprintable area on the page). It affects only the current page.
 You typically first send msgPrintGetPrintableArea to get the margins that
 the printJob is currently using. Then you can set the flags argument to
 prAdjustActualForUnprintable, and send the structure on to this message.

 Automatic tiling by the printJob always adjusts the user-set (print metrics)
 margins to account for the unprintable area on the page.  

 Typically graphics (non-flow) applications will desire this type of
 adjustment, while word processing (flow) apps won't since it may cause 
 data reformatting. 
 Sometimes, as with text, it is more user-friendly not to adjust (and let the
 data get clipped) so that the source of the problem is obvious to the user.  
 Auto adjustment may induce unwanted visual changes and obscure their source.

 Developers: You do not need to handle this message.
*/
#define msgPrintSetPrintableArea	MakeMsg(clsPrint, 13)

#define prAdjustActualForUnprintable flag0	// make sure actual margins
											// account for hardware limits

typedef struct PRINTABLE_AREA {
	U16				flags;					 
	PRINT_MARGINS	printMetricsMargins;	// user-set margins
	PRINT_MARGINS	unprintableMargins;		// hardware limitations
	PRINT_MARGINS	actualMargins;			// actual margins used by print
											// layout driver
} PRINTABLE_AREA, *P_PRINTABLE_AREA;


/****************************************************************************
 msgPrintGetPrintableArea	takes PRINTABLE_AREA,  returns STATUS
    Sent to the print job during printing to request margin information.

 Flags are ignored.

 Developers: You do not need to handle this message.
*/
#define msgPrintGetPrintableArea	MakeMsg(clsPrint, 14)


#endif	// PRINT_INCLUDED

