/****************************************************************************
 File: prefs.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.14  $
   $Author:   bschwart  $
     $Date:   25 Feb 1992 14:58:48  $

 Next up: 28

 This file contains the API definition for clsPreferences.
 clsPreferences inherits from clsObject.
 clsPreferences provides a shell to access system preferences.

 theSystemPreferences is a well-known instance of clsPreferences.
 theSystemPreferences provides access to read and write system wide
 preferences.

 clsPreferences supports a set of preferences.  Preferences are stored
 as resources in the "current" system preferences resource file.  An
 instance of clsPreferences, known as theSystemPreferences, is created
 at boot time.  This should be the only instance of clsPreferences
 in the system.

 Preferences are named by well known resource id's (RES_ID's).   This
 header file contains some predefined preference id's to simplify things.
 When defining new preferences, use the class that originated the
 preferences.

 Clients can get and set preferences by accessing the well known
 object theSystemPreferences.

 Preferences are stored in a resource file.  Any request to read or write
 a preference will force a read or write to a file.  This
 minimizes the amount of space required to store preferences.
 theSystemPreferences will respond to any resource file messaged defined
 in resfile.h and process them appropriately.

 Remember, to read and write system preferences simply use the messages
 msgResReadData and msgResWriteData (or msgResUpdateData).
 theSystemPreferences forwards the msg to the current system preferences
 resource file.

 As an example of reading a system preference:
//{
	U16 lineHeight;
	RES_READ_DATA read;

	read.resId = prLineHeight;
 	read.heap = 0;
	read.pData = &lineHeight;
	read.length = SizeOf(U16);
	ObjectCall(msgResReadData, theSystemPreferences, &read);
//}

 An example of writing a system preference:
//{
	U16 lineHeight;
	RES_WRITE_DATA write;

	write.resId = prLineHeight;
	write.pData = &lineHeight;
	write.length = SizeOf(U16);
	write.agent = resDefaultResAgent;
	ObjectCall(msgResWriteData, theSystemPreferences, &write);
//}

 theSystemPreferences "knows" about certain preferences (listed in this
 file below) and performs whatever interaction is required to activate the
 new preference.  It also handles certain system wide notification and
 actions when certain preferences change. For example, clsPreferences
 will cause the system to be re-drawn and re-fonted when the
 system preference for the font changes.

 clsPreferences will notify all observers when a preference has
 (potentially) changed.  This will allow various objects to observe
 theSystemPreferences, and react to the preference changes.

 Whenever a number of preferences are being changed, clients may wish
 to send msgPrefsWritingMany, followed by the preference writes, and
 then msgPrefsWritingDone. clsPreferences will use these messages to
 delay any layout that may occur as a result of writing preferences
 that cause layout.  clsPreferences will also send these messages
 to observers, allowing them to delay expensive operations until the
 preference changes are complete. As an example, when the preference
 set changes, msgPrefsWritingMany, followed by msgPrefsPreferenceChanged
 for each preference, followed by msgPrefsWritingDone is sent to
 the observers.

 clsPreferences supports the concept of different sets of preferences.
 A set of preferences is stored in a single resource file in a well-known
 preferences directory managed by theInstalledPreferences.  clsPreferences
 supports messages to change the current preference set to another one
 that is already filed. In addition, clsPreferences allows a preference
 set to start "clean".  When PenPoint first starts up (or during a
 warm boot), theSystemPreferences will contain the set of preferences
 associated with the "current" preference set managed by
 theInstalledPreferences.  If no current set exists,
 theSystemPreferences will start with a "clean" preference set. When
 a preference set changes, clsPreferences will notify the observers
 of the changed preferences.  This is because clsPreferences is notified
 via msgIMCurrentChanged from the install manager (see instlmgr.h).

 To change the set of preference set programmatically, one must communicate
 with theIntallManager.  An example code fragment to change a preference
 set. See instlmgr.h for details:
//{
	IM_INSTALL install;
	install.locator.uid = theBootVolume;
	rn.fs.locator.pPath = "\\PenPoint\\prefs\\PREFERENCESET";
	install.exist = imExistReactivate;
	install.listAttrLabel = 0;
	install.listHandle = 0;
	ObjectCall(msgIMInstall, theInstalledPrefs, &install);
	ObjectCall(msgIMSetCurrent, theInstalledPrefs, install.handle);
//}

****************************************************************************/
#ifndef PREFS_INCLUDED
#define PREFS_INCLUDED

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef RESFILE_INCLUDED
#include <resfile.h>
#endif

#ifndef SYSGRAF_INCLUDED
#include <sysgraf.h>
#endif

#ifndef OS_INCLUDED
#include <os.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Known Preferences in the System						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*
 * The following are the predefined resource names, the data
 * that reading and writing will return, and some predefined
 * return values for certain preferences.
 */
/***************************************************************************/
/****  System Font  ****/
/*
 * prSystemFont is the resource id for the system font.
 * Reads and writes of this id use a P_PREF_SYSTEM_FONT. This resource will
 * affect the returned value from PrefsSysFontInfo.
 *
 * Changing this resource (via msgResWriteData) will cause the system
 * to layout after notification of observers, which is expensive.
 * This is done by doing an ObjectPost of msgPrefsLayoutSystem to self.
 * As a result, clsPreferences will compare this resource to the previous
 * value to prevent layout and observer notification if the write did
 * not change the value.
 */
#define tagPrSystemFont			MakeWknResId(clsPreferences, 1)
#define prSystemFont			tagPrSystemFont

/***************************************************************************/
/****  Field Font  ****/
/*
 * prUserFont is the resource id for the field (user) font.
 * Reads and writes of this id use a P_PREF_SYSTEM_FONT. This preference
 * will affect the returned data from PrefsSysFontInfo.
 *
 * Changing this resource (via msgResWriteData) will cause the system
 * to layout after notification of observers, which is expensive.
 * This is done by doing an ObjectPost of msgPrefsLayoutSystem to self.
 * As a result, clsPreferences will compare this resource to the previous
 * value to prevent layout and observer notification if the write did
 * not change the value.
 */
#define tagPrUserFont			MakeWknResId(clsPreferences, 2)
#define	prUserFont				tagPrUserFont

/*
 * This data structure is the what is read in and written when
 * reading and writing when the resId is prSystemFont or prUserFont.
 * It contains a font specification, and a font scale to use.
 */
typedef struct PREF_SYSTEM_FONT {
  SYSDC_FONT_SPEC		spec;		// Font spec
  SCALE				    scale;		// Scale: same for system and user font
} PREF_SYSTEM_FONT, *P_PREF_SYSTEM_FONT;

/***************************************************************************/
/****  Orientation  ****/
/*
 * prOrientation is the resource id for the screen orientation. Reads and
 * writes of this id use a a P_U8, whose values are defined below.
 *
 * Changing this resource (via msgResWriteData) will cause the system
 * to layout after notification of observers, which is expensive.
 * This is done by doing an ObjectPost of msgPrefsLayoutSystem to self.
 * As a result, clsPreferences will compare this resource to the previous
 * value to prevent layout and observer notification if the write did
 * not change the value.
 */
#define tagPrOrientation		MakeWknResId(clsPreferences, 3)
#define prOrientation			tagPrOrientation

#define prPortrait              0		// Portrait mode
#define prLandscape             1		// Landscape mode
#define prPortraitReversed      2		// Portrait mode (rotated 180 degrees)
#define prLandscapeReversed     3		// Landscape mode (rotated 180 degrees)

/***************************************************************************/
/****  Bell  ****/
/*
 * prBell is the resource id for ringing the warning bell.  It reads and
 * writes a P_U8, whose values are defined below. prBell is
 */
#define tagPrBell				MakeWknResId(clsPreferences, 5)
#define	prBell					tagPrBell

#define prBellOn                0		// Ring the bell
#define prBellOff               1		// Don't ring the bell

/***************************************************************************/
/****  Writing Style  ****/
/*
 * prWritingStyle is the resource id for the handwriting preference
 * style. Reads and writes of this id use a P_U8, whose values
 * are defined below.
 */
#define tagPrWritingStyle		MakeWknResId(clsPreferences, 6)
#define tagPrPrintingStyle		MakeWknResId(clsPreferences, 6)
#define prWritingStyle			tagPrWritingStyle
#define prPrintingStyle			tagPrPrintingStyle		// old name

#define prMixedCase             0		// Mixed case writer
#define prCapsOnly              1		// All caps writer

/***************************************************************************/
/****  Date Format  ****/
/*
 * prDateFormat is the resource id for the desired date format.  Reads and
 * writes use a P_U8, whose values are defined below. This preference
 * will affect the format of the string returned from PrefsDateToString.
 */
#define tagPrDateFormat			MakeWknResId(clsPreferences, 7)
#define prDateFormat			tagPrDateFormat

#define prDateMDYFull           0   //  January 15, 1990
#define prDateMDYAbbre          1   //  Jan. 15, 1990
#define prDateMDYSlash          2   //  1/15/90
#define prDateMDYHyphe          3   //  1-15-90
#define prDateMDYDot			8	//  1.15.90
#define prDateDMYFull           4   //  15 January 1990
#define prDateDMYAbbre          5   //  15 Jan. 1990
#define prDateDMYSlash          6   //  15/1/90
#define prDateDMYHyphe          7   //  15-1-90
#define prDateDMYDot			9	//	15.1.90


/***************************************************************************/
/****  Gesture Timeout  ****/
/*
 * prGestureTimeout is the resource id for the gesture timeout, and
 * is measured in 1/100's of a second.  Reads and writes of this id use
 * a P_U16 whose meaning is 1/100's of a second.
 */
#define tagPrGestureTimeout		MakeWknResId(clsPreferences, 9)
#define prGestureTimeout		tagPrGestureTimeout

/***************************************************************************/
/****  Line Height  ****/
/*
 * prLineHeight is the resource id for the ruled line writing line height in
 * edit pads. Reads and writes of this id use a P_U16, whose meaning is
 * 1/100th's of an inch.  Changing this preference only affects newly
 * created ruled pads.
 */
#define tagPrLineHeight			MakeWknResId(clsPreferences, 10)
#define prLineHeight			tagPrLineHeight

/***************************************************************************/
/****  Auto Suspend  ****/
/*
 * tagPrAutoSuspend is the resource id for auto suspend timeout.
 * Reads and writes of this id use a P_U16, whose units are minutes.  If the
 * value is 0, the machine will not be auto suspended.
 *
 * Machines that do not support auto suspend use the auto suspend preference
 * for the auto shutdown timeout.
 */
#define tagPrAutoSuspend	MakeWknResId(clsPreferences, 11)

/***************************************************************************/
/****  Auto Shutdown  ****/
/*
 * tagPrAutoShutdown is the resource id for auto shutdown timeout.
 * Reads and writes of this id use a P_U16, whose units are hundredths of
 * hours.  If the value is 0, the machine will not auto shutdown.
 *
 * Machines that do not support auto suspend use the auto suspend timeout
 * prefrence for auto shutdown.
 */
#define tagPrAutoShutdown	MakeWknResId(clsPreferences, 28)


/***************************************************************************/
/****  Power Management  ****/
/*
 * prPowerManagement is the resource id that indicates if PenPoint should
 * attempt to limit the computer's power consumption by turning off inactive
 * devices
 */
#define tagPrPowerManagement	MakeWknResId(clsPreferences, 27)
#define prPowerManagement		tagPrPowerManagement

#define prPowerManagementOff        0	// power management not attempted
#define prPowerManagementOn         1	// power management attempted

/***************************************************************************/
/****  Floating Allowed  ****/
/*
 * prDocFloating is the resource id that indicates if documents can
 * be floated.  Reads and writes of this id use a P_U8, whose meaning is defined
 * below.
 */
#define tagPrDocFloating	MakeWknResId(clsPreferences, 12)
#define prDocFloating		tagPrDocFloating

#define prDocFloatingOff        0		// document floating not allowed
#define prDocFloatingOn         1		// document floating allowed


/***************************************************************************/
/****  Zooming Allowed  ****/
/*
 * prDocZooming is the resource id that indicates if documents can
 * be zoomed. Reads and writes of this id use a P_U8, whose meaning is defined
 * below.
 */
#define tagPrDocZooming		MakeWknResId(clsPreferences, 13)
#define prDocZooming		tagPrDocZooming

#define prDocZoomingOff        0		// document zooming not allowed
#define prDocZoomingOn         1		// document zooming allowed

/***************************************************************************/
/****  Left/Right Handed  ****/
/*
 * prHandPreference is the resource id that indicates a left handed or
 * right handed user. Reads and writes of this id use a P_U8, whose
 * meaning is defined below.
 *
 * Changing this resource (via msgResWriteData) will cause the system
 * to layout after notification of observers, which is expensive.
 * This is done by doing an ObjectPost of msgPrefsLayoutSystem to self.
 * As a result, clsPreferences will compare this resource to the previous
 * value to prevent layout and observer notification if the write did
 * not change the value.
 */
#define tagPrHandPreference		MakeWknResId(clsPreferences, 14)
#define prHandPreference		tagPrHandPreference

#define prLeftHanded           0		// Left Handed writer
#define prRightHanded          1		// Right Handed writer

/***************************************************************************/
/****  Scroll Margins Style  ****/
/*
 * prScrollMargins is the resource id that indicates a "full" vs. "light"
 * scroll bars. Reads and writes of this id use a P_U8, whose
 * meaning is defined below.
 *
 * Changing this resource (via msgResWriteData) will cause the system
 * to layout after notification of observers, which is expensive.
 * This is done by doing an ObjectPost of msgPrefsLayoutSystem to self.
 * As a result, clsPreferences will compare this resource to the previous
 * value to prevent layout and observer notification if the write did
 * not change the value.
 */
#define tagPrScrollMargins		MakeWknResId(clsPreferences, 26)
#define prScrollMargins			tagPrScrollMargins

#define prScrollMarginsFull    0
#define prScrollMarginsLight   1

/***************************************************************************/
/****  Character Box Width  ****/
/*
 * prCharBoxWidth is the resource indicating the width of char boxes
 * for boxed writing fields.  Reads and writes of this id use a P_U8,
 * whose meaning is the width of the box in points. This preference
 * only affects newly created character boxes.
 */
#define tagPrCharBoxWidth		MakeWknResId(clsPreferences, 15)
#define prCharBoxWidth			tagPrCharBoxWidth

/***************************************************************************/
/****  Character Box Height  ****/
/*
 * prCharBoxHeight is the resource id indicating the height of char boxes
 * for boxed writing fields. Reads and writes of this id use a P_U8, whose meaning
 * is the height of the char box in points. This preference only
 * affects newly created character boxes.
 */
#define tagPrCharBoxHeight		MakeWknResId(clsPreferences, 16)
#define prCharBoxHeight			tagPrCharBoxHeight

/***************************************************************************/
/****  Hand Writing Timeout  ****/
/*
 * prHWXTimeout is the resource id indicating the handwriting timeout.
 * Reads and writes of this id use a P_U16 whose meaning is 1/100's of a
 * second.
 */
#define tagPrHWXTimeout		MakeWknResId(clsPreferences, 17)
#define prHWXTimeout		tagPrHWXTimeout

/***************************************************************************/
/****  Input Pad Style  ****/
/*
 * prInputPadStyle is the resource id indicating the preferred style
 * of handwriting pads. Reads and writes of this id use a P_U8, whose meaning is
 * defined below.
 */
#define tagPrInputPadStyle		MakeWknResId(clsPreferences, 18)
#define prInputPadStyle			tagPrInputPadStyle

#define prInputPadStyleBoxed        	0	// Pad styles are boxed
#define prInputPadStyleRuled        	1	// Pad styles are Ruled
#define prInputPadStyleRuledAndBoxed	2	// Pad styles are boxed-->ruled

#define prInputPadStyleSegmented    	0 	// Obsolete


/***************************************************************************/
/****  Hold Timeout  ****/
/*
 * prPenHoldTimeout is the resource id for the press hold timeout. Reads
 * and writes of this id use a P_U16 whose meaning is 1/100's of a second.
 */
#define tagPrPenHoldTimeout		MakeWknResId(clsPreferences, 19)
#define prPenHoldTimeout		tagPrPenHoldTimeout


/***************************************************************************/
/****  Pen Cursor  ****/
/*
 * prPenCursor is the resource id for whether the cursor is off or on.
 * Reads and writes of this id use a P_U8, whose meaning is defined below.
 */
#define tagPrPenCursor			MakeWknResId(clsPreferences, 20)
#define prPenCursor				tagPrPenCursor

#define prPenCursorOff              0		// Pen cursor should be off
#define prPenCursorOn               1		// Pen cursor should be on

/***************************************************************************/
/****  Time Format  ****/
/*
 * prTimeFormat is the resource id for the preferred time format (military
 * or civilian).  Reads and writes of this id use a P_U8, whose meaning is
 * defined below.  This preference will affect the returned string from
 * PrefsTimeToString.
 */
#define tagPrTimeFormat			MakeWknResId(clsPreferences, 21)
#define prTimeFormat			tagPrTimeFormat

#define prTime12Hour                0		// Display 12 hour times
#define prTime24Hour                1		// Display 24 hour times

/***************************************************************************/
/****  Display Seconds  ****/
/*
 * prTimeSeconds is the resource id indicating if seconds should be displayed
 * or not. Reads and writes of this id use a P_U8, whose meaning is defined
 * below. This preference will affect the returned string from
 * PrefsTimeToString.
 */
#define tagPrTimeSeconds		MakeWknResId(clsPreferences, 22)
#define prTimeSeconds			tagPrTimeSeconds

#define prTimeSecondsDisplay        0		// Display seconds in time
#define prTimeSecondsOff            1		// Don't display seconds in time

/***************************************************************************/
/****  Time  ****/
/*
 * prTime is the resource id for the system time.  Reads and writes of this
 * ID use a P_PREF_TIME_INFO, containing the current time information.
*/
#define tagPrTime				MakeWknResId(clsPreferences, 23)
#define prTime					tagPrTime

typedef union PREF_TIME_MODE {
    OS_SET_TIME_MODE writeMode;     // In: which attributes to set (for write only)
} PREF_TIME_MODE;

typedef struct PREF_TIME_INFO {
    PREF_TIME_MODE  mode;           // In: read or write mode
    OS_DATE_TIME    dateTime;       // In/Out: date and time information
} PREF_TIME_INFO, *P_PREF_TIME_INFO;


/***************************************************************************/
/****  Primary Input  ****/
/*
 * prPrimaryInput is the resource id defining the primary input device.
 * Reads and writes of this id use a P_U8, whose meaning is defined below.
 */
#define tagPrPrimaryInput		MakeWknResId(clsPreferences, 24)
#define prPrimaryInput			tagPrPrimaryInput

#define prPrimaryInputPen           0	// Primary input is with the pen
#define prPrimaryInputKbd           1	// Primary input is with a keyboard

/***************************************************************************/
/****  Unrecognized Character  ****/
/*
 * prUnrecCharacter is the resource id used for the unrecognized character
 * glyph.  Reads and writes of this id use a P_U8, whose meaning is defined
 * below.
 */
#define tagPrUnrecCharacter		MakeWknResId(clsPreferences, 25)
#define prUnrecCharacter		tagPrUnrecCharacter

#define prUnrecCharacterQuestion	0
#define prUnrecCharacterUnder		1

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages											   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew 	takes P_PREFS_NEW, returns STATUS
    category: class message
	Creates a new preferences object.

 This message should not be called by clients.
 Creates a preferences object.  If pPrefSet is pNull, the list will
 start out empty.  Otherwise, pPrefSet is expected to be an already
 installed file title in the preferences directory.
*/
typedef struct PREFS_NEW_ONLY {
    P_CHAR        pPrefSet;   // Preference set name
} PREFS_NEW_ONLY, *P_PREFS_NEW_ONLY;

#define	prefsNewFields		\
	objectNewFields	 		\
	PREFS_NEW_ONLY	 		prefs;

typedef struct PREFS_NEW {
	prefsNewFields
} PREFS_NEW, *P_PREFS_NEW;


/****************************************************************************
 msgPrefsPreferenceChanged		takes P_PREF_CHANGED, returns STATUS
	category: observer notification
	Sent to observers when a preference has changed.

 Sent to observers.  Notifies observers that a given preference
 has changed. Notifies with the manager (usually theSystemPreferences,
 as there are no other pre-defined instances of clsPreferences),
 and the RES_ID of the preference that has changed.
*/
#define msgPrefsPreferenceChanged   MsgNoError(MakeMsg(clsPreferences, 1))

typedef struct PREF_CHANGED {
  OBJECT            manager;        // Sender of the notification (theSystemPreferences)
  RES_ID            prefId;         // resId of preference that changed
} PREF_CHANGED, *P_PREF_CHANGED;

/****************************************************************************
 msgPrefsLayoutSystem	takes NULL, returns STATUS
	Causes the system to re-layout and re-paint.

 Causes the entire system to layout. If msgPrefsWritingMany has not
 been called, posted to self when clsPreferences
 receives msgResWriteData and a new value has been written for
 prSystemFont, prUserFont, prOrientation, prHandPreference, or
 prScrollMargins.  If msgPrefsWritingMany has been called,
 the layout will occur when msgPrefsWritingDone is called.
 Will be sent to observers when immediately before a layout
 of the system occurs due to a preference change.

 See Also
	msgPrefsWritingMany
	msgPrefsWritingDone
*/
#define msgPrefsLayoutSystem					MakeMsg(clsPreferences, 5)

/****************************************************************************
 msgPrefsWritingMany	takes NULL, returns STATUS
	Indicates several preferences are to be written in succession.

 Causes clsPreferences to delay the self-posting of msgPrefsLayoutSystem
 until it receives msgPrefsWritingDone. Useful when writing several
 preference changes at once, and the client does not want the system
 laying out several times. If, after this message is received, a msgResWrite
 of prSystemFont, prUserFont, prOrientation, prHandPreference, or
 prScrollMargins is received, clsPreferences will self-post
 msgPrefsLayoutSystem when msgPrefsWritingDone is received. After
 msgPrefsWritingDone is received, any other msgResWrite of these preferences
 will cause an immediate layout unless this message is sent again. Will be
 sent to observers to allow them to be aware that several preferences
 are being written.

 See Also
	msgPrefsWritingDone
	msgPrefsLayoutSystem
*/
#define msgPrefsWritingMany					MakeMsg(clsPreferences, 6)

/****************************************************************************
 msgPrefsWritingDone	takes NULL, returns STATUS
	Indicates completion of writing several preferences.

 Causes the system to layout if necessary by self-posting msgPrefsLayoutSystem.
 You should send this message in conjunction with msgPrefsWritingMany to
 indicate that writing of successive preferences is complete. If a
 msgResWrite of prSystemFont, prUserFont, prOrientation, prHandPreference,
 or prScrollMargins with a new value has been done, layout will occur at
 this time. Will be sent to observers to indicate that a series of
 preferences writes have been completed.

 See Also
	msgPrefsWritingMany
	msgPrefsLayoutSystem
*/
#define msgPrefsWritingDone						MakeMsg(clsPreferences, 7)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Public Functions									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 PrefsSysFontInfo	returns void
	Passes back the system and user font information.

 This function can be used to read all font information stored in
 the preferences file at one time.  Equivalent functionality exists
 with msgResRead.  This function is provided for convenience.
*/
typedef struct PREF_SYSTEM_FONT_INFO {
  U8                    scale;
  U16                   sysFontId;
  U16                   userFontId;
} PREF_SYSTEM_FONT_INFO, *P_PREF_SYSTEM_FONT_INFO;

void EXPORTED PrefsSysFontInfo (
  P_PREF_SYSTEM_FONT_INFO pFontInfo);

/****************************************************************************
 PrefsDateToString	returns P_CHAR
	Returns a pointer to the string containing a formatted date.

 This function will return a string containing the ASCII representation
 of the formatted date based on the current user-preference for date. Puts
 the date into the string passed in. The longest possible string is 18
 characters (19 including the terminating 0) given the CURRENT formats.
 If additional formats are added, this may increase.
*/

#define prefsMaxDate	19
P_CHAR EXPORTED PrefsDateToString (
  P_OS_DATE_TIME pTime,
  P_CHAR pStr);

/****************************************************************************
 PrefsTimeToString   returns P_CHAR
	Returns a pointer to the string containing a formatted time.

 This function will return a string containing the ASCII representation of the
 time based on the current user preferences for time. Puts the time into the
 string passed in, and returns the string pointer. The longest possible
 string is 10 characters (11 including the terminating 0) given the
 current time formats. If additional formats are added, this may increase.
*/
#define prefsMaxTime	11
P_CHAR EXPORTED PrefsTimeToString (
  P_OS_DATE_TIME pTime,
  P_CHAR pStr);

#endif
