/****************************************************************************
 File: picseg.h

 Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.59  $
   $Author:   dlow  $
     $Date:   27 Feb 1992 18:34:14  $

 This file contains the API definition for clsPicSeg (Picture Segments).
 clsPicSeg inherits from clsSysDrwCtx.
 clsPicSeg provides a database and storage for drawing primitives.

 A Picture Segment creates a display list from the stream of messages
 defined by drawing context. The graphic elements in a PicSeg are called
 grafics. The display list can repaint to the same window or store the
 grafics and later repaint it to another window. It also provides a
 move/copy transfer type for grafics.

 The Picture Segment stores the following shapes as defined by clsSysDrwCtx:
 rectangle, ellipse, Bezier, polyline, polygon, sector rays, arc rays,
 chord rays, text. In addition, it defines a spline, and object types as
 an enhancement to the drawing context. It doesn't store images or raster
 operations such as CopyRect and XOR. Raster operations like XOR, AND,
 dynamic and fast modes defined by the drawing context apply to the whole
 display list. Similarly, transformations scale, translate, rotate and units
 apply to the PicSeg before drawing the list. The PicSeg stores the grafics
 in Logical Unit Coordinates as defined by the drawing context.

 PicSeg's provide display query messages allowing changes to grafic shapes
 it stores. The grafics in a picture segment are ordered; it keeps track
 of the current grafic. You can retrieve, alter, reorder, and delete
 individual grafics.

 Common uses of PicSeg's:

 PicSeg's generally used as the Data Object of a View (clsView). A drawing
 View (like clsGrafPaper) translates the input strokes into grafics and
 draws them to the PicSeg, treating the PicSeg just like a Drawing Context.
 When the View gets msgWinRepaint it sends msgPicSegPaint to the PicSeg.

 The PicSeg's file data as an Object so they can be used as resources.
 A Drawing View could file many PicSegs with different resource ids to the
 same file. Latter a display View could look up the different PicSegs in
 the resource file and display them.

 PicSeg's are used to Move/Copy grafic data between Views. The transfer
 (xfer) mechanism uses an intermedate global PicSeg for grafics.

****************************************************************************/
#ifndef PICSEG_INCLUDED
#define PICSEG_INCLUDED

#ifndef SYSGRAF_INCLUDED
#include <sysgraf.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****  Data Collection and Drawing Modes  ****/
/*
 * The PicSeg flags deterimine what to do with a draw messages.
 * By default a message like msgDcDrawRectangle causes the PicSeg to store
 * the rectangle in the display list and draw it on the window set by
 * msgDcSetWindow. The following flags can prevent one or
 * both of these thing from happening (picseg.flags).
 */
#define picSegAdd			flag0		// on if PicSeg should add grafics.
#define picSegDraw			flag1		// on if PicSeg should draw grafics
#define picSegSendDestroy	flag2		// on ObjectCall(msgDestroy, ...)
										// to an object grafic when it is 
										// deleted from the PicSeg or if the 
										// PicSeg is freed.
/*
 * The first grafic in the display list is 0. The last can be set by using
 * msgPicSegSetCurrent with picSegTopGrafic or asking for the current number
 * of grafics and then setting the current grafic.
 */
#define picSegTopGrafic	0x7FFFFFFF	// theoretical maximum number of grafics


/****  OpCodes  ****/
/*
 * Each grafic in the PicSeg is given an OpCode that idenifies what type
 * of data is stored in the pData member of PIC_SEG_GRAFIC.
 */

typedef U16						OP_CODE;
typedef P_U16					P_OP_CODE;

#define opCodeMaskInvisible		0x1000
											// grafic.pData
#define opCodePolyline	 		100			// PIC_SEG_POLYLINE
#define opCodeRectangle			101			// PIC_SEG_RECT
#define opCodeEllipse			102			// PIC_SEG_ELLIPSE
#define opCodePolygon 			103			// PIC_SEG_POLYGON
#define opCodeSpline			104			// PIC_SEG_SPLINE
#define opCodeArcRays			105			// PIC_SEG_ARC_RAYS
#define opCodeSectorRays 		106			// PIC_SEG_ARC_RAYS
#define opCodeChordRays			107			// PIC_SEG_ARC_RAYS
#define opCodeText			 	 55			// PIC_SEG_TEXT
#define opCodeObject		 	150			// PIC_SEG_OBJECT

/*
 * The basic grafic used with msgPicSegGetGrafic. The pData allocated
 * in the a heap and must be freed by creator of the PicSeg.
 */
typedef struct {
	OP_CODE			opCode;				// the type of grafic stored in pData
	P_UNKNOWN		pData;				// pointer to the grafic data
}PIC_SEG_GRAFIC, * P_PIC_SEG_GRAFIC;

/*
 * Every grafic provides the basic painting attributes.
 */
typedef struct {
	SYSDC_PATTERN	linePat,				// the line pattern 
					fillPat;				// the fill pattern 
	SYSDC_RGB		foregroundRGB,			// the foreground color
					backgroundRGB;			// the background color
	U16 			lineThickness;			// the line width
} PIC_SEG_PAINT, * P_PIC_SEG_PAINT;

/*
 * The polyline, polygon, and spline grafics provide line attributes.
 */
typedef struct {
	U8 join;
	U8 cap;
	U8 miterLimit;
	U8 spare;
} PIC_SEG_PLINE_TYPE, * P_PIC_SEG_PLINE_TYPE;

/*
 * Text style attributes.
 */
typedef struct PIC_SEG_FONT_STYLE{
	U16					alignChr	: 3,	 // see sysDcAlignChr???
						underline	: 2,	 // 0, 1, 2
						strikeout	: 2,	 // 0, 1
						spare		: 9;	 // spare - default 0
} PIC_SEG_FONT_STYLE, P_PIC_SEG_FONT_STYLE;

/*
 * The grafic.pData provided with grafic.opCode == opCodeText.
 */
typedef struct PIC_SEG_TEXT{
	PIC_SEG_PAINT		paint;
	RECT32				rectangle;
	SYSDC_FONT_SPEC		fontSpec;		// unique font
	PIC_SEG_FONT_STYLE	style;			
	SIZE16				size;			// size of text
	XY32				cp;				// text position
	U16					length;			// length of text
	U8					text[1];		// null terminated text
} PIC_SEG_TEXT, * P_PIC_SEG_TEXT;

/*
 * The grafic.pData provided with grafic.opCode == opCodeEllipse.
 */
typedef struct {
	PIC_SEG_PAINT		paint;
	RECT32				ellipse;
} PIC_SEG_ELLIPSE, * P_PIC_SEG_ELLIPSE;

/*
 * The grafic.pData provided with grafic.opCode == opCodeRectangle.
 */
typedef struct {
	PIC_SEG_PAINT		paint;
	RECT32				rectangle;
	S16					radius;				// The rectangle radius
											// 0 for square corners.
} PIC_SEG_RECT, * P_PIC_SEG_RECT;

/*
 * The grafic.pData provided with grafic.opCode == opCodePolyline.
 * The pData is of variable size depending on the number of points
 * (pData->count). For Example, the third point is pData->points[3].
 * The size of pData is:
 * (sizeof(PIC_SEG_POLYLINE) + sizeof(XY32) * ((pData->count)-1)).
 */
typedef struct {
	PIC_SEG_PAINT		paint;
	PIC_SEG_PLINE_TYPE	type;
	U16					count;			// number of points 
	XY32				points[1];		// variable number of points
} PIC_SEG_POLYLINE, * P_PIC_SEG_POLYLINE;

/*
 * The grafic.pData provided with grafic.opCode == opCodePolygon.
 * The pData is of varible size depending on the number of points
 * (pData->count). For Example, the third point is pData->points[3].
 * The size of pData is:
 * (sizeof(PIC_SEG_POLYGON) + sizeof(XY32) * ((pData->count)-1)).
 */
typedef struct {
	PIC_SEG_PAINT		paint;
	PIC_SEG_PLINE_TYPE	type;
	U16					count;			// number of points
	XY32				points[1];		// variable number of points
} PIC_SEG_POLYGON, * P_PIC_SEG_POLYGON;

/*
 * The grafic.pData provided with grafic.opCode == opCodeSpline.
 * A spline is a continuous number of four point Bezier curves.
 * The first Bezier is defined by the first four points in pData->points.
 * The second Bezier starts at pData->points[3]. The count minus one is
 * a multiple of three. msgDcDrawBezier stores as a spline.
 * The pData is of varible size depending on the number of points
 * (pData->count). For Example, the third point is pData->points[3].
 * The size of pData is:
 * (sizeof(PIC_SEG_SPLINE) + sizeof(XY32) * ((pData->count)-1)).
 */
typedef struct {
	PIC_SEG_PAINT		paint;
	PIC_SEG_PLINE_TYPE	type;
	U16					count;			// number of points 
	XY32				points[1];		// variable number of points
} PIC_SEG_SPLINE, * P_PIC_SEG_SPLINE;

/*
 * The grafic.pData provided with grafic.opCode == opCodeArcRays,
 * opCodeChordRays, or opCodeSectorRays.
 */
typedef struct {
	PIC_SEG_PAINT		paint;
	RECT32				bounds;
	XY32				rays[2];
} PIC_SEG_ARC_RAYS, * P_PIC_SEG_ARC_RAYS;

/*
 * The grafic.pData provided with grafic.opCode == opCodeObject.
 */

typedef struct {
	PIC_SEG_PAINT		paint;
	RECT32				rectangle;
	OBJECT				object;
} PIC_SEG_OBJECT, * P_PIC_SEG_OBJECT;

#define maxPolylineSize	((0xFFFF / sizeof(XY32)) - sizeof(PIC_SEG_POLYLINE))

/*
 * 
 */
typedef struct PIC_SEG_METRICS {
	U16					flags;
	MESSAGE				units;					// information only
	S32					numberGrafics;			// information only
	S32					currentGrafic;			// information only

	SYSDC_PATTERN		fillPat;				// attributes of the next
	SYSDC_PATTERN		linePat;				// drawn grafic
	SYSDC_RGB			foregroundRGB;
	SYSDC_RGB			backgroundRGB;
	SYSDC_LINE			line;

	SYSDC_PATTERN		clearFillPat;		   	// clear 
	SYSDC_PATTERN		clearLinePat;		   	
	SYSDC_RGB			clearForegroundRGB;
	SYSDC_RGB			clearBackgroundRGB;
											
	SYSDC_FONT_SPEC		fontSpec;				// font stuff 
	SIZE16				fontSize;
	PIC_SEG_FONT_STYLE	fontStyle;

	S32					reserved[5];
	S32					spare[8];				// reserved
} PIC_SEG_NEW_ONLY, PIC_SEG_METRICS,
  *P_PIC_SEG_NEW_ONLY, *P_PIC_SEG_METRICS;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDump								takes S32, returns STATUS
	category: class message
	Dumps a PicSeg.	Debug version only!

 pArgs == 0 everything and dc.
 pArgs == 1 PicSeg and metrics and does not Dump ancestor.
 pArgs == 2 PicSeg metrics only and does not Dump ancestor.
 pArgs == 3 PicSeg database only and does not Dump ancestor.
*/

/****************************************************************************
 msgNew								takes P_PIC_SEG_NEW, returns STATUS
	category: class message
	Creates a new PicSeg.
*/
#define	picSegNewFields		\
	sysdcNewFields			\
	PIC_SEG_NEW_ONLY	picSeg;

typedef struct PIC_SEG_NEW {
	picSegNewFields
} PIC_SEG_NEW, *P_PIC_SEG_NEW;

/****************************************************************************
 msgNewDefaults						takes P_PIC_SEG_NEW, returns STATUS
	category: class message
	Initializes a PIC_SEG_NEW structure to default values.

 Defaults:
//{
	picSeg.flags = picSegDraw | picSegAdd | picSegSendDestroy
	picSeg.units = msgDcUnitsPoints
	picSeg.currentGrafic = -1
	picSeg.fillPat = sysDcPatBackground
	picSeg.linePat = sysDcPatForeground
	picSeg.backgroundRGB.all = SysDcGrayRGB(255)
	picSeg.foregroundRGB.all = SysDcGrayRGB(0)

	picSeg.line.cap = 0
	picSeg.line.join = 0
	picSeg.line.miterLimit = 10
	picSeg.line.radius = 0
	picSeg.line.thickness = 1

	picSeg.clearFillPat = sysDcPatNil
	picSeg.clearLinePat = sysDcPatNil
	picSeg.clearForegroundRGB = SysDcGrayRGB(255)
	picSeg.clearBackgroundRGB = SysDcGrayRGB(0)

	picSeg.fontSpec.id = Nil
	picSeg.fontSpec.attr.group = sysDcGroupDefault
	picSeg.fontSpec.attr.weight = sysDcWeightNormal
	picSeg.fontSpec.attr.aspect = sysDcAspectNormal
	picSeg.fontSpec.attr.italic = false
	picSeg.fontSpec.attr.monospaced = false
	picSeg.fontSpec.attr.encoding = sysDcEncodeHWX850

	picSeg.fontSize.w = 1
	picSeg.fontSize.h = 1
	picSeg.fontStyle.alignChr = 0
	picSeg.fontStyle.underline = sysDcAlignChrBaseline
	picSeg.fontStyle.strikeout = 0
	picSeg.fontStyle.spare = 0
//}
*/

/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	category: class message
	Restores the PicSeg metrics and grafics and sets the DC state.

 The Restore doesn't connect the PicSeg to a window. Before using the PicSeg
 it must be set to a window with msgDcSetWindow.
*/

/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	category: class message
	Saves the PicSeg metrics and grafics and the DC units and LUC matrix.

 The Save doesn't save the window connected to the PicSeg.
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Drawing Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 * Messages of clsSysDrwCtx used by clsPicSeg:
 * 
 * All of the following messages draw the shape and add it as a grafic
 * to end of the of the PicSeg display list, provided the add and draw flags
 * are turned on. 
 * 
 * msgDcDrawEllipse, msgDcDrawRectangle, msgDcDrawPolyline, msgDcDrawPolygon,
 * msgDcDrawSectorRays, msgDcDrawArcRays, msgDcDrawChordRays, msgDcDrawBezier,
 * msgDcDrawText
 * 
 * PicSeg text defaults: spaceChar, spaceExtra, otherExtra
 * 
 * All of the following messages change the DC and also the PicSeg state.
 * PicSeg converts the x,y font scale to 16 bits dc units.
 * 
 * msgDcSetForegroundRGB, msgDcSetBackgroundRGB, msgDcSetLinePat,
 * msgDcSetFillPat, msgDcSetLine, msgDcSetLineThickness, msgDcOpenFont
 * msgDcScaleFont, msgDcIdentityFont, msgDcUnits...
*/

/****************************************************************************
 msgPicSegPaint								takes pNull, returns STATUS
	Paints the grafics in the PicSeg.

 Object Call either msgWinBeginPaint or msgWinBeginRepaint before
 using this message.
*/
#define msgPicSegPaint							MakeMsg(clsPicSeg, 7)

/****************************************************************************
 msgPicSegDrawSpline			   		takes P_PIC_SEG_SPLINE, returns STATUS
	Adds and draws the grafic to the end of the display list.
*/
#define msgPicSegDrawSpline			   			MakeMsg(clsPicSeg, 104)

/****************************************************************************
 msgPicSegDrawObject		 		takes P_PIC_SEG_OBJECT, returns STATUS
	Adds and draws an object to the PicSeg  display list.
*/
#define msgPicSegDrawObject						MakeMsg(clsPicSeg, 121)

/****************************************************************************
 msgPicSegPaintObject	  		takes P_PIC_SEG_PAINT_OBJECT, returns STATUS
	Sent by the PicSeg to objects in its database so they can draw themselves.
*/
#define msgPicSegPaintObject						MakeMsg(clsPicSeg, 46)

typedef struct {
	PIC_SEG_PAINT		paint;
	RECT32				rectangle;
	OBJECT				object;
	OBJECT				picSeg;
	S32					reserved[6];
} PIC_SEG_PAINT_OBJECT, * P_PIC_SEG_PAINT_OBJECT;

/****************************************************************************
 msgPicSegDrawGrafic	  			 	takes P_PIC_SEG_GRAFIC, returns STATUS
	Draws a grafic from the PicSeg.

 The grafic opCode must be set to one of the opCode's defined by PicSeg's.
 Can be used for HitTest on a specific grafic.
 Never adds the grafic to the PicSeg.
 Responds to flags picSegDraw.
*/
#define msgPicSegDrawGrafic						MakeMsg(clsPicSeg, 10)

/****************************************************************************
 msgPicSegDrawGraficIndex	   			   takes S32 index, returns STATUS
	Sets the current grafic to index and draws it.

	Can be used for HitTest on a specific grafic.
*/
#define msgPicSegDrawGraficIndex				MakeMsg(clsPicSeg, 11)

/****************************************************************************
 msgPicSegDrawGraficList	   		   takes P_PIC_SEG_LIST, returns STATUS
	Draws all the grafics indexed by the list.
*/
#define msgPicSegDrawGraficList					MakeMsg(clsPicSeg, 8)

typedef struct {
	S32 	count;			// number of grafic in list to draw
	P_S32	pIndex;			// pointer to the list of grafics
}PIC_SEG_LIST, * P_PIC_SEG_LIST;

/****************************************************************************
 msgPicSegAddGrafic	  			 	takes P_PIC_SEG_GRAFIC, returns STATUS
	Adds a grafic to the PicSeg and Draws the grafic.

 The grafic opCode must be set to one of the opCode's defined by PicSeg's.
 Responds to flags picSegAdd and picSegDraw.
*/
#define msgPicSegAddGrafic						MakeMsg(clsPicSeg, 9)

/****************************************************************************
 msgPicSegGetMetrics 			   takes P_PIC_SEG_METRICS, returns STATUS
	Passes back the metrics of the PicSeg.
*/
#define msgPicSegGetMetrics						MakeMsg(clsPicSeg, 3)

/****************************************************************************
 msgPicSegSetMetrics			   takes P_PIC_SEG_METRICS, returns STATUS
	Sets the metrics of the PicSeg.

 You cannot set picseg.numberGrafics.
*/
#define msgPicSegSetMetrics						MakeMsg(clsPicSeg, 4)

/****************************************************************************
 msgPicSegSetFlags							takes S32, returns STATUS
	Sets the PicSeg flags.
*/
#define msgPicSegSetFlags							MakeMsg(clsPicSeg, 5)

/****************************************************************************
 msgPicSegGetFlags							takes P_S32, returns STATUS
	Gets the PicSeg flags.
*/
#define msgPicSegGetFlags							MakeMsg(clsPicSeg, 6)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Hit Test 										   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 msgPicSegHitTest       takes P_PIC_SEG_HIT_LIST, returns STATUS
	Performs a hit test on the PicSeg, passing back a single grafic index.
	
 index - in:  First grafic to start hit test hit stops at grafic 0.
 Use picSegTopGrafic for starting at the top most grafic.
 out: The grafic hit if status is stsDcHitOn or stsDcHitIn. Otherwise 0.

 STATUS return:
	stsDcHitOn  if the line intersects the hit rectangle 
	stsDcHitIn  if the rectangle is inside a closed figure
	stsDcHitOut if there was no hit

 msgWinBeginPaint must be sent to the window first.
 msgWinEndPaint must be sent to the window after.

*/
#define msgPicSegHitTest				MakeMsg(clsPicSeg, 23)

typedef struct {
	RECT32 			rect;		// rectangle for the hit test
	S32  			index;		// in start grafic - out end grafic
} PIC_SEG_HIT_LIST, * P_PIC_SEG_HIT_LIST;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				Editing the PicSeg Display List				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 msgPicSegErase					   takes nothing, returns STATUS
	Deletes all grafics.
*/
#define msgPicSegErase							MakeMsg(clsPicSeg, 24)

/****************************************************************************
 msgPicSegDelete				   takes S32, returns STATUS
	Deletes a grafic, takes a grafic Index. Sends msgDestroy to
	objects in the PicSeg.
*/
#define msgPicSegDelete							MakeMsg(clsPicSeg, 26)

/****************************************************************************
 msgPicSegRemove				   takes S32, returns STATUS
	Deletes a grafic, takes a grafic Index. Does not send msgDestroy to
	objects in the PicSeg.
*/
#define msgPicSegRemove							MakeMsg(clsPicSeg, 45)

/****************************************************************************
 msgPicSegDelta					   takes P_PIC_SEG_GRAFIC, returns STATUS
	Changes the current grafic.
*/
#define msgPicSegDelta							MakeMsg(clsPicSeg, 27)

/****************************************************************************
 msgPicSegGetGrafic			 takes P_PIC_SEG_GRAFIC, returns STATUS
	Gets the current grafic.
	
 Data must be freed by caller.
*/
#define msgPicSegGetGrafic						MakeMsg(clsPicSeg, 28)

/****************************************************************************
 msgPicSegSetCurrent			   takes S32, returns STATUS
	Sets the current grafic index.
	
 Specifying picSegTopGrafic sets the current grafic to the last grafic
 in the list.
*/
#define msgPicSegSetCurrent						MakeMsg(clsPicSeg, 30)

/****************************************************************************
 msgPicSegGetCurrent			 			takes P_S32, returns STATUS
	Gets the index of the current grafic.
*/
#define msgPicSegGetCurrent						MakeMsg(clsPicSeg, 31)

/****************************************************************************
 msgPicSegGetCount						takes P_S32, returns STATUS
	Gets the number of grafics in the PicSeg.
*/
#define msgPicSegGetCount						MakeMsg(clsPicSeg, 32)

/****************************************************************************
 msgPicSegMakeInvisible							takes S32, returns STATUS
	Makes the given grafic invisible.

 Changes the grafics opCode by oring in opCodeMaskInvisible.
*/
#define msgPicSegMakeInvisible					MakeMsg(clsPicSeg, 33)

/****************************************************************************
 msgPicSegMakeVisible					   		takes S32, returns STATUS
	Makes the given grafic visible.

 Changes the grafics opCode by masking out opCodeMaskInvisible.
*/
#define msgPicSegMakeVisible					MakeMsg(clsPicSeg, 34)

/****************************************************************************
 msgPicSegChangeOrder			   takes S32, returns STATUS
	Changes the order of the grafics in the display, Moving the current
	grafic to the given index.


 If the given index is less than the current index, then the grafics
 in between shift forward.

 If the given index is greater than the current index, then the grafics
 in between shift backward.
*/
#define msgPicSegChangeOrder					MakeMsg(clsPicSeg, 35)

/****************************************************************************
 msgPicSegSizeof						takes P_PIC_SEG_GRAFIC, returns S32
	Returns the size of the (PIC_SEG_GRAFIC).pData in bytes.
*/
#define msgPicSegSizeof							MakeMsg(clsPicSeg, 39)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				Messages used for move copy								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 * You can move and copy grafics in picture segments using the
 * selection manager XFER mechanism type xferPicSegObject.
 * The PicSeg is a data object and only helps define the method.
 * The PicSeg itself does not have the selection. Usually the View,
 * using the PicSeg as its data object, responds
 * to move and copy messages. The selected View puts xferPicSegObject
 * on the list when it receives msgXferList. With a match the receiving
 * View creates a global heap PicSeg and sets up the XFER_OBJECT:
*/
/*
//{
	XFER_OBJECT			xferObject;
	OBJECT				picSeg;
	MAT					matrix;

	memset(&xferObject, 0, sizeof(XFER_OBJECT));
	xferObject.id = xferPicSegObject;
	xferObject.receiver = self;

	StsJmp(ObjectSendUpdate(msgXferGet, sel, &xferObject,\
	  (U32)sizeof(XFER_OBJECT)), sts, error);

	xferPicSeg = xferObject.uid;
	ObjectCall(msgDcSetWindow, xferPicSeg, (P_ARGS)self);
	ObjectCall(msgPicSegScaleUnits, xferPicSeg, (P_ARGS)psMetrics.units);
	matrix.m31 = pTip->x - bounds.origin.x;
	matrix.m32 = pTip->y - bounds.origin.y;
	MatIdentity(matrix);
	ObjectCall(msgPicSegTransform, xferPicSeg, &matrix);
	ObjectCall(msgPicSegCopy, picSeg, (P_ARGS)xferPicSeg);
	ObjectCall(msgDestroy, xferPicSeg, pNull);
//}
*/
/*
 * The receiving View then ObjectSends msgXferGet to the selection.
 * The selected View takes msgXferGet sets the xfer PicSeg's metrics
 * to its own and puts the selected grafics into the global PicSeg.
 * The receiving View must rebind the xfer PicSeg to a window using
 * msgDcSetWindow. Then transform the xfer PicSeg with msgPicSegScaleUnits
 * and msgPicSegTransform. The xferPicSeg is copied into the receiving View's
 * PicSeg with msgPicSegCopy. The global PicSeg is then freed by the
 * receiving View.
*/

#define tagPicSeg						MakeTag(clsPicSeg,0)

/****************************************************************************
 msgPicSegScaleUnits			   takes MESSAGE, returns STATUS
	Scales all coordinates in the PicSeg from the old units to the new
	units, then sets the units of the PicSeg to the new units.

 Valid arguments: msgDcUnitsMetric,	msgDcUnitsMil, msgDcUnitsPoints,
 msgDcUnitsTwips, msgDcUnitsPen, msgDcUnitsPen, msgDcUnitsDevice,
 msgDcUnitsLayout.

 Invalid arguments:	msgDcUnitsWorld.
*/
#define msgPicSegScaleUnits						MakeMsg(clsPicSeg, 36)


/****************************************************************************
 msgPicSegTransform					   		takes MAT, returns STATUS
	Transforms all coordinates in the PicSeg database with the
	provided matrix.

 Doesn't change line thickness, text size and rect radius.  Thus this
 message is best used for Rotation and Translation only.
*/
#define msgPicSegTransform						MakeMsg(clsPicSeg, 37)

/****************************************************************************
 msgPicSegCopy						   		takes OBJECT, returns STATUS
	Copies the contents of the specified PicSeg to self.

 Takes no account for units, scale, rotate and translate differences.
*/
#define msgPicSegCopy							MakeMsg(clsPicSeg, 38)
#endif	// PIC_SEG_INCLUDED





