/****************************************************************************
 File: osheap.h													

 (C) Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.35  $
   $Author:   ctaylor  $
	 $Date:   24 Feb 1992 10:39:44  $

 This file describes the heap memory management routines.

 Heaps are used to allocate local and shared memory efficiently.

 The functions described in this file are contained in PENPOINT.LIB. 
****************************************************************************/

/**** Introduction ****/
/*
 Heaps allocate regions of virtual memory and manage the allocation and
 freeing of smaller blocks within those regions.  
 
 Heaps have many different characteristics which are specified when the
 heap is created (see OSHeapCreate).  For example, heaps can be shared (i.e.
 put in the shared memory space) or local.

 A heap is identified by a heap handle.  PenPoint pre-defines two heap
 handles for each process, as described below.  OSHeapCreate also returns 
 the handle of a new heap.  Most heap routines take the heap handle as a
 parameter to identify the heap.
*/

/**** Pre-defined Heaps ****/
/*
 PenPoint pre-defines two heaps for every process.  These heaps can be used
 without calling OSHeapCreate.

 osProcessHeapId is the handle for the pre-defined local heap in each
 process.
 
 osProcessSharedHeapId is the handle for the shared heap.  The shared heap
 behavior is the same as the local heap except that the shared heap resides
 in shared memory.  Blocks allocated from the shared heap are accessible
 from any process.  
*/

/**** Quick Start ****/
/*
 Many clients call only the following functions, using one of the two
 pre-defined heaps.
	-:	OSHeapBlockAlloc
	-:	OSHeapBlockFree

 Clients who need to create their own heaps also call the following
 functions:
	-:	OSHeapCreate
	-:	OSHeapDelete
*/

/**** Debugging Flags ****/
/*
 Heap Manager debugging flag set is '*'.  Defined flags are:

	   1:  Validate heap before OSHeapBlockAlloc and before OSHeapBlockFree
	   2:  Display message for each heap block allocate and free
	   4:  Display message for each heap create and delete
	  10:  Validate heap after OSHeapBlockAlloc and after OSHeapBlockFree
	  20:  Display messages about internal region operation (private)
	1000:  Display messages about the internal workings (private)
	8000:  Enter the debugger after printing warnings.
*/

/**** Memory Overhead ****/
/*
 A heap consists of the memory allocated by the client plus the structures
 needed by the heap manager itself to maintain the heap.  This section
 describes the overhead imposed by these structures.

 A heap is constructed as a collection of REGIONS.  The overhead for a
 region is 36 bytes. By default, regions are 16Kb long; however, a request
 larger than ~16K causes the creation of a special region whose size is a
 multiple of 4K and large enough to handle the request.

 Each region have any number of allocated blocks within it.  The overhead
 of an allocated block (beyond the size requested) is 4 bytes, plus 0-3
 bytes as necessary to pad the whole block up the nearest 32-bit boundary.
*/

#ifndef OSHEAP_INCLUDED
#define OSHEAP_INCLUDED

#ifndef GO_INCLUDED
  #include <go.h>
#endif
#ifndef OSTYPES_INCLUDED
  #include <ostypes.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Heap attributes for OSHeapCreate */
Enum16(OS_HEAP_MODE) {

  osHeapLocal = 0,				// heap is local to the owning process
  osHeapShared = flag0,			// heap is accessible by all processes

  osHeapReadWrite = 0,			// heap is writable
  osHeapReadOnly = flag1,		// heap is only readable

  osHeapOptSpace = 0,			// heap is optimized for space
  osHeapOptTime = flag2,		// heap is optimized for speed

  osHeapWaitForMem = 0,			// wait for memory to become available
  osHeapOutOfMemErrOK = flag3	// doesn't wait, returns out-of-memory error

  // flags 5-10 reserved as supervisor flags

};

/* Heap information */
typedef struct OS_HEAP_BLOCK_INFO {

	SIZEOF		numBlocks;		// total number of blocks
	SIZEOF		totalSize;		// total # bytes in all blocks
	SIZEOF		minSize;		// # bytes in smallest block
	SIZEOF		maxSize;		// # bytes in largest block

} OS_HEAP_BLOCK_INFO, * P_OS_HEAP_BLOCK_INFO;

typedef struct OS_HEAP_INFO {				// info on a given heap

	OS_HEAP_BLOCK_INFO		alloc;			// info for allocated blocks
	OS_HEAP_BLOCK_INFO		free;			// info for free blocks

	U32						numRegions;		// # regions in heap
	U32						committedSize;	// # bytes committed
	U32						decommittedSize;// # bytes decommitted
	U32						reservedSize;	// # bytes reserved

	U32						numOwners;		// # tasks which have heap open
	OS_HEAP_MODE			heapMode;		// Mode used in heap creation

} OS_HEAP_INFO, * P_OS_HEAP_INFO;


#define OSTaskSharedHeapId(t) ((OS_HEAP_ID)OSTaskProcess(t))


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Functions								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 OSHeapCreate	returns STATUS
	Creates a heap.

 The size of the initial region allocated by the heap manager is a
 parameter to OSHeapCreate.  If the amount of memory required by the heap
 is more than the size of the initial region, the heap manager allocates
 additional regions of 16K or the last request size, whichever is larger. 
 An initial region size of 0 will default to 16K.

 Return Value
	stsOSRequestTooBig:		The requested size is greater than maxS32.
	stsOutOfMem:			The heap cannot be created because there is
							not enough memory available within the system.
	stsBadParam:			The mode parameter specified an illegal mode.

 See Also
	OSHeapDelete 
*/
STATUS EXPORTED OSHeapCreate(
	OS_HEAP_MODE	mode,		// heap create mode
	SIZEOF			size,		// initial region size
	P_OS_HEAP_ID	pHeapId		// Out: heap id
);


/****************************************************************************
 OSHeapDelete	returns STATUS
	Deletes a heap.  Frees all memory allocated by clients and by the heap
	manager.

 Even heap blocks that are still allocated are deleted. 
 
 If other tasks have opened the heap (using OSHeapOpen), the heap is not
 actually deleted until all tasks that have opened the heap have closed it
 (using OSHeapClosed).  Note that this routine is similar to calling
 OSHeapClose with the current task.

 Return Value
	stsOSInvalidHeapId:  The heapId was invalid or inaccessible.

 See Also
	OSHeapCreate
	OSHeapOpen
	OSHeapClose
*/
STATUS EXPORTED OSHeapDelete(
	OS_HEAP_ID	heapId			// heap id of heap to delete
);


/****************************************************************************
 OSHeapAllowError	returns OS_HEAP_ID
	Changes the "out of memory" behavior of heap block allocation.

 Normally when a heap block is requested, the heap manager returns only
 when the memory is available.	Calling OSHeapAllowError changes the heap so
 that if the system has insufficient memory the heap manager returns
 immediately with stsOutOfMem.
*/
#define OSHeapAllowError(heap)	\
	((OS_HEAP_ID)((U32)(heap)|osHeapIdOutOfMemErrOKBit))
#define	osHeapIdOutOfMemErrOKBit	flag0


/****************************************************************************
 OSHeapClear	returns STATUS
	Clears a heap.	Deletes all the allocated heap blocks but not the heap.

 Return Value
	stsOSHeapOpen:  		Heap has multiple owners and cannot be cleared.
	stsOSInvalidHeapId:	The heapId was invalid or inaccessible.

 See Also
	OSHeapDelete
*/
STATUS EXPORTED OSHeapClear(
	OS_HEAP_ID	heapId			// heap id of heap to clear
);

/****************************************************************************
 OSHeapBlockAlloc	returns STATUS
	Allocates a block within the heap.

 The memory for the heap block is obtained from the list of regions in the
 heap.  If a heap allocate request is larger than the available space in
 the region, a new region is allocated for the request.

 The newly allocated block is at least as large as the requested length.
 Sometimes, the heap manager allocates a block larger than the requested
 size.  Heap blocks are always allocated on 32-bit boundaries.

 Heap blocks are allocated on behalf of the creator of the heap.  Even if
 the allocate occurs in a different task than the creator, the new memory
 is owned by the creator of the heap.

 WARNING.  This function expects a valid heap identifier.  Using an invalid
 heap identifer can cause unpredictable results (including a page fault).
 A heapId for a heap that has been deleted is considered to be invalid.

 See Also
	OSHeapBlockFree
	OSHeapBlockSize

 Return Value
	stsOSRequestTooBig:			The requested block size greater than maxS32.
	stsOutOfMem:				The heap cannot grow any bigger because the
								system is out of memory.
	stsOSInvalidHeapId:			The heapId given is invalid.
	stsOSHeapIntegrityError:	The heap has been corrupted (heap flag 1).
*/
STATUS EXPORTED OSHeapBlockAlloc(
	OS_HEAP_ID	heapId,			// heap id
	SIZEOF		size,			// size of block to allocate
	PP_UNKNOWN	ppHeapBlock		// Out: pointer to new heap block
);

/****************************************************************************
 OSHeapBlockFree	returns STATUS
	Frees a heap block.

 WARNING.  This function expects a valid heap block.  Using an invalid
 heap block can cause unpredictable results (including a page fault).

 See Also
	OSHeapBlockAlloc

 Return Value
	stsOSInvalidHeapId:			The heapId given is invalid.
	stsOSHeapIntegrityError:	The heap has been corrupted (heap flag 1)
								or heap block pointer is bad (debug only).
	stsBadParam:				The heap block pointer is bad (debug only).
*/
STATUS EXPORTED OSHeapBlockFree(
	P_UNKNOWN	pHeapBlock		// pointer to heap block
);

/****************************************************************************
 OSHeapBlockResize	returns STATUS
	Resizes a heap block.

 The heap block is resized to the new size.  This may be slightly faster
 than allocating a new block and copying the original block's contents.

 After the call the heap block may be identified with a new pointer value,
 which is returned in *ppHeapBlock.
 
 The actual size of the new heap block may be slightly larger than the
 request.

 WARNING.  This function expects a valid heap block.  Using an invalid
 heap block can cause unpredictable results (including a page fault).
*/
STATUS EXPORTED OSHeapBlockResize(
	SIZEOF		newSize,		// new size to allocate
	PP_UNKNOWN	ppHeapBlock		// Out: New pointer is returned here.
);

/****************************************************************************
 OSHeapId	returns OS_HEAP_ID
	Passes back the heap id from which a heap block has been allocated.

 WARNING.  This function expects a valid heap block.  Using an invalid
 heap block can cause unpredictable results (including a page fault).
*/
OS_HEAP_ID EXPORTED OSHeapId(
	P_UNKNOWN	pHeapBlock		// pointer to a heap block
);

/****************************************************************************
 OSHeapBlockSize	returns STATUS
	Passes back the size of the heap block.

 The size returned is the actual size of the heap block.  This may be
 slightly larger than the requested size.

 WARNING.  This function expects a valid heap block.  Using an invalid
 heap block can cause unpredictable results (including a page fault).

 See Also
	OSHeapBlockAlloc
	OSHeapBlockResize
*/
STATUS EXPORTED OSHeapBlockSize(
	P_UNKNOWN	pHeapBlock,		// pointer to the heap block
	P_SIZEOF	pSize			// Out: size of the heap block
);

/****************************************************************************
 OSHeapPoke returns STATUS
	Stores 32 bits of client info in the heap header.

 The client info is not interpreted by the heap manager.

 There is only client info field per heap;  if more than one call is made
 to OSHeapPoke, the most recent caller determines the value stored.

 WARNING.  This function expects a valid heap identifier.  Using an invalid
 heap identifer can cause unpredictable results (including a page fault). 
 An heapId for a heap that has been deleted is considered to be invalid.
*/
STATUS EXPORTED OSHeapPoke(
	OS_HEAP_ID	heapId,			// heap id
	P_UNKNOWN	info			// uninterpreted pointer stored in heap header
);

/****************************************************************************
 OSHeapPeek returns STATUS
	Passes back the client info previously set via OSHeapPoke().

 WARNING.  This function expects a valid heap identifier.  Using an invalid
 heap identifer can cause unpredictable results (including a page fault). A
 heapId for a heap that has been deleted is considered to be invalid.
*/
STATUS EXPORTED OSHeapPeek(
	OS_HEAP_ID	heapId,			// heap id
	PP_UNKNOWN	pInfo			// Out: pointer stored by OSHeapPoke
);


/****************************************************************************
 OSHeapInfo returns STATUS
	Passes back information on a heap.

 Return Value
	stsOSInvalidHeapId:			The heapId was invalid or inaccessible.
	stsOSHeapIntegrityError:	The heap has been corrupted.
								Under debug version additional info is 
								printed.
*/
STATUS EXPORTED OSHeapInfo(
	OS_HEAP_ID		heapId,			// heap id
	SIZEOF			heapInfoSize,	// size of heap info buffer
	P_OS_HEAP_INFO	pHeapInfo		// Out: heap info buffer
);

/****************************************************************************
 OSHeapOpen returns STATUS
	Adds the specified task as an owner of the specified heap.

 Heaps are owned by the task that creates them.  When the task is destroyed
 the heap is automatically destroyed.  If one task wants to access another
 task's heap, the heap should be opened.  Opening a heap is not required,
 but if the task owning the heap is destroyed while the second task is
 accessing the heap, the second task will fault.

 Memory resources allocated in the heap are not actually destroyed until
 the last owner of the heap deletes the heap.  Note that if the heap is
 opened multiple times by the same owner, a corresponding OSHeapClose or
 OSHeapDelete must occur for each before resources are deallocated.

 The kernel automatically destroys heap resources when all of the owners of
 the heap have terminated.

 The heap is automatically opened on the behalf of the creator during an
 OSHeapCreate.

  Return Value
	stsOSInvalidHeapId:		The heap must be a shared heap to be opened,
							the heapId was invalid or inaccessible.

 See Also
	OSHeapCreate
	OSHeapClose
*/
STATUS EXPORTED OSHeapOpen(
	OS_HEAP_ID heapId,				// heap id
	OS_TASK_ID taskId				// task to add as an owner
);


/****************************************************************************
 OSHeapClose returns STATUS
	Remove the specifed task as an owner of the specified heap.

 When the heap has been closed by the last owner, the heap is automatically
 deleted.

 Return Value
	stsOSInvalidHeapId:  The heapId was invalid or inaccessible.

 See Also
	OSHeapClose
	OSHeapCreate
*/
STATUS EXPORTED OSHeapClose(
	OS_HEAP_ID heapId,				// heap id
	OS_TASK_ID taskId				// task to remove as an owner
);


/****************************************************************************
 OSHeapEnumerate	returns STATUS
	Enumerates all the heaps in the given process.

 For each heap in the current process, OSHeapEnumerate calls the supplied
 callback procedure.  This routine is supplied with a heapId and its mode.

 OSHeapEnumerate continues until it has exhausted all the heaps in the
 current process or the callback routine returns an error status.  If the
 callback procedure returns an error status, processing is terminated and
 the error status is returned to the caller of OSHeapEnumerate.

 Return Value
	stsOSInvalidHeapId:  The heapId was invalid or inaccessible.

 See Also
	OSHeapWalk
*/

typedef STATUS FunctionPtr(P_OS_HEAP_ENUMERATE) (
	OS_HEAP_ID			heapId,			// next heap
	OS_HEAP_MODE		heapMode,		// mode of heap
	P_UNKNOWN			clientData		// client data of OSHeapEnumerate
);

STATUS EXPORTED OSHeapEnumerate(
	P_OS_HEAP_ENUMERATE pEnumProc,
	P_UNKNOWN			clientData		// passed EnumProc on each call
);


/****************************************************************************
 OSHeapWalk		returns STATUS
	Traverses the given heap.

 For each allocated block in the given heap, calls the supplied callback
 routine, providing the address and size of the block. OSHeapWalk continues
 until it has exhausted all allocated blocks in the heap or the callback
 routine returns an error status. If the callback procedure returns an
 error status, processing is terminated and the error status is immediately
 returned to the caller of OSHeapWalk.

 Return Value
	stsOSInvalidHeapId:  The heapId was invalid or inaccessible.

 See Also
	OSHeapEnumerate
*/

typedef struct OS_HEAP_WALK_INFO {

	P_UNKNOWN	pBlock;			// address of heap block
	U32			size;			// size of block
	BOOLEAN		inUse;			// true if the block is allocated
	
	P_UNKNOWN	clientData;		// set to the client data of OSHeapWalk

	// The following fields are only supported by a debugging version of
	// PenPoint's kernel.  Changing their value modifies the heap block.

	BOOLEAN		marked;			// true if the block was marked w/OSHeapMark
	OS_TASK_ID	owner;			// last task to allocate or free this block
	P_UNKNOWN	caller;			// address of the last OSHeapBlockAlloc/Free

} OS_HEAP_WALK_INFO, * P_OS_HEAP_WALK_INFO;

typedef STATUS FunctionPtr(P_OS_HEAP_WALK)(P_OS_HEAP_WALK_INFO pInfo);

STATUS EXPORTED OSHeapWalk(
	OS_HEAP_ID		heapId,			// heap to walk
	P_OS_HEAP_WALK	pWalkProc,		// procedure to call for each heap block
	P_UNKNOWN		clientData		// passed directly to pWalkProc
);


/****************************************************************************
 OSHeapMark		returns STATUS
	Marks all the allocated blocks in given heap.

 Combining OSHeapMark with OSHeapWalk provides a simple means to track
 down storage leaks.  For example:
//{
	// Program is in a known state
	OSHeapMark(myHeap);

	// Lots of OSHeapBlockAlloc/Free calls
	OSHeapBlockAlloc(myHeap, xx, &blk);
	OSHeapBlockFree(blk);

	// Program is back to the known state.
	// Any unmarked heap blocks probably indictate a storage leak
	OSHeapWalk(myHeap, MyHeapWalker);
//}

 Return Value
	stsOSInvalidHeapId:  The heapId was invalid or inaccessible.

 See Also
	OSHeapWalk
*/
STATUS EXPORTED OSHeapMark(
	OS_HEAP_ID heapId				// heap to mark
);	


/****************************************************************************
 OSHeapPrint	returns STATUS
	Prints debugging info about the given heap.

 OSHeapPrint is only available in a debugging version of the PenPoint
 kernel.  This request is not supported in production versions of Penpoint.

 OSHeapPrint assumes the heap is not corrupted;  in other words,
 OSHeapPrint does not duplicate any of the integrity tests done by
 OSHeapInfo.

 Return Value
	stsOSInvalidHeapId:  The heapId was invalid or inaccessible.
*/

/* Flags for OSHeapPrint */
typedef enum OS_HEAP_PRINT_FLAGS {

	osHeapSuppressFree =		flag0,	// Don't print the free blocks
	osHeapSuppressInUse	=		flag1,	// Don't print the allocated blocks
	osHeapSuppressMarked =		flag2,	// Don't print the marked blocks
	osHeapSuppressUnmarked =	flag3,	// Don't print the unmarked blocks
	osHeapSuppressSummary =		flag4,	// Don't print the heap summary
	osHeapDisplayRegions =		flag5,	// Print regions in heap

	osHeapPrintAll =			0,		// Display summary and all blocks
	osHeapPrintSummaryOnly =			// Display summary
		osHeapSuppressFree|osHeapSuppressInUse|
		osHeapSuppressMarked|osHeapSuppressUnmarked,

	// Show blocks created since the last call to OSHeapMark 
	osHeapPrintActiveBlocks = osHeapSuppressFree|osHeapSuppressMarked

} OS_HEAP_PRINT_FLAGS;


STATUS EXPORTED OSHeapPrint(OS_HEAP_ID heapId, OS_HEAP_PRINT_FLAGS suppress);

#endif
