/****************************************************************************
 File: note.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.35  $
   $Author:   sbartlet  $
     $Date:   26 Mar 1992 09:20:56  $

 This file contains the API for clsNote.

 clsNote inherits from clsFrame.
 Provides the UI for system- and app-modal messages to the user.
****************************************************************************/


/**** Debugging Flags ****/
/*
 The clsNote debugging flag is 'K'.  Defined values are:

    flag15 (0x8000):    general
*/

#ifndef NOTE_INCLUDED
#define NOTE_INCLUDED

													#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
													#endif
													#ifndef WIN_INCLUDED
#include <win.h>
													#endif
													#ifndef FRAME_INCLUDED
#include <frame.h>
													#endif
													#ifndef TKTABLE_INCLUDED
#include <tktable.h>
													#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Common #defines and typedefs					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef WIN		NOTE, *P_NOTE;

// Tags of component windows within a note.
#define tagNoteTitle		MakeTag(clsNote, 1)
#define tagNoteTkTable		MakeTag(clsNote, 2)
#define tagNoteCmdBar		MakeTag(clsNote, 3)

// Note flags
#define nfSystemModal      flag0
#define nfAutoDestroy      flag1
#define nfSystemTitle      flag2  // use system title; ignore pTitle
#define nfAppTitle         flag3  // use app title; ignore pTitle
#define nfUnformattedTitle flag9  // use pTitle as is (not "Note from <pTitle>")
#define nfTimeout          flag4  // dismiss on timeout or input
#define nfNoWordWrap       flag5  // don't word wrap content labels
#define nfResContent       flag6  // pContentEntries is P_NOTE_RES_ID
#define nfNoBeep           flag7  // disable prefs-controlled beeping
#define nfExplicitCancel   flag8  // note will ignore cmdBar buttons
#define nfDefaultSysFlags  \
					(nfSystemModal | nfAutoDestroy | nfSystemTitle | nfNoBeep)
#define nfDefaultAppFlags  (nfAppTitle | nfNoBeep)
#define nfDefaultFlags     nfDefaultSysFlags

typedef struct NOTE_METRICS {
	U16              flags;           // looks and filter flags
	MESSAGE          autoDismissMsg;  // returned iff win dismissed
	OBJECT           modalFilter;     // filter or objNull for default
	OS_MILLISECONDS  timeout;         // timeout or 0 for user pref
	OBJECT           client;          // client for msgNoteDone
	U32              spare;           // reserved
} NOTE_METRICS, *P_NOTE_METRICS;

/****************************************************************************
 msgNew			takes P_NOTE_NEW, returns STATUS
	category: class message
	Creates a note.

 The fields you commonly set are:
	pArgs->note.flags:				appropriate flags
	pArgs->note.autoDismissMsg:		arg for msgNoteCancel
	pArgs->note.timeout:			timeout if desired
	pArgs->note.client:				client if app-modal

 clsNote will create all the appropriate interior windows, then self-send
 msgWinLayout to size and place all the windows.  After that, if either
 the x or y of the note's origin is 0, clsNote will delta the new instance
 so that when it is inserted as a child of theRootWindow the note will
 appear in a reasonable location.

 To display and activate the note, use msgNoteShow.
*/

typedef struct {
	NOTE_METRICS		metrics;
	P_CHAR				pTitle;
	P_UNKNOWN			pContentEntries;	// used to create the content
	P_TK_TABLE_ENTRY	pCmdBarEntries;		// used to create the command bar
	U32					spare;				// reserved
} NOTE_NEW_ONLY, *P_NOTE_NEW_ONLY;

/*
 If nfSystemModal is on, then the client is ignored.
 If nfSystemModal is off, then msgNoteShow returns immediately,
 and the client will be sent msgNoteDone when the note is dismissed.
 
 If pTitle will be used (nfSystemTitle and nfAppTitle are off), the title
 will appear as follows:
	"Note from <pTitle>..." if nfUnformattedTitle is off
	"<pTitle>" if nfUnformattedTitle is on
*/

#define noteNewFields	\
	frameNewFields		\
	NOTE_NEW_ONLY		note;
        
typedef struct NOTE_NEW {
	noteNewFields
} NOTE_NEW, *P_NOTE_NEW;

typedef struct {
	RES_ID	resId;		// resId for a string table resource
	U32		index;		// index within that table of a string
	U32		spare;		// reserved (unused)
} NOTE_RES_ID, *P_NOTE_RES_ID;

/*
 clsNote will use msgResReadData to read the string from either
 OSThisApp()'s APP_METRICS.resList, or theSystemResFile if OSThisApp()
 returns objNull.
*/

/*
 Since clsNote will make a label from the string and clsLabel will break
 word-wrapped labels at newlines ('\n'), you may embed newlines in the
 string to force line breaks.
*/


/****************************************************************************
 msgNewDefaults			takes P_NOTE_NEW, returns STATUS
	category: class message
	Initializes the NOTE_NEW structure to default values.

 Zeroes out pArgs->note and sets:
//{
	pArgs->win.flags.style
		|= wsSaveUnder | wsShrinkWrapWidth | wsShrinkWrapHeight;
	pArgs->border.style.resize = false;
	pArgs->border.style.drag = bsDragNone;
	pArgs->customLayout.style.limitToRootWin = true;
	pArgs->frame.style.closeBox = false;
	pArgs->frame.style.zoomable = false;

	pArgs->note.metrics.flags = nfDefaultFlags;
//}
*/

/****************************************************************************
 msgNoteGetMetrics			takes P_NOTE_METRICS, returns STATUS
	Get the metrics of a note.
*/
#define msgNoteGetMetrics	MakeMsg(clsNote, 1)


/****************************************************************************
 msgNoteSetMetrics			takes P_NOTE_METRICS, returns STATUS
	Set the metrics of a note.

 clsNote will destroy any previous filter object if the filter is changed.
*/
#define msgNoteSetMetrics	MakeMsg(clsNote, 2)


/****************************************************************************
 msgNoteShow				takes P_MESSAGE, returns STATUS
	Displays a note.

 If nfSystemModal is on, then the send of this message will block until
 the note is dismissed.  At that time, msgNoteShow will set *pArgs
 to the message sent by the button that was hit (or autoDismissMsg if the
 win was dismissed by its modal filter).  Be aware that the entire
 input system (and therefore the window system) will be blocked while
 msgNoteShow is waiting for completion.

 If nfSystemModal is off, then msgNoteShow returns immediately.  It is
 the app's responsibility to implement whatever notion of "modality"
 is appropriate.  Usually this means remembering that the app should
 be "modal" and waiting for msgNoteDone to be sent to the note's client
 (which should usually be the app object).  Although the note will
 filter all the input to the app and discard that input not directed
 at the note, the app must still respond to messages from the app
 framework.  When nfSystemModal is off, the *pArgs to msgNoteShow
 is not set.
*/
#define msgNoteShow			MakeMsg(clsNote, 3)


/****************************************************************************
 msgNoteDone				takes MESSAGE, returns STATUS
	This is the message sent to clients when a note is dismissed.

 msgNoteDone is only sent if nfSystemModal is off.

 The parameter message is the message sent by the button that was hit
 (or autoDismissMsg if the win was dismissed by its modal filter).
*/
#define msgNoteDone			MakeMsg(clsNote, 4)


/****************************************************************************
 msgNoteCancel				takes P_MESSAGE, returns STATUS
	Informs a note that it should take itself down.

 This will be posted to a note when:
	o	it receives msgButtonNotify from its command bar, or
	o	it receives msgModalFilterDismissWin from its filter.
 The method code will do all the final cleanup, including extracting
 the note window (and destroying it if nfAutoDestroy was on).
 The *pArgs message will either be returned to the original code
 that called msgNoteShow (if nfSystemModal is on), or passed to
 msgNoteDone (if nfSystemModal is off).

 This message is only interesting to subclasses of clsNote.
 It should not be used by normal clients.
*/
#define msgNoteCancel		MakeMsg(clsNote, 5)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from Other Classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgFree	takes OBJ_KEY, returns STATUS
	Sent as the last of three msgs to destroy an object.

 clsNote will use InputFilterRemove() to take its filter out of the input
 system's list of filters if the filter is active.  clsNote will then
 send msgDestroy to its filter if the note had created it (as opposed to
 the client passing in a filter).
*/


/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.

 clsNote will restore its flags, autoDismissMsg, and timeout.
*/


/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Causes an object to file itself in an object file.

 clsNote will file its flags, autoDismissMsg, and timeout.  It will not
 file its modalFilter or client.
*/


/****************************************************************************
 msgWinSend 	takes P_WIN_SEND, returns STATUS
    Sends a message up a window ancestry chain.

 The note may respond by posting itself msgNoteCancel (passing a pointer
 to its autoDismissMsg), depending on the pArgs->msg and the
 nfExplicitCancel flag.
*/


/****************************************************************************
 msgWinLayoutSelf 	takes P_WIN_METRICS, returns STATUS
    Tells a window to layout its children (sent during layout).

 If wsLayoutResize is on and nfNoWordWrap is off and the note is
 shrinkwrapping in width, the note might further adjust the results of the
 default layout (obtained by just calling ancestor).  The note's width
 will be forced wider if the height of the initial layout is taller than
 dictated by the 'golden section' ratio of h/w = 0.618.
*/


/****************************************************************************
 msgGWinGesture		takes P_GWIN_GESTURE, returns STATUS
	Self-sent to process the gesture.

 clsNote will just return the result of calling its ancestor if the note
 has buttons (i.e., NOTE_NEW_ONLY had a non-null pCmdBarEntries).

 Otherwise, the note will post itself msgNoteCancel, passing a pointer
 to its autoDismissMsg.  Although clsNote should check the nfExplicitCancel
 flag, it does not yet do so for msgGWinGesture (although this may change
 in the future).

 See Also
	msgNoteCancel:	tells a note to take itself down.
*/


/****************************************************************************
 msgModalFilterDismissWin			takes nothing, returns STATUS
	category: third-party notification
	Sent to the subTreeRoot if the win should be dismissed.

 The note will respond by posting itself msgNoteCancel, passing a pointer
 to its autoDismissMsg.
*/


/****************************************************************************
 msgTimerNotify takes P_TIMER_NOTIFY, returns nothing
    category: advisory message
    Notifies the client that the timer request has elapsed.  

 A note may receive this when a non-zero NOTE_METRICS.timeout was specified
 and the note was displayed via msgNoteShow.  If this msgTimerNotify does
 indeed signify that the note should take itself down, the note will do
 so by posting itself msgNoteCancel (passing a pointer to its autoDismissMsg).
*/


#endif
