/****************************************************************************
 File: mbutton.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.59  $
   $Author:   kcatlin  $
     $Date:   10 Mar 1992 13:25:16  $

 This file contains the API definition for clsMenuButton.

 clsMenuButton inherits from clsButton.
 Menu buttons support an optional pull-down or pull-right pop-up menu.
****************************************************************************/
#ifndef MBUTTON_INCLUDED
#define MBUTTON_INCLUDED

												#ifndef BUTTON_INCLUDED
#include <button.h>
												#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT MENU_BUTTON;

/****  Submenu Types  ****/
#define	mbMenuNone			0	// no sub-menu defined
#define	mbMenuPullDown 		1	// sub-menu is pull-down
#define	mbMenuPullRight		2  	// sub-menu is pull-right
#define	mbMenuPopup			3  	// sub-menu is popup
#define	mbMenuSibling		4  	// sub-menu is a window sibling
//							5  	// unused (reserved)
//							.. 	// unused (reserved)
//							7 	// unused (reserved)

/****  Menu Actions  ****/
#define	mbAction1Tap		0	// menu up/down on xgs1Tap or msgPenUp
#define	mbAction2Tap		1	// menu up/down on xgs2Tap

typedef struct MENU_BUTTON_STYLE {
	U16	subMenuType			: 3,	// sub-menu type
		getWidth			: 1,	// self-send msgMenuButtonProvideWidth
		getMenu				: 1,	// send msgMenuButtonProvideMenu to client
		enableMenu			: 1,	// send msgControlEnable to menu
		menuAction			: 2,	// action to bring up/down menu
		menuIsUp			: 1,	// read-only: true => menu is up
		spare				: 7;	// unused (reserved)
} MENU_BUTTON_STYLE, *P_MENU_BUTTON_STYLE;

/*
 Default MENU_BUTTON_STYLE:
//{
 	subMenuType		= mbMenuNone
 	getWidth			= false
 	getMenu			= false
 	enableMenu		= false
   menuAction		= mbAction1Tap
 	menuIsUp			= false
//}
*/

typedef struct MENU_BUTTON_PROVIDE_MENU {
	MENU_BUTTON		menuButton;		// In: requestor
	WIN				menu;			// In/Out: uid of menu
	U32				spare;			// reserved (unused)
} MENU_BUTTON_PROVIDE_MENU, *P_MENU_BUTTON_PROVIDE_MENU;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew		takes P_MENU_BUTTON_NEW, returns STATUS
	category: class message
	Creates a menu button window.

 The fields you commonly set are:
	pArgs->menuButton.style.subMenuType:	kind of subMenu
	pArgs->menuButton.menu:					uid of menu window

 If pArgs->menuButton.style.subMenuType is mbMenuPullRight,
 sets pArgs->label.style.decoration to lsDecorationPullRight.

 If pArgs->menuButton.style.subMenuType is not mbMenuNone,
 sets pArgs->button.style.contact to bsContactToggle.

 If pArgs->menuButton.menu is not objNull, it self-sends msgWinSetPopup 
 with WIN_METRICS parameters of child = menu;
*/

typedef struct MENU_BUTTON_NEW_ONLY {
	MENU_BUTTON_STYLE	style;	// overall style
	WIN					menu;	// sub-menu or objNull
	U32					spare1;	// unused (reserved)
	U32					spare2;	// unused (reserved)
} MENU_BUTTON_NEW_ONLY, *P_MENU_BUTTON_NEW_ONLY;

#define	menuButtonNewFields		\
	buttonNewFields				\
	MENU_BUTTON_NEW_ONLY		menuButton;

typedef struct MENU_BUTTON_NEW {
	menuButtonNewFields
} MENU_BUTTON_NEW, *P_MENU_BUTTON_NEW;


/****************************************************************************
 msgNewDefaults				takes P_MENU_BUTTON_NEW, returns STATUS
	category: class message
	Initializes the MENU_BUTTON_NEW structure to default values.

 Zeroes out pArgs->menuButton and sets:
//{
	pArgs->win.flags.style |= wsFileNoBounds;

	pArgs->border.style.edge = bsEdgeNone;
	pArgs->border.style.join = bsJoinSquare;
	pArgs->border.style.shadow = bsShadowNone;

	pArgs->gWin.style.gestureEnable = false;

	pArgs->control.style.showDirty = false;

	pArgs->label.style.xAlignment = lsAlignLeft;
	pArgs->label.style.yAlignment = lsAlignBottom;
//}
*/


/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Causes an object to file itself in an object file.

	If the menu button has a menu, and the menu has wsSendFile on, msgSave 
	will be sent to the menu passing along pArgs.
*/

/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.
	
	clsMenuButton restores the instance from the file.  If the menu button
	had a menu when filed, the menu is restored and the following is done:
		Sends msgTkTableSetManager, with pArgs of self to the menu.
		Self-sends msgWinSetPopup with WIN_METRICS parameters of
			child	= menu;
*/

/****************************************************************************
 msgFree	takes OBJ_KEY, returns STATUS
	Sent as the last of three msgs to destroy an object.

	If the menu button has a menu, msgDestroy is sent to the menu.
*/


/****************************************************************************
 msgMenuButtonGetStyle		takes P_MENU_BUTTON_STYLE, returns STATUS
	Passes back the current style values.
*/
#define msgMenuButtonGetStyle	MakeMsg(clsMenuButton, 1)


/****************************************************************************
 msgMenuButtonSetStyle		takes P_MENU_BUTTON_STYLE, returns STATUS
	Sets the style values.

 Note that style.menuIsUp is read-only -- pArgs->menuIsUp will be ignored.

 If style.subMenuType changes the following is done:
	-: if style.subMenuType is mbMenuPullRight, the label.style.decoration
	   is set to to lsDecoratePullRight, otherwise it is set to lsDecorateNone.

	-: if the menu button has a menu, button.style.contact is set to
	   bsContactToggle, otherwise bsContactMomentary.

	-: if the menu button has a menu, self-sends msgWinSetPopup
	   with WIN_METRICS parameters of
			child	= menu;
*/
#define msgMenuButtonSetStyle	MakeMsg(clsMenuButton, 2)


/****************************************************************************
 msgMenuButtonGetMenu		takes P_MENU, returns STATUS
	Passes back the menu, objNull if none.
  
 Note that this message does not result in msgMenuButtonProvideMenu
 to the menu button's client, even if style.getMenu is true.  To retrieve
 the menu that will be shown, send msgMenuButtonProvideMenu to the menu
 button.

 See Also
	msgMenuButtonProvideMenu
*/
#define msgMenuButtonGetMenu	MakeMsg(clsMenuButton, 3)


/****************************************************************************
 msgMenuButtonSetMenu		takes MENU, returns STATUS
	Sets the menu.

 The submenu is only used if style.subMenuType is not mbMenuNone.
 Note that the old menu, if any, is not freed.  If the new menu is not objNull,
 self-sends msgWinSetPopup with WIN_METRICS parameters of
	child	= menu;
*/
#define msgMenuButtonSetMenu	MakeMsg(clsMenuButton, 4)


/****************************************************************************
 msgMenuButtonProvideWidth		takes P_S32, returns STATUS
	category: self-sent
	Self-sent when style.getWidth is true.

 Subclasses should respond with the desired width of the menu button.
 clsMenuButton responds with self's current window width.
*/
#define msgMenuButtonProvideWidth	MakeMsg(clsMenuButton, 7)


/****************************************************************************
 msgMenuButtonInsertMenu		takes WIN, returns STATUS
	category: self-sent
	Self-sent when style.menuAction is detected.

 Subclasses should respond by inserting pArgs into the window tree.
 If style.subMenuType is mbMenuSibling, clsMenuButton responds by inserting
 pArgs as a window sibling to self.  Otherwise, msgMenuShow(true), is sent
 to pArgs.
*/
#define msgMenuButtonInsertMenu	MakeMsg(clsMenuButton, 10)


/****************************************************************************
 msgMenuButtonExtractMenu		takes WIN, returns STATUS
	category: self-sent
	Self-sent when style.menuAction is detected.

 Subclasses should respond by extracting pArgs from the window tree.
 clsMenuButton responds by sending msgMenuShow(false) to pArgs.
 If style.subMenuType is mbMenuSibling, clsMenuButton responds by sending
 msgWinExtract to pArgs.  Otherwise, msgMenuShow(false), is sent
 to pArgs.
*/
#define msgMenuButtonExtractMenu	MakeMsg(clsMenuButton, 11)

/****  msgMenuButtonShowMenu Arguments  ****/
Enum16(MENU_BUTTON_SHOW_MENU) {
	mbShowToggle	= 0,	// toggle the state of the menu
	mbShowExtract	= 1,	// take down the menu
	mbShowInsert	= 2		// put up the menu
};

/****************************************************************************
 msgMenuButtonShowMenu			takes MENU_BUTTON_SHOW_MENU, returns STATUS
	Puts up or takes down the menu.

 If the menu is currently up, and pArgs is mbShowToggle or mbShowExtract, 
 does the following:
	-: self-sends msgMenuButtonExtractMenu(menu).
	-: if style.getMenu is true, sends msgMenuButtonMenuDone with the following
	   MENU_BUTTON_PROVIDE_MENU parameters to the menu button's client:
//{	   	
			menuButton	= self;
			menu		= menu;
//}	   	

 If the menu is currently down, and pArgs is mbShowToggle or mbShowInsert,
 does the following:
	-: if style.subMenuType is not mbMenuSibling and the menu has
	   wsLayoutDirty set in its WIN_METRICS.flags.style, sends msgWinLayout
	   with the following WIN_METRICS parameters to the menu:
//{	   	
	   		options	= wsLayoutResize;
//}	   	

	-: if style.getMenu is true, sends msgMenuButtonProvideMenu with the
	   following MENU_BUTTON_PROVIDE_MENU parameters to the menu button's
	   client (and then the resulting MENU_BUTTON_PROVIDE_MENU.menu will 
	   be used):
//{	   	
	   		menuButton	= self;
			menu		= menu;
//}	   	

	-: if style.enableMenu is true, the process of the selection owner is
	   compared against the process of OSThisApp().  The menu is sent
	   msgControlEnable with the following CONTROL_ENABLE parameters:
//{
	   		root	= self;
			enable	= true if processes match, false otherwise
//}

	-: msgMenuButtonPlaceMenu is self-sent with the following WIN_METRICS
	   parameters:
//{
	   		bounds.size		= current menu size;
			bounds.origin	= origin of self, in theRootWindow space;
//}

	-: msgWinDelta is sent to the menu to position it at the resulting origin.

	-: msgTkTableSetManager(self) is sent to the menu.

	-: self-sends msgMenuButtonInsertMenu(menu).

 Note that if style.subMenuType is mbMenuSibling, msgWinLayout is not sent
 to self's parent.  The caller must do this to insure the correct layout.
*/
#define msgMenuButtonShowMenu	MakeMsg(clsMenuButton, 5)


/****************************************************************************
 msgMenuButtonPlaceMenu			takes P_WIN_METRICS, returns STATUS
	category: self-sent
	Self-sent whenever a menu button needs to position its associated menu.

 If this message reaches clsMenuButton, that class will do some default
 positioning.  In the message arguments:
	bounds.origin		In: origin of menu *button* wrt/theRootWindow
						Out: origin of *menu* wrt/theRootWindow
	bounds.size			In: size of menu

 Since clsMenuButton uses msgMenuShow to display the menu, and that
 message always ensures that the menu lies within theRootWindow, there's
 no need in the method for msgMenuButtonPlaceMenu to check the bounds
 of the menu against theRootWindow.
*/
#define msgMenuButtonPlaceMenu	MakeMsg(clsMenuButton, 6)


/****************************************************************************
 msgMenuButtonProvideMenu	takes P_MENU_BUTTON_PROVIDE_MENU, returns STATUS
	category: client responsibility
	Sent to the client if style.getMenu is true.

 clsMenuButton will send this message to the client of the menu button just
 before showing the menu.  The MENU_BUTTON_PROVIDE_MENU parameters will be
 set as follows:
 	menuButton	= uid of menu button needing the menu
	menu		= uid of last provided or set (via msgMenuButtonSetMenu) menu

 The client may modify the passed menu or supply a different menu uid.  If
 the client makes changes to the menu that invalidate its layout or supplies
 a different uid, the client should lay out the menu before returning.  If
 the client changes the uid of the menu, clsMenuButton will self-send
 msgMenuButtonSetMenu(pArgs->menu) (i.e. the menu button will remember the
 provided menu for future use).  The client will be sent msgMenuButtonMenuDone
 when the menu button is finished with the menu.

 Note that this message can also be sent to a menu button to retrieve the 
 actual menu that would be shown by the menu button.  If style.getMenu
 is true, clsMenuButton will send msgMenuButtonProvideMenu to the menu
 button's client.  In this case, the caller must send msgMenuButtonMenuDone
 to the menu button when finished with the menu.

 See Also
 	msgMenuButtonMenuDone
*/
#define msgMenuButtonProvideMenu	MakeMsg(clsMenuButton, 8)


/****************************************************************************
 msgMenuButtonMenuDone		takes P_MENU_BUTTON_PROVIDE_MENU, returns STATUS
	category: client responsibility
	Sent to the client if style.getMenu is true.

 clsMenuButton will send this message to the menu button's client when the
 menu button has taken down the menu and style.getMenu is true.  Note that
 clsMenuButton does remember the uid of the menu even after sending this
 message.  If the client destroys the menu, msgMenuButtonSetMenu(objNull)
 should be sent to the menu button.

 If clsMenuButton receives this message, and style.getMenu is true, this
 message will be forwarded to the menu button's client.
*/
#define msgMenuButtonMenuDone		MsgNoError(MakeMsg(clsMenuButton, 9))

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from Other Classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgWinLayoutSelf takes P_WIN_METRICS, returns STATUS
	Tell a window to layout its children.

 If the menu button has a menu, and style.getWidth is true and
 pArgs->options has wsLayoutResize set and the menu button has
 wsShrinkWrapWidth on, clsButton self-sends msgMenuButtonProvideWidth
 to compute pArgs->bounds.size.w.
*/

/****************************************************************************
 msgWinSend 	takes WIN_SEND, returns STATUS
    Sends a message up a window ancestry chain.

 clsMenuButton acts as the manager for its menu, and looks for msgMenuDone
 to be sent via msgWinSend.

 If style.subMenuType is not mbMenuNone and pArgs->msg is msgMenuDone,
 and the menu is currently up, and pArgs->data[0] is not self,
 clsMenuButton will do the following:
  	-: take down the menu as in msgMenuButtonShowMenu.
  	-: self-send msgButtonSetNoNotify, false to turn off the menu button.
  	-: ObjectCallAncestor() to all the msgWinSend to continue up the
  	   window tree.

 If pArgs->data[0] is self, nothing is done and stsOK is returned without
 calling ancestor.  This allows, for example, prevents a menu button with
 a pull-right menu from taking down the menu containing the menu button.
*/

/****************************************************************************
 msgGWinGesture:	takes P_GWIN_GESTURE, returns STATUS
	Called to process the gesture.

 If pArgs->msg is xgs2Tap and style.menuAction is mbAction2Tap and
 style.subMenuType is not mbMenuNone, the menu will be inserted/removed
 as in msgMenuButtonShow.

 clsMenuButton will notify its manager after any gesture.

 clsMenuButton self-sends msgButtonNotifyManager with the following
 BUTTON_NOTIFY parameters:
//{
		msg		= msgMenuDone;
		data	= self;
		button	= self;
//}

 This is followed by ObjectCallAncestor(), to allow the gesture to be
 processed normally.
*/

/****************************************************************************
 msgControlSetClient		takes UID, returns STATUS
	Sets the control metrics.client.

 clsMenuButton will send msgTkTableSetClient(pArgs) to the menu.
*/

#endif	// MBUTTON_INCLUDED
