/****************************************************************************
 File: inbxsvc.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.7  $
     $Date:   30 Mar 1992 11:00:06  $

 This file contains the API definition for clsINBXService.
 clsINBXService inherits from clsIOBXService.
 Provides default behavior for Inbox Services.
****************************************************************************/
#ifndef INBXSVC_INCLUDED
#define INBXSVC_INCLUDED

#ifndef IOBXSVC_INCLUDED
#include <iobxsvc.h>
#endif

/****  1. Introduction  ****/

/*
 In PenPoint, input operations are handled by a special class of
 services known as the "inbox services."  While most input operations
 are triggered by an external event such as an incoming fax image from
 a remote fax machine, some input operations may require that the
 PenPoint computer be one that initiates the communication process.
 For example, a fax input service may wish to periodically "poll" a
 "store-and-forward" facility in order to receive a fax image.  Thus,
 an inbox service implements the "deferred input" feature in PenPoint:
 This concept permits a user to specify input operations regardless of
 the readiness of input devices.  If the input device (e.g., a
 data/fax modem, a LAN connection, etc.) is not available or not
 connected, the input process is deferred until the input device
 becomes ready.
*/

/****  2. Passive vs. Active Inbox Services  ****/

/*
 The simplest type of inbox services are those who passively wait for
 an input event to happen.  That is, after the input operation is
 initiated by a remote agent such as a fax machine, the inbox service
 running on a PenPoint computer detects the input event and then
 receives the incoming data stream.  This type of inbox services do
 not initiate an input operation by themselves.  Typically, when such
 a service is enabled by the user, it simply becomes the owner of the
 I/O device.  A simple fax inbox service, for example, becomes the
 owner of the fax modem and sets it up to start receiving fax images
 whenever a phone call comes in.  While the inbox service owns the I/O
 device, no other services can transmit or receive data through the
 same device.  (For more details on the notion of service ownership,
 see the service API in service.h.)

 Some inbox services may want to actively "solicitate" input from a
 remote agent.  For example, a service that queries a remote database
 will have to establish the communication link between the PenPoint
 computer and the remote database server.  For this type of services,
 clsINBXService provides default behaviors to manage the state of the
 I/O device (connected or disconnected), the permission to initiate
 input operation (whether the service is enabled or disabled), as well
 as automatic polling behavior similar to that of an outbox service.
 Thus, the user can "defer" the input operation until it becomes
 possible to establish a communication link with a remote agent.  See
 the API for clsOBXService for a detailed discussion of the deferred
 input/output protocol.  Note, however, that to enable such
 outbox-like behavior, the polling flag must be turned on when the
 service is created.  I.e., in msgNewDefaults, you should set

//{
    pArgs->iobxsvc.in.autoPoll          = true;
//}

*/

/****  3. Inbox Documents  ****/

/*
 Normally, documents can be automatically created in an inbox section
 as the end result of an input event.  For example, a fax inbox
 section may create a document containing the fax images receieved in
 the fax modem.  Such documents are normal PenPoint documents.  Their
 contents have nothing to do with the input device or where the
 document came from.  

 Sometimes an inbox document contains not only data, but also some
 control information about the input operation to be performed.  For
 example, taking advantage of the "deferred input" feature, the user
 may construct a specific query statement for an online database and
 put it into the appropriate inbox section before the PenPoint machine
 is physically connected to the remote database.  When the input
 service becomes ready, the query statement is sent to the remote
 database, and the result is put into either another document or the
 same document containing the query statements.  This type of inbox
 documents is very similar to the outbox document that controls the
 actual output operation.  Again, for more information about the
 deferred input/output protocol, see obxsvc.h.

 Note that the deferred I/O protocol implemented by clsINBXService
 assumes that an input operation is controlled by an inbox document:
 an assumption that may be too cumbersome and confusing for many
 services.  If such is the case, an inbox service can simply store the
 input control information (e.g., a database query statement) with the
 service itself.  When the service receives msgINBXSvcPollDocuments,
 it simply handles the input operation directly and bypasses the rest
 of the protocol.
*/

/****  4. Services that Handle Input and/or Output  ****/

/*
 clsINBXService deals only with input operations.  For those services
 that want to handle output operations or both input and output at the
 same time, two other classes, clsOBXService and clsIOBXService, are
 provided by PenPoint.  In fact, clsINBXService and clsOBXService are
 implemented as a subclass (hence a subset) of clsIOBXService.
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                              Class Messages                             *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNewDefaults             takes P_INBXSVC_NEW, returns STATUS
    category: class message
    Initializes the P_INBXSVC_NEW structure to default values.

    Zeroes out pArgs->inbxsvc and sets...
    pArgs->iobxsvc.in.autoPoll          = false;
    pArgs->inbxsvc.sectionClass         = clsNBToc;

*/

typedef struct INBXSVC_NEW_ONLY {
    OBJECT  sectionClass;   // class of the inbox section
                            // This must be clsNBToc or a subclass of it.
    U32     unused1;
    U32     unused2;
    U32     unused3;
} INBXSVC_NEW_ONLY, *P_INBXSVC_NEW_ONLY;

#define inbxServiceNewFields \
    ioSvcNewFields           \
    INBXSVC_NEW_ONLY         inbxsvc;

typedef struct INBXSVC_NEW {
    inbxServiceNewFields
} INBXSVC_NEW, *P_INBXSVC_NEW;

/****************************************************************************
 msgNew                     takes P_INBXSVC_NEW, returns STATUS
    category: class message
    Creates a new inbox service object.

*/

/****************************************************************************
 msgINBXSvcSwitchIcon   takes nothing, returns STATUS
    category: class message
    Toggles the inbox icon (to empty or filled) if neccessary.

 Check the content of the inbox notebook.  Show the "filled" icon
 if any document is found.  Show the "emtpy" icon otherwise.
*/
#define msgINBXSvcSwitchIcon                    msgIOBXSvcSwitchIcon

/****************************************************************************
 msgINBXDocGetService   takes P_INBX_DOC_GET_SERVICE, returns STATUS
    category: class message
    Gets the service name.

 Get the name of the service associated with an inbox document.
 If the document has not been placed into an inbox section, stsFailed
 is returned.

 Note that the document must be at the top level of an inbox section.
 That is, if the document is embedded within another document, which
 is in an inbox section, stsFailed will be returned.
*/
#define msgINBXDocGetService                    msgIOBXDocGetService

typedef struct INBX_DOC_GET_SERVICE {

    OBJECT          document;               // In: document uid
    CHAR            svcName[nameBufLength]; // Out: service name

} INBX_DOC_GET_SERVICE, *P_INBX_DOC_GET_SERVICE;

/****************************************************************************
 msgINBXDocInInbox  takes P_INBX_DOC_IN_INBOX, returns STATUS
    category: class message
    Checks if a document is in a section in the Inbox.

 This message can be sent to clsINBXService to check if a PenPoint
 document represented by pArgs->uuid is already in the input queue of
 an inbox service inheriting from pArgs->svcClass.  stsOK is returned
 if it is, stsFailed otherwise.  If pArgs->svcClass is objNull, stsOK
 is returned if the document is anywhere in the Inbox notebook.
*/
#define msgINBXDocInInbox                       msgIOBXDocInIOBox

typedef struct INBX_DOC_IN_INBOX {

    UUID            uuid;               // In: document uuid
    CLASS           svcClass;           // In: service class to check for

} INBX_DOC_IN_INBOX, *P_INBX_DOC_IN_INBOX;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *              Messages Sent to an Inbox Service Instance                *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgINBXSvcMoveInDoc    takes P_INBXSVC_MOVE_COPY_DOC, returns STATUS
    Moves a document into the inbox section.

 Superclass behavior is to move the document located at pArgs->source
 into the input queue associated with the inbox service.  For
 example, set pArgs->sequence to 1 to move the document to the top
 of the queue.  Set it to maxU16 to move the document to the bottom of
 the queue.

 After the document is moved (or copied) to the input queue, it is
 considered to be in a state ready for input, even though the service
 may not be connected at the time.  Client should not alter the document
 in any way once it has been moved to the input queue.

 Subclasses can provide their own behavior if they wish.  Remember to
 use the class message msgINBXSvcSwitchIcon to change the inbox icon.
*/
#define msgINBXSvcMoveInDoc                     msgIOBXSvcMoveInDoc

typedef struct INBXSVC_MOVE_COPY_DOC {
    FS_LOCATOR          source;     // In: Location of source document.
    U16                 sequence;   // In: Sequence number to move/copy
                                    //     in front of.
} INBXSVC_MOVE_COPY_DOC, *P_INBXSVC_MOVE_COPY_DOC;


/****************************************************************************
 msgINBXSvcCopyInDoc    takes P_INBXSVC_MOVE_COPY_DOC, returns STATUS
    Copies a document into the Inbox section.

 Same as msgINBXSvcMoveInDoc, except that the document is copied
 to the input queue.
*/
#define msgINBXSvcCopyInDoc                     msgIOBXSvcCopyInDoc

/****************************************************************************
 msgINBXSvcGetTempDir   takes P_OBJECT, returns STATUS
    Passes back a handle for a temporary directory.

 This message is provided for clients who may want ot prepare their
 input document before moving it into the input queue.  The handle
 of an "official" temporary directory is passed back and it can be
 used as temporary storage for documents, data, etc.  Clients are
 responsible for deleting temporary files when they are done.  The
 directory will be flushed after a warm boot.
*/
#define msgINBXSvcGetTempDir                    msgIOBXSvcGetTempDir

/****************************************************************************
 msgINBXSvcPollDocuments        takes nothing, returns STATUS
    Polls all documents in an input queue and input those who are ready.

 This message tells the inbox service to look through its input
 queue and send out the first document ready for input.  The service
 will first make sure that it is enabled and is connected to the
 designated input port.  If these conditions are met, it will then
 self-send msgINBXSvcNextDocument to locate the next document ready for
 input.

 If msgINBXSvcNextDocument returns stsOK, indicating that a
 document is ready for input, this message proceeds to self-send
 msgINBXSvcLockDocument to lock the document, and finally
 msgINBXSvcInputStart to initiate the input process.

 If msgINBXSvcNextDocument returns stsINBXSvcDocReady, indicating
 that the section is not empty but none of the documents are ready
 for input, this message self-sends msgINBXSvcScheduleDocument
 to schedule the document passed back in pArgs at a later time.

 Subclasses normally do not process this message.

 See Also
    msgINBXSvcNextDocument
    msgINBXSvcLockDocument
    msgINBXSvcUnlockDocument
    msgINBXSvcInputStart
    msgINBXSvcScheduleDocument
*/
#define msgINBXSvcPollDocuments                 msgIOBXSvcPollDocuments

/****************************************************************************
  msgINBXSvcNextDocument        takes P_INBXSVC_DOCUMENT, returns STATUS
     category: self-sent
     Passes back the next document ready for input.

 Superclass behavior is to start from the top of the input queue and
 locate the first document ready for input.  If one is found,
 information about the document is passed back in pArgs.  The same
 pArgs will be passed to messages msgINBXSvcLockDocument and
 msgINBXSvcInputStart.  By default, a document is ready for input
 when it is closed.  If the document is open, it will receive
 msgINBXDocInputStartOK and it should return stsOK to indicate that it
 is ready for input.

 Subclasses can provide their own behavior if they wish.  Return
 stsINBXSvcSectionEmpty to give the superclass an opportunity to
 change the inbox icon from filled to empty.

 Return Value
    stsOK:                  A document is ready for input.
    stsINBXSvcSectionEmpty: The input queue is empty.
    stsINBXSvcDocNotReady:  No document in the input queue is ready.
    Service-Specific Error Returns.

 See Also
    msgINBXSvcPollDocuments
*/
#define msgINBXSvcNextDocument                  msgIOBXSvcNextDocument

typedef struct INBXSVC_DOCUMENT {

    OBJECT      uid;                        // uid of the doc
    OBJECT      dir;                        // app dir of the doc
    OBJECT      docClass;                   // class of the doc
    U16         sequence;                   // sequence of the doc
    CHAR        pName[nameBufLength];       // name of this doc
    P_UNKNOWN   pDocData;                   // subclass's private data

} INBXSVC_DOCUMENT, *P_INBXSVC_DOCUMENT;

/****************************************************************************
  msgINBXSvcLockDocument        takes P_INBXSVC_DOCUMENT, returns STATUS
     category: self-sent
     Locks the document in preparation for input.

 This message is a place holder for subclasses that may require
 additional preparatory work to be performed on a document before it
 is ready for input.  For example, a document may have to be "locked"
 so that it can not be opened during the input process.  This message
 may be used for other purposes as well.  For example, an inbox
 service may decide to store a light-weight "shadow" document (e.g., a
 report designator for a database application) in the input queue
 until it is chosen for input.  The service then handles this message
 by converting the shadow document to a real one (e.g., the actual
 report).

 The superclass behavior for this message is to stamp the document
 directory with the filesystem attribute iobxsvcDocInputInProgress.  This
 stamp will prevent any gestures over the document from being
 processed.  This means that once a document is locked for input it
 can not be deleted, renamed, etc. via gestures.

 See Also
    msgINBXSvcUnlockDocument
*/
#define msgINBXSvcLockDocument                  msgIOBXSvcLockDocument

/****************************************************************************
  msgINBXSvcUnlockDocument      takes P_INBXSVC_DOCUMENT, returns STATUS
     category: self-sent
     Unlocks a document that was previously locked.

 This message is a place holder for subclasses that may require
 additional "cleanup" work to be performed on a document before it
 is put back to the input queue.

 The superclass behavior for this message is to remove the
 iobxsvcDocInputInProgress stamp on the document directory.

 See Also
    msgINBXSvcLockDocument
*/
#define msgINBXSvcUnlockDocument                msgIOBXSvcUnlockDocument

/****************************************************************************
  msgINBXSvcScheduleDocument        takes P_INBXSVC_DOCUMENT, returns STATUS
     category: self-sent
     Schedules a document that is not ready for input

 This message is sent when msgINBXSvcNextDocument locates a document in
 the input queue but the document is not ready for input.

 Subclasses should provide their own behavior.  The default behavior is
 to release the ownership of the target service (i.e., become disabled),
 with the expectation that the user must manually schedule the document
 later on (by re-enabling the section.)

 See Also
    msgINBXSvcNextDocument
*/
#define msgINBXSvcScheduleDocument              msgIOBXSvcScheduleDocument

/****************************************************************************
  msgINBXSvcInputStart      takes P_INBXSVC_DOCUMENT, returns STATUS
     category: self-sent
     Starts the input process for a document in the input queue.

 Superclass behavior is to activate the inbox document if it isn't
 already active, and then send msgINBXDocInputStart to the
 document instance.

 Subclasses can provide their own behavior if they wish.
*/
#define msgINBXSvcInputStart                    msgIOBXSvcIOStart

/****************************************************************************
  msgINBXSvcInputCancel     takes nothing, returns STATUS
     Cancels the input process.

 This message is sent to the service when the caller wishes to
 cancel any input operation in progress.  The service responds
 to this message by sending msgINBXDocInputCancel to an active
 inbox document.  After the document is cancelled, the service
 will post an error note to the user if there are other documents
 waiting to be processed.  The user then decides whether or not
 the service should proceed to send the remaining documents.

 Subclasses do not normally process this message.
*/
#define msgINBXSvcInputCancel                   msgIOBXSvcIOCancel

/****************************************************************************
  msgINBXSvcInputCleanUp        takes P_INBX_DOC_INPUT_DONE, returns STATUS
     category: self-post.
     Cleans up after the current input is done.

 This message is posted to self as a result of the service receiving
 msgINBXDocInputDone, which is sent by the inbox document when it
 finishes the input operation.  The inbox document will be either
 deleted or marked as specified in pArgs, and when everything is
 properly cleaned up the service will post msgINBXSvcPollDocuments to
 self to see if anything else is waiting for input.

 Subclasses do not normally process this message.

 See Also
    msgINBXDocInputDone
*/
#define msgINBXSvcInputCleanUp                  msgIOBXSvcIOCleanUp

Enum32(INBX_DOC_EXIT_BEHAVIOR) {

    inbxDocExitDoNothing,
    inbxDocExitDelete,
    inbxDocExitMarkAsFailed,
    inbxDocExitMarkAsCancelled

};

typedef struct INBX_DOC_INPUT_DONE {

    INBX_DOC_EXIT_BEHAVIOR  behavior;   // exit behavior
    P_UNKNOWN               pDocData;   // Unused: document specific data

} INBX_DOC_INPUT_DONE, *P_INBX_DOC_INPUT_DONE;

/****************************************************************************
 msgINBXSvcStateChanged     takes OBJECT, returns STATUS
     category: observer notification.
     Tells observers that the service state just changed.

 Informs observers that the state of a service has just changed.  pArgs is
 the UID of the service.
*/
#define msgINBXSvcStateChanged                  msgIOBXSvcStateChanged

/****************************************************************************
 msgINBXSvcQueryState       takes P_INBXSVC_QUERY_STATE, returns STATUS
     Passes back the state of the service.

 This message is typically used to query what state the service
 instance is in.
*/
#define msgINBXSvcQueryState                    msgIOBXSvcQueryState

typedef struct {
    BOOLEAN     enabled;                // true if the service is enabled.
    CHAR        status[nameBufLength];  // text describing the status of
                                        // the service.
    CHAR        docName[nameBufLength]; // document being processed
    P_UNKNOWN   pStateData;             // subclass's private data
} INBXSVC_QUERY_STATE, *P_INBXSVC_QUERY_STATE;

/****************************************************************************
 msgINBXSvcGetEnabled       takes P_BOOLEAN, returns STATUS
    Gets the enabled state of the service.

 Subclasses can override this message and redefine the notion of
 "enabled."  The default behavior of the superclass is to equate
 "enabled" with the ownership of the target service (i.e., input
 device).  That is, the service is "enabled" when it owns the target
 service.  By appending to or replacing the default behavior, a
 subclass can define additional conditions which must be met before a
 service is considered enabled.
*/
#define msgINBXSvcGetEnabled                    msgIOBXSvcGetEnabled

/****************************************************************************
 msgINBXSvcSetEnabled       takes BOOLEAN, returns STATUS
    Sets the enabled state of the service.

 This message is sent to the service in response to service
 notification messages msgSvcOwnerAcquired and msgSvcOwnerReleased.
 Subclasses can provide their own behavior and thereby redefine the
 notion of "enabled" for the service.  If they do, they must pass this
 message up to the ancestor so that observers of the inbox service
 will be properly notified.
*/
#define msgINBXSvcSetEnabled                    msgIOBXSvcSetEnabled

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                      Inbox Document Messages                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgINBXDocInputStartOK         takes nothing, returns STATUS
    Asks the inbox document if it is OK to start the input process

 When an inbox service finds an opened document in the inbox
 section, it sends this message to the document instance, asking
 whether it's OK to start the input operation while the document
 remains open.  When the document receives this message, it should
 return stsOK to give the service permission to begin the input
 process.  An error status, including stsNotUnderstood, is taken to
 mean that the document instance vetos the request and the service
 will not start the input process.
*/
#define msgINBXDocInputStartOK                  msgIOBXDocIOStartOK

/****************************************************************************
 msgINBXDocInputStart           takes nothing, returns STATUS
    Tells an inbox document to start the input process.

 This message is sent by the inbox service to a document.  The
 document should respond to this message by starting the input
 process.
*/
#define msgINBXDocInputStart                    msgIOBXDocIOStart

/****************************************************************************
 msgINBXDocInputDone        takes P_INBX_DOC_INPUT_DONE, returns STATUS
    category: client responsibility
    Tells the inbox service that input is finished.

 When the input process is finished, the inbox document in charge of
 the input should send this message to the inbox service.  This
 message must be sent even if the input process has been aborted.
 The pArgs for this message tells the inbox service what to do with
 the inbox document.  If inbxDocExitDelete is specified, the document
 will be removed from the inbox.  In all other cases the document
 will be unlocked and left in the inbox.  If either
 inbxDocExitMarkAsCancelled or inbxDocExitMarkAsFailed are specified,
 the name of the document will be altered to provide visual indication
 for the user that the input process has not completed successfully.

 See Also
    msgINBXDocGetService
    msgINBXDocInputCancel
    msgINBXDocInputCleanUp
*/
#define msgINBXDocInputDone                     msgIOBXDocIODone

/****************************************************************************
 msgINBXDocInputCancel          takes nothing, returns STATUS
    Tells an inbox document to cancel the input process.

 This message is used by the inbox service to inform a document that
 it should cancel the input process.  The document should handle this
 message by terminating its input operation and then sending
 msgINBXDocInputDone to the service with pArgs->behavior set to
 inbxDocExistMarkAsCancelled.

*/
#define msgINBXDocInputCancel                   msgIOBXDocIOCancel

/****************************************************************************
 msgINBXDocStatusChanged       takes P_INBX_DOC_STATUS_CHANGED, returns STATUS
    category: client responsibility
    Tells the inbox service that the document status is changed.

 This message is sent by the inbox document to the service whenever
 its status has just changed.  This status is displayed on Status
 column for the inbox section, in the Inbox notebook.
*/
#define msgINBXDocStatusChanged                 msgIOBXDocStatusChanged

typedef struct INBX_DOC_STATUS_CHANGED {

    CHAR        status[nameBufLength];  // Text describing document state
    P_UNKNOWN   pDocData;               // Unused: document-specific data

} INBX_DOC_STATUS_CHANGED, *P_INBX_DOC_STATUS_CHANGED;

#endif  // INBXSVC_INCLUDED



