/****************************************************************************
 File: go.h

 (C) Copyright 1992 by Go Corporation, All Rights Reserved.

 $Revision:   1.38  $
   $Author:   ctaylor  $
	 $Date:   20 Feb 1992 15:25:22  $

 This file contains PenPoint's standard #defines, types and intrinsics.
 Essentially all PenPoint source files must include this file.

 The functions described in this file are contained in PENPOINT.LIB. 
****************************************************************************/

#ifndef GO_INCLUDED
#define GO_INCLUDED

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Standard Definitions						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Static Declarations
 *
 * Functions declared STATIC (rather than static) will, when compiled with
 * DEBUG defined, appear in map files.
*/
#ifndef DEBUG
  #define STATIC		static
#else
  #define STATIC
#endif

/*
 * Function Scope Definitions
 *    -:  LOCAL:			Scope is module wide
 *    -:  GLOBAL:			Scope is subsystem wide
 *    -:  EXPORTED:		Scope is ring wide (either ring0 OR ring3)
 *    -:  EXPORTED0:		Scope is system wide.  For public ring0 functions.
 *    -:  RINGCHELPER:	Scope is system wide.  For private ring0 functions.
*/
#define LOCAL			STATIC PASCAL
#define GLOBAL			PASCAL
#define EXPORTED		PASCAL
#define EXPORTED0		PASCAL
#define RINGCHELPER		PASCAL


/* Null values */
#ifndef M_I86			// 32 bit compiler
 #define NULL	0
#else					// 16 bit compiler
 #define NULL	0L
#endif

#define null			0
#define pNull			((P_UNKNOWN)0)
#define ppNull			((PP_UNKNOWN)0)
#define Nil(type)		((type)0)

/* Boolean operators */
#define AND				&&
#define OR				||
#define NOT				!
#define MOD				%

/* 
 * Bit flags.
 *
 * These flags can be used with FlagOn, FlagOff, FlagSet, and FlagClr.
*/
#define flag0			(0x0001)
#define flag1			(0x0002)
#define flag2			(0x0004)
#define flag3			(0x0008)
#define flag4			(0x0010)
#define flag5			(0x0020)
#define flag6			(0x0040)
#define flag7			(0x0080)
#define flag8			(0x0100)
#define flag9			(0x0200)
#define flag10			(0x0400)
#define flag11			(0x0800)
#define flag12			(0x1000)
#define flag13			(0x2000)
#define flag14			(0x4000)
#define flag15			(0x8000)
#define flag16			(0x00010000L)
#define flag17			(0x00020000L)
#define flag18			(0x00040000L)
#define flag19			(0x00080000L)
#define flag20			(0x00100000L)
#define flag21			(0x00200000L)
#define flag22			(0x00400000L)
#define flag23			(0x00800000L)
#define flag24			(0x01000000L)
#define flag25			(0x02000000L)
#define flag26			(0x04000000L)
#define flag27			(0x08000000L)
#define flag28			(0x10000000L)
#define flag29			(0x20000000L)
#define flag30			(0x40000000L)
#define flag31			(0x80000000L)

/* Limits */
#define maxU8			((U8)0xFF)
#define minS8			((S8)0x80)
#define maxS8			((S8)0x7F)
#define maxU16			((U16)0xFFFF)
#define minS16			((S16)0x8000)
#define maxS16			((S16)0x7FFF)
#define maxU32			((U32)0xFFFFFFFF)
#define minS32			((S32)0x80000000)
#define maxS32			((S32)0x7FFFFFFF)

/* Name limits */
#define maxNameLength	32
#define nameBufLength	(maxNameLength+1)

/* Enums */
/*	
 Different compilers allocate different amounts of space for an enum.  To
 avoid portability problems, use the Enum16 and Enum32 macros.  They
 guarantee that the enum is 16 bits or 32 bits, respectively.  Example:

//{
	Enum16(PRIMARY_COLOR) {
		red,
		green,
		blue
	}
//}
*/
#define Enum16(name) typedef S16 name, * P_##name; enum name
#define Enum32(name) typedef S32 name, * P_##name; enum name

/* Calling Conventions */
#if defined __WATCOMC__
 #define PASCAL			__pascal
 #define CDECL			__cdecl
 #define Unused(x)		(void)(x)
 #define FunctionPtr(fn) (PASCAL * fn)
 #define CFunctionPtr(fn) (CDECL * fn)
 #if defined __386__
 #pragma aux pascal "^" parm routine []\
   value struct float struct caller [eax] modify [eax ecx edx gs];
 #pragma aux cdecl "_*" parm caller []\
   value struct float struct caller [eax] modify [eax ecx edx gs];
 #endif
#elif defined __HIGHC__
 #define PASCAL			_CC(_REVERSE_PARMS|_CALLEE_POPS_STACK)
 #define CDECL			// Default for the compiler
 #define Unused(x)
 #define FunctionPtr(fn) PASCAL (* fn)
 #define CFunctionPtr(fn) CDECL (* fn)
#else
 #define PASCAL			pascal
 #define CDECL			cdecl
 #define Unused(x)		(void)(x)
 #define FunctionPtr(fn) (* PASCAL fn)
 #define CFunctionPtr(fn) (* CDECL fn)
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								  Typedefs								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Unsigned integers */
typedef unsigned char	U8,	 * P_U8,  ** PP_U8;		//  8-bit unsigned
typedef unsigned short	U16, * P_U16, ** PP_U16;	// 16-bit unsigned
#ifndef M_I86			// 32 bit compiler
 typedef unsigned int	U32, * P_U32, ** PP_U32;	// 32-bit unsigned
#else					// 16 bit compiler
 typedef unsigned long	U32, * P_U32, ** PP_U32;	// 32-bit unsigned
#endif

/* Signed integers */
typedef signed char		S8,	 * P_S8,  ** PP_S8;		//  8-bit signed
typedef signed short	S16, * P_S16, ** PP_S16;	// 16-bit signed
#ifndef M_I86			// 32 bit compiler
 typedef signed int		S32, * P_S32, ** PP_S32;	// 32-bit signed
#else					// 16 bit compiler
 typedef signed long	S32, * P_S32, ** PP_S32;	// 32-bit signed
#endif

/*
 * Wide characters.  In PenPoint 1.0 these are 8 bit values.  In PenPoint 2.0
 * and forward they are 16 bit values.
*/
typedef U8				CHAR;
typedef P_U8			P_CHAR;
typedef P_CHAR*			PP_CHAR;

/* 8 bit Characters */
typedef U8				CHAR8;	  // These are guaranteed to stay 8-bit
typedef P_U8			P_CHAR8;
typedef P_CHAR8*		PP_CHAR8;

/* 16 bit Characters */
typedef U16				CHAR16;	  // These are guaranteed to stay 16-bit
typedef P_U16			P_CHAR16;
typedef P_CHAR16*		PP_CHAR16;

/* Strings */
typedef U8				STRING;
typedef P_U8			P_STRING;
typedef P_STRING*		PP_STRING;

/*
 * SIZEOF is the type returned by the SizeOf.  It is guaranteed to be 32 bits.
*/
typedef U32				SIZEOF, * P_SIZEOF;

/* Pointer to an opaque entity */
typedef void*			P_UNKNOWN;
typedef P_UNKNOWN*		PP_UNKNOWN;

/* Generic pointer to procedure */
typedef P_UNKNOWN FunctionPtr(P_PROC)();

/* True/False values */
Enum16(BOOLEAN) {
	FALSE = 0,
	TRUE  = 1,
	False = 0,
	True  = 1,
	false = 0,
	true  = 1
};


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								  Intrinsics							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define Abs(v)			((v)<0?(-(v)):(v))
#define Max(a,b)		((a)>(b)?(a):(b))
#define Min(a,b)		((a)<(b)?(a):(b))
#define Odd(v)			((v)&1)
#define Even(v)			(!Odd(v))

#define LowU16(dw)		((U16)(U32)(dw))
#define HighU16(dw)		((U16)((U32)(dw)>>16))
#define LowU8(w)		((U8)(w))
#define HighU8(w)		((U8)((U16)(w)>>8))
#define MakeU16(lb,hb)	(((U16)(hb)<<8)|(U16)(lb))
#define MakeU32(lw,hw)	(((U32)(hw)<<16)|(U32)(lw))

#define FlagOn(f,v)		(!FlagOff(f,v))
#define FlagOff(f,v)	(!((v)&(f)))
#define FlagSet(f,v)	((v)|(f))
#define FlagClr(f,v)	((v)&(~(f)))

#define OutRange(v,l,h)	((v)<(l)||(v)>(h))
#define InRange(v,l,h)	((v)>=(l)&&(v)<=(h))

#define SizeOf(t)		((SIZEOF)sizeof(t))


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Commonly Used Class Manager Types					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * A variable of type OBJECT identifies an object.  The type UID is
 * interchangeable with OBJECT.
 *
 * A variable of type TAG identifies one of the following:
 *    -:  Tag
 *    -:  Message
 *    -:  Error status (values less than 0)
 *    -:  Warning status (values greater than or equal to 0)
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						   Well-known UID Structure						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 A UID is constructed as:
	-:	Version:	7 bits
	-:	Admin:		20 or 19 bits
	-:	Scope:		1 or 2 bits
	-:	Layout:

//{
			00000000001111111111222222222233
			01234567890123456789012345678901
			---------------------------------
	Name:	0|	Ver | 0|	Admin+Scope
			+-------+-------+-------+-------+
	Size:	1|	 7	| 3|	20+1 or 19+2
			---------------------------------
//}
*/

typedef P_UNKNOWN	UID, * P_UID;

typedef UID			OBJECT, * P_OBJECT, ** PP_OBJECT;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						   Well-known UID Macros						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Create a well-known UID */
#define MakeWKN(admin,version,scope) \
	((UID)((U32)(0x7F&(version))<<24|(U32)(admin)<<1+(scope&1)|scope))

/* Create a well-known UID */
#define MakeGlobalWKN(admin,version)	MakeWKN(admin,version,wknGlobal)

/* Create a process-global well-known UID */
#define MakeProcessGlobalWKN(admin,version) \
	MakeWKN(admin,version,wknProcessGlobal)

/* Create a private well-known UID */
#define MakePrivateWKN(admin,version)	MakeWKN(admin,version,wknPrivate)

/* Extract the admin number plus the scope information */
#define WKNValue(wkn)	(0x1FFFFF&(U32)wkn)

/* Extract the admin number */
#define WKNAdmin(wkn)	(WKNValue(wkn)>>1+((U32)wkn&1))

/* Extract the version number */
#define WKNVer(wkn)		((U32)(wkn)>>24)

/* Extract the scope */
#define WKNScope(wkn)	((U32)(wkn)&-((U32)(wkn)&1)&3)

/* Magic constants */
#define wknGlobal			0
#define wknProcessGlobal	1
#define wknPrivate			3

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Tag Structure								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Tags are created using a well-known Administered value and a tag number in
 the range 0-255.
	-:	X:		1 bit.  0 for tag or Warning Status;  1 for an Error Status.
	-:	TagNum:	8 bits
	-:	Flags:	2 bits
	-:	Admin:	20 or 19 bits
	-:	Scope:	1 or 2 bits
	-:	Layout:

//{
			00000000001111111111222222222233
			01234567890123456789012345678901
			---------------------------------
	Name:	X| tagNum|F|	Admin+Scope
			+-------+-------+-------+-------+
	Size:	1|	 8	 |2|	20+1 or 19+2
			---------------------------------
//}

*/

typedef S32			TAG, * P_TAG;			// Tags are always positive
typedef S32			STATUS, * P_STATUS;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Tag Macros								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Create a tag */
#define MakeTag(wkn,tagNum)	(((TAG)(tagNum)&0xFF)<<23|WKNValue(wkn))

/* Create a tag with flags */
#define MakeTagWithFlags(wkn,i,f) (MakeTag(wkn,i)|((U32)(f)&3)<<21)

/* Extract the tag num */
#define TagNum(tag)		((U32)(tag)<<1>>24)
#define Tag(tag)		TagNum(tag)

/* Extract the tag num and flags together */
#define TagAndFlags(tag) ((U32)(tag)<<1>>22)

/* Extract only the tag flags */
#define TagFlags(tag)	(TagAndFlags(tag)&3)

/* Extract the tag admin */
#define TagAdmin(tag)	WKNAdmin(tag)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							   Status Macros							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Create an error status */
#define MakeStatus(wkn,sts) ((STATUS)(0x80000000|MakeTag(wkn,sts)))

/* Create a warning status */
#define MakeWarning(wkn,sts)	((STATUS)MakeTag(wkn,sts))

/* Extract the status num from a STATUS */
#define Sts(sts)		Tag(sts)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							   Debugging Macros							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define StsRet(se,s)	if (((s) = StsWarn(se)) < stsOK) return s; else

#define StsJmp(se,s,x)	if (((s) = StsWarn(se)) < stsOK) goto x; else

#define StsOK(se,s)		(((s) = StsWarn(se)) >= stsOK)

#define StsFailed(se,s) (((s) = StsWarn(se)) < stsOK)

#define StsChk(se,s)	(((s) = (se)) < stsOK)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Status Printing Macros						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 StsWarn	returns nothing
	Prints status warning message.

 When DEBUG is defined during compilation, the StsWarn macro prints a
 status warning message if the status is less than stsOK (an error).  When
 DEBUG is not defined during compilation, StsWarn simply evaluates its
 expression.

 See Also
	StsPrint
*/
#if defined DEBUG || defined CLSMGR_COMPILE

#define StsWarn(se)		StsWarning(se,__FILE__,__LINE__)

#else // if not DEBUG

#define StsWarn(se)		(se)

#endif // DEBUG


/****************************************************************************
 StsPrint	returns nothing
	Prints status warning message.

 When DEBUG is defined during compilation, the StsPrint macro prints
 a status warning message regardless of the value of the status.  When
 DEBUG is not defined during compilation, StsPrint does nothing.

 See Also
	StsWarn
*/

#if defined DEBUG || defined CLSMGR_COMPILE

#define StsPrint(s)		StatusWarning(s,__FILE__,__LINE__)

#else // if not DEBUG

#define StsPrint(s)

#endif // DEBUG


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Status Values							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

// Next up: 11

/* Classes used to create generic status values (see uid.h) */
#define clsGO					MakeWKN(14,1,wknGlobal)
#define clsOS					MakeWKN(16,1,wknGlobal)
#define clsGOMath				MakeWKN(162,1,wknGlobal)

/* Values */
#define stsBadParam				MakeStatus(clsGO, 1)
#define stsNoMatch				MakeStatus(clsGO, 2)
#define stsEndOfData			MakeStatus(clsGO, 3)
#define stsFailed				MakeStatus(clsGO, 4)
#define stsTimeOut				MakeStatus(clsGO, 5)
#define stsRequestNotSupported	MakeStatus(clsGO, 6)
#define stsReadOnly				MakeStatus(clsGO, 7)
#define stsIncompatibleVersion	MakeStatus(clsGO, 8)
#define stsNotYetImplemented	MakeStatus(clsGO, 9)
#define stsOutOfMem				MakeStatus(clsGO, 10)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						   Non-Error Status Values						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

// Next up: 4

#define stsOK				MakeWarning(0, 0)
#define stsRequestDenied	MakeWarning(clsGO, 1)
#define stsRequestForward	MakeWarning(clsGO, 2)	// also stsMessageIgnored
#define stsTruncatedData	MakeWarning(clsGO, 3)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								GO Math Support	 						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Conceptually these declarations should be in gomath.h.  They are defined
 here instead to ease the load on the compiler symbol tables.
*/
typedef S32			FIXED;
typedef FIXED*		P_FIXED;
FIXED PASCAL		FxMakeFixed(S16 whole, U16 frac);


//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Private			 						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#if defined DEBUG || defined CLSMGR_COMPILE

/****************************************************************************
 StsWarning	returns STATUS
	Prints status warning message if sts is less than stsOK.

 Clients should not call this routine directly.  Use the StsWarn macro.
*/
STATUS EXPORTED StsWarning(
	STATUS		sts,
	P_STRING	fn,
	U16			ln);

/****************************************************************************
 StatusWarning	returns nothing
	Prints status warning message regardless of the value of sts.

 Clients should not call this routine directly.  Use the StsPrint macro.
*/
void EXPORTED StatusWarning(
	STATUS		sts,
	P_STRING	fn,
	U16			ln);

#endif

//REFGEN ENDIGNORE

#endif // GO_INCLUDED
