/****************************************************************************
 dvscreen.h

 Copyright 1992, GO Corporation,   All Rights Reserved

 The public definition of the MIL screen device interface.

 $Revision:   1.10  $
   $Author:   thall  $
     $Date:   24 Mar 1992 15:09:28  $
****************************************************************************/
#ifndef DVSCREEN_INCLUDED
#define DVSCREEN_INCLUDED


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 This section contains definitions specific to the screen device. 
 The definitions contained here are data structures that are necessary
 for the MIL screen specific functions along with standard
 MIL device data structures with additional fields specific to the
 screen device.
*/

/**** Screen Device Specific Function Numbers ****/

/*
 The following definitions are the device specific functions supported
 by the screen device. The function numbers start immediately after
 the last function number used by the functions that are common to all
 MIL devices. These function numbers correspond to the Function Transfer
 Table (FTT) entries.
*/

#define milScreenGetDevParameters   milLastCommonFunction+1
#define milScreenSetMode	    	milLastCommonFunction+2
#define milScreenPutBitMap	    	milLastCommonFunction+3
#define milScreenDrawLine	    	milLastCommonFunction+4

#define fnCountScreen		    fnCountBase+4

/**** Screen Device Function Transfer Table ****/
/*
 This structure is the declaration of the Screen Device Function
 Transfer Table. It includes one entry for each of the MIL driver
 functions common to all MIL devices along with one entry for
 each of the device specific functions for the MIL Screen device.
 Each entry contains fields that define the size of the request
 block for that function, the staging requirements of the function,
 and the address of the actual function. 
*/

typedef struct MIL_FTT_SCREEN {
    funcTransTableCommonFields
    MIL_FUNCTION_DESC functionDesc[fnCountScreen];
} MIL_FTT_SCREEN, *P_MIL_FTT_SCREEN;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 					Screen Device Specific Functions	   	 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 The following function definitions are the screen device specific
 functions. Access to these functions is provided by the FTT defined
 above. All of the functions defined below will be passed 3 pointers.

	pCommonData:
		This is a pointer to the common data area.

	pRB:
		This is a pointer to the function request block.

	pDeviceBlock:
		This is a pointer to the device block for the device

 All of the functions return STATUS. Any STATUS other than staging 
 statuses will cause the request to terminate. The staging statuses
 will result in the specified staging operation to be performed.
*/


/****************************************************************************
 milScreenGetDevParameters		returns STATUS
 	Provides device specific information about the screen device.

 This function is a single stage function that provides device and mode
 specific information about the screen device.

*/ 

/**** Device Specific Request Block Fields ****/
/*

 The milScreenGetDevParameters funtion request block contains a data 
 structure that defines the screen device specific parameters and 
 another data structure that defines the current mode specific
 parameters. A field in the request block identifies the current mode
 associated with the mode related parameters.

	forMode:
		This field identifies the current active display mode.

	screenDevParms:
		This field is a data structure that contains the device
		specific information for this screen device. The data
		structure is defined below.

	screenModeParms:
		This field is a data structure that contains the mode
		specific information for this screen device. The data
		structure is defined below.

*/

/*
 screenDevParms

 The screenDevParms field of the request block is a data structure
 that contains the following fields.

	screenFlags:
 		The screen flags indentify the screen orientation. The 
		definitions of the screenFlags values is provided below.

	screenMemLogAddress:
		This field contains the address of the screen memory.

	screenMemLength:
		This field contains the length of the display memory
		in bytes.

	numModes:
		This field indicates the number of modes supported by
		the screen device.

	currentMode:
		This field indicates the current active display mode.

*/
typedef struct MIL_SCREEN_DEVICE_PARAMETERS {
    U16 screenFlags;
    U32 screenMemLogAddress;
    U32 screenMemLength;
    U16 numModes;
    U16 currentMode;
    U16 reservedU16[2];
    U32 reservedU32[2];
} MIL_SCREEN_DEVICE_PARAMETERS, *P_MIL_SCREEN_DEVICE_PARAMETERS;

/*
 screenModeParms

 The screenModeParms field of the request block is a data structure
 that contains the following fields.

	horzPixelsPerMeter:
		This field defines the horizontal size of a pixel. The size
		is specified as the number pixels per meter.

	vertPixelsPerMeter:
		This field defines the vertical size of a pixel. The size
		is specified as the number pixels per meter.

	horzAspect:
		This field defines the horizontal aspect value.

	vertAspect:
		This field defines the vertical aspect value.

	horzSize:
		This field is the actual horizontal screen dimension specified
		in millimeters.

	vertSize:
		This field is the actual vertical screen dimension specified 
		in millimeters.

	horzPixels:
		This field is the total number of pixels in the horizontal
		dimension.

	vertPixels:
		This field is the total number of pixels in the vertical
		dimension.

	pixelPlanes:
		This field specifies the number of planes used to display
		information pixels on the screen.

	pixelBytesPerRow:
		This field specifies the number of bytes in a row of
		pixels for a single plane.

	pixelColorResolution:
		This field specifies the number of colors supported for this
		display.

	inkPlanes:
		This field specifies the number of display planes used to 
		display pen ink.

	inkBytesPerRow:
		This field specifies the number of bytes in a pixel row in
		any of the ink planes.

	inkColorResolution:
		This field specifies the number of supported colors for ink.

*/

typedef struct MIL_SCREEN_MODE_PARAMETERS {
    U32 horzPixelsPerMeter;
    U32 vertPixelsPerMeter;
    U16 horzAspect;
    U16 vertAspect;
    U32 horzSize;
    U32 vertSize;
    U32 horzPixels;
    U32 vertPixels;
    U16 pixelPlanes;
    U32 pixelBytesPerRow;
    U16 pixelColorResolution;
    U16 inkPlanes;
    U32 inkBytesPerRow;
    U16 inkColorResolution;
    U16 reservedU16[2];
    U32 reservedU32[2];
} MIL_SCREEN_MODE_PARAMETERS, *P_MIL_SCREEN_MODE_PARAMETERS;


/*

 screenFlags

 Initial screen orientation: for a conventional VGA screen these are:


                 North
             ---------------
             |             |
             |             |
             |             |
             |             |
             |             |
             |             |
       West  |             | East
             |             |
      (top   |             | (bottom
       in    |             |  in
       char. |             |  character
       mode) |             |  mode)
             |             |
             |             |
             |             |
             ---------------
             ^    South    ^
             ^             ^
             ^             last VGA scan line
             ^
             first VGA scan line

 This information is simply passed to the screen driver DLL. The DLL
 is where this orientation is given meaning. The PutBitMap function
 should also examine these bits if more than one initial orintation
 is supported.

*/

#define milScreenInitialTopMask     (flag0|flag1)
#define milScreenInitialTopNorth    0
#define milScreenInitialTopEast     (flag0)
#define milScreenInitialTopSouth    (flag1)
#define milScreenInitialTopWest     (flag0|flag1)

#define milScreenBlack 0
#define milScreenWhite 0xff

/*
 rbPublicFieldsScreenGetDevParameters

 This define merges the device specific data structure with the standard
 MIL device parameters reserved fields.

*/

#define rbPublicFieldsScreenGetDevParameters \
    U16 forMode; \
    MIL_SCREEN_DEVICE_PARAMETERS screenDevParms; \
    MIL_SCREEN_MODE_PARAMETERS screenModeParms; \
    U16 funcReservedU16[2]; \
    U32 funcReservedU32[2];

/*
 
 milScreenGetDevParameters Request Block

*/

typedef struct MIL_RB_PUB_SCREEN_GET_DEV_PARAMETERS {
    milRequestBlockCommonFields
    rbPublicFieldsScreenGetDevParameters
} MIL_RB_PUB_SCREEN_GET_DEV_PARAMETERS, *P_MIL_RB_PUB_SCREEN_GET_DEV_PARAMETERS;


/****************************************************************************
 milScreenSetMode		returns STATUS
 	Sets the specified display mode.

 This function is a single stage function that sets the display to the
 specified display mode.

*/ 

/**** Device Specific Request Block Fields ****/
/*
 The following declaration defines the device and function specific
 fields for this function.

	screenMode:
		This input field specifies the display mode to be initiated.

*/

#define rbPublicFieldsScreenSetMode \
    U16 screenMode; \
    U16 funcReservedU16[2]; \
    U32 funcReservedU32[2];
/*
 
 milScreenSetMode Request Block

*/

typedef struct MIL_RB_PUB_SCREEN_SET_MODE {
    milRequestBlockCommonFields
    rbPublicFieldsScreenSetMode
} MIL_RB_PUB_SCREEN_SET_MODE, *P_MIL_RB_PUB_SCREEN_SET_MODE;


/****************************************************************************
 milScreenPutBitmap		returns STATUS
 	Displays a bitmap.

 This function displays a specified bitmap on the screen.
 The Screen PutBitMap function is only used to display
 monchrome images to indicate progress or error conditions before
 ImagePoint is functional. Because they may be used in error conditions
 these two functions are required to be single-stage functions.

 The coordinate system uses the first quadrant. In other words the lower
 left corner is 0,0 with increasing values moving up and to the right.

*/
 
/**** Device Specific Request Block Fields ****/
/*
 The following declaration defines the device and function specific
 fields for this function.

	pBitMap:
		This input field is a pointer to a data structure that 
		contains fields that describe and contain the bitmap. This
		structure is defined below.

	row:
		This field specifies the screen row where the bitmap is
		to be positioned. This row represents the bottom of the
		image.

	col:
		This field specifies the screen column where the bitmap
		is to be positioned. This position represents the left
		side of the image.

*/

/*

 pBitMap

 The following data structure describes and contains a bitmap. It is 
 made up of the following fields.

	height:
		This is the height of the image in pixels.

	width:
		This field is the width of the image in pixels.

	bits:
		This field is an array of pixels that make up the bitmap.
 		The bytes of the bit-maps are formatted into scan-lines
		padded to quad-byte boundaries, one bit per pixel. As most
		PenPoint machines initially operate in portrait orientation
		on hardware where the scan-lines run along the long axis,
		the bit-maps are oriented to be easily drawn in this
		configuration. With North being the top of the screen the
		first scan-line in the bit-map is the right edge of the image
		with the first pixel (the MSB) going at the bottom right corner
		of the image.

*/
 
typedef struct MIL_SCREEN_BIT_MAP {
    U16 height;
    U16 width;
    U32 bits[];
} MIL_SCREEN_BIT_MAP, *P_MIL_SCREEN_BIT_MAP;

/*
 rbPublicFieldsScreenPutBitMap

 This define merges the device specific data structure with the standard
 MIL device parameters reserved fields.

*/

#define rbPublicFieldsScreenPutBitMap \
    P_MIL_SCREEN_BIT_MAP pBitMap; \
    U16 row; \
    U16 col; \
    U16 funcReservedU16[2]; \
    U32 funcReservedU32[2];

/*
 
 milScreenPutBitMap Request Block

*/

typedef struct MIL_RB_PUB_SCREEN_PUT_BIT_MAP {
    milRequestBlockCommonFields
    rbPublicFieldsScreenPutBitMap
} MIL_RB_PUB_SCREEN_PUT_BIT_MAP, *P_MIL_RB_PUB_SCREEN_PUT_BIT_MAP;



/****************************************************************************
 milScreenDrawLine		returns STATUS
 	Displays a line.

 This function draws line on the screen based on the coordinates specified.
 The Screen Drawline function is only used to display
 monchrome images to indicate progress or error conditions before
 ImagePoint is functional. Because they may be used in error conditions
 these two functions are required to be single-stage functions.

 The coordinate system uses the first quadrant. In other words the lower
 left corner is 0,0 with increasing values moving up and to the right.

*/

/**** Device Specific Request Block Fields ****/
/*
 The following declaration defines the device and function specific
 fields for this function.

	row:
		This field specifies the screen row where the line
		starts.

	col:
		This field specifies the screen column where the starts.

	height:
		This field specifies the height of the line. This field
		could be considered to describe the height of a box where
		the line to be drawn is a diagonal line.

	width:
		This field specifies the width of the line. This field
		could be considered to describe the width of a box where
		the line to be drawn is a diagonal line. 

*/

#define rbPublicFieldsScreenDrawLine \
    U16 row; \
    U16 col; \
    U16 height; \
    U16 width; \
    U16 color; \
    U16 funcReservedU16[2]; \
    U32 funcReservedU32[2];

/*
 
 milScreenDrawLine Request Block

*/

typedef struct MIL_RB_PUB_SCREEN_DRAW_LINE {
    milRequestBlockCommonFields
    rbPublicFieldsScreenDrawLine
} MIL_RB_PUB_SCREEN_DRAW_LINE, *P_MIL_RB_PUB_SCREEN_DRAW_LINE;

#endif

