/****************************************************************************
 dvpower.h

 Copyright 1992, GO Corporation,   All Rights Reserved

 The public definition of the power device interface.

 $Revision:   1.8  $
   $Author:   unknown  $
     $Date:   25 Mar 1992 16:39:48  $
****************************************************************************/
#ifndef DVPOWER_INCLUDED
#define DVPOWER_INCLUDED

/**** Introduction ****/
/*
 PenPoint utilizes the power management device to handle global power 
 management over the system. It will communicate to each device, at the
 appropriate times, the proper power state to enter (using the 
 milBase SetDevPowerState function). In addition, PenPoint handles other
 specific aspects of power management.

 A PenPoint machine needs to support software control of the power-up
 cycle. This allows PenPoint to turn the machine on to perform scheduled
 tasks or respond to external events such as an incoming call on a modem.
 PenPoint also requires control of the power-down process to complete 
 various tasks before responding to the user's power down request. Based
 on user preferences, PenPoint can also set the machine to the standby
 or power off states after being idle for a defined period.

 PenPoint and the MIL support four global power states.

	fullPower:
		The CPU is clocked and in use. This state persists until an
		explicit request is made to the MIL to change it.

	idlePower:
		The CPU has no tasks to run and may be idle (through a reduced
		or stopped clock) or halted. Any interrupt should cause the 
		hardware to return the system from this state to the 
		fullPower state. The MIL will hide whatever interaction this
		state has with DMA. In other words, PenPoint can set this
		power state without worrying about whether DMA is occuring.

	standbyPower:
		Power to the CPU and peripherals is removed but main memory
		is kept powered and intact. The system must be reset to leave
		this state.

	noPower:
		Power for all hardware is removed and memory is lost. The
		system must be powered-up to leave this state.

 The function milPowerGetDevParameters returns standbyPowerSupported
 and noPowerSupported in the powerFlags field of the request block as
 an indication of support of standby and or power-off features. The flag
 idleStateSavesPower in the same field indicates the idle state reduces
 power consumption.

 When the standbyPower state is set, an indicator should be set in the 
 MIL in a reserved byte in non-volatile RAM and/or main memory flagging
 the validity of main memory. The location and value of these indicators
 must be known to the selftest code to prevent it from applying a 
 destructive test of main memory. These indicators should be explicitly
 cleared when read. By clearing the indicators as soon as possible
 we allow a possible hardware reset to force a cold boot.

 Penpoint starts-up or boots at a number of levels. In order of increasing
 time, these are:

	hot-boot:
		Memory is intact when penpoint starts so PenPoint does not have
		to reinitialize memory.

	swap-boot:
		Memory is invalid when PenPoint starts but a swap device exits
		that contains the memory context. PenPoint is reloaded and pulls
		in code and the swap file as needed.

	warm-boot:
		Memory is valid and PenPoint is running. PenPoint must rebuild
		all tasks and processes. This type of re-boot is initiated by the
 		user and only necessary when the state of the machine's processes 
		is suspect.

	cold-boot:
		Memory is invalid and there iis no saved context or swap device.
		The boot code optionally loads the MIL and the PenPoint kernal
		from the boot media and branches to the MIL's start-up entry 
		point.

 On power-up, PenPoint would be passed the type of boot and the source 
 of the power up request. If the main memory tags are invalid a cold boot
 is performed. If the memory tags are OK the routine determines the boot
 type by checking the modem, RTC alarm, or standby button. If the source is
 the standby button is held for 5 seconds the system will perform a warm
 boot. Otherwise, it will perform a hot-boot. Note that this test must be
 conducted by the power up code early during initialization to catch the 
 state of the standby button. A machine without a hardwired reset button 
 but with a power button - could trigger reset if the power button was 
 held for five seconds on a power-up.

 For typical hardware, the boot type would be identified using the
 following scheme.

// {
		if <main memory tags valid> {
			<clear tags>
			if <modem ring>
				powerUpRequestSource = modem
			else if <alarm occurred>
				powerUpRequestSource = alarm
			else if <standby button held>
				powerUpRequestSource = button
			else
				powerUpRequestSource = unknown

			if <standby button held for five seconds>
				bootType = warmBoot
			else
				bootType = hotBoot
		}
		else
			bootType = coldBoot
// }


				

				

*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 This section contains definitions specific to the power device. 
 The definitions contained here are data structures that are necessary
 for the MIL power specific functions along with standard
 MIL device data structures with additional fields specific to the
 power device.
*/

/**** Power Device Specific Function Numbers ****/

/*
 The following definitions are the device specific functions supported
 by the power device. The function numbers start immediately after
 the last function number used by the functions that are common to all
 MIL devices. These function numbers correspond to the Function Transfer
 Table (FTT) entries.

*/

#define milPowerGetDevParameters	milLastCommonFunction+1
#define milPowerSetPowerState		milLastCommonFunction+2
#define milPowerSetBatteryLevel		milLastCommonFunction+3
#define milPowerReadReqEventCont	milLastCommonFunction+4
#define milPowerBatteryEventCont	milLastCommonFunction+5

#define fnCountPower				fnCountBase+5

/**** Power Device Function Transfer Table ****/
/*
 This structure is the declaration of the Power Device Function
 Transfer Table. It includes one entry for each of the MIL driver
 functions common to all MIL devices along with one entry for
 each of the device specific functions for the MIL Power device.
 Each entry contains fields that define the size of the request
 block for that function, the staging requirements of the function,
 and the address of the actual function. 
*/


typedef struct MIL_FTT_POWER {
	funcTransTableCommonFields
	MIL_FUNCTION_DESC functionDesc[fnCountPower];
} MIL_FTT_POWER, *P_MIL_FTT_POWER;



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 					Power Device Specific Functions	   	 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 The following function definitions are the power device specific
 functions. Access to these functions is provided by the FTT defined
 above. All of the functions defined below will be passed 3 pointers.

	pCommonData:
		This is a pointer to the common data area.

	pRB:
		This is a pointer to the function request block.

	pDeviceBlock:
		This is a pointer to the device block for the device

 All of the functions return STATUS. Any STATUS other than staging 
 statuses will cause the request to terminate. The staging statuses
 will result in the specified staging operation to be performed.
*/


/****************************************************************************
 milPowerGetDevParameters		returns STATUS
 	Provides device specific information about the power device.

 This function is a single stage function that provides device
 specific information about the power device. It returns information
 on the capabilities and current state of the power hardware.

 The flags standbyPowerSupported and noPowerSupported indicate support
 for these two power states. The existence of an actual standby button
 is flagged by standbyButtonSupported. The availability of unlimited
 power can cause PenPoint to alter its power management strategy. If
 the connection of the battery charger is detected, then 
 chargerConnectedSupported should be set and chargerConnected should 
 reflect the attachment of the charger. If the idle power state does 
 indeed reduce power consumption, then the flag idleStateSavesPower 
 will be set.

 A given MIL implementation can describe and monitor as many batteries
 as necessary. Each battery is assigned a unit number with the main
 battery typically being unit zero. Additional batteries are commonly
 used to power the real time clock or keep memory alive while the
 main battery is being swapped. The name of each battery can be obtained
 (like any other device unit) by accessing the array indicated by 
 pUnitNameList in the device data block.

 milPowerGetDevParameters will return a description in batteryDesc of
 the battery indicated by the unit field. The ability to monitor the 
 actual battery voltage is indicated by the flag rawVoltsSupported
 in the field batteryFlags. If this flag is true, then the fields
 maxMillivolts, warnMillivolts, failMillivolts, and currentMillivolts
 will contain valid data. The validity of secondsOfBatteryLeft and 
 maxSeconds are indicated by secondsLeftSupported. The flag 
 setLevelSupport indicates that the function milPowerSetBatteryLevel
 is supported for this battery (unit).

*/

/**** Function Specific Definitions ****/
/*
 Power State Definitions

 The following values are defined for the different power states.

	milPowerFullPower:
		The CPU is clocked and in use. This state persists until an
		explicit request is made to the MIL to change it.
		

	milPowerIdlePower:
		The CPU has no tasks to run and may be idle (through a reduced
		or stopped clock) or halted. Any interrupt should cause the 
		hardware to return the system from this state to the 
		fullPower state. The MIL will hide whatever interaction this
		state has with DMA. In other words, PenPoint can set this
		power state without worrying about whether DMA is occuring.

	milPowerStandbyPower:
		Power to the CPU and peripherals is removed but main memory
		is kept powered and intact. The system must be reset to leave
		this state.

	milPowerNoPower:
		Power for all hardware is removed and memory is lost. The
		system must be powered-up to leave this state.

	milPowerForceReset:
		Force a system reset.

*/

Enum16(MIL_POWER_STATE) {
	milPowerFullPower,
	milPowerIdlePower,
	milPowerStandbyPower,
	milPowerNoPower,
	milPowerForceReset
};

/*
 Battery and Power State Event Definitions
 
 The following events are issued to indicate changes in the battery or
 external power state.

	milPowerBatteryWarn:
		This event indicates the battery level is low.

	milPowerBatteryFail:
		This event indicates the battery has failed.

	milPowerChargerConnected:
		This event indicates the battery charger has been connected.

	milPowerChargerRemoved:
		This event indicates the battery charger has been removed.

*/ 

Enum16(MIL_BATTERY_EVENT) {
	milPowerBatteryWarn,
	milPowerBatteryFail,
	milPowerChargerConnected,
	milPowerChargerRemoved
};

/*
 Power State Request Events

 The follow request are issued to indicate a change in the system power
 management state.

	milPowerOffReq:
		A power off operation should be performed.

	milPowerStandbyReq:
		A power standby operation should be performed.

	milPowerWarmBootReq:
		A warm boot operation should be performed.

	milPowerForceResetReq
		A system reset should be performed.

*/

Enum16(MIL_POWER_REQ_EVENT) {
	milPowerOffReq,
	milPowerStandbyReq,
	milPowerWarmBootReq,
	milPowerForceResetReq
};

/*
 Reset Type Definitions

 The following definitions define the different reset operations that can
 be performed.

	milPowerColdReset:
		The reset is a full reset.

	milPowerWarmReset:
		The reset is a warm reset.

	milPowerHotReset:
		The reset is a hot reset.

*/

	
Enum16(MIL_RESET_TYPE) {
	milPowerColdReset,
	milPowerWarmReset,
	milPowerHotReset
};

/*
 Power Flags

 The following definitions indicate the supported power modes.

	milStandbyPowerSupported:
		This bit indicates the power hardware supports standby mode.

	milNoPowerSupported:
		This bit indicates the power hardware supports a no power state.

	milStandbyButtonSupported:
		This bit indicates a standby button is supported.

	milChargerConnectedSupported:
		This bit indicates the hardware can detect when the battery
		charger is connected or disconnected.

	milIdleStateSavesPower:
		This bit indicates the idle state will save power.

	milChargerConnected:
		This bit indicates the battery charger is connected.

*/

#define milStandbyPowerSupported         flag0
#define milNoPowerSupported              flag1
#define milStandbyButtonSupported        flag2
#define milChargerConnectedSupported     flag3
#define milIdleStateSavesPower           flag4
#define milChargerConnected              flag5

/* 
 Battery Flags

 The following definitions define the capabilities of the battery
 hardware.

	milRawVoltsSupported:
		This bit indicates the hardware can monitor the battery voltage.
		
	milPercentLeftSupported:
		This bit indicates the remaining battery level can be detected.

	milSecondsLeftSupported:
		This bit indicates the remaining batter level can be returned as
 		the time left.

	milSetLevelSupported:
		This bit indicates the driver supports the milPowerSetBatteryLevel
		function.

*/
#define milRawVoltsSupported             flag0
#define milPercentLeftSupported          flag1
#define milSecondsLeftSupported          flag2
#define milSetLevelSupported             flag3

/*
 Power Unit Parameters Data Structure

 The following data structure defines the unit specific information
 for the power hardware device.

	batteryFlags:
		These flags identify the capabilities and characteristics
		of the battery. The values for these flags are defined above.

	maxMillivolts:
		This field indicates the millivolts value for a fully charged
		battery.

	warnMillivolts:
		This field identifies the level at which a low battery warning
		event will be issued.

	failMillivolts:
		This field identifies the level at which a battery fail event
		will be issued.

	currentMillivolts:
		This field indicates the current level of the battery in 
		millivolts.

	percentOfBatteryLeft:
		This field indicates the percentage of the battery life that
		is left.

	maxSeconds:
		This field indicates the total number of seconds in a fully
		charged battery.

	secondsOfBatteryLeft:
		This field indicates the amount of time left in the current
		battery.

*/	

typedef struct MIL_POWER_UNIT_PARAMETERS {
	U16 batteryFlags;
	U16 maxMillivolts;
	U16 warnMillivolts;
	U16 failMillivolts;
	U16 currentMillivolts;
	U16 percentOfBatteryLeft;
	U16 maxSeconds;
	U16 secondsOfBatteryLeft;
} MIL_POWER_UNIT_PARAMETERS, *P_MIL_POWER_UNIT_PARAMETERS;


/*
 Power Device Parameters Data Structure

 The following data structure defines the device specific information
 for the power hardware device. It contains the following fields.

	powerFlags:
		These flags identify the capabilities and characteristics
		of the power hardware. The values for these flags are defined
		above.

	powerOnType:
		This field indicates the type of the last power on operation.

	powerOnSourceLogicalId:
		This field contains the ID of the power on source. 

	powerOnSourceDeviceId:
		This field contains the ID of the device that initiated
		the power on.

	powerState:
		This field contains the current power state.

*/


typedef struct MIL_POWER_DEVICE_PARAMETERS {
	U16 powerFlags;
	MIL_RESET_TYPE powerOnType;
	U16 powerOnSourceLogicalId;
	TAG powerOnSourceDeviceId;
	MIL_POWER_STATE powerState;
	U16 reservedU16[2];
	U32 reservedU32[2];
} MIL_POWER_DEVICE_PARAMETERS, *P_MIL_POWER_DEVICE_PARAMETERS;


/**** Device Specific Request Block Fields ****/
/*

 The milPowerGetDevParameters function request block contains a two
 data structures that defines the power device and unit specific
 parameters. The request block contains the following fields.

*/	

#define rbPublicFieldsPowerGetDevParameters \
	MIL_POWER_DEVICE_PARAMETERS powerDevParms; \
	MIL_POWER_UNIT_PARAMETERS powerUnitParms; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milPowerGetDevParameters Request Block

*/

typedef struct MIL_RB_PUB_POWER_GET_DEV_PARAMETERS {
	milRequestBlockCommonFields
	rbPublicFieldsPowerGetDevParameters
} MIL_RB_PUB_POWER_GET_DEV_PARAMETERS, *P_MIL_RB_PUB_POWER_GET_DEV_PARAMETERS;



/****************************************************************************
 milPowerSetPowerState		returns STATUS
 	Sets the system power state.

 This function is a single stage function that sets the power state
 to the state specified.

*/
/**** Function Specific Request Block Fields ****/
/*

 The milPowerSetPowerState funtion request block contains the
 following fields.

	powerState:
		This field is the power state that should be set. The
		supported power states are defined above.

*/

#define rbPublicFieldsPowerSetPowerState \
	MIL_POWER_STATE powerState;

/*
 
 milPowerSetPowerState Request Block

*/

typedef struct MIL_RB_PUB_POWER_SET_POWER_STATE {
	milRequestBlockCommonFields
	rbPublicFieldsPowerSetPowerState
} MIL_RB_PUB_POWER_SET_POWER_STATE, *P_MIL_RB_PUB_POWER_SET_POWER_STATE;



/****************************************************************************
 milPowerSetBatteryLevel		returns STATUS
 	Sets the battery level variable.

 This function is a single stage function that sets the battery level
 variable to the value specified.

*/
/**** Function Specific Request Block Fields ****/
/*

 The milPowerSetBatteryLevel funtion request block contains the
 following fields.

	percentOfBatteryLeft:
		This field is the value that should be set as the current
		amount of battery power left as a percentage of the total
		battery life.

*/

#define rbPublicFieldsPowerSetBatteryLevel \
	U16 percentOfBatteryLeft;

/*
 
 milPowerSetBatteryLeft Request Block

*/

typedef struct MIL_RB_PUB_POWER_SET_BATTERY_LEVEL {
	milRequestBlockCommonFields
	rbPublicFieldsPowerSetBatteryLevel
} MIL_RB_PUB_POWER_SET_BATTERY_LEVEL,
*P_MIL_RB_PUB_POWER_SET_BATTERY_LEVEL;



/****************************************************************************
 milPowerReadReqEventCont		returns STATUS
 	Checks power state.

 This function is a continuous function that checks the state of the
 power hardware.

*/
/**** Function Specific Request Block Fields ****/
/*

 The milPowerReadReqEventCont funtion request block contains the
 following fields.

	pPowerReadReqEventFunc:
		This field is a pointer to a function that is called if a
		power related event occurs.

	powerReqEvent:
		This field indicates the power event.

*/

#define rbPublicFieldsPowerReadReqEventCont \
	P_MIL_ASYNC_EVENT_FUNC pPowerReadReqEventFunc; \
	MIL_POWER_REQ_EVENT powerReqEvent;

/*
 
 milPowerReadReqEventCont Request Block

*/


typedef struct MIL_RB_PUB_POWER_READ_REQ_EVENT_CONT{
	milRequestBlockCommonFields
	rbPublicFieldsPowerReadReqEventCont
} MIL_RB_PUB_POWER_READ_REQ_EVENT_CONT,
	*P_MIL_RB_PUB_POWER_READ_REQ_EVENT_CONT;


/****************************************************************************
 milPowerBatteryEventCont		returns STATUS
 	Checks battery state.

 This function is a continuous function that checks the state of the
 battery.

*/
/**** Function Specific Request Block Fields ****/
/*

 The milPowerBatteryEventCont funtion request block contains the
 following fields.

	batteryEvent:
		This field is set to the battery event that has occured.

	pPowerBatteryEventFunc:
		This field is a pointer to a function that should be 
		called when the battery state indicates an event should
		be issued.

*/


#define rbPublicFieldsPowerBatteryEventCont \
	MIL_BATTERY_EVENT batteryEvent; \
	P_MIL_ASYNC_EVENT_FUNC pPowerBatteryEventFunc;


/*
 
 milPowerBatteryEventCont Request Block

*/


typedef struct MIL_RB_PUB_POWER_BATTERY_EVENT_CONT {
	milRequestBlockCommonFields
	rbPublicFieldsPowerBatteryEventCont
} MIL_RB_PUB_POWER_BATTERY_EVENT_CONT,
*P_MIL_RB_PUB_POWER_BATTERY_EVENT_CONT;

#endif
