/****************************************************************************
 dvasyncs.h

 Copyright 1992, GO Corporation,   All Rights Reserved

 The public definition of the async sio device interface.

 $Revision:   1.10  $
   $Author:   unknown  $
	 $Date:   25 Mar 1992 16:39:40  $
****************************************************************************/
#ifndef DVASYNCS_INCLUDED
#define DVASYNCS_INCLUDED

#ifndef MIL_INCLUDED
#include <mil.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 This section contains definitions specific to the async device. 
 The definitions contained here are data structures that are necessary
 for the MIL async specific functions along with standard
 MIL device data structures with additional fields specific to the
 async device.
*/

/**** Async Device Specific Function Numbers ****/

/*
 The following definitions are the device specific functions supported
 by the async device. The function numbers start immediately after
 the last function number used by the functions that are common to all
 MIL devices. These function numbers correspond to the Function Transfer
 Table (FTT) entries.

*/

#define milAsyncSIOGetDevParameters	 milLastCommonFunction+1
#define milAsyncSIOSetLineParameters milLastCommonFunction+2
#define milAsyncSIOSetModemControl	 milLastCommonFunction+3
#define milAsyncSIOCheckBuffer		 milLastCommonFunction+4
#define milAsyncSIOEventCont		 milLastCommonFunction+5
#define milAsyncSIOSendBreak		 milLastCommonFunction+6

#define fnCountAsyncSIO				fnCountBase+6


/**** Async Device Function Transfer Table ****/
/*
 This structure is the declaration of the Async Device Function
 Transfer Table. It includes one entry for each of the MIL driver
 functions common to all MIL devices along with one entry for
 each of the device specific functions for the MIL Async device.
 Each entry contains fields that define the size of the request
 block for that function, the staging requirements of the function,
 and the address of the actual function. 
*/

typedef struct MIL_FTT_ASYNCSIO {
	funcTransTableCommonFields
	MIL_FUNCTION_DESC functionDesc[fnCountAsyncSIO];
} MIL_FTT_ASYNCSIO, *P_MIL_FTT_ASYNCSIO;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 					Async Device Specific Functions	   	 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 The following function definitions are the async device specific
 functions. Access to these functions is provided by the FTT defined
 above. All of the functions defined below will be passed 3 pointers.

	pCommonData:
		This is a pointer to the common data area.

	pRB:
		This is a pointer to the function request block.

	pDeviceBlock:
		This is a pointer to the device block for the device

 All of the functions return STATUS. Any STATUS other than staging 
 statuses will cause the request to terminate. The staging statuses
 will result in the specified staging operation to be performed.
*/


/****************************************************************************
 milAsyncGetDevParameters		returns STATUS
 	Provides device specific information about the async device.

 This function is a single stage function that provides device
 specific information about the async device.
*/

/**** Function Specific Definitions ****/
/*
 Baud Rate Flags

 The following flags are used to indicate the baud rates supported
 by the async device.

*/

#define milRate50Supported		 flag0
#define milRate75Supported		 flag1
#define milRate110Supported		 flag2
#define milRate150Supported		 flag3
#define milRate300Supported		 flag4
#define milRate600Supported		 flag5
#define milRate1200Supported	 flag6
#define milRate2400Supported	 flag7
#define milRate4800Supported	 flag8
#define milRate9600Supported	 flag9
#define milRate19200Supported	 flag10
#define milRate38400Supported	 flag11
#define milRate57600Supported	 flag12


/*
 Modem Control Flags

 The following flag definitions are used in the modemControl,
 supportedModemControl, and dropLineToStop fields.
*/
 
#define milDataTerminalReady	flag0
#define milRequestToSend		flag1
#define	milAux1				  	flag2
#define	milAux2				  	flag3

/* 
 Stop Bit Values

 The following definitions define the possible stop bit configurations.

*/

#define milOneStopBit			 0
#define milTwoStopBits			 1
#define milOneAndaHalfStopBits	 2

/*
 Word Length Values

 The following definitions are used to configure the size of the 
 transmitted or received data.

*/

#define milFiveBitWord			 0
#define milSixBitWord			 1
#define milSevenBitWord			 2
#define milEightBitWord			 3

/*
 Parity Values

 The following values are used to configure the parity control of the
 data.

*/
#define milNoParity				 0
#define milOddParity			 1
#define milEvenParity			 2
#define milOneParity			 3
#define milZeroParity			 4

/*
 Supported Line Parameter Flags

 The following flags are used to identify support of various configurable
 characteristics of the async device.

*/
#define milOneStopBitSupported				flag0
#define milTwoStopBitsSupported				flag1
#define milOneAndaHalfStopBitsSupported		flag2

#define milFiveBitWordSupported				flag3
#define milSixBitWordSupported				flag4
#define milSevenBitWordSupported			flag5
#define milEightBitWordSupported			flag6

#define milNoParitySupported				flag7
#define milOddParitySupported				flag8
#define milEvenParitySupported				flag9
#define milOneParitySupported				flag10
#define milZeroParitySupported				flag11

/*
 Modem Status Flags

 The following flag definitions are used in the modemStatus, 
 supportedModemStatus, and stopWhenDropped fields.

*/

#define milDeltaCTS				 flag0
#define milDeltaDSR				 flag1
#define milTrailingEdgeRI		 flag2
#define milDeltaDCD				 flag3
#define milClearToSend			 flag4
#define milDataSetReady			 flag5
#define milRingIndicator		 flag6
#define milDataCarrierDetect	 flag7

/*
 Line Status Flags

 The following flags are used in the lineStatus and
 supportedLineStatus fields.

*/

#define milDataReady				flag0
#define milOverrunError				flag1
#define milParityError				flag2
#define milFramingError				flag3
#define milBreakReceived			flag4
#define milFIFOError				flag7

/*
 Transmit and Receive Status Flags

 The following field definitions are used in the receiveTransmitStatus
 field.

*/

#define	milAsyncSioRxFrozen			flag0
#define	milAsyncSioTxFrozen			flag1
#define	milAsyncSioConnect			flag2
#define	milAsyncSioBreakOn			flag3
#define	milAsyncSioTxTimeOutOn		flag4
#define	milAsyncSioInEventCallBack	flag5	// internal condition only:
											// always false!
/*
 Async Device Parameters

 The following data structure contains information about the async device.
 This data structure contains the following fields.

	asyncSIOFlags:
  		This field contains bits that indicate capabilities, status,
		and modes of operation of the async device. No bits are
		defined for this field at this time.

	supportedModemControl:
		This field contains flags that identify the modem control features
		supported by the async device. The flags are defined above.

	supportedModemStatus:
		This field contains flags that identify the modem status features
		supported by the async device. The flags are defined above.

	supportedLineStatus:
		This field contains flags that identify the line status features
		supported by the async device. The flags are defined above.

	supportedLineParameters:
		This field contains flags that identify the line control features
		supported by the async device. The flags are defined above.

	supportedBaudRates:
		This field contains flags that identify the baud rates
		supported by the async device. The flags are defined above.

	baudRateGeneratorInpFreq:
		This field contains a value that represents the input frequency
		of the baud rate generator.

	internalBaudRateClkDivide:
		This field contains a value that represents the baud rate clock
		divisor.

*/
typedef struct MIL_ASYNC_SIO_DEVICE_PARAMETERS {
	U16 asyncSIOFlags;
	U16 supportedModemControl;
	U16 supportedModemStatus;
	U16 supportedLineStatus;
	U16	supportedLineParameters;
	U16 supportedBaudRates;
	U32 baudRateGeneratorInpFreq;
	U16 internalBaudRateClkDivide;
	U16 reservedU16[2];
	U32 reservedU32[2];
} MIL_ASYNC_SIO_DEVICE_PARAMETERS, *P_MIL_ASYNC_SIO_DEVICE_PARAMETERS;

/*
 Async Unit Parameters

 The following data structure contains information about the async unit.
 This data structure contains the following fields.

	baudRate:
		This field contains the current baud rate.

	modemControl:
		This field contains the current modem control state. The bits
		in this field are defined above.

	stopBits:
		This field indicates the current configuration setting for the
		stop bits.

	wordLength:
		This field specifies the word length in the current configuration.
		Possible value for this field are defined above.

	parityType:
		This field specifies the parity type in the current configuration.
		Possible value for this field are defined above.

	modemStatus:
		This field indicates the current modem status.
		Possible value for this field are defined above.

	lineStatus:
		This field indicates the current line status.
		Possible value for this field are defined above.

	receiveTransmitStatus:
		This field indciates the current receive transmit status.
		Possible value for this field are defined above.

	stopWhenReceived:
		This field indicates the value used for flow control that indicates
		transmission should stop.

	startWhenReceived:
		This field indicates the value used for flow control that indicates
		transmission can start again.

	sendToStop:
		This field indicates the value that must be sent to stop the 
		sender from sending more data. This is used for flow control.

	sendToStart:
		This field indicates the value to send that will tell the sending
		system to resume sending data.

	stopWhenDropped:
		This field indicates status lines that will stop transmission
		if the designated line drops. The values for this field are 
		defined above.

	dropLineToStop:
		This field indicates a control line that must be dropped to 
		stop a sending system from sending more data. The values for
		this field are defined above.

	sendStartTimeout:
		This field indicates the timeout value while waiting for a 
		start value.

	sendStopLevel:
		This field indicates the receive buffer level at which the
		stop value is sent to stop data flow.

	sendStartLevel:
		This field indicates the receive buffer level at which the
		start value is sent after a stop value has been sent

	dropLineLevel:
		This field indicates the receive buffer level at which the
		flow control line is dropped to stop data flow.

	raiseLineLevel:
		This field indicates the receive buffer level at which the
		flow control line is rasied to start data flow.


*/
typedef struct MIL_ASYNC_SIO_UNIT_PARAMETERS {
	U32 baudRate;
	U16 modemControl;
	U16 stopBits;
	U16 wordLength;
	U16 parityType;
	U16 modemStatus;
	U16 lineStatus;
	U16	receiveTransmitStatus;
	U16	stopWhenReceived;
	U16	startWhenReceived;
	U16	sendToStop;
	U16	sendToStart;
	U16	stopWhenDropped;
	U16	dropLineToStop;
	U32	sendStartTimeout;
	U32	sendStopLevel;
	U32	sendStartLevel;
	U32	dropLineLevel;
	U32	raiseLineLevel;
	U16 reservedU16[2];
	U32 reservedU32[2];
} MIL_ASYNC_SIO_UNIT_PARAMETERS, *P_MIL_ASYNC_SIO_UNIT_PARAMETERS;

/**** Device Specific Request Block Fields ****/
/*

 The milAsyncGetDevParameters funtion request block contains the 
 following fields.

	asyncsioDevParameters:
		This field is a dtat structure that contains device specific
		information about the async device. The definition for this
		data structure is given above.

	asyncsioUnitParms:
		This field is a dtat structure that contains unit specific
		information about the async device. The definition for this
		data structure is given above.


*/

#define rbPublicFieldsAsyncSIOGetDevParameters \
	MIL_ASYNC_SIO_DEVICE_PARAMETERS asyncsioDevParms; \
	MIL_ASYNC_SIO_UNIT_PARAMETERS asyncsioUnitParms; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milAsyncGetDevParameters Request Block

*/

typedef struct MIL_RB_PUB_ASYNC_SIO_GET_DEV_PARAMETERS {
	milRequestBlockCommonFields
	rbPublicFieldsAsyncSIOGetDevParameters
} MIL_RB_PUB_ASYNC_SIO_GET_DEV_PARAMETERS, *P_MIL_RB_PUB_ASYNC_SIO_GET_DEV_PARAMETERS;


/****************************************************************************
 milAsyncSIOSetLineParameters		returns STATUS
 	Sets line parameters for the async device.

 This function is a single stage function that sets the line parameters
 to the configuration specified.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milAsyncSIOSetLineParameters function request block contains the 
 following fields.

	baudRate:
		This field is the requested baud rate.

	stopBits:
		This field is the requested stop bit configuration. Values for
		this field are defined above.

	wordLength:
		This field is the requested word length configuration. Values for
		this field are defined above.

	parityType:
		This field is the requested parity configuration. Values for
		this field are defined above.

*/

#define rbPublicFieldsAsyncSIOSetLineParameters \
	U32 baudRate; \
	U16 stopBits; \
	U16 wordLength; \
	U16 parityType; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milAsyncSetLineParameters Request Block

*/

typedef struct MIL_RB_PUB_ASYNC_SIO_SET_LINE_PARAMETERS {
	milRequestBlockCommonFields
	rbPublicFieldsAsyncSIOSetLineParameters
} MIL_RB_PUB_ASYNC_SIO_SET_LINE_PARAMETERS, *P_MIL_RB_PUB_ASYNC_SIO_SET_LINE_PARAMETERS;


/****************************************************************************
 milAsyncSIOSetModemControl		returns STATUS
 	Sets modem control parameters for the async device.

 This function is a single stage function that sets the modem control
 parameters to the configuration specified.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milAsyncSIOSetModemControl function request block contains the 
 following fields.

	modemControl:
		This field specifies the requested modem control configuration.
		The values for this field are defined above.

*/

#define rbPublicFieldsAsyncSIOSetModemControl \
	U16 modemControl; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2]; \
/*
 
 milAsyncSetModemControl Request Block

*/

typedef struct MIL_RB_PUB_ASYNC_SIO_SET_MODEM_CONTROL {
	milRequestBlockCommonFields
	rbPublicFieldsAsyncSIOSetModemControl
} MIL_RB_PUB_ASYNC_SIO_SET_MODEM_CONTROL, *P_MIL_RB_PUB_ASYNC_SIO_SET_MODEM_CONTROL;



/****************************************************************************
 milAsyncSIOCheckBuffer		returns STATUS
 	Causes driver to check receive and send buffers.

 This function has no parameters, it simply tells the MIL that the 
 characters have been added to the send buffer or removed from the
 receive buffer. Such a call will be needed to start a new send or
 re-start reception which is paused due to flow-control. This function
 can be called at any time.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milAsyncSIOCheckBuffer function request block does not contain
 any function specific fields.

*/

#define rbPublicFieldsAsyncSIOCheckBuffer \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];
/*
 
 milAsyncSIOCheckBuffer Request Block

*/

typedef struct MIL_RB_PUB_ASYNC_SIO_CHECK_BUFFER {
	milRequestBlockCommonFields
	rbPublicFieldsAsyncSIOCheckBuffer
} MIL_RB_PUB_ASYNC_SIO_CHECK_BUFFER, *P_MIL_RB_PUB_ASYNC_SIO_CHECK_BUFFER;


/****************************************************************************
 milAsyncSIOEventCont		returns STATUS
 	Handles the SIO events.

 This continuous function is started with two circular buffers, one for
 sending from and one for receiving to. In normal operation the 
 buffered bytes will be sent and/or received without calling the async.
 event "call-back" function. The call-back will only be made for 
 "interesting" events. The event will be encoded in "sioEvent".
 
 One class of events cover the state of the send and receive buffers.
 Event "sioSendWarnEvent" will occur when the population of the send 
 buffer is less than "sendLowWarnLevel". This event can be disabled by
 setting "sendLowWarnLevel" to zero. Event "sioSendEmptyEvent"
 will occur when the send buffer is empty. 
 
 Event "sioReceiveWarnEvent" is returned when the population of the
 receive buffer is equal to "receiveFullWarnLevel". This event can
 be disabled by setting "receiveFullWarnLevel" to maxU32.
 "sioReceiveFullEvent" is returned when the receive buffer is completely
 full and "sioReceiveOverflowEvent" when a byte is received when there
 is no room in the buffer.

 Another type of event allows the async. event routine to examine each
 change in line status (including each received byte). This mode is 
 enabled when the caller sets "reportLineChanges" to true. In this
 type of call-back the "receivedByte" field is only valid if the 
 "milDataReady" flag is set in "lineStatus". The call-back routine can 
 intercept the received by setting "bufferReceivedByte" to false. 
 Otherwise the byte will be buffered if "bufferReceivedByte" is true
 when the call-back returns.

 The async. serial device supports full duplex send and receive with
 optional flow-control in both directions. Flow control can be done
 using the control lines (DTR and/or RTS) and/or character-
 based flow control (XON/XOFF, ACK/NAK, etc.). 
 
 For receive control the byte "sendToStop" will be transmitted when
 the receive buffer INCREASES to "sendStopLevel" bytes. The restart
 of transmision will be requested by sending "sendToStart" when the
 receive buffer DECREASES to "sendStartLevel". If "sendStartTimeout"
 is non-zero and that many milliseconds pass without transmission
 restart then another "sendToStart" byte will be transmitted. The
 difference between "sendStopLevel" and "sendStartLevel" provides
 hysteresis in flow control.  In a similar fashion " the control
 lines in "dropLineToStop" will be lowered when the receive buffer
 INCREASES to "dropLineLevel" bytes and raised when the population
 DECREASES to "raiseLineLevel". Setting "sendStopLevel" and 
 "sendStartLevel" to maxU32 will inhibit the transmision of any
 flow-control characters from this device.

 The sending of bytes from this device is controlled by the fields
 "stopWhenReceived", "startWhenReceived" and "stopWhenDropped".
 Setting "stopWhenReceived" or "startWhenReceived" and "stopWhenDropped" 
 to maxU16 will disable any response to incoming flow-control characters
 or lines.

*/

/**** Function Specific Definitions ****/
/*
 Async Serial Events

 The following definitions make up the async serial events

*/

Enum16(MIL_ASYNC_SERIAL_EVENT) {
	sioSendWarnEvent		= flag0,
	sioSendEmptyEvent		= flag1,
	sioReceiveWarnEvent		= flag2,
	sioReceiveFullEvent		= flag3,
	sioReceiveOverflowEvent = flag4,
	sioLineChangeEvent		= flag5
};

/**** Device Specific Request Block Fields ****/
/*

 The milAsyncSIOEventCont function request block contains the 
 following fields.

	sioEvent:
		This field identifies the sio event that occured.

	modemStatus:
		This field indicates the current modem status.

	lineStatus:
		This field indicates the current line status.

	receivedByte:
		This field contains the received byte if the event was
		a receive event.

	reportLineChanges:
		This field, if set, causes line changes to be reported.

	bufferReceivedByte:
				
	sendBuffer:
		This field specifies the send buffer.

	sendLowWarnLevel:
		This field identifies the receive buffer level at which
 		a warning event is sent to the call back function.

	receiveBuffer:
		This field specifies the receive buffer for incomming data.

	receiveFullWarnLevel:
		This field specifies the receive buffer level at which
		a full buffer warning event is sent to the call back function.

	sendToStop:
		This field is the current stop value that must be sent to
		stop data flow.

	sendToStart:
		This field is the current start value that must be sent to
		start data flow.

	sendStopLevel:
		This field is the buffer level at which a stop value will be
		sent.

	sendStartLevel:
		This field is the buffer level at which a start value will be
		sent.

	sendStartTimeout:
		This is the timeout value that is used to time the delay between
		receiving a stop value and a start value.

	dropLineToStop:
		This field indicates lines that will cause data flow to
		stop if the lines drop.

	dropLineLevel:
		This field indicates the buffer level at which the flow control
		line will be dropped.

	raiseLineLevel:
		This field indicates the buffer level at which the flow control
		line will be rasied.

	stopWhenReceived:
		This field indicates the value that, when received, will cause
		the device to stop sending.

	startWhenReceived:
		This field indicates the value that, when received, will cause
		the device to start sending.

	stopWhenDropped:
		This field identifies the status line that will cause the device
		to stop sending when it drops.

	pAsyncSIOEventFunc:
		This field is a pointer to a function that is called if a 
		special even occurs.
*/

#define rbPublicFieldsAsyncSIOEventCont \
	MIL_ASYNC_SERIAL_EVENT sioEvent; \
	U16 modemStatus; \
	U16 lineStatus; \
	U16 receivedByte; \
	BOOLEAN reportLineChanges; \
	BOOLEAN bufferReceivedByte; \
	MIL_CIRCULAR_BYTE_BUFFER sendBuffer; \
	U32 sendLowWarnLevel; \
	MIL_CIRCULAR_BYTE_BUFFER receiveBuffer; \
	U32 receiveFullWarnLevel; \
	U16 sendToStop; \
	U16	sendToStart; \
	U32 sendStopLevel; \
	U32 sendStartLevel; \
	U32 sendStartTimeout; \
	U16 dropLineToStop; \
	U32 dropLineLevel; \
	U32 raiseLineLevel; \
	U16 stopWhenReceived; \
	U16 startWhenReceived; \
	U16 stopWhenDropped; \
	P_MIL_ASYNC_EVENT_FUNC pAsyncSIOEventFunc; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milAsyncSIOEventCont Request Block

*/

typedef struct MIL_RB_PUB_ASYNC_SIO_EVENT_CONT {
	milRequestBlockCommonFields
	rbPublicFieldsAsyncSIOEventCont
} MIL_RB_PUB_ASYNC_SIO_EVENT_CONT, *P_MIL_RB_PUB_ASYNC_SIO_EVENT_CONT;


/****************************************************************************
 milAsyncSendBreak		returns STATUS
 	Sends a break.

 This function causes the async device to send a break.

*/

/**** Device Specific Request Block Fields ****/
/*

 The milAsyncSendBreak function request block contains the 
 following fields.

	milliseconds:
		This field indicates the number of milliseconds to hold the
		break.

*/

#define rbPublicFieldsAsyncSIOSendBreak \
	U16 milliseconds; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milAsyncSendBreak Request Block

*/


typedef struct MIL_RB_PUB_ASYNC_SIO_SEND_BREAK {
	milRequestBlockCommonFields
	rbPublicFieldsAsyncSIOSendBreak
} MIL_RB_PUB_ASYNC_SIO_SEND_BREAK, *P_MIL_RB_PUB_ASYNC_SIO_SEND_BREAK;

#endif
