/****************************************************************************
 File: control.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.77  $
   $Author:   cmeyer  $
     $Date:   18 Mar 1992 14:46:30  $

 This file contains the API definition for clsControl.

 clsControl inherits from clsBorder.

 clsControl implements the previewing and client notification behavior of
 several UI components.   clsControl is an abstract class -- it is never
 instantiated directly.
****************************************************************************/

/**** Debugging Flags ****/
/*
 The clsControl debugging flag is '%'.  Defined values are:

    flag8  (0x0100):    msgControlEnable info
*/

#ifndef CONTROL_INCLUDED
#define CONTROL_INCLUDED

													#ifndef BORDER_INCLUDED
#include <border.h>
													#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT			CONTROL;

/****  Dynamic Enable Styles  ****/
/* Use one of these values in control's style.dynamicEnable. */
#define csDynamicNone		0	// no dynamic determination of "enabled"
#define csDynamicClient		1	// send msgControlProvideEnable to client
#define csDynamicObject		2	// send msgControlProvideEnable to "object"
#define csDynamicPargs		3	// set "enabled" from pArgs

typedef struct CONTROL_STYLE {
	U16	enable			: 1,	// if enabled, a control responds to input
		previewGrab		: 1,	// grab input when previews start
		previewRepeat	: 1,	// previews repeat on time-out
		previewing	   	: 1,	// msgControlBeginPreview has been sent out
		dirty			: 1,	// dirty status
		previewEnable	: 1,	// self-send msgControlBeginPreview, etc.
		showDirty		: 1,	// visuals reflect dirty state
		dynamicEnable	: 2,	// how "enable" value is determined
		private1		: 1,	// reserved
		spare			: 6;	// unused (reserved)
} CONTROL_STYLE, *P_CONTROL_STYLE;

/*
 Default CONTROL_STYLE:
//{
	enable			= true
	previewGrab		= true
	previewRepeat	= false
	previewing	   	= false
	dirty			= false
	previewEnable	= false
	showDirty		= true
//}
*/

typedef struct CONTROL_STRING {
	P_CHAR		pString;
	U16			len;
	U32			spare;		// unused (reserved)
} CONTROL_STRING, *P_CONTROL_STRING;

/* Advisory return values for subclasses */
#define stsControlCancelPreview			MakeWarning(clsControl, 13)
#define stsControlCancelRepeat			MakeWarning(clsControl, 1)

/****************************************************************************
 msgNew		takes P_CONTROL_NEW, returns STATUS
	category: class message
	Creates a control window.

 Note that setting pArgs->control.style.enable to false does not result in
 pArgs->border.style.look set to bsLookInactive.  If you change style.enable
 after msgNew (via msgControlSetStyle or msgControlSetEnable), the
 border.style.look will be changed to match. 
*/

typedef struct CONTROL_NEW_ONLY {
	CONTROL_STYLE		style;	// overall style
	OBJECT				client;	// client to notify
	U32					spare;	// unused (reserved)
} CONTROL_NEW_ONLY, CONTROL_METRICS,
  *P_CONTROL_NEW_ONLY, *P_CONTROL_METRICS;

#define	controlNewFields		\
	borderNewFields				\
	CONTROL_NEW_ONLY		control;

typedef struct CONTROL_NEW {
	controlNewFields
} CONTROL_NEW, *P_CONTROL_NEW;

/****************************************************************************
 msgNewDefaults				takes P_CONTROL_NEW, returns STATUS
	category: class message
	Initializes the CONTROL_NEW structure to default values.

 Zeroes pArgs->control and sets
//{
	pArgs->win.flags.style |= wsFileInline;

	pArgs->border.style.previewAlter = bsAlterBackground;
	pArgs->border.style.selectedAlter = bsAlterBackground;

	pArgs->control.style.enable = true;
	pArgs->control.style.showDirty = true;
//}
*/

/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Causes an object to file itself in an object file.

 If the client of the control is OSThisApp(), this is remembered and
 reinstated in msgRestore.  In any case, the client is not saved.
*/

/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.
	
 clsControl restores the instance from the file.  If the client of the
 control was OSThisApp() when filed, the client is set to OSThisApp(),
 otherwise objNull.
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                  Messages Clients Send to Controls					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgControlGetMetrics		takes P_CONTROL_METRICS, returns STATUS
	Passes back the current metrics.
*/
#define msgControlGetMetrics	MakeMsg(clsControl, 1)

/****************************************************************************
 msgControlSetMetrics		takes P_CONTROL_METRICS, returns STATUS
	Sets the metrics.
*/
#define msgControlSetMetrics	MakeMsg(clsControl, 2)

/****************************************************************************
 msgControlGetStyle		takes P_CONTROL_STYLE, returns STATUS
	Passes back the current style values.
*/
#define msgControlGetStyle		MakeMsg(clsControl, 3)

/****************************************************************************
 msgControlSetStyle		takes P_CONTROL_STYLE, returns STATUS
	Sets the style values.

 If	style.enable changes, the control does the following:
    -:  self-sends msgBorderSetLook, with pArgs of bsLookActive if
        style.enable is true, bsLookInactive otherwise.
    -:  self-sends msgControlCancelPreview, pNull if style.enable is false.
*/
#define msgControlSetStyle		MakeMsg(clsControl, 4)

/****************************************************************************
 msgControlGetClient		takes P_UID, returns STATUS
	Passes back metrics.client.
*/
#define msgControlGetClient		MakeMsg(clsControl, 5)

/****************************************************************************
 msgControlSetClient		takes UID, returns STATUS
	Sets metrics.client.
*/
#define msgControlSetClient		MakeMsg(clsControl, 6)

/****************************************************************************
 msgControlGetDirty		takes P_BOOLEAN, returns STATUS
	Passes back true if the control has been altered since dirty was set 
	false.
*/
#define msgControlGetDirty		MakeMsg(clsControl, 15)

/****************************************************************************
 msgControlSetDirty		takes BOOLEAN, returns STATUS
	Sets style.dirty.
*/
#define msgControlSetDirty		MakeMsg(clsControl, 16)

/****************************************************************************
 msgControlGetEnable		takes P_BOOLEAN, returns STATUS
	Passes back style.enable.
*/
#define msgControlGetEnable		MakeMsg(clsControl, 17)

/****************************************************************************
 msgControlSetEnable		takes BOOLEAN, returns STATUS
	Sets style.enable.

 Responds to changes in style.enable as in msgControlSetStyle.
*/
#define msgControlSetEnable		MakeMsg(clsControl, 18)

/****************************************************************************
 msgControlEnable				takes P_CONTROL_ENABLE, returns STATUS
	The control re-evaluates whether it is enabled.

 This is commonly used with menu buttons that need to be enabled/disabled
 according to some constraints known to the sender.  For example,
 clsMenuButton sends msgControlEnable to its menu before showing the menu,
 which results in each control in the menu receiving msgControlEnable with
 appropriate parameters.  See msgMenuButtonShowMenu (mbutton.h) for sample
 usage.

 clsControl responds to msgControlEnable as follows:

    -:  If style.dynamicEnable is csDynamicNone, simply returns stsOK.

    -:  If style.dynamicEnable is csDynamicPargs, style.enable is set to
        pArgs->enable.

    -:  If style.dynamicEnable is csDynamicClient and metrics.client is
        objNull, does not change enable and returns stsOK.
		
    -:  If style.dynamicEnable is csDynamicObject and pArgs->object is
        objNull, sets style.enable to false (as in msgControlSetEnable) and
        returns stsOK.  

 The cases that remain are style.dynamicEnable of csDynamicClient or
 csDynamicObject, and a non-null object.

    -:  If the object is not owned by OSThisProcess(), sets
        style.enable to false (as in msgControlSetEnable) and returns
        stsOK.  Otherwise, sends msgControlProvideEnable with the following
        CONTROL_PROVIDE_ENABLE parameters:
//{
			root	= pArgs->root;
			control	= self;
			tag		= self's WIN_METRICS.tag;
			enable	= current value of style.enable;
//}

    -:  If the object responds to msgControlProvideEnable with
        stsNotUnderstood, sets style.enable to true (as in
        msgControlSetEnable) and returns stsOK.  Otherwise, sets
        style.enable to CONTROL_PROVIDE_ENABLE.enable (as in
        msgControlSetEnable) and returns stsOK.

 See Also
	msgControlProvideEnable
*/
#define msgControlEnable		MakeMsg(clsControl, 19)

typedef struct CONTROL_ENABLE {
	WIN			root;		// In: originator
	OBJECT		object;		// In: object for msgControlProvideEnable
	BOOLEAN		enable;		// In: value to use iff csDynamicPargs
	U32			spare;		// reserved (unused)
} CONTROL_ENABLE, *P_CONTROL_ENABLE;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Subclass Responsibility Messages					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgControlGetValue		takes P_S32	, returns STATUS
	Passes back the current "value" of the control.

 In response to this message clsControl returns stsNotUnderstood.
*/
#define msgControlGetValue			MakeMsg(clsControl, 7)

/****************************************************************************
 msgControlSetValue		takes S32	, returns STATUS
	Sets the current "value" of the control.

 In response to this message clsControl returns stsNotUnderstood.
*/
#define msgControlSetValue			MakeMsg(clsControl, 8)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                  Messages Controls Send to Self						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgControlBeginPreview		takes P_INPUT_EVENT, returns STATUS
	category: self-sent
	Self-sent when msgPenDown is received.

 clsControl responds with stsOK.  pArgs is pNull if the preview is not
 caused by an input event.
*/
#define msgControlBeginPreview		MakeMsg(clsControl, 10)

/****************************************************************************
 msgControlUpdatePreview		takes P_INPUT_EVENT, returns STATUS
	category: self-sent
	Self-sent when msgPenMoveDown is received.
	
 clsControl responds with stsOK.  pArgs is pNull if the preview is not
 caused by an input event.
*/
#define msgControlUpdatePreview		MakeMsg(clsControl, 11)

/****************************************************************************
 msgControlRepeatPreview		takes P_INPUT_EVENT, returns STATUS
	category: self-sent
	Self-sent if style.repeatPreview is true.
	Initial delay is 600ms, then immediate repeat until msgPenUp.
	
 clsControl responds with stsOK.

 Subclasses can return stsControlCancelRepeat to prevent the next
 msgControlRepeatPreview.

 pArgs is pNull if the preview is not caused by an input event.
*/
#define msgControlRepeatPreview		MakeMsg(clsControl, 12)

/****************************************************************************
 msgControlCancelPreview		takes P_INPUT_EVENT, returns STATUS
	category: self-sent
	Self-sent when style.previewGrab is false and msgPenExitDown is received.
	Clients or subclasses can send this to a control to cancel existing 
	preview.

 Sets style.previewing to false.

 pArgs is pNull if the preview is not caused by an input event.
*/
#define msgControlCancelPreview		MakeMsg(clsControl, 13)

/****************************************************************************
 msgControlAcceptPreview		takes P_INPUT_EVENT, returns STATUS
	category: self-sent
	Self-sent when msgPenUp is received.

 If gestures are enabled this message is not sent until msgGWinGesture is
 received with xgs1Tap.
	
 clsControl responds with stsOK.

 pArgs is pNull if the preview is not caused by an input event.
*/
#define msgControlAcceptPreview		MakeMsg(clsControl, 14)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                  Messages Controls Send to Client					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgControlProvideEnable		takes P_CONTROL_PROVIDE_ENABLE, returns STATUS
	Sent out to client or "object" during processing of msgControlEnable.
*/
#define msgControlProvideEnable		MakeMsg(clsControl, 20)

typedef struct CONTROL_PROVIDE_ENABLE {
	WIN			root;		// In: originator
	CONTROL		control;	// In: sending control
	TAG			tag;		// In: tag of sending control
	BOOLEAN		enable;		// In/Out: enabled value for control
	U32			spare;		// unused (reserved)
} CONTROL_PROVIDE_ENABLE, *P_CONTROL_PROVIDE_ENABLE;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages Defined by Other Classes					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgInputEvent 					takes P_INPUT_EVENT, returns STATUS
	Notification of an input event. 

 clsControl first calls ancestor, then responds as follows. (In each of
 these cases, see below for status return value.)

    -:  If pArgs->flags has evBorderTaken set (see border.h), assumes
        clsBorder used the event and returns status.

    -:  If style.enable is false, or style.previewEnable is false, or the
        event is not a pen event, returns status returned by ancestor.

    -:  If pArgs->devCode is msgPenDown, self-sends msgControlBeginPreview
        passing along pArgs.  If return status is stsControlCancelPreview,
        returns status.  If style.previewRepeat is true, and return status
        is not stsControlCancelRepeat, the control repeats preview after
        600ms delay. Sets style.previewing to true.

    -:  If pArgs->devCode is msgPenMoveDown, self-sends
        msgControlUpdatePreview passing	along pArgs.  If return status is
        stsControlCancelPreview, sets style.previewing to false and returns
        status.

    -:  If pArgs->devCode is msgPenUp, checks GWIN_STYLE.gestureEnable.  If
        true, does nothing and returns status.  Otherwise, self-sends
        msgControlAcceptPreview	passing	along pArgs and returns
        stsInputTerminate.

    -:  If pArgs->devCode is msgPenExitDown and style.previewGrab is true
        or style.previewing is false or GWIN_STYLE.gestureEnable is true,
        does nothing and returns status. Otherwise, self-sends
        msgControlCancelPreview passing	along pArgs and returns
        stsInputTerminate.

 clsControl returns stsInputGrabTerminate if no error was encountered and
 style.previewing and style.previewGrab are true after processing the input
 event.  Otherwise, the status returned by ObjectCallAncestor() is
 returned.
*/

/****************************************************************************
 msgGWinGesture		takes P_GWIN_GESTURE, returns STATUS
	Called to process the gesture.

 If ObjectCallAncestor() returns stsOK, clsControl self-sends 
 msgControlCancelPreview and returns stsOK.

 If pArgs->msg is xgs1Tap and style.previewEnable is true, self-sends
 msgControlAcceptPreview and returns stsOK.

 All other gestures result in msgGWinForwardedGesture to the control
 client, followed by msgControlCancelPreview to self.
*/

/****************************************************************************
 msgGWinAbort	takes void, returns STATUS
	Clears the translation state of the GWin.  

 clsControl responds to this by self-sending msgControlCancelPreview if the
 receiver is currently previewing.
*/

/****************************************************************************
 msgGWinGestureDone 	takes P_GWIN_GESTURE, returns STATUS
    category: self-sent
    Sent to signal the end of a gesture.

 clsControl responds to this by self-sending msgControlCancelPreview if the
 receiver is currently previewing.
*/

/****************************************************************************
 msgBorderGetDirty	takes P_BOOLEAN, returns STATUS
 	Passes back true if any child responds to msgBorderGetDirty with true;
	otherwise passes back false.

 clsControl responds by self-sending msgControlGetDirty.  If the control is
 dirty, true is passed back.  Otherwise, this message is passed on to
 clsControl's ancestor.  clsBorder will respond by passing back true if any
 child of this control is dirty.
*/

/****************************************************************************
 msgBorderSetDirty	takes BOOLEAN, returns STATUS
	Sends msgBorderSetDirty(pArgs) to each child.
 
 clsControl will call ancestor (to allow clsBorder to dirty any children),
 then self-send msgControlSetDirty(pArgs).
*/

#endif	// CONTROL_INCLUDED
