/****************************************************************************
 File: appdir.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.21  $
   $Author:   pstahl  $
     $Date:   03 Mar 1992 11:46:16  $

 This file contains the API definition for clsAppDir.

 clsAppDir inherits from clsDirHandle.
 Provides management for document directories.

 "AppDir" stands for Application Directory Handle.
****************************************************************************/

/**** Introduction ****/
/*
 Application directory nodes represent documents in the document hierarchy.
 Application directories are where documents store their resource files
 and any other files they use.  Attributes on application directories
 specify useful information about each document.

 clsAppDir is used to manage the various file system attributes associated
 with a document in PenPoint.  It includes definitions of these attributes
 and messages to manage them.  clsAppDir also provides support for
 enumerating embedded documents via the filesystem.  This is similar to
 the file system's FSReadDir facilities, but clsAppDir filters out all
 files and directories that are not documents.

 A document can find its application directory by self sending
 msgAppGetMetrics.  The application directory's uid will be passed back
 in the dir field of the APP_METRICS structure.  See app.h for more
 information.

 Application directories are created automatically for documents during
 AppInit time by the Application Framework.  Application classes generally
 should never create or destroy application directories themselves.
*/

#ifndef APPDIR_INCLUDED
#define APPDIR_INCLUDED

#ifndef APP_INCLUDED
#include <app.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT APP_DIR, *P_APP_DIR;

/****  File System Attributes  ****/
/*
 * These attributes are stamped on every document directory.
 */
#define appAttrClass			FSMakeFix32Attr(clsAppDir, 1)
#define appAttrSequence			FSMakeFix32Attr(clsAppDir, 4)
#define appAttrNumChildren		FSMakeFix32Attr(clsAppDir, 3)
#define appAttrFlags			FSMakeFix64Attr(clsAppDir, 6)
#define appAttrBookmark			FSMakeStrAttr(clsAppDir, 9)
#define appAttrAuthor			FSMakeStrAttr(clsAppDir, 10)
#define appAttrComments			FSMakeStrAttr(clsAppDir, 11)
#define appAttrClassName		FSMakeStrAttr(clsAppDir, 12)
//REFGEN BEGINIGNORE
// xxx delete these
//REFGEN ENDIGNORE
#define appAttrGlobalSequence	FSMakeFix32Attr(clsAppDir, 4)

/****  Application Directory Flags  ****/
/*
 This structure defines the application directory flags.
 They are stamped on a document directory with appAttrFlags.
 This structure is used in the flags field of APP_DIR_ATTRS.
 */
typedef struct APP_DIR_FLAGS {

	U16		application		: 1;	// true = this is an application.
	U16		newInstance		: 1;	// true = new app instance.
	U16		disabled		: 1;	// true = app is disabled, don't activate.
	U16		bookmark		: 1;	// true = app has a tab
	U16		readOnly		: 1;	// True = app is read only.
	U16		deletable		: 1;	// true = app can be deleted.
	U16		movable			: 1;	// true = app can be moved.
	U16		copyable		: 1;	// true = app can be copied.
	U16		reserved1		: 8;	// Reserved.		
	U16		reserved2		: 16;	// Reserved.		
	U16		reserved3		: 16;	// Reserved.		
	U16		reserved4		: 16;	// Reserved.		

} APP_DIR_FLAGS, *P_APP_DIR_FLAGS;

/****  Application Directory Attributes Structure  ****/
/*
 This structure is used to specify and pass back the directory attributes
 in one chunk.
    -: appClass     The document's application class (sub-class of clsApp).
    -: uuid         The document's uuid. Can be used in msgNew to 
                    clsDirHandle
                    or clsAppDir to open a handle on a document directory.
    -: sequence     The 1-based position of a document within its embeddor.
                    If the document is in a notebook, this is the document's
                    position within its section.
    -: numChildren  The total number of embedded children.
 */
typedef struct APP_DIR_ATTRS {
	CLASS			appClass;		// Application class.
	UUID			uuid;			// Application uuid.
	U32				sequence;		// Local sequence number.	
	U32				numChildren;	// Number of child apps (recursive).
	APP_DIR_FLAGS	flags;			// Flags.
} APP_DIR_ATTRS, *P_APP_DIR_ATTRS;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew     takes P_FS_NEW, returns STATUS
    category: class message
    Creates a new AppDir.

 See fs.h for the FS_NEW structure definition.

 clsAppDir has no method for msgNewDefaults.  See fs.h for a description
 of clsDirHandle's handler for msgNewDefaults.
*/

/****************************************************************************
 msgAppDirGetAttrs	takes P_APP_DIR_GET_SET_ATTRS, returns STATUS
	Passes back a document's application directory attributes.

 If you are interested in only one of the attributes, use the individual
 msgAppDirGet... messages described below.  They're generally faster.
*/
#define msgAppDirGetAttrs				MakeMsg(clsAppDir, 1)

typedef struct APP_DIR_GET_SET_ATTRS {
	P_STRING		pPath;	// in: Path relative to target directory.
	APP_DIR_ATTRS	attrs;	// in/out: Application directory attributes.
} APP_DIR_GET_SET_ATTRS, *P_APP_DIR_GET_SET_ATTRS;

/****************************************************************************
 msgAppDirSetAttrs	takes P_APP_DIR_GET_SET_ATTRS, returns STATUS
	Specifies a document's application directory attributes.

 If you are interested in only one of the attributes, use the individual
 msgAppDirSet... messages described below.  They're generally faster.
*/
#define msgAppDirSetAttrs				MakeMsg(clsAppDir, 2)

/****************************************************************************
 msgAppDirGetFlags	takes P_APP_DIR_GET_SET_FLAGS, returns STATUS
	Passes back a document's application directory flags.
*/
#define msgAppDirGetFlags				MakeMsg(clsAppDir, 3)

typedef struct APP_DIR_GET_SET_FLAGS {
	P_STRING		pPath;	// in: Path relative to target directory.
	APP_DIR_FLAGS	flags;	// in/out: Application directory control flags.
} APP_DIR_GET_SET_FLAGS, *P_APP_DIR_GET_SET_FLAGS;

/****************************************************************************
 msgAppDirSetFlags	takes P_APP_DIR_GET_SET_FLAGS, returns STATUS
	Specifies a document's application directory flags.
*/
#define msgAppDirSetFlags				MakeMsg(clsAppDir, 4)

/****************************************************************************
 msgAppDirGetClass	takes P_APP_DIR_UPDATE_CLASS, returns STATUS
	Passes back a document's application class.
*/
#define msgAppDirGetClass				MakeMsg(clsAppDir, 5)

typedef struct APP_DIR_UPDATE_CLASS {
	P_STRING	pPath;		// in: Path relative to target directory.
	CLASS		appClass;	// in/out: Application directory class.
} APP_DIR_UPDATE_CLASS, *P_APP_DIR_UPDATE_CLASS;

/****************************************************************************
 msgAppDirSetClass	takes P_APP_DIR_UPDATE_CLASS, returns STATUS
	Specifies a document's application class.
*/
#define msgAppDirSetClass				MakeMsg(clsAppDir, 6)

/****************************************************************************
 msgAppDirGetUUID	takes P_APP_DIR_UPDATE_UUID, returns STATUS
	Passes back an application directory's uuid.
*/
#define msgAppDirGetUUID				MakeMsg(clsAppDir, 7)

typedef struct APP_DIR_UPDATE_UUID {
	P_STRING	pPath;		// in: Path relative to target directory.
	UUID		uuid;		// in/out: Application directory uuid.
} APP_DIR_UPDATE_UUID, *P_APP_DIR_UPDATE_UUID;

/****************************************************************************
 msgAppDirSetUUID	takes P_APP_DIR_UPDATE_UUID, returns STATUS
	Specifies an application directory's uuid.
*/
#define msgAppDirSetUUID				MakeMsg(clsAppDir, 8)

/****************************************************************************
 msgAppDirGetUID	takes P_APP_DIR_UPDATE_UID, returns STATUS
	Passes back an application directory's uid.
*/
#define msgAppDirGetUID					MakeMsg(clsAppDir, 9)

typedef struct APP_DIR_UPDATE_UID {
	P_STRING	pPath;	// in: Path relative to target directory.
	UID			uid;	// in/out: App directory uid.
} APP_DIR_UPDATE_UID, *P_APP_DIR_UPDATE_UID;

/****************************************************************************
 msgAppDirSetUID	takes P_APP_DIR_UPDATE_UID, returns STATUS
	Specifies an application directory's uid.
*/
#define msgAppDirSetUID					MakeMsg(clsAppDir, 10)

/****************************************************************************
 msgAppDirGetSequence	takes P_APP_DIR_UPDATE_SEQ, returns STATUS
	Passes back an application directory's sequence number.

 If the document is in a notebook, the sequence number is a 1-based
 position within the section.
*/
#define msgAppDirGetSequence			MakeMsg(clsAppDir, 11)

typedef struct APP_DIR_UPDATE_SEQUENCE {
	P_STRING	pPath;		// in: Path relative to target directory.
	U32			sequence;	// in/out: Application directory sequence.
} APP_DIR_UPDATE_SEQUENCE, *P_APP_DIR_UPDATE_SEQUENCE;

/****************************************************************************
 msgAppDirSetSequence	takes P_APP_DIR_UPDATE_SEQUENCE, returns STATUS
	Specifies an application directory's sequence number.

 If the document is in a notebook, the sequence number is a 1-based
 position within the section.
*/
#define msgAppDirSetSequence			MakeMsg(clsAppDir, 12)

/****************************************************************************
 msgAppDirGetNumChildren  takes P_APP_DIR_UPDATE_NUM_CHILDREN, returns STATUS
	Passes back the total number of embedded children of a document.
*/
#define msgAppDirGetNumChildren			MakeMsg(clsAppDir, 22)

typedef struct APP_DIR_UPDATE_NUM_CHILDREN {

	P_STRING	pPath;			// in: Path relative to target directory.
	U32			numChildren;	// in/out: App directory attr numchildren.

} APP_DIR_UPDATE_NUM_CHILDREN, *P_APP_DIR_UPDATE_NUM_CHILDREN;

/****************************************************************************
 msgAppDirSetNumChildren  takes P_APP_DIR_UPDATE_NUM_CHILDREN, returns STATUS
	Specifies the total number of embedded children of a document.
*/
#define msgAppDirSetNumChildren			MakeMsg(clsAppDir, 23)

/****************************************************************************
 msgAppDirGetGlobalSequence	takes P_APP_DIR_GET_GLOBAL_SEQUENCE, returns STATUS
	Passes back an application directory's global sequence number.

 The global sequence number is the 1-based position of a document within
 its clsRootContainerApp embeddor (i.e., the document's page number in
 the notebook).
*/
#define msgAppDirGetGlobalSequence		MakeMsg(clsAppDir, 21)

typedef struct APP_DIR_GET_GLOBAL_SEQUENCE {
	P_STRING	pPath;			// in: Path relative to target directory.
	U32			globalSequence;	// in/out: App directory global sequence.
} APP_DIR_GET_GLOBAL_SEQUENCE, *P_APP_DIR_GET_GLOBAL_SEQUENCE;

/****************************************************************************
 msgAppDirGetBookmark	takes P_APP_DIR_GET_BOOKMARK, returns STATUS
	Passes back an document's application tab.
	
 If the application directory has no tab (appDirFlags.bookmark==false),
 msgAppDirGetBookmark will return stsOK and pArgs->label will be unchanged.
 For this reason it is recommended that you drop a null byte into
 pArgs->label[0] before calling msgAppDirGetBookmark.  Then, if the
 application directory has no tab, you will get back a null string.
*/
#define msgAppDirGetBookmark			MakeMsg(clsAppDir, 13)

typedef struct APP_DIR_GET_BOOKMARK {
	P_STRING	pPath;				  // in: Path relative to target directory.
	char		label[nameBufLength]; // out: tab label.
} APP_DIR_GET_BOOKMARK, *P_APP_DIR_GET_BOOKMARK;

/****************************************************************************
 msgAppDirSetBookmark	takes P_APP_DIR_SET_BOOKMARK, returns STATUS
	Specifies a document's application tab.  
	
 clsAppDir sends msgAppChanged to observers of clsApp as a result of
 this message.  See app.h for a description of msgAppChanged.

 If label[0] is NULL, clsAppDir uses the default label, which is the
 name of the document.
*/
#define msgAppDirSetBookmark			MakeMsg(clsAppDir, 14)

typedef struct APP_DIR_SET_BOOKMARK {
	BOOLEAN		on;					  // in: Turn bookmark on or off.
	P_STRING	pPath;				  // in: Path relative to target directory.
	char		label[nameBufLength]; // in/out: tab label.
} APP_DIR_SET_BOOKMARK, *P_APP_DIR_SET_BOOKMARK;

/****************************************************************************
 msgAppDirGetNextInit	takes P_APP_DIR_NEXT, returns STATUS
	Initializes an APP_DIR_NEXT structure.

 Send this message to an application directory to prepare it for an
 ensuing msgAppDirGetNext loop.
*/
#define msgAppDirGetNextInit			MakeMsg(clsAppDir, 15)

/****************************************************************************
 msgAppDirGetNext	takes P_APP_DIR_NEXT, returns STATUS
	Passes back the attributes of the next application directory.

 Send this message to an application directory in a loop to get the
 appDirAttrs for each embedded document (not recursive), ordered by
 sequence number.

 You generally do not change the values in the APP_DIR_NEXT structure
 between calls to msgAppDirGetNext.  Doing so jeopardizes the traversal
 of the embedded documents.
*/
#define msgAppDirGetNext				MakeMsg(clsAppDir, 16)

typedef struct APP_DIR_NEXT {
	APP_DIR_ATTRS	attrs;		// out: attrs for next child.
	P_STRING		pName;		// out: name of next child.
	U32				fsFlags;	// out: fs flags for next child (see fs.h)
	P_UNKNOWN		pFirst;		// out: first app dir to examine
	P_UNKNOWN		pNext;		// out: next app dir to examine
	P_UNKNOWN		handle;		// out: current app dir
} APP_DIR_NEXT, *P_APP_DIR_NEXT;

/****************************************************************************
 msgAppDirReset	takes P_APP_DIR_NEXT, returns STATUS
	Frees resources after a series of msgAppDirGetNext messages.

 You must send this message to the application directory after the
 msgAppDirGetNext loop has completed.  Failing to do so can cause
 internally allocated memory not to be deallocated.
*/
#define msgAppDirReset					MakeMsg(clsAppDir, 17)

/****************************************************************************
 msgAppDirSeqToName	takes P_APP_DIR_SEQ_TO_NAME, returns STATUS
	Passes back the name of the embedded document with a specified
    sequence number.
*/
#define msgAppDirSeqToName				MakeMsg(clsAppDir, 18)

typedef struct APP_DIR_SEQ_TO_NAME {
	U32			sequence;		// in: Sequence number.
	P_STRING	pName;			// out: Buffer for name. 
								// Must be nameBufLength long.
} APP_DIR_SEQ_TO_NAME, *P_APP_DIR_SEQ_TO_NAME;

/****************************************************************************
 msgAppDirGetDirectNumChildren	takes P_U32, returns STATUS
	Passes back the number of directly embedded documents (not recursive).
*/
#define msgAppDirGetDirectNumChildren	MakeMsg(clsAppDir, 19)

/****************************************************************************
 msgAppDirGetTotalNumChildren	takes P_U32, returns STATUS
	Passes back the total number of embedded documents (recursive).
*/
#define msgAppDirGetTotalNumChildren	MakeMsg(clsAppDir, 20)

#endif	// APPDIR_INCLUDED

