/****************************************************************************
 File: tttdata.c

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 You may use this Sample Code any way you please provided you 
 do not resell the code and that this notice (including the above 
 copyright notice) is reproduced on all copies.  THIS SAMPLE CODE 
 IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, AND GO CORPORATION 
 EXPRESSLY DISCLAIMS ALL IMPLIED WARRANTIES, INCLUDING BUT NOT 
 LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE. IN NO EVENT WILL GO CORPORATION BE LIABLE TO YOU 
 FOR ANY CONSEQUENTIAL,INCIDENTAL,OR INDIRECT DAMAGES ARISING OUT OF 
 THE USE OR INABILITY TO USE THIS SAMPLE CODE.

 $Revision:   1.6  $
 $Author:   kcatlin  $
 $Date:   30 Jan 1992 19:14:56  $

 This file contains the implementation of clsTttData.
****************************************************************************/

#ifndef DEBUG_INCLUDED
#include <debug.h>
#endif

#ifndef RESFILE_INCLUDED
#include <resfile.h>
#endif

#ifndef TTTDATA_INCLUDED
#include <tttdata.h>
#endif

#ifndef TTTPRIV_INCLUDED
#include <tttpriv.h>
#endif

#ifndef OSHEAP_INCLUDED
#include <osheap.h>
#endif

#ifndef UNDO_INCLUDED
#include <undo.h>
#endif

#include <stdlib.h>
#include <string.h>

#include <methods.h>


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          Defines, Types, Globals, Etc	 		   	   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


typedef struct TTT_DATA_INST {

	TTT_DATA_METRICS	metrics;

} TTT_DATA_INST,
  * P_TTT_DATA_INST,
  * * PP_TTT_DATA_INST;


//
// CURRENT_VERSION is the file format version written by this implementation.
// MIN_VERSION is the minimum file format version readable by this 
// implementation.  MAX_VERSION is the maximum file format version readable
// by this implementation.
//
#define CURRENT_VERSION 0
#define MIN_VERSION		0
#define MAX_VERSION		0


typedef struct TTT_DATA_FILED_0 {

	TTT_SQUARES	squares;

} TTT_DATA_FILED_0, * P_TTT_DATA_FILED_0;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          Utility Routines							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
	TttDataFiledData0FromInstData

	Computes filed data from instance data.
****************************************************************************/
STATIC void PASCAL
TttDataFiledData0FromInstData(
	P_TTT_DATA_INST		pInst,
	P_TTT_DATA_FILED_0	pFiled)
{
	memcpy(pFiled->squares, pInst->metrics.squares,
			sizeof(pFiled->squares));
} /* TttDataFiledData0FromInstData */


/****************************************************************************
	TttDataInstDataFromFiledData0

	Computes instance data from filed data.
****************************************************************************/
STATIC void PASCAL
TttDataInstDataFromFiledData0(
	P_TTT_DATA_FILED_0	pFiled,
	P_TTT_DATA_INST		pInst)
{
	memcpy(pInst->metrics.squares, pFiled->squares,
			sizeof(pInst->metrics.squares));
} /* TttDataInstDataFromFiledData0 */


/****************************************************************************
	TttDataNotifyObservers
	
	Sends notifications.
****************************************************************************/
#define DbgTttDataNotifyObservers(x) \
	TttDbgHelper("TttDataNotifyObservers",tttDataDbgSet,0x1,x)

STATIC STATUS PASCAL
TttDataNotifyObservers(
	OBJECT					self,
	P_ARGS					pArgs)
{
	OBJ_NOTIFY_OBSERVERS	nobs;
	STATUS 					s;

	DbgTttDataNotifyObservers((""))

	nobs.msg = msgTttDataChanged;
	nobs.pArgs = pArgs;
	nobs.lenSend = SizeOf(TTT_DATA_CHANGED);
	ObjCallJmp(msgNotifyObservers, self, &nobs, s, Error);

	DbgTttDataNotifyObservers(("return stsOK"))
	return stsOK;

Error:
	DbgTttDataNotifyObservers(("Error; return 0x%lx",s))
	return s;
} /* TttDataNotifyObservers */


	   
/****************************************************************************
	TttDataRecordStateForUndo
	
	Records current state with undo manager.
	Assumes that a transaction is already open.
****************************************************************************/
#define DbgTttDataRecordStateForUndo(x) \
	TttDbgHelper("TttDataRecordStateForUndo",tttDataDbgSet,0x2,x)

STATIC STATUS PASCAL
TttDataRecordStateForUndo(
	OBJECT				self,
	TAG					undoTag,
	PP_TTT_DATA_INST	pData)
{
	UNDO_ITEM			item;
	STATUS 				s;

	DbgTttDataRecordStateForUndo((""))

	ObjCallJmp(msgUndoBegin, theUndoManager, (P_ARGS)undoTag, s, Error);
	item.object = self;
	item.subclass = clsTttData;
	item.flags = 0;
	item.pData = &((*pData)->metrics);
	item.dataSize = SizeOf(((*pData)->metrics));
	ObjCallJmp(msgUndoAddItem, theUndoManager, &item, s, Error);
	ObjCallJmp(msgUndoEnd, theUndoManager, pNull, s, Error);

	DbgTttDataRecordStateForUndo(("return stsOK"))
	return stsOK;

Error:
	DbgTttDataRecordStateForUndo(("Error; return 0x%lx",s))
	return s;
} /* TttDataRecordStateForUndo */


/****************************************************************************
	TttDataPrivSetMetrics

	Sets metrics and (optionally) records information
	needed to undo the set.  Assumes an undo transaction is open.
****************************************************************************/
#define DbgTttDataPrivSetMetrics(x) \
	TttDbgHelper("TttDataPrivSetMetrics",tttDataDbgSet,0x4,x)

STATIC STATUS PASCAL
TttDataPrivSetMetrics(
	OBJECT				 self,
	P_TTT_DATA_METRICS	 pArgs,
	PP_TTT_DATA_INST	 pData,
	BOOLEAN				 recordUndo)
{
	STATUS				 s;

	DbgTttDataPrivSetMetrics((""))

	//
	// Perhaps record undo information
	//
	if (recordUndo) {
		StsJmp(TttDataRecordStateForUndo(self, pArgs->undoTag, pData), 
				s, Error);
	}

	//
	// Change data.
	//
	(*pData)->metrics = *pArgs;

	DbgTttDataPrivSetMetrics(("returns stsOK"))
	return stsOK;

Error:
	DbgTttDataPrivSetMetrics(("Error; return 0x%lx",s))
	return s;
} /* TttDataPrivSetMetrics */


/****************************************************************************
	TttDataPrivSetSquare

	Sets a square and (optionally) records
	information	needed to undo the set.  Assumes an undo transaction
	is open.
****************************************************************************/
#define DbgTttDataPrivSetSquare(x) \
	TttDbgHelper("TttDataPrivSetSquare",tttDataDbgSet,0x8,x)

STATIC STATUS PASCAL
TttDataPrivSetSquare(
	OBJECT					self,
	P_TTT_DATA_SET_SQUARE	pArgs,
	PP_TTT_DATA_INST		pData,
	BOOLEAN					recordUndo)
{
	STATUS					s;

	DbgTttDataPrivSetSquare((""))

	//
	// Perhaps record undo information
	//
	if (recordUndo) {
		StsJmp(TttDataRecordStateForUndo(self, pNull, pData), s, Error);
	}

	//
	// Change data.
	//
	(*pData)->metrics.squares[pArgs->row][pArgs->col] = pArgs->value;

	DbgTttDataPrivSetSquare(("returns stsOK"))
	return stsOK;

Error:
	DbgTttDataPrivSetSquare(("Error; return 0x%lx",s))
	return s;
} /* TttDataPrivSetSquare */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          Message Handlers	  				   		   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
	TttDataNewDefaults
	
	Respond to msgNewDefaults.
****************************************************************************/
#define DbgTttDataNewDefaults(x) \
	TttDbgHelper("TttDataNewDefaults",tttDataDbgSet,0x10,x)

MsgHandlerWithTypes(TttDataNewDefaults, P_TTT_DATA_NEW, PP_TTT_DATA_INST)
{
	U16 row;
	U16 col;
	
	DbgTttDataNewDefaults((""))

	for (row=0; row<3; row++) {
		for (col=0; col<3; col++) {
			pArgs->tttData.metrics.squares[row][col] = tttBlank;
		}
	}
	pArgs->tttData.metrics.undoTag = 0;

	DbgTttDataNewDefaults(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;
} /* TttDataNewDefaults */


/****************************************************************************
	TttDataInit
	
	Initialize instance data of new object.

	Note: clsmgr has already initialized instance data to zeros.
****************************************************************************/
#define DbgTttDataInit(x) \
	TttDbgHelper("TttDataInit",tttDataDbgSet,0x20,x)

MsgHandlerWithTypes(TttDataInit, P_TTT_DATA_NEW, PP_TTT_DATA_INST)
{
	P_TTT_DATA_INST	pInst;
	STATUS			s;

	DbgTttDataInit((""))

	//
	// Initialize for error recovery.
	//
	pInst = pNull;

	//
	// Allocate, initialize, and record instance data.
	//
	StsJmp(OSHeapBlockAlloc(osProcessHeapId, SizeOf(*pInst), &pInst), \
			s, Error);
	pInst->metrics = pArgs->tttData.metrics;
	ObjectWrite(self, ctx, &pInst);

	DbgTttDataInit(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	if (pInst) {
		OSHeapBlockFree(pInst);
	}
	DbgTttDataInit(("Error; returns 0x%lx",s))
	return s;
} /* TttDataInit */


/****************************************************************************
	TttDataFree
	
	Respond to msgFree.

	Note:  Always return stsOK, even if a problem occurs.  This is
	(1) because there's nothing useful to do if a problem occurs anyhow
	and (2) because the ancestor is called after this function if and
	only if stsOK is returned, and it's important that the ancestor
	get called.
****************************************************************************/
#define DbgTttDataFree(x) \
	TttDbgHelper("TttDataFree",tttDataDbgSet,0x40,x)

MsgHandlerWithTypes(TttDataFree, P_ARGS, PP_TTT_DATA_INST)
{
	STATUS	s;

	DbgTttDataFree((""))

	StsJmp(OSHeapBlockFree(*pData), s, Error);

	DbgTttDataFree(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	DbgTttDataFree(("Error; return 0x%lx",s))
	return s;
} /* TttDataFree */


/****************************************************************************
	TttDataSave
	
	Save self to a file.
****************************************************************************/
#define DbgTttDataSave(x) \
	TttDbgHelper("TttDataSave",tttDataDbgSet,0x80,x)

MsgHandlerWithTypes(TttDataSave, P_OBJ_SAVE, PP_TTT_DATA_INST)
{
	TTT_DATA_FILED_0	filed;
	STATUS			 	s;

	DbgTttDataSave((""))

	StsJmp(TttUtilWriteVersion(pArgs->file, CURRENT_VERSION), s, Error);
	TttDataFiledData0FromInstData(*pData, &filed);
	StsJmp(TttUtilWrite(pArgs->file, SizeOf(filed), &filed), s, Error);

	DbgTttDataSave(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	DbgTttDataSave(("Error; return 0x%lx",s))
	return s;
} /* TttDataSave */


/****************************************************************************
	TttDataRestore
	
	Restore self from a file.

	Note: clsmgr has already initialized instance data to zeros.
****************************************************************************/
#define DbgTttDataRestore(x) \
	TttDbgHelper("TttDataRestore",tttDataDbgSet,0x100,x)

MsgHandlerWithTypes(TttDataRestore, P_OBJ_RESTORE, PP_TTT_DATA_INST)
{
	P_TTT_DATA_INST		pInst;
	TTT_DATA_FILED_0	filed;
	TTT_VERSION			version;
	STATUS 				s;

	DbgTttDataRestore((""))

	//
	// Initialize for error recovery.
	//
	pInst = pNull;

	//
	// Read version, then read filed data.  (Currently there's only
	// only one legitimate file format, so no checking of the version
	// need be done.)
	//
	// The allocate instance data and convert filed data.
	//
	StsJmp(TttUtilReadVersion(pArgs->file, MIN_VERSION, MAX_VERSION, \
			&version), s, Error);
	StsJmp(TttUtilRead(pArgs->file, SizeOf(filed), &filed), s, Error);
	StsJmp(OSHeapBlockAlloc(osProcessHeapId, SizeOf(*pInst), &pInst), \
			s, Error);
	TttDataInstDataFromFiledData0(&filed, pInst);

	ObjectWrite(self, ctx, &pInst);
	DbgTttDataRestore(("returns stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	if (pInst) {
		OSHeapBlockFree(pInst);
	}
	DbgTttDataRestore(("Error; returns 0x%lx",s))
	return s;
} /* TttDataRestore */


/****************************************************************************
	TttDataDump
	
	Respond to msgDump.
****************************************************************************/

#ifdef DEBUG

MsgHandlerWithTypes(TttDataDump, P_ARGS, PP_TTT_DATA_INST)
{
	Debugf("TttDataDump: [%s %s %s] [%s %s %s] [%s %s %s]",
			TttUtilStrForSquareValue((*pData)->metrics.squares[0][0]),
			TttUtilStrForSquareValue((*pData)->metrics.squares[0][1]),
			TttUtilStrForSquareValue((*pData)->metrics.squares[0][2]),
			TttUtilStrForSquareValue((*pData)->metrics.squares[1][0]),
			TttUtilStrForSquareValue((*pData)->metrics.squares[1][1]),
			TttUtilStrForSquareValue((*pData)->metrics.squares[1][2]),
			TttUtilStrForSquareValue((*pData)->metrics.squares[2][0]),
			TttUtilStrForSquareValue((*pData)->metrics.squares[2][1]),
			TttUtilStrForSquareValue((*pData)->metrics.squares[2][2]));
	return stsOK;
	MsgHandlerParametersNoWarning;
} /* TttDataDump */

#endif // DEBUG


/****************************************************************************
	TttDataGetMetrics
****************************************************************************/
#define DbgTttDataGetMetrics(x) \
	TttDbgHelper("TttDataGetMetrics",tttDataDbgSet,0x200,x)

MsgHandlerWithTypes(TttDataGetMetrics, P_TTT_DATA_METRICS, PP_TTT_DATA_INST)
{
	DbgTttDataGetMetrics((""))

	*pArgs = (*pData)->metrics;

	DbgTttDataGetMetrics(("returns stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;
} /* TttDataGetMetrics */


/****************************************************************************
	TttDataSetMetrics
****************************************************************************/
#define DbgTttDataSetMetrics(x) \
	TttDbgHelper("TttDataSetMetrics",tttDataDbgSet,0x400,x)

MsgHandlerWithTypes(TttDataSetMetrics, P_TTT_DATA_METRICS, PP_TTT_DATA_INST)
{
	BOOLEAN	transactionOpen;
	STATUS	s;

	DbgTttDataSetMetrics((""))

	// Steps:
	//		* Initialize for error recovery.
	//		* Begin the undo transaction.
	//		* Change the data and record undo information.
	//		* End the undo transaction.
	//		* Notify observers.
	//
	transactionOpen = false;
	ObjCallJmp(msgUndoBegin, theUndoManager, (P_ARGS)pArgs->undoTag, s, Error);
	transactionOpen = true;
	StsJmp(TttDataPrivSetMetrics(self, pArgs, pData, true), s, Error);
	ObjCallJmp(msgUndoEnd, theUndoManager, pNull, s, Error);
	transactionOpen = false;
	StsJmp(TttDataNotifyObservers(self, pNull), s, Error);

	DbgTttDataSetMetrics(("returns stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	if (transactionOpen) {
		ObjCallJmp(msgUndoEnd, theUndoManager, pNull, s, Error);
				//
				// FIXME: This should abort, not end, the transaction.
				// The abort functionality should be available in M4.8.
				//
	}
	DbgTttDataSetMetrics(("Error; return 0x%lx",s))
	return s;
} /* TttDataSetMetrics */


/****************************************************************************
	TttDataSetSquare

	Handles both msgTttDataSetMetrics and msgTttDataSetSquare
****************************************************************************/
#define DbgTttDataSetSquare(x) \
	TttDbgHelper("TttDataSetSquare",tttDataDbgSet,0x800,x)

MsgHandlerWithTypes(TttDataSetSquare, P_TTT_DATA_SET_SQUARE, PP_TTT_DATA_INST)
{
	TTT_DATA_CHANGED	changed;
	BOOLEAN				transactionOpen;
	STATUS				s;

	DbgTttDataSetSquare(("row=%ld col=%ld value=%s", \
			(U32)(pArgs->row), (U32)(pArgs->col), \
			TttUtilStrForSquareValue(pArgs->value)))

	// Steps:
	//		* Initialize for error recovery.
	//		* Begin the undo transaction.
	//		* Change the data and record undo information.
	//		* End the undo transaction.
	//		* Notify observers.
	//
	transactionOpen = false;
	ObjCallJmp(msgUndoBegin, theUndoManager, pNull, s, Error);
	transactionOpen = true;
	StsJmp(TttDataPrivSetSquare(self, pArgs, pData, true), s, Error);
	ObjCallJmp(msgUndoEnd, theUndoManager, pNull, s, Error);
	transactionOpen = false;
	changed.row = pArgs->row;
	changed.col = pArgs->col;
	StsJmp(TttDataNotifyObservers(self, &changed), s, Error);

	DbgTttDataSetSquare(("returns stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	if (transactionOpen) {
		ObjCallJmp(msgUndoEnd, theUndoManager, pNull, s, Error);
				//
				// FIXME: This should abort, not end, the transaction.
				// The abort functionality should be available in M4.8.
				//
	}
	DbgTttDataSetSquare(("Error; return 0x%lx",s))
	return s;
} /* TttDataSetSquare */


/****************************************************************************
	TttDataRead
	
	Handles msgTttDataRead
****************************************************************************/
#define DbgTttDataRead(x) \
	TttDbgHelper("TttDataRead",tttDataDbgSet,0x1000,x)

#define N_BYTES 9

MsgHandlerWithTypes(TttDataRead, P_TTT_DATA_READ, PP_TTT_DATA_INST)
{
	STREAM_READ_WRITE	read;
	U8					buf[N_BYTES+1];
	U16					row;
	U16					col;
	STATUS				s;

	DbgTttDataRead((""))

	//
	// Read in the 9 bytes that must be present.  If there are fewer
	// than 9 bytes, treat the attempt to read as a failure.
	//
	read.numBytes = SizeOf(buf) - 1;
	read.pBuf = buf;
	ObjCallJmp(msgStreamRead, pArgs->fileHandle, &read, s, Error);
	buf[read.count] = '\0';
	DbgTttDataRead(("read.count=%ld buf=<%s>", (U32)read.count,buf))

	//
	// Now convert the buffer contents to reasonable square values.
	//
	for (row=0; row<3; row++) {
		for (col=0; col<3; col++) {
			int ch;
			ch = buf[(row*3)+col];
			if ((ch == 'X') OR (ch == 'x')) {
				(*pData)->metrics.squares[row][col] = tttX;
			} else if ((ch == 'O') OR (ch == 'o')) {
				(*pData)->metrics.squares[row][col] = tttO;
			} else {
				(*pData)->metrics.squares[row][col] = tttBlank;
			}
		}
	}

	StsJmp(TttDataNotifyObservers(self, pNull), s, Error);

	pArgs->successful = true;
	DbgTttDataRead(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	DbgTttDataRead(("Error; return 0x%lx",s))
	return s;
} /* TttDataRead */


/****************************************************************************
	TttDataUndoItem
****************************************************************************/
#define DbgTttDataUndoItem(x) \
	TttDbgHelper("TttDataUndoItem",tttDataDbgSet,0x2000,x)

MsgHandlerWithTypes(TttDataUndoItem, P_UNDO_ITEM, PP_TTT_DATA_INST)
{
	STATUS	s;

	DbgTttDataUndoItem((""))

	if (pArgs->subclass != clsTttData) {
		DbgTttDataUndoItem(("not clsTttData; give to ancestor"))
		return ObjectCallAncestorCtx(ctx);
	}

	StsJmp(TttDataPrivSetMetrics(self, pArgs->pData, pData, false),
			s, Error);

	StsJmp(TttDataNotifyObservers(self, pNull), s, Error);

	DbgTttDataUndoItem(("returns stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	DbgTttDataUndoItem(("Error; return 0x%lx",s))
	return s;
} /* TttDataUndoItem */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          Installation	  							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
	ClsTttDataInit
	
	Install the class.
****************************************************************************/

STATUS PASCAL
ClsTttDataInit (void)
{
	CLASS_NEW		new;
	STATUS			s;

	ObjCallJmp(msgNewDefaults, clsClass, &new, s, Error);
	new.object.uid			= clsTttData;
	new.object.key			= 0;
	new.cls.pMsg			= clsTttDataTable;
	new.cls.ancestor		= clsObject;
	new.cls.size			= SizeOf(P_TTT_DATA_INST);
	new.cls.newArgsSize	= SizeOf(TTT_DATA_NEW);
	ObjCallJmp(msgNew, clsClass, &new, s, Error);

	return stsOK;

Error:
	return s;
} /* ClsTttDataInit */

