/****************************************************************************
 File: vsearch.h

 (C) Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.22  $
   $Author:   twiegman  $
	 $Date:   24 Feb 1992 17:06:16  $


 This file contains the API for clsVolSearch.

 clsVolSearch inherits from clsObject.
 Provides file system ui support, including formatting & duplicating disks.
 theVolSearcher is the only instance of clsVolSearch.

 The categories of functionality provided by theVolSearcher are:

 - Reformatting/duplicating a volume:

 These are sent from the disk viewer when a user selects the format or
 duplicate volume items from the volume menu.  The user is lead thru a
 series of system notes to get the information and for disk swapping.

 - Searching for a volume (because it doesn't exist or is write protected):

 This is sent from the file system when a file system request internally
 returns a stsFSVolDisconnected or stsFSVolReadOnly.
****************************************************************************/
#ifndef VSEARCH_INCLUDED
#define VSEARCH_INCLUDED


/* Include file dependencies */


#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef OSTYPES_INCLUDED
#include <ostypes.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef FS_INCLUDED
#include <fs.h>
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 These defines and enums define the text for the notes displayed by the
 volSearcher.  The resources are stored in the system resource file.
*/

/* Defines */

/* Resource ids */
#define vsResUIStrings			MakeTag (clsVolSearch, 1)


/* Types */

/* Resource string numbers */
Enum16(VS_STRING_IDS) {
	vsFindVolumeStrsBase		= 0,
	vsFindGenVolumeStr			= 0,
	vsFindDiskVolumeStr			= 1,
	vsFindRemoteVolumeStr		= 2,

	vsWriteProtectedVolumeStr	= 3,

	vsCancelButtonStr			= 4,
	vsContinueButtonStr			= 5,

	vsPercentDoneStr			= 6,

	vsFmtNoticeStr				= 7,
	vsFmtChooseSizeStr			= 8,
	vsFmtWarningStr				= 9,
	vsFmtAskForNameStr			= 10,
	vsFmtBlankNameErrStr		= 11,
	vsFmtBadCharErrStr			= 12,
	vsFmtInProgressStr			= 13,

	vsDupInProgressStr			= 14,
	vsDupInsertSrcDiskStr		= 15,
	vsDupInsertDstDiskStr		= 16,
	vsDupWriteProtectedStr		= 17,
	vsDupReadingStr				= 18,
	vsDupWritingStr				= 19,

	vsFormattingMediaStr		= 20,
};


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgVSFormatVolume			takes P_VS_FORMAT_VOLUME, returns STATUS
	Reformats an existing volume.

 The volumeRootDir must be the actual root of the volume to be formatted and
 there cannot be any other handles open on the volume or an error will be
 returned.  pVolumeName will be the initial name when the user is asked to
 provide a name or will be the name if the user is not asked to provide a
 name (controlled by the withName flag).  The warning message can be
 controlled with the noWarning flag.  And the choose a size interaction can
 be controlled with the maxSize flag.

 Return Value
	stsRequestNotSupported:	The volume does not support formatting.
*/

typedef struct VOL_FORMAT_VOLUME {
	OBJECT	volumeRootDir;					// Root dir of volume to format
	CHAR	pVolumeName[nameBufLength];		// Suggested or actual name
	U16		reserved:13,					// Reserved
			noWarning:1,					// Do not warn about dangers
			maxSize:1,						// Format to maximum possible size
			withName:1;						// Name forced to be pVolumeName
	U32		reserved1;
	U32		reserved2;
} VOL_FORMAT_VOLUME, * P_VOL_FORMAT_VOLUME;

#define msgVSFormatVolume				MakeMsg(clsVolSearch, 5)


/****************************************************************************
 msgVSDuplicateVolume		takes dir/file handle of a volume, returns STATUS
	Copy an existing volume from one floppy disk to another floppy disk.

 Return Value
	stsRequestNotSupported:	The volume does not support duplicating.
*/
#define msgVSDuplicateVolume			MakeMsg(clsVolSearch, 6)


/****************************************************************************
 msgVSFormatMedia			takes block device object, returns STATUS
	Formats unformatted media that does belong to any volume.

 This message is sent by theBlockDeviceManager when it receives a block
 device reset all and in the process discovers unformatted media on a device.
*/
#define msgVSFormatMedia				MakeMsg(clsVolSearch, 7)


/****************************************************************************
 msgVSUpdateVolumes			takes BOOLEAN, returns STATUS
	Requests theVolSearcher to update all volumes.

 This message requests the volSearcher to ask all volume classes to update
 their list of volumes.  This may result in volumes being installed, removed,
 connected or disconnected.  Interested parties should become observers of
 theFileSystem and look for msgFSVolChanged (see fs.h).  The argument passed
 should be true to update all volumes.

 This message can only be sent via ObjectSendXXX.
*/
#define msgVSUpdateVolumes				MakeMsg(clsVolSearch, 8)


/****************************************************************************
 msgVSFormatCompleteNotify	takes BOOLEAN, returns STATUS
	Notifies observers of theVolSearcher that a format has completed.

 The argument passed to the observer indicates whether the format was
 successful or not.  False would be returned if there was an error or
 if the format was cancelled.
*/
#define msgVSFormatCompleteNotify		MakeMsg(clsVolSearch, 20)


/****************************************************************************
 msgVSNameVolume			takes root dir handle of volume, returns STATUS
	Prompts user to name an unlabelled volume and adds new name.

 This message is used by volumes that have discovered unlabeled volumes.
 This message can only be sent via ObjectPostXXX.
*/
#define msgVSNameVolume				MakeMsg(clsVolSearch, 9)

//REFGEN BEGINIGNORE

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Private Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Typical usage of searching for a volume:

 Set fields of volSearchInfo.
 ObjSendUpdateWarn msgVSSearchForVolume
 This will identify the note as active vs passive
 (i.e. does it have buttons that user will respond to
  or does the volume in question have to be polled)
 If volume request because of write protected, eject volume first.
 ObjPostAsyncWarn  msgVSSearchNoteShow
 while (not found) {
 	If active note ObjSendUpdateWarn msgVSSearchNoteResponse
	If user cancelled note, done.
	If user confirmed note or must poll ObjSendU32 msgVSUpdateVolumes
	Check to see if volume exists and is connected
		(via msgFSGetInstalledVolumes, msgFSGetVolMetrics)
	If volume request satisfied, done
	Eject disks to allow for insertion of volume requested
		(via msgFSGetInstalledVolumes, msgFSEjectMedia)
	Delay a few seconds (typically 2)
 }
 ObjSendUpdateWarn msgVSSearchComplete
*/


/****************************************************************************
 msgNewDefaults				takes P_VOL_SEARCH_NEW, returns STATUS
	category: class message
	Initializes the VOL_SEARCH_NEW structure to default values.

 This is used in the creation of the well known object theVolSearcher.
 NOTE: Clients cannot create another instance of clsVolSearch.
*/


/****************************************************************************
 msgNew						takes P_VOL_SEARCH_NEW, returns STATUS
	category: class message
	Creates a volSearch object.
	
 This is used to create the well known object theVolSearcher.
 NOTE: Clients cannot create another instance of clsVolSearch.
*/

#define vsNewFields		\
	objectNewFields

typedef struct VOL_SEARCH_NEW {
	vsNewFields
} VOL_SEARCH_NEW, *P_VOL_SEARCH_NEW;


/****************************************************************************
 msgVSSearchForVolume		takes P_VOL_SEARCH_INFO, returns STATUS
	Notifies theVolSearcher that a volume search should begin.

 The user may be asked to insert a "disk" or remove write protection from a
 disk depending on the vsNeedWritableVolume flag.  Both volType and volClass
 are used as filters in searching for a volume.	 pVolName contains the name
 of the volume to be requested.  The out value postBootSearch can either be
 true, indicating that a real system modal note was displayed (with cancel
 and continue buttons) or false, indicating that a simplified "pre-boot" note
 (without any buttons) was displayed.  If only a preboot note was displayed
 then the caller must poll for the volume being inserted.

 Return Value
	stsFailed:	VolSearcher is busy searching, formatting or duplicating.
*/
#define msgVSSearchForVolume			MakeMsg(clsVolSearch, 0)

Enum16(VS_SEARCH_FLAGS) {
	vsNeedWritableVolume	= flag0,
	vsSearchFlagsDefault	= 0
};

typedef struct VOL_SEARCH_INFO {
	VS_SEARCH_FLAGS		flags;						// Find vs write protected
	FS_VOL_TYPE			volType;					// Type of vol to find
	CLASS				volClass;					// Class of vol to find
	U32					spare1;						// For future use
	U32					spare2;						// For future use
	U32					spare3;						// For future use
	U16					spare4;						// For future use
	BOOLEAN				postBootSearch;				// Out: System booted
	U8					pVolName [nameBufLength];	// Name of vol to find
} VOL_SEARCH_INFO, *P_VOL_SEARCH_INFO;


/****************************************************************************
 msgVSSearchNoteShow		takes void, returns STATUS
	Displays the 'insert volume: xxx' note.

 This message can only be sent via ObjectPostAsync!

 If the vsNeedWritableVolume flag was set in volSearchInfo.flags
 when msgVSSearchForVolume was sent to theVolSearcher, the user
 will be prompted with a message that the volume is write protected,
 remove write protection and reinsert.
*/
#define msgVSSearchNoteShow				MakeMsg(clsVolSearch, 1)


/****************************************************************************
 msgVSSearchNoteResponse	takes P_MESSAGE, returns STATUS
	Checks to see if the note has been responded to and the result.

 This message requests the volSearcher to check to see if the user has
 responded to the search note.  If the note is still up then the returned
 message will be msgNull.  If the user has responded to the note, then
 the returned message will be msgVSSearchComplete (the user wishes to
 continue) or msgVSSearchCancel (the search is being terminated).

 This message can only be sent via ObjectSendXXX.
*/
#define msgVSSearchNoteResponse			MakeMsg(clsVolSearch, 2)


/****************************************************************************
 msgVSSearchComplete		takes void, returns STATUS
	Notifies theVolSearcher that the volume search has ended.

 This message cleans up the system note asking for the volume.
*/
#define msgVSSearchComplete				MakeMsg(clsVolSearch, 3)


/****************************************************************************
 msgVSSearchCancel			takes void, returns STATUS
	Sent by cancel button when search is cancelled.
*/
#define msgVSSearchCancel				MakeMsg(clsVolSearch, 4)


/****************************************************************************
 msgVSSearchPeriodicCheck	takes void, returns STATUS
 	Self sent by theVolSearcher every few seconds so background volume
	updates can take place.  These volume updates detect volume insertion
	and removal and also give the file system a periodic ping.
*/
#define msgVSSearchPeriodicCheck		MakeMsg(clsVolSearch, 40)


/****************************************************************************
 msgVSFormatFirstChoice - msgVSFormatLastChoice:
	These are returned when the user chooses a format capacity button.
*/
#define vsFormatChoiceBase				51
#define msgVSFormatFirstChoice			MakeMsg(clsVolSearch, 51)
#define msgVSFormatLastChoice			MakeMsg(clsVolSearch, 54)


/****************************************************************************
 msgVSFormatBegin, msgVSFormatContinue, msgVSFormatEnd:
	These are used to control the processing of a format of a volume.
*/
#define msgVSFormatBegin				MakeMsg(clsVolSearch, 58)
#define msgVSFormatContinue				MakeMsg(clsVolSearch, 59)
#define msgVSFormatEnd					MakeMsg(clsVolSearch, 60)


/****************************************************************************
 msgVSDuplicateContinue, msgVSDuplicateCheck, msgVSDuplicateEnd:
	These are used to control the processing of the duplication of a volume.
*/
#define msgVSDuplicateContinue			MakeMsg(clsVolSearch, 65)
#define msgVSDuplicateCheck				MakeMsg(clsVolSearch, 66)
#define msgVSDuplicateEnd				MakeMsg(clsVolSearch, 67)

//REFGEN ENDIGNORE

#endif	// VSEARCH_INCLUDED
