/****************************************************************************
 File: volgodir.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.21  $
   $Author:   twiegman  $
	 $Date:   24 Feb 1992 17:05:44  $

 This file contains declarations for the common part of godir volumes.
 Examples of these include clsVolDisk and clsVolTOPS.

 Information in this file is useful if you are trying to understand the
 format of PenPoint.dir files or if you are writing an installable volume.
****************************************************************************/
#ifndef VOLGODIR_INCLUDED
#define VOLGODIR_INCLUDED


/* Include file dependencies for this include file */

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef OS_INCLUDED
#include <os.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef FS_INCLUDED
#include <fs.h>
#endif

#ifndef VOL_INCLUDED
#include <vol.h>
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Defines */

/* GO directory related defines */

#define goNameIndex				0
#define goDirSearchFromFirst	0L
#define goDirHeaderBufSize		112	// Min space for 3 max names plus some.


/* Types */

/* General types */

/* Enumerated constants for searching for particular directory entries */
Enum16(GO_DIR_FINDTYPE) {
	gdFindEmpty			= 0,
	gdFindNextName		= 1,
	gdFindNativeName	= 2,
	gdFindGoDirName		= 3
};

/*
 Note that this can also be treated as an array of U32, using the tag part
 of the associated fsAttr as the index into the array, except flags and
 unused together form a special case of a U32!!!
*/
typedef struct VOLGODIR_CMN_ATTRS {
	FS_NODE_FLAGS		flags;
	U16					unused;			// Was sequence
	FS_DATE_TIME		dateCreated;
	FS_DATE_TIME		dateModified;
	FS_FILE_SIZE		fileSize;
} VOLGODIR_CMN_ATTRS, *P_VOLGODIR_CMN_ATTRS;

/* GO directory related types */

/*
 Each directory entry is identified as either erased (e) or full (f).
*/
Enum16(GO_DIR_ENTRY_TYPES) {
	goDirUnusedEntry	= 'e',
	goDirNodeEntry		= 'f'
};

typedef struct GO_DIR_USER_ATTR {
	FS_ATTR_LABEL		label;			// file system attribute label.
	U16					size;			// size of value field.
	U8					value;			// a U32, string or var length attr.
} GO_DIR_USER_ATTR, *P_GO_DIR_USER_ATTR;

typedef struct GO_DIR_ENTRY_HEADER {
	U8					type;			// 'e': erased or 'f' for file/dir.
	U16					size;			// Actual size on disk is modulo 32.
} GO_DIR_ENTRY_HEADER, *P_GO_DIR_ENTRY_HEADER;

/*
 Go name is located at goDirEntry.buf, always the first entry.
 The define goNameIndex can be used to index to the name.
 It is important that the size of GO_DIR_ENTRY is modulo 32.
*/
typedef struct GO_DIR_ENTRY {
	GO_DIR_ENTRY_HEADER	hdr;
	U16					numUserAttrs;	// Number of user attributes.
	U8					nativeNameIndex;// Offset to native file name.
	U8					rsrvdForLater;	// UNUSED SPARE.
	U8					userAttrsIndex;	// Offset to first user attr.
	FS_NODE_FLAGS		flags;
	U16					rsrvdForLater2;	// WAS SEQUENCE
	FS_DATE_TIME		dateCreated;
	U8					buf [goDirHeaderBufSize];	// Min space for names.
} GO_DIR_ENTRY, *P_GO_DIR_ENTRY, **PP_GO_DIR_ENTRY;


/* VNode types */

/* VNode related type declarations */
Enum16(VOLGODIR_VNODE_FLAGS) {
	gdfPenPointDir		= flag1,	// This is a PenPoint.Dir file
	gdfRootDir			= flag2,
	gdfNodeCorrupt		= flag3,
	gdfNodeModified		= flag4,
	gdfHasGoDirParent	= flag5,
	gdfHasGoDirSister	= flag6,
	gdfNoGoDirSister	= flag7
};

typedef struct VOLGODIR_VNODE_COMMON {
	U16						refCount;
	U16						numUserAttrs;
	U32						goDirPos;
	VOLGODIR_VNODE_FLAGS	flags;
	VOLGODIR_CMN_ATTRS		attrs;
} VOLGODIR_VNODE_COMMON;

typedef struct VOLGODIR_VNODE {
	struct VOLGODIR_VNODE  *pNext;
	VOLGODIR_VNODE_COMMON	cmn;
} VOLGODIR_VNODE, *P_VOLGODIR_VNODE, **PP_VOLGODIR_VNODE;

/* Penpoint dir cache */

typedef struct GO_DIR_CACHE {
	U32					size;			// How much of data is valid?
	U32					base;			// Position in penpoint dir.
	P_VOLGODIR_VNODE	owner;			// Cache for which dir.
	U8					buffer [512];	// Fixed size buffer.
} GO_DIR_CACHE, *P_GO_DIR_CACHE;

/* VolInfo types */

/* This is the instance data for a GO dir volume object */
typedef struct VOLGODIR_INFO {
	// Common volume info...
	struct VOLGODIR_INFO   *pNext;
	FS_VOL_HEADER			hdr;
	VOL_COMMON				cmn;

	// Pointer to the low level volumes routines...
	struct VOLGODIR_RTNS   *pRtns;

	// Head of the vnode chain...
	P_VOLGODIR_VNODE		pFirstVNode;

	// Buffer used by the GO DIR volume part - does not need to be inited...
	GO_DIR_ENTRY			goDirEntry;

	// GO DIR buffer & info...
	GO_DIR_CACHE			goDirCache;

	// Beyond this point each volume will have their own info...
	//	.
	//	.
	//	.

} VOLGODIR_INFO, *P_VOLGODIR_INFO;


/* Exported routine that returns pointer GoDirShell entrypoint table */

P_VOL_RTNS EXPORTED GoDirShellEntrypoint (void);


/* Typedefs for functions supported by each godir lower level volume */

/****************************************************************************
 LVStatus	returns STATUS
	Has a volume check for readiness.

 Possible return status are stsOK, stsFSVolDisconnected, other errors.
 If status is okay, should indicate if volume has changed.
*/
typedef STATUS FunctionPtr(P_LVOL_STATUS) (
	P_VOLGODIR_INFO		pVolInfo,
	P_BOOLEAN			pChanged		// In/Out: Has volume changed?
);
#define LVStatus(pVolInfo, pChanged) \
	((pVolInfo)->pRtns->pLVolStatus) \
		(pVolInfo, pChanged)

/****************************************************************************
 LVSetVolName	returns STATUS
	Requests for a volume to set/change its volume name.
*/
typedef STATUS FunctionPtr(P_LVOL_SET_VOL_NAME) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_STRING			pName			// Vol name
);
#define LVSetVolName(pVolInfo, pName) \
	((pVolInfo)->pRtns->pLVolSetVolName) \
		(pVolInfo, pName)

/****************************************************************************
 LVUpdateInfo	returns STATUS
	Requests for a volume to update its user accessable volume info.
*/
typedef STATUS FunctionPtr(P_LVOL_UPDATE_INFO) (
	P_VOLGODIR_INFO		pVolInfo		// Vol Info
);
#define LVUpdateInfo(pVolInfo) \
	((pVolInfo)->pRtns->pLVolUpdateInfo) \
		(pVolInfo)

/****************************************************************************
 LVSpecificMsg	returns STATUS
	Passes a volume specific message down to a volume.
*/
typedef STATUS FunctionPtr(P_LVOL_SPECIFIC_MSG) (
	P_VOLGODIR_INFO		pVolInfo,
	P_VOLGODIR_VNODE	pVNode,			// Handle of vnode
	MESSAGE				msg,			// Message
	P_UNKNOWN			pArgs			// In/Out: Arguments for message
);
#define LVSpecificMsg(pVolInfo, pVNode, msg, pArgs) \
	((pVolInfo)->pRtns->pLVolSpecificMsg) \
		(pVolInfo, pVNode, msg, pArgs)


/****************************************************************************
 LVNGet	returns STATUS
	Gets a vnode given pVolInfo, dirVNode and name of node in the directory.
*/
typedef STATUS FunctionPtr(P_LVNODE_GET) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pDirVNode,		// VNode of parent directory
	P_STRING			pFileName,		// Name of file node
	P_UNKNOWN			pVolSpecific,	// Vol specific info
	PP_VOLGODIR_VNODE	ppVNode			// Out: Returned vnode handle
);
#define LVNGet(pVolInfo, pDirVNode, pFileName, pVolSpecific, ppVNode) \
	((pVolInfo)->pRtns->pLVNodeGet) \
		(pVolInfo, pDirVNode, pFileName, pVolSpecific, ppVNode)

/****************************************************************************
 LVNGetAndOpenParent	returns STATUS
	Gets a vnode's parent given pVolInfo and a vnode and open it.
*/
typedef STATUS FunctionPtr(P_LVNODE_GET_OPEN_PARENT) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode,			// VNode to get parent of
	PP_VOLGODIR_VNODE	ppDirVNode,		// Out: VNode handle of parent
	P_BOOLEAN			pComplete		// Out: Did the vnode already exist?
);
#define LVNGetAndOpenParent(pVolInfo, pVNode, ppDirVNode, pComplete) \
	((pVolInfo)->pRtns->pLVNodeGetAndOpenParent) \
		(pVolInfo, pVNode, ppDirVNode, pComplete)

/****************************************************************************
 LVNGetAndOpenByDirId	returns STATUS
	Gets a dir vnode given pVolInfo and the directory's dirID.

 Note: pDirVNode could be null.  If it isn't then it can be used.
*/
typedef STATUS FunctionPtr(P_LVNODE_GET_OPEN_BY_DIR_ID) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pDirVNode,		// VNode of parent of dir
	U32					dirId,			// Dir ID of vnode to get & open
	PP_VOLGODIR_VNODE	ppDirVNode,		// Out: Returned vnode handle of dir
	P_BOOLEAN			pComplete		// Out: Did the vnode already exist?
);
#define LVNGetAndOpenByDirId(pVolInfo, pDirVNode, dirId, ppDirVNode, pComplete) \
	((pVolInfo)->pRtns->pLVNodeGetAndOpenByDirId) \
		(pVolInfo, pDirVNode, dirId, ppDirVNode, pComplete)

/****************************************************************************
 LVNRelease	returns STATUS
	Releases a vnode.
*/
typedef STATUS FunctionPtr(P_LVNODE_RELEASE) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode			// VNode to release
);
#define LVNRelease(pVolInfo, pVNode) \
	((pVolInfo)->pRtns->pLVNodeRelease) \
		(pVolInfo, pVNode)

/****************************************************************************
 LVNOpen	returns STATUS
	Opens a vnode.
*/
typedef STATUS FunctionPtr(P_LVNODE_OPEN) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode,			// VNode to open
	P_STRING			pName,			// Name of node
	VNODE_ACCESS		access			// R/W, exclusivity, etc.
);
#define LVNOpen(pVolInfo, pVNode, pName, access) \
	((pVolInfo)->pRtns->pLVNodeOpen) \
		(pVolInfo, pVNode, pName, access)

/****************************************************************************
 LVNClose	returns STATUS
	Closes a vnode.
*/
typedef STATUS FunctionPtr(P_LVNODE_CLOSE) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode			// VNode to close
);
#define LVNClose(pVolInfo, pVNode) \
	((pVolInfo)->pRtns->pLVNodeClose) \
		(pVolInfo, pVNode)


/****************************************************************************
 LVNCreate	returns STATUS
	Creates a file or directory within the directory given.
*/
typedef STATUS FunctionPtr(P_LVNODE_CREATE) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pDirVNode,		// Directory where new node belongs
	P_STRING			pName,			// Name of new file/dir
	FS_NODE_FLAGS		fileType		// Create a dir or a file
);
#define LVNCreate(pVolInfo, pDirVNode, pName, fileType) \
	((pVolInfo)->pRtns->pLVNodeCreate) \
		(pVolInfo, pDirVNode, pName, fileType)

/****************************************************************************
 LVNDelete	returns STATUS
	Deletes a file system node; either a dir or a file node.
*/
typedef STATUS FunctionPtr(P_LVNODE_DELETE) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode,			// VNode to release
	BOOLEAN				visible			// At root of hierarchical delete?
);
#define LVNDelete(pVolInfo, pVNode, visible) \
	((pVolInfo)->pRtns->pLVNodeDelete) \
		(pVolInfo, pVNode, visible)

/****************************************************************************
 LVNMove	returns STATUS
	Moves a file or directory to a directory w/ the new (old) name.
*/
typedef STATUS FunctionPtr(P_LVNODE_MOVE) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pSrcDirVNode,	// Dir of source node
	P_VOLGODIR_VNODE	pSrcVNode,		// Source node
	P_VOLGODIR_VNODE	pDstDirVNode,	// Dir of dest
	P_STRING			pDstName		// Name to give the dest node
);
#define LVNMove(pVolInfo, pSrcDirVNode, pSrcVNode, pDstDirVNode, pDstName) \
	((pVolInfo)->pRtns->pLVNodeMove) \
		(pVolInfo, pSrcDirVNode, pSrcVNode, pDstDirVNode, pDstName)

/****************************************************************************
 LVNReadDir	returns STATUS
	Returns the next entry from the specified directory.
*/
typedef STATUS FunctionPtr(P_LVNODE_READ_DIR) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pDirVNode,		// Directory to read from
	P_U32				pDirPos,		// In/Out: Current position
	P_STRING			pName			// Out: Name of the node
);
#define LVNReadDir(pVolInfo, pDirVNode, pDirPos, pName) \
	((pVolInfo)->pRtns->pLVNodeReadDir) \
		(pVolInfo, pDirVNode, pDirPos, pName)

/****************************************************************************
 LVNDirPosDeleteAdjust	returns STATUS
	Makes any necessary adjustment to the dirPos after a node has been deleted.
*/
typedef STATUS FunctionPtr(P_LVNODE_DIR_POS_DEL_ADJUST) (
	P_VOLGODIR_INFO		pVolInfo,
	P_VOLGODIR_VNODE	dirVNode,		// Handle of directory vnode
	P_VOLGODIR_VNODE	vnode,			// Handle of deleted vnode
	P_U32				pDirPos			// In/Out: Dir pos data before delete
);
#define LVNDirPosDeleteAdjust(pVolInfo, dirVNode, vnode, pDirPos) \
	((pVolInfo)->pRtns->pLVNodeDirPosDelAdjust) \
		(pVolInfo, dirVNode, vnode, pDirPos)

/****************************************************************************
 LVNGetDirId	returns STATUS
	Returns a well known constant dir id that represents this directory.
*/
typedef STATUS FunctionPtr(P_LVNODE_GET_DIR_ID) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode,			// Return dir id of this dir vnode
	P_U32				pDirId			// In/Out: The directory's id
);
#define LVNGetDirId(pVolInfo, pVNode, pDirId) \
	((pVolInfo)->pRtns->pLVNodeGetDirId) \
		(pVolInfo, pVNode, pDirId)


/****************************************************************************
 LVNName	returns STATUS
	Returns the name a file system node.
*/
typedef STATUS FunctionPtr(P_LVNODE_NAME) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode,			// VNode to get name of
	P_STRING			pName			// In/Out: Name
);
#define LVNName(pVolInfo, pVNode, pName) \
	((pVolInfo)->pRtns->pLVNodeName) \
		(pVolInfo, pVNode, pName)

/****************************************************************************
 LVNGetNumAttrs	returns STATUS
	Returns the number of non-standard attributes, given the vnode.
*/
typedef STATUS FunctionPtr(P_LVNODE_GET_NUM_ATTRS) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode,			// VNode of node to read from
	P_U16				pNumAttrs		// Out: num of attrs to get
);
#define LVNGetNumAttrs(pVolInfo, pVNode, pNumAttrs) \
	((pVolInfo)->pRtns->pLVNodeGetNumAttrs) \
		(pVolInfo, pVNode, pNumAttrs)

/****************************************************************************
 LVNGetAttrInfo	returns STATUS
	Gets a node's attributes, given the vnode.

 Which common attributes and which arrays of the label/value/size arrays
 that need to be filled in are defined by the flgs field.  Which particular
 elements of each (label/value/size) array to be filled in is defined by the
 pWhich byte array.  If num is 0 or pWhich is null then no label/value/size
 array elements should be filled in.  If an element of pWhich is maxU8 then
 the corresponding label/value/size array element should be filled in.
 If the data is known and set then the pWhich array element should be set to
 1 after setting the values.
*/
typedef STATUS FunctionPtr(P_LVNODE_GET_ATTR_INFO) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode,			// VNode of node to read from
	U16					num,			// Num of attrs to get
	VNODE_ATTR_FLAGS	flgs,			// Get which common attrs
	P_VNODE_CMN_ATTRS	pCmn,			// Common attrs
	P_U8				pWhich,			// Which user defined attrs
	P_FS_ATTR_LABEL		pLbls,			// In/Out: attribute labels
	P_FS_ATTR_VALUE		pVals,			// In/Out: attribute values
	P_FS_ATTR_SIZE		pSizs			// In/Out: attribute sizes
);
#define LVNGetAttrInfo(pVolInfo, pVNode, num, flgs, pCmn, pWhich, pLbls, pVals, pSizs) \
	((pVolInfo)->pRtns->pLVNodeGetAttrInfo) \
		(pVolInfo, pVNode, num, flgs, pCmn, pWhich, pLbls, pVals, pSizs)

/****************************************************************************
 LVNSetAttrInfo	returns STATUS
	Sets a node's attributes, given the vnode.

 Which common attributes and which arrays of the label/value/size arrays
 that need to be stored are defined by the flgs field.  Which particular
 elements of each (label/value/size) array to be filled in is defined by the
 pWhich byte array.  If num is 0 or pWhich is null then no label/value/size
 array elements should be stored.  If an element of pWhich is maxU8 then
 the corresponding label/value/size array element should be stored.
 If the data is stored successfully then the pWhich array element should be
 set to 1.
*/
typedef STATUS FunctionPtr(P_LVNODE_SET_ATTR_INFO) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode,			// VNode of node to read from
	U16					num,			// Num of attrs to set
	VNODE_ATTR_FLAGS	flgs,			// Set which common attrs
	P_VNODE_CMN_ATTRS	pCmn,			// Common attrs
	P_U8				pWhich,			// Which user defined attrs
	P_FS_ATTR_LABEL		pLbls,			// In/Out: attribute labels
	P_FS_ATTR_VALUE		pVals,			// In/Out: attribute values
	P_FS_ATTR_SIZE		pSizs			// In/Out: attribute sizes
);
#define LVNSetAttrInfo(pVolInfo, pVNode, num, flgs, pCmn, pWhich, pLbls, pVals, pSizs) \
	((pVolInfo)->pRtns->pLVNodeSetAttrInfo) \
		(pVolInfo, pVNode, num, flgs, pCmn, pWhich, pLbls, pVals, pSizs)


/****************************************************************************
 LVNRead	returns STATUS
	Transfers n bytes from position m in a file to a buffer.
*/
typedef STATUS FunctionPtr(P_LVNODE_READ) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode,			// VNode of node to read from
	U32					filePos,		// Starting point of read
	U32					numBytes,		// Number of bytes to be read
	P_U8				pReadBuffer,	// Destination of bytes read
	P_U32				pCount			// Out: Actual number of bytes read
);
#define LVNRead(pVolInfo, pVNode, filePos, numBytes, pReadBuffer, pCount) \
	((pVolInfo)->pRtns->pLVNodeRead) \
		(pVolInfo, pVNode, filePos, numBytes, pReadBuffer, pCount)

/****************************************************************************
 LVNWrite	returns STATUS
	Transfers n bytes from a buffer to position m in a file.
*/
typedef STATUS FunctionPtr(P_LVNODE_WRITE) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode,			// VNode of node to write to
	U32					filePos,		// Starting point of the write
	U32					numBytes,		// Number of bytes to write
	P_U8				pWriteBuffer,	// Destination of bytes to write
	P_U32				pCount			// Out: Actual number of bytes written
);
#define LVNWrite(pVolInfo, pVNode, filePos, numBytes, pWriteBuffer, pCount) \
	((pVolInfo)->pRtns->pLVNodeWrite) \
		(pVolInfo, pVNode, filePos, numBytes, pWriteBuffer, pCount)

/****************************************************************************
 LVNGetSize	returns STATUS
	Returns the size of a file.
*/
typedef STATUS FunctionPtr(P_LVNODE_GET_SIZE) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode,			// VNode of node to change size of
	P_FS_FILE_SIZE		pSize			// The size of the file
);
#define LVNGetSize(pVolInfo, pVNode, pSize) \
	((pVolInfo)->pRtns->pLVNodeGetSize) \
		(pVolInfo, pVNode, pSize)

/****************************************************************************
 LVNSetSize	returns STATUS
	Adjusts the size of a file.
*/
typedef STATUS FunctionPtr(P_LVNODE_SET_SIZE) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode,			// VNode of node to change size of
	FS_FILE_SIZE		newSize			// The new size
);
#define LVNSetSize(pVolInfo, pVNode, newSize) \
	((pVolInfo)->pRtns->pLVNodeSetSize) \
		(pVolInfo, pVNode, newSize)


/****************************************************************************
 LVNFlush	returns STATUS
	Flushes a file.
*/
typedef STATUS FunctionPtr(P_LVNODE_FLUSH) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_VOLGODIR_VNODE	pVNode			// VNode of node to flush
);
#define LVNFlush(pVolInfo, pVNode) \
	((pVolInfo)->pRtns->pLVNodeFlush) \
		(pVolInfo, pVNode)

/****************************************************************************
 LVNativeName	returns BOOLEAN
	Returns the native file system form of this name.

 A return of true implies that the name was not changed (was native),
 and a return of false implies that the name was changed to be native.
*/
typedef BOOLEAN FunctionPtr(P_LV_NATIVE_NAME) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	P_STRING			pName			// In/Out: Name
);
#define LVNativeName(pVolInfo, pName) \
	((pVolInfo)->pRtns->pLVNativeName) \
		(pVolInfo, pName)

/****************************************************************************
 LDirIdGetParent	returns STATUS
	Gets the dir id of the parent of a node (also identified by dir id).
*/
typedef STATUS FunctionPtr(P_LDIRID_GET_PARENT) (
	P_VOLGODIR_INFO		pVolInfo,		// Vol Info
	U32					node,			// Node identified by dir id
	P_U32				pParent,		// In/Out: dir id of parent
	P_BOOLEAN			pParentIsRoot	// In/Out: parent is root
);
#define LDirIdGetParent(pVolInfo, node, pParent, pParentIsRoot) \
	((pVolInfo)->pRtns->pLDirIdGetParent) \
		(pVolInfo, node, pParent, pParentIsRoot)


/**** This is the definition for the table of volume routines ****/

typedef struct	VOLGODIR_RTNS {
	P_LVOL_STATUS					pLVolStatus;
	P_LVOL_SET_VOL_NAME				pLVolSetVolName;
	P_LVOL_UPDATE_INFO				pLVolUpdateInfo;
	P_LVOL_SPECIFIC_MSG				pLVolSpecificMsg;

	P_LVNODE_GET					pLVNodeGet;
	P_LVNODE_GET_OPEN_PARENT		pLVNodeGetAndOpenParent;
	P_LVNODE_GET_OPEN_BY_DIR_ID		pLVNodeGetAndOpenByDirId;
	P_LVNODE_RELEASE				pLVNodeRelease;
	P_LVNODE_OPEN					pLVNodeOpen;
	P_LVNODE_CLOSE					pLVNodeClose;

	P_LVNODE_CREATE					pLVNodeCreate;
	P_LVNODE_DELETE					pLVNodeDelete;
	P_LVNODE_MOVE					pLVNodeMove;
	P_LVNODE_READ_DIR				pLVNodeReadDir;
	P_LVNODE_DIR_POS_DEL_ADJUST		pLVNodeDirPosDelAdjust;
	P_LVNODE_GET_DIR_ID				pLVNodeGetDirId;

	P_LVNODE_NAME					pLVNodeName;
	P_LVNODE_GET_NUM_ATTRS			pLVNodeGetNumAttrs;
	P_LVNODE_GET_ATTR_INFO			pLVNodeGetAttrInfo;
	P_LVNODE_SET_ATTR_INFO			pLVNodeSetAttrInfo;

	P_LVNODE_READ					pLVNodeRead;
	P_LVNODE_WRITE					pLVNodeWrite;
	P_LVNODE_GET_SIZE				pLVNodeGetSize;
	P_LVNODE_SET_SIZE				pLVNodeSetSize;

	P_LVNODE_FLUSH					pLVNodeFlush;
	P_LV_NATIVE_NAME				pLVNativeName;
	P_LDIRID_GET_PARENT				pLDirIdGetParent;

} VOLGODIR_RTNS, *P_VOLGODIR_RTNS;


#endif // VOLGODIR_INCLUDED
