/****************************************************************************
 undo.h

 Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.24  $
   $Author:   bschwart  $
     $Date:   09 Mar 1992 12:44:40  $

 This file contains the API definition for theUndoManager.  theUndoManager
 is the wknProcessGlobal instance of clsUndo.

 clsUndo inherits from clsList.

 The functions described in this file are contained in MISC.LIB. 
****************************************************************************/

/**** Introduction ****/
/*
 theUndoManager provides a centralized facility for managing undo
 information.  theUndoManager supports undo of user interface actions.

 An undoable operation, or "undo transaction," is a collection of "undo
 items."  Typically an undoable operation is a small UI action (e.g.
 deleting some text).
 
 When the user issues an "Undo" command the most recent undo transaction
 will be undone.  A typical scenario goes something like this:

    -:	In response to some user interface action, a message handler begins
        an undo transaction with msgUndoBegin and then sends messages which
        manipulate the application's data.

    -:	As the data manipulation routines do their work, they add undo items
        to the undo transaction via msgUndoAddItem.

    -:	When the user interface handler regains control, the transaction is
        closed with msgUndoEnd.

    -:	At some later date, the transaction might be undone.  theUndoManager
        undoes a transaction by sending msgUndoItem to each item in the
        transaction (in the reverse order in which they were added).

    -:	If the transaction is not undone, but instead falls off the end of
        the undo transaction queue, then the transaction is freed.  (A
        transaction is also freed if the application is terminated.) 
        theUndoManager frees a transaction by sending msgUndoFreeItemData
        to each item in the transaction.  (But see the comments near the
        typedef UNDO_ITEM for some circumstances under which theUndoManager
        doesn't send msgUndoFreeItemData but instead frees the item
        itself.)
*/

/**** Common Messages ****/
/*
 Typical application code will send the following messages to
 theUndoManager:

    -:	msgUndoBegin
    -:	msgUndoEnd
	-:	msgUndoAddItem

 Typical application code will receive the following messages from
 theUndoManager:
    -:	msgUndoItem
    -:	msgUndoFreeItemData

 See the individual descriptions of each of these messages for more
 information.
*/

/**** Debugging Flags ****/
/*
 Undo's debugging flag set is 'U.'  Defined flags are:

	0001:  Show messages sent to theUndoManager.
	0002:  Show clsUndo initialization.
	0004:  Show msgUndoAddItem.
	0008:  Show undoing a undo transaction.
	0010:  Show creating a undo transaction.
	0020:  Show destroying an undo transaction.
*/

/**** The Current Transaction ****/
/*
 At any time, there is at most one current undo transaction open.
 The current undo transaction includes:

	-:	a unique id of type UNDO_ID

    -:	the OS_TASK_ID of the task that issued the msgUndoBegin that began
        the transaction

    -:	a nesting count which is the number of msgUndoBegin's minus the
        number of msgUndoEnd's.  (See the section "Nesting of msgUndoBegin
        and msgUndoEnd.")

    -:	a heap with local scope from which clients can allocate space for
        undo information

    -:	a list of undo items added to the transaction so far.
*/

/**** The Undo Queue ****/
/*
 theUndoManager maintains a queue of undo transactions.  By default
 theUndoManager has a queue length of 2, but an application can set the
 limit by sending msgUndoLimit to theUndoManager.
												
 Your code should not depend on any particular queue size.
*/

/**** Nesting of msgUndoBegin and msgUndoEnd ****/
/*
 In response to msgUndoBegin, theUndoManager opens a new transaction if
 there is no open transaction;  otherwise it simply increments a "nesting
 count."  The nesting count is decremented when theUndoManager receives
 msgUndoEnd.  When the count becomes zero, the transaction is closed. 

 This allows you to write code that doesn't know whether it there is an
 open transaction or not.  If the code wants to record undo information, it
 can simply send a msgUndoBegin / msgUndoEnd pair.  If there was no open
 transaction, the result is that one will be created.  And if there is one
 open, then the code's items will be added to that one.

 It is vital that every msgUndoBegin have a matching msgUndoEnd!

 To guard against erroneous code never terminating the current transaction,
 and thus having that transaction slowly consume all of system memory,
 there is a bounds on the depth of nesting permitted.  (This bounds is
 approximately 1000.)  If the bounds is exceeded, the open transaction is
 automatically closed.
*/

/**** Memory Management ****/
/*
 Each undo item records the information necessary to undo and/or free
 itself.

 Often this information has to be remembered in allocated memory or objects
 that must be freed once the item can no longer be undone.  For instance,
 an undoable operation might involve deleting an object.  However, you
 probably don't want to destroy the object until you're sure that the
 operation can't be undone.  But eventually that object has to be
 destroyed.

 Normally theUndoManager will send msgUndoFreeItemData to the object stored
 in each UNDO_ITEM.  The handler should respond by freeing any resources
 associated with the item.  Typically those resources are pointed to by
 item.pData.

 But there are five ways in which you and theUndoManager can cooperate so
 that theUndoManager can free the resources for you.

    -:	If ufDataIsHeapNode is set in item.flags, then item.pData must
        point to a heap block.  theUndoManager will free item.pData by
        calling OSHeapBlockFree(item.pData).

    -:	If ufDataInUndoHeap is set in item.flags, then item.pData must
        point to heap block allocated from the current transaction's heap. 
        theUndoManager will free item.pdata when it destroys the
        transactions's heap.

    -:	If ufDataIsObject is set in item.flags, then item.pData must be
        an object UID.  theUndoManager will free item.pdata by calling
        ObjectSend(msgDestroy, item.pData, ...).  (See the section "Freeing
        Undone Items" for one reason NOT to use this variation.)

    -:	If ufDataIsSimple is set in item.flags, then item.pData is
        treated as a 32 bit value.  There is no need for theUndoManager to
        do anything to free item.pData.

    -:	If none of the above flags is set in item.flags, and
        item.dataSize is non-zero, then when the item is added to the
        transaction (with msgUndoAddItem) theUndoManager copies
        item.dataSize bytes from item.pData into a block allocated from the
        current transaction's heap.  theUndoManager then frees item.pData
        when it destroys the transactions's heap.
*/

/**** Freeing Undone Items ****/
/*
 Even an item that has been undone will be freed.  It might be
 automatically freed by theUndoManager, as described in the section on
 Memory Management, or it might be freed by sending msgUndoFreeItemData to
 item.object.

 Often freeing an item's data is done the same way regardless of whether
 the item has been undone or not.  But there are cases where the difference
 is very important.  Here's an example.  Assume that the undoable operation
 includes deleting an object.  If the operation is undone, then the object
 is "put back" into the application.  

 If the item IS undone, then the object should NOT be destroyed when the
 item is freed.  But if the operation IS NOT undone, then the object should
 be destroyed when the object is destroyed.

 For items that need to free the item's data differently in these two
 cases, the fact that the item has been undone should be recorded in the
 item when msgUndoItem is received.  Then the code responding to
 msgUndoFreeItemData can check this recorded value.  (One convenient place
 to record this value is in the item's ufClient flags.)
*/

/**** Adding Items When No Transaction is Open ****/
/*
 When theUndoManager is undoing a transaction, there is no current open
 transaction.  But, as described in the typical scenario above, data
 manipulation routines will attempt to add items anyhow.  Therefore it is
 CRITICAL that your code check the value returned from msgUndoAddItem and
 handle it properly.

 There are several ways to do this, but here's one convenient approach. 
 (This approach works ONLY if you DON'T use any of theUndoManager's memory
 management functionality.)

 If you're not using the memory management facilities of theUndoManager,
 then you're most likely allocating memory to hold the client data part of 
 an undo item.  That memory has been allocated before calling
 msgUndoAddItem and must be freed if the msgUndoAddItem fails. 
 Conveniently, an item's client data can be freed by sending
 msgUndoFreeItemData to the object stored in item.object.

 Simply define a utility routine that attempts to add an item, and which
 frees the item if adding fails.  Then always use that routine to add
 items.  The routine will look something like:

//{
	if (ObjectCall(msgUndoAddItem, theUndoManager, pItem) < stsOK) {
		return ObjCallWarn(msgUndoFreeItemData, pItem->object, pItem);
	} else {
		return stsOK;
	}
//}
*/

/**** Subclass Issues ****/
/*
 A class and any number of its ancestors may contribute items to an undo
 transaction.

 Thus, every msgUndoFreeItemData handler should first check that 
 item.subclass is the expected value.  If it isn't, the message should be
 passed onto the ancestor.  So a msgUndoFreeItemData handler should look
 something like:
//{
	MsgHandlerWithTypes(RTItemUndoFreeItemData, P_UNDO_ITEM, PP_DATA)
	{
		if (pArgs->subclass != clsRTItem) {
			return ObjectCallAncestorCtx(ctx);
		} else {
			...
		}
	}
//}
*/

/**** Flushing the Undo Queue ****/
/*
 There may be "points of no return" in an application's execution beyond
 which undoing previous operations is impossible or non-sensical.  (For
 instance, it may not be possible to undo operations if the application's
 data files are saved via msgAppSave.)

 You should flush the queue when one of these "points of no return" is
 encountered.  The queue can be flushed by performing the following three
 steps: (1) get the current undo limit via msgUndoGetMetrics, (2) send
 msgUndoLimit with a pArgs of 0 (which actually flushes the queue), and (3)
 send msgUndoLimit, but this time with the limited returned by the previous
 call to msgUndoGetMetrics.
*/

/**** Aborting a Transaction ****/
/*
 Sometimes it is necessary to abort an operation part way through.  (For
 instance, the user might not confirm the operation.)  If this happens, you
 should abort the then the undo transaction with msgUndoAbort.  See the
 comments on msgUndoAbort for more information.
*/

#ifndef UNDO_INCLUDED
#define UNDO_INCLUDED

#ifndef	LIST_INCLUDED
#include <list.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  						Types and Constants							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef STATUS	UNDO_ID;				// A transaction's id.

#define stsUndoAbortingTransaction MakeStatus(clsUndo, 1)

#define stsUndoDataFreed		   MakeWarning(clsUndo, 1)

#define	undoStateNil		0
#define	undoStateBegun		flag0
#define	undoStateUndoing	flag1
#define	undoStateRedoing	flag2		// Not implemented
#define	undoStateAborting	flag3

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Exported Functions                                                     *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

STATUS PASCAL 
InitClsUndo(void);

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Message Arguments		                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** UNDO_ITEM ****/
typedef struct UNDO_ITEM {
	OBJECT				object;		// In:  object that undoes/frees item
	OBJECT				subclass;	// In:	See "Subclass Issues" section
	U16					flags;		// In:	See "Memory Management" section
	P_UNKNOWN			pData;		// In:	See "Memory Management" section
	SIZEOF				dataSize;	// In:	See "Memory Management" section
} UNDO_ITEM, *P_UNDO_ITEM;

/*
 * The following flags are used in the flags field of an UNDO_ITEM.
*/
#define ufReserved        (0xff00)
#define ufClient          (flag0|flag1|flag2|flag3)
#define ufDataType        (flag4|flag5|flag6|flag7|ufReserved)
#define ufDataInUndoHeap  flag4
#define ufDataIsHeapNode  flag5
#define ufDataIsObject    (flag5|flag4)
#define ufDataIsSimple    (flag6|flag4)


/**** Other Message Arguments ****/
typedef struct UNDO_METRICS {
	UNDO_ID				id;					// In:Out	Nil => get current
	OS_HEAP_ID			heapId;				// Out
	U16					state;				// Out
	U16					transactionCount;	// Out
	U16					itemCount;			// Out
	U32					limit;				// Out
	U32					resId;				// Out
	U32					info;				// Reserved
} UNDO_METRICS, *P_UNDO_METRICS;			   		   


#define	undoNewFields		\
	listNewFields			\
	UNDO_NEW_ONLY	undo;


typedef struct UNDO_NEW_ONLY {
	U32				reserved;				// Reserved for expansion
	P_UNKNOWN		pReserved;				// Reserved for expansion
	U32				maxTransactions;
} UNDO_NEW_ONLY, *P_UNDO_NEW_ONLY;


typedef struct UNDO_NEW {
	undoNewFields
} UNDO_NEW, *P_UNDO_NEW;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  							Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Next: 11; recycled: none */

/****************************************************************************
 msgUndoAbort		takes pNull, returns STATUS
	Aborts the current undo transaction.

 The current transaction is flagged as being aborted.  Until the
 transaction is closed, any attempted msgUndoAddItem, msgUndoBegin, and
 msgUndoEnd (including the one that finally closes the transaction) will
 fail and return stsUndoAbortingTransaction.  Once the msgUndoEnd that
 closes the transaction is received, any remaining undo items in the
 aborted transaction are freed.
*/
#define	msgUndoAbort			MakeMsg(clsUndo, 10)


/****************************************************************************
 msgUndoAddItem		takes P_UNDO_ITEM, returns STATUS
	Adds a new item to the current undo transaction if and only if it 
	is still open.

 theUndoManager returns stsFailed if an open transaction does not exist. 
 Any other error status indicates that there are not enough resources
 available to add the item.
*/
#define	msgUndoAddItem			MakeMsg(clsUndo, 0)


/****************************************************************************
 msgUndoBegin		takes RES_ID, returns STATUS or UNDO_ID
	Creates a new undo transaction if there is no current transaction,
	or increments the nesting count if there is a current transaction.

 See the "Nesting of msgUndoBegin and msgUndoEnd" section for information
 about how to send this message.

 Return Value
	stsFailed:		Nesting limit exceeded.
	stsOK:			Returned status is actually the id of the new (or
					currently open) transaction.  Cast it to type UNDO_ID.

 The RES_ID for a transaction is determined by the first msgUndoBegin with
 a non-null argument.  The string identified by the RES_ID of the current
 undo transaction is used as the string for the "Undo" menu item.  The
 RES_ID should specify a resGrpTK string resource list.  (This is analogous
 to the quick help strings that are found in the resGrpQHelp string
 resource list.)
*/
#define	msgUndoBegin			MakeMsg(clsUndo, 1)


/****************************************************************************
 msgUndoCurrent		takes pNull, returns STATUS
	Undoes the most recent undo transaction.

 msgUndoCurrent undoes the most recent transaction.  If a transaction is
 currently open the transaction is closed first, and then undone.

 It is unusual for a client to send this message.  The only real reason for
 sending this message is if some piece of client code is implementing an
 alternative UI mechanism to invoke the undo mechanism.
*/
#define	msgUndoCurrent			MakeMsg(clsUndo, 2)


/****************************************************************************
 msgUndoEnd		takes pNull, returns STATUS
	Decrements the nesting count of (and thus may end) the current
	transaction.

 See the "Nesting of msgUndoBegin and msgUndoEnd" section for information
 about how to send this message.

 Return Value
	stsFailed:	 No open transaction.
*/
#define	msgUndoEnd				MakeMsg(clsUndo, 3)


/****************************************************************************
 msgUndoGetMetrics		takes P_UNDO_METRICS, returns STATUS
	Passes back the metrics associated with an undo transaction.

 Only an pArgs->id of Nil(UNDO_ID), representing the current undo
 transaction, is supported.  

 Return Value
	stsFailed:	The specified transaction does not exist or there is 
				in sufficient memory available to manipulate it.
*/
#define	msgUndoGetMetrics		MakeMsg(clsUndo, 4)


/****************************************************************************
 msgUndoLimit		takes U32, returns STATUS
	Sets the maximum number of remembered undo transactions.

 The default undo limit is 2.  If your application wants to support a
 longer undo history, send msgUndoLimit to theUndoManager with the desired
 limit.

 If there are more transactions in the queue than the new limit, the extra
 transactions will be freed.  Setting the limit to 0 flushes all 
 transactions and effectively disables undo until the limit is set to some
 non-zero value.

 msgUndoLimit always returns stsOK.
*/
#define	msgUndoLimit		MakeMsg(clsUndo, 8)


/****************************************************************************
 msgUndoRedo		takes pNull, returns STATUS
	Not implemented.

 Not implemented.  Do not send this message.
*/
#define	msgUndoRedo				MakeMsg(clsUndo, 5)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  						Client Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgUndoItem		takes P_UNDO_ITEM, returns STATUS
	Sent to pArgs->object to have the item undone.

 Note that the item will be freed in a separate step later.
*/
#define	msgUndoItem				MakeMsg(clsUndo, 6)


/****************************************************************************
 msgUndoFreeItemData		takes P_UNDO_ITEM, returns STATUS
	Sent to pArgs->object to have pArgs->pData freed.

 See the "Memory Management," "Subclass Issues" and "Freeing Undone Items"
 sections for information about how to respond to this message.
*/
#define	msgUndoFreeItemData		MakeMsg(clsUndo, 7)

#endif	// ifndef UNDO_INCLUDED

