/****************************************************************************
 txtdata.h

 Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.224  $
   $Author:   msapsfor  $
     $Date:   24 Feb 1992 11:48:00  $

 This file contains the API definition for clsText.

 clsText inherits from clsObject.

 clsText is the Data Object for the Text subsystem.  These objects hold
 characters, their attributes and embedded objects.

 The functions described in this file are contained in TEXT.LIB. 
****************************************************************************/

/**** Road Map ****/
/*
 Clients manipulating the character contents of the textData might use:
	-:	msgTextGet
	-:	msgTextGetBuffer
	-:	msgTextModify
*/
/*
 Clients manipulating the attributes stored in textData might use:
	-:	msgTextChangeAttrs
	-:	msgTextClearAttrs
	-:	msgTextGetAttrs
	-:	msgTextInitAttrs
	-:	msgTextPrintAttrs
	-:	TextInitCharAttrs()
	-:	TextInitCharMask()
	-:	TextInitParaAttrs()
	-:	TextInitParaMask()
	-:	TextDeleteMany()
	-:	TextInsertOne()
*/
/*
 Clients manipulating a textData's embedded objects might use:
	-:	msgTextEmbedObject
	-:	msgTextExtractObject
	-:	msgTextEnumEmbeddedObjects

 Clients needing to work with words, sentences or paragraphs might use:
	-:	msgTextSpan
	-:	msgTextSpanType
*/
/*
 Clients needing to import or export text might use:
	-:	msgTextRead
	-:	msgTextWrite

 Clients observing a textData might want to handle:
	-:	msgTextAffected
	-:	msgTextReplaced
*/


/**** Characters and Encodings ****/
/*
 Text data objects hold bytes representing characters using the encoding
 specified in tencode.h.  In PenPoint 1.0, this encoding is derived from
 the IBM-PC's code page 850, and uses one byte per character.  There are
 characters representing line, paragraph, and page breaks.

 Characters are indexed starting from zero.
*/


/**** Formatting Information ****/
/*
 Text data objects also hold "formatting" or "attribute" information.  
 The types of attributes stored are:
    -:  character attributes such as font face, size and weight
    -:  paragraph attributes such margins, first line offset, first line
        offset
	-:	tab attributes for a paragraph
    -:  embedded object info (specifically the embedded object's uid)
    -:  link termination (specifically the destination information for
        marks)

 Attributes "tile" ranges of characters.  In other words, no character can
 have two different sets of character attributes associated with it,
 although it can have both character and paragraph attributes.  This tiling
 is enforced by the textData.  
 
 Any character that does not have explicit character or paragraph
 attributes takes on the "default" character or paragraph attributes of the
 data object.  There are messages to inspect, enumerate, and modify all the
 attributes, including the defaults.
*/


/**** Relation to UI Classes ****/
/*
 A textData only provides storage for characters and attributes.  It does
 not provide any user interface (UI).  The UI is provided by an instance of
 clsTextView.  
 
 To assist the class providing the UI, the textData provides notifications
 whenever either the characters or the attributes are modified.
*/


/**** Implementation Note ****/
/*
 clsText is actually composed of three layers of classes.  Clients need not
 be concerned by these layers, and should not rely on their existence as
 they may disappear in future releases.

 clsTextBlock (usually referred to as clsText) is a descendant of 
 clsTextMarkStore, which in turn is a descedant of clsTextChar.
*/

#ifndef TXTDATA_INCLUDED
#define TXTDATA_INCLUDED $Revision:   1.224  $ 

#ifndef						CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef						BYTARRAY_INCLUDED
#include <bytarray.h>							// For BYTE_INDEX
#endif

#ifndef						GEO_INCLUDED
#include <geo.h>								// Required by sysfont.h
#endif

#ifndef						SYSFONT_INCLUDED
#include <sysfont.h>							// For SYSDC_FONT_ATTR
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  					Types and Constants: Atoms						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Atoms are used as parameters to many of textData messages.  All valid
 * atoms are defined below.
*/
typedef U16 ATOM;

#define	atomChar		((ATOM) 1)
#define	atomWord		((ATOM) 2)
#define	atomLine		((ATOM) 3)
#define	atomSentence	((ATOM) 4)
#define	atomPara		((ATOM) 5)
#define	atomDivision	((ATOM) 6)
#define	atomDoc			((ATOM) 7)
#define	atomMisc		((ATOM) 8)
#define	atomEmbedded	((ATOM) 9)
#define	atomParaTabs	((ATOM)10)
#define	atomLink		((ATOM)11)
#define atomWSDelimit	((ATOM)12)
#define atomClient1		((ATOM)28)
#define atomClient2		((ATOM)29)
#define atomClient3		((ATOM)30)
#define atomClient4		((ATOM)31)

#define minValidAtom	atomChar
#define maxValidAtom	atomClient4


/****************************************************************************
 AtomGetName	returns STATUS
	Passes back a pointer to the string value of the atom.

 Most clients and subclasses do not use this function.  It is occasionally
 useful for debugging.

 Return Value
	stsBadParam:	atom is out of the range of valid atoms
	stsOK:			atom is within the valid range.  *ppString may still
					be NULL if the atom falls into one of the gaps.
*/
STATUS EXPORTED
AtomGetName(
	ATOM		atom,
	PP_STRING	ppString);


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  				Types and Constants:  Character Indices				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Character Indices ****/
typedef	U32					TEXT_INDEX;
typedef	TEXT_INDEX *		P_TEXT_INDEX;
#define maxTEXT_INDEX		maxU32

/*
 * Some messages and functions which take a TEXT_INDEX as a parameter may use
 * special values to achieve certain effects.  Each message and function
 * description indicates which special values can be used.
*/
#define lpoTEXT_INDEX	(maxTEXT_INDEX-maxU16)
#define lastTEXT_INDEX	(lpoTEXT_INDEX-1)
#define infTEXT_INDEX	(maxTEXT_INDEX-1)
#define mInfTEXT_INDEX	maxTEXT_INDEX


/*
 * "Magic" value for msgTextChangeAttrs, msgTextGetAttrs and
 * msgTextInitAttrs.
*/
#define textDefaultAttrs	infTEXT_INDEX


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  			Types and Constants:  Character Attributes				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * The prefixes "TA_" and "ta" indicate that an identifier is related to
 * "text attributes."
*/

/* Use these in the alignBase field of a TA_CHAR_ATTRS. */
typedef enum {							// Must fit in 2 bits
	taNormalLineBase	= 0,
//REFGEN BEGINIGNORE
	taSuperScriptBase	= 1,			// Not Implemented
	taSubScriptBase 	= 2,			// Not Implemented
	taSpareAlignBase	= 3				// Reserved
//REFGEN ENDIGNORE
} TA_ALIGN_BASE;


/* Character Attributes */
typedef struct TA_CHAR_ATTRS {
	U16				size;				// Font size in twips.  Not all 
										// values are available -- some are 
										// rounded down.  Max of 160*20 twips.
	U16				tacSpare	: 8,	// Reserved.
					highlight	: 1,
					smallCaps	: 1,
					upperCase	: 1,
					strikeout	: 1,
					underlines	: 2,	// As defined in sysfont.h. Must be 
										// 0, 1, or 2.
					alignBase	: 2;	// Use a TA_ALIGN_BASE value. Only
										// taNormalLineBase is implemented.
	SYSDC_FONT_SPEC	font;
} TA_CHAR_ATTRS, *P_TA_CHAR_ATTRS;


/* 
 * Character Attributes Mask.
 *
 * The highlight and encoding fields contain extra bits.  These bits are
 * automatically zero-ed by assigning a legitimate values to the field.
*/
typedef struct {		   			// Must fit in 32 bits
	U16			tacSpare	: 8,	// Reserved. Should be set to 0.
				highlight	: 2,	// true or false (and 1 spare bit)
				size		: 1,
				smallCaps	: 1,
				upperCase	: 1,
				strikeout	: 1,
				underlines	: 1,
				alignBase	: 1;
	U16			id			: 1,	// mask bit for attrs.font.id
				group		: 1,	// mask bit for attrs.font.attr.group
				weight		: 1,	// mask bit for attrs.font.attr.weight
				aspect		: 1,	// mask bit for attrs.font.attr.aspect
				italic		: 1,	// mask bit for attrs.font.attr.italic
				monospaced	: 1,	// mask bit for attrs.font.attr.monospaced
				encoding	: 10;	// mask bit for attrs.font.attr.encoding
									// true or false (and 9 spare bits)
} TA_CHAR_MASK, *P_TA_CHAR_MASK;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  				Types and Constants:  Tab Attributes				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Each paragraph can have up to TA_MAX_TABS tab stops.  A paragraph without
 its own explicit tab stops "inherits" the document's "default" tab stops.

 Paragraphs that desire uniformly spaced tab stops can compactly define the
 stops by setting at least two explicit stops and then setting repeatAtEnd
 to true.  This has the effect of defining an unlimited number of implicit
 stops, each of which follows the prior stop by the distance between the
 last two explicit stops.

 NOTE:  Even though each tab store has a type and leader, only the type
 taTabLeft and the leader taLeadSpace are implemented.
*/

typedef enum {						// Must fit in 2 bits
	taTabLeft		= 0,
	taTabCenter		= 1,			// Not Implemented
	taTabRight		= 2,			// Not Implemented
	taTabDecimal	= 3				// Not Implemented
} TA_TAB_TYPE;


typedef enum {						// Must fit in 2 bits
	taLeadSpace		= 0,
	taLeadDot		= 1,			// Not Implemented
	taLeadDash		= 2,			// Not Implemented
	taLeadUnderline	= 3				// Not Implemented
} TA_TAB_LEADER;


/*
 * Tab Stop.
 *
 * The type and leader fields contain extra bits.  These bits are
 * automatically zero-ed by assigning a legitimate values to the field.
*/
typedef struct TA_TAB_STOP {
	U16			x;					// In twips
	U8			type;				// TA_TAB_TYPE (and 6 spare bits)
	U8			leader;				// TA_TAB_LEADER (and 6 spare bits)
} TA_TAB_STOP, *P_TA_TAB_STOP;


/*
 * The maximum number of tab stops for a paragraph.
*/
#define TA_MAX_TABS	31


/*
 * Tab Stops.
 *
 * The count and repeatAtEnd fields contain extra bits.  These bits are
 * automatically zero-ed by assigning a legitimate values to the field.
*/
typedef struct TA_TABS {
	U16				count		: 8,	// Number of tab stops, in the range 
										// 0..TA_MAX_TABS.  (plus 3
										// spare bits.)
					repeatAtEnd	: 8;	// true or false (and 7 spare bits)
	TA_TAB_STOP		tabs[1];			// Actually variable size array
} TA_TABS, *P_TA_TABS;


/*
 * Another representation of tab stops.
*/
typedef struct TA_MANY_TABS {
	U16				count		: 8,	// Number of tab stops, in the range 
										// 0..TA_MAX_TABS.  (plus 3
										// spare bits.)
					repeatAtEnd	: 8;	// true or false (and 7 spare bits)
	TA_TAB_STOP		tabs[TA_MAX_TABS];
} TA_MANY_TABS, *P_TA_MANY_TABS;


#define textNoTabs	((P_TA_MANY_TABS)1)		// Not Implemented



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  				Types and Constants:  Paragraph Attributes			   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Use these in the alignment field of a TA_PARA_ATTRS. */
typedef enum {							// Must fit in 2 bits
	taParaLeft		= 0,
	taParaCenter	= 1,
	taParaRight		= 2,
	taParaSpare		= 3					// Reserved
} TA_PARA_ALIGN;


/*
 * Paragraph Attributes.
 *
 * All of the fields in TA_PARA_ATTRS that are linear measurements are in
 * twips.
 *
 * The alignment and justify fields contain extra bits.  These bits are
 * automatically zero-ed by assigning a legitimate values to the field.
*/
typedef struct TA_PARA_ATTRS {
	U16			alignment	: 8,	// TA_PARA_ALIGN (and 6 spare bits)
				justify		: 8;	// 0 or 1.  (0x80 is used internally,
									// so there are 6 spare bits.)
	U16			lineHeight;			// The special value textUseMaxHeightOnLine 
									// causes the line height to be as high 
									// as the highest thing in the line. 
									// Don't use zero!
	U16			interLineHeight;
	U16			beforeSpacing;		// Adds to previous paragraphs's 
									// afterSpacing
	U16			afterSpacing;
	S16			firstLineOffset;	// Add to leftMargin to get the effective
									// left margin for the first line of the
									// paragraph.
	U16			leftMargin;
	U16			rightMargin;
} TA_PARA_ATTRS, *P_TA_PARA_ATTRS;

/* Special lineHeight value */
#define	textUseMaxHeightOnLine	maxU16


/* 
 * Paragraph Attribute Mask
 *
 * The lineHeight, interLineHeight, beforeSpacing and afterSpacing fields
 * contain extra bits.  These bits are automatically zero-ed by assigning a
 * legitimate values to the field.
*/
typedef struct {					// Must fit in 32 bits
	U16		alignment			: 1,
			justify				: 1,
			firstLineOffset		: 1,
			leftMargin			: 1,
			rightMargin			: 1,
			lineHeight			: 3,		// 0 or 1 (2 spare bits)
			interLineHeight		: 8;		// 0 or 1 (7 spare bits)
	U16		beforeSpacing		: 8,		// 0 or 1 (7 spare bits)
			afterSpacing		: 8;		// 0 or 1 (7 spare bits)
} TA_PARA_MASK, *P_TA_PARA_MASK;


//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				Types and Constants:  Document Attributes				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Obsolete.  Do not use. */
typedef struct TA_DOC_ATTRS {
	P_UNKNOWN	printer;
	OBJECT		dc;
	ATOM		selType;
	TEXT_INDEX	selFirst;
	TEXT_INDEX	selLength;
} TA_DOC_ATTRS, *P_TA_DOC_ATTRS;

//REFGEN ENDIGNORE

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  				Types and Constants:  Embedding						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef struct TEXT_EMBED_OBJECT {
	TEXT_INDEX	first;
	OBJECT		toEmbed;
	U8			clientFlags;
	U8			action;				// One of the values below (6 spare bits)
} TEXT_EMBED_OBJECT, *P_TEXT_EMBED_OBJECT;

/* Use these in the action field of a TEXT_EMBED_OBJECT. */
#define textEmbedCopy	0		// For internal use only.
#define textEmbedFree	1		// For internal use only.
#define textEmbedInsert	2
#define textEmbedMove	3		// For internal use only.


/*
 * The fields of this structure are described in the comments for
 * msgTextEnumEmbeddedObjects.
*/
typedef struct TEXT_ENUM_EMBEDDED {
	TEXT_INDEX				first;
	TEXT_INDEX				length;
	U16						flags;		// One ofthe values below
	U16						max;
	U16						count;
	P_TEXT_EMBED_OBJECT		pItems;
} TEXT_ENUM_EMBEDDED, *P_TEXT_ENUM_EMBEDDED;


/*
 * The prefix "tee" indicates that an identifier is related to 
 * "TEXT_ENUM_EMBEDDED."
*/

/* Use these in the flags field of a TEXT_ENUM_EMBEDDED. */
#define teeFloat	flag0					// Include floating embedded 
											// objects. (These will be
											// children of theRootWindow.)
#define teeInline	flag1					// Include embedded objects 
#define teeDefault	(teeFloat|teeInline)
//REFGEN BEGINIGNORE
#define teePrivate	(flag15|flag14|flag13|flag12|flag11|flag10|flag9|flag8)
//REFGEN ENDIGNORE

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  				Types and Constants: Import/Export					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * More information about the fields of this structure is in the comments for
 * for msgTextRead.
 *
 * The freeAfter and inputIsObject fields contain extra bits.  These bits are
 * automatically zero-ed by assigning a legitimate values to the field.
*/
typedef struct TEXT_READ {
	TEXT_INDEX	first;
	P_UNKNOWN	input;
	U16			embeddedAction:	2,
				freeAfter:		6,	// true or false (and 5 spare bits)
				inputIsObject:	8;	// true or false (and 7 spare bits)
	TAG			format;
} TEXT_READ, *P_TEXT_READ;


/*
 * More information about the fields of this structure is in the comments for
 * for msgTextWrite.
 *
 * The flags and outputIsObject fields contain extra bits.  These bits are
 * automatically zero-ed by assigning legitimate values to the fields.
*/
typedef struct TEXT_WRITE {
	TEXT_INDEX	first;
	TEXT_INDEX	length;
	P_UNKNOWN	output;
	U16			flags;				// One of the values below (and 13 
									// spare bits)
	TAG			format;
	U8			outputIsObject;
} TEXT_WRITE, *P_TEXT_WRITE;


/*
 * The prefix "tw" indicates that an identifier is related to "text write."
*/

/*
 * Use these in the flags field of a TEXT_WRITE.  They are described in the
 * comments for msgTextWrite.
*/
#define twExtractEmbedded	flag0
#define twTempFile			flag1
#define twForUndo			flag3


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  					Other Types and Constants						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT TEXT_DATA;

/* Resource ids */
#define textResDefaultCharAttrs		MakeWknResId(clsText, 1)
#define textResDefaultParaAttrs		MakeWknResId(clsText, 2)	// Not Impl.
#define textResDefaultParaTabs		MakeWknResId(clsText, 3)	// Not Impl.


//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  							Private									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Everything in this section is private, and for internal use only.  It is
 * subject to change without notice.
*/

typedef enum {					// Used as a SET: must fit in 8 bits
	tmFirstMoves	= flag0,
	tmLastMoves		= flag1,
	tmFreeWhenEmpty	= flag2,
	tmFreeClient	= flag3,
	tmFreeWhenPileup= flag4,
	tmFreed			= flag7
} TM_FLAGS;

typedef P_UNKNOWN TM_ID;

#define _TEXT_MARK_COMMON		\
	TM_ID		mark;			\
	U8			flags;			\
	U8			clientFlags;	\
	P_UNKNOWN	clientData;

typedef struct TEXT_MARK {
	TEXT_INDEX	first;
	TEXT_INDEX	length;
	_TEXT_MARK_COMMON
	P_UNKNOWN	clientData2;
	ATOM		tag;
} TEXT_MARK, *P_TEXT_MARK;

typedef struct TM_FREED {
	TEXT_DATA		sender;
	TEXT_MARK		tm;
} TM_FREED, *P_TM_FREED;

#define TM_ALLOCATED	TM_FREED
#define P_TM_ALLOCATED	P_TM_FREED

//REFGEN ENDIGNORE


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  						Public Functions and Macros					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Utility Functions ****/

/****************************************************************************
 TextDeleteMany	returns STATUS
	Deletes characters from a textData.

 The return values are the same as those for msgTextModify.
*/
STATUS EXPORTED
TextDeleteMany(
	const OBJECT		dataObj,
	const TEXT_INDEX	pos,			// first character to delete
	const TEXT_INDEX	length);		// number to delete


/****************************************************************************
 TextInsertOne	returns STATUS
	Inserts one character into a textData.

 The return values are the same as those for msgTextModify.
*/
STATUS EXPORTED
TextInsertOne(
	const OBJECT		dataObj,
	const TEXT_INDEX	pos,				// position at which to insert
	const CHAR			toInsert);			// character to insert


/****************************************************************************
 TextFindNextParaTab	returns STATUS
	Passes back the next tab stop to the right of the passed-in stop.

 Note that if p->repeatAtEnd is true, there are effectively an infinite
 number of tab stops.

 Return Value
	stsNoMatch:		no tabs, or this is the last tab.
*/
STATUS EXPORTED
TextFindNextParaTab(
	const P_TA_TABS			p,
	const P_TA_TAB_STOP		pTab,
	const P_U16				pIndex);


/**** Attribute and Mask Initialization Routines ****/

/****************************************************************************
 TextInitCharAttrs		returns nothing
	Initialzes a character attribute structure.

 This function reads the default character attributes from the process's
 resource list (using the resource id textResDefaultCharAttrs), or sets all
 values to 0 if the resource cannot be found.

 See Also
	msgTextChangeAttrs
*/
void EXPORTED
TextInitCharAttrs(
	P_TA_CHAR_ATTRS	p);


/****************************************************************************
 TextInitCharMask		returns nothing
	Initialzes a character attribute mask to all zeros.

 See Also
	msgTextChangeAttrs
*/
void EXPORTED
TextInitCharMask(
	P_TA_CHAR_MASK	p);


/****************************************************************************
 TextInitParaAttrs		returns nothing
	Initialzes a paragraph attribute structure to all zeros.

 See Also
	msgTextChangeAttrs
*/
void EXPORTED
TextInitParaAttrs(
	P_TA_PARA_ATTRS	p);


/****************************************************************************
 TextInitParaMask		returns nothing
	Initialzes a paragraph attribute mask to all zeros.

 See Also
	msgTextChangeAttrs
*/
void EXPORTED
TextInitParaMask(
	P_TA_PARA_MASK	p);


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  						Message Arguments							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * The prefix "TD_" indicates that an identifier is related to "text data."
 *
 * The prefix "tdm" indicates that an identifier is related to "text data
 * metrics."
*/

typedef struct TD_METRICS {
	U16			flags;					// One of the values below
	U16			spareBits;				// Reserved.
	P_UNKNOWN	spares[2];				// Reserved.
} TD_METRICS, *P_TD_METRICS;

/* Use these in the flags field of a TD_METRICS. */
#define tdmCanUndo			flag8		// if on, textData supports undo
#define tdmFileCharsOnOwn	flag1		// Not Implemented
#define tdmReadOnly			flag0		// characters cannot be modified
//REFGEN BEGINIGNORE
#define tdmReserved0		flag14		// Reserved.
#define tdmReserved1		flag15		// Reserved.
//REFGEN ENDIGNORE


/*
 * expectedSize is a hint about the expected number of characters in a 
 * textData.  An accurate hint can improve performance.
*/
typedef struct TD_NEW_ONLY {
	TD_METRICS	metrics;
	TEXT_INDEX	expectedSize;
	U16			expectedTagCount;		// Private. For internal use only.
} TD_NEW_ONLY, *P_TD_NEW_ONLY;


typedef struct TD_NEW {
	OBJECT_NEW_ONLY		object;
	TD_NEW_ONLY			text;
} TD_NEW, *P_TD_NEW;


typedef struct TEXT_BUFFER {
	TEXT_INDEX	first;				// In
	TEXT_INDEX	length;				// In
	TEXT_INDEX	bufLen;				// In
	P_CHAR		buf;				// In:Out via *buf
	TEXT_INDEX	bufUsed;			// Out
} TEXT_BUFFER, *P_TEXT_BUFFER;


typedef enum {						// Used as a SET
	tdForward	= 1,
	tdBackward	= 2
} TEXT_DIRECTION;


typedef struct TEXT_SPAN {
	TEXT_INDEX				first;			// In:Out
	TEXT_INDEX				length;			// In:Out
	ATOM					type;			// In:Out (for msgTextSpanType)
	TEXT_DIRECTION			direction;		// In
	BOOLEAN					needPrefix;		// In
	BOOLEAN					needSuffix;		// In
	U16						prefixLength;	// Out: valid if and only if 
											// needPrefix is true
	U16						suffixLength;	// Out: valid if and only if
											// needSuffix is true
	U8						firstNormal;	// Out: 0 or 1 (7 spare bits)
	U8						lastNormal;		// Out: 0 or 1 (7 spare bits)
	U32						spares[4];		// Reserved
} TEXT_SPAN, *P_TEXT_SPAN;


typedef struct TEXT_SPAN_AFFECTED {
	OBJECT		sender;
	U32			changeCount;
	TEXT_INDEX	first;
	TEXT_INDEX	length;
} TEXT_SPAN_AFFECTED, *P_TEXT_SPAN_AFFECTED;


typedef struct TEXT_REPLACED {
	TEXT_SPAN_AFFECTED	span;
	TEXT_INDEX			bytesTakenFromBuf;
} TEXT_REPLACED, *P_TEXT_REPLACED;


typedef struct TEXT_AFFECTED {
	TEXT_SPAN_AFFECTED	span;
	U16					remeasure;
	P_UNKNOWN			spare;
} TEXT_AFFECTED, *P_TEXT_AFFECTED;


typedef struct TEXT_COUNTER_CHANGED {
	OBJECT		sender;
	U32			changeCount;
	U32			oldCount;
} TEXT_COUNTER_CHANGED, *P_TEXT_COUNTER_CHANGED;


typedef struct TEXT_CHANGE_ATTRS {
	ATOM				tag;
	TEXT_INDEX			first;
	TEXT_INDEX			length;
	P_UNKNOWN			pNewMask;
	P_UNKNOWN			pNewValues;
} TEXT_CHANGE_ATTRS, *P_TEXT_CHANGE_ATTRS;


typedef struct TEXT_GET_ATTRS {
	ATOM				tag;
	TEXT_INDEX			first;
	TEXT_INDEX			length;			// Not defined.
	P_UNKNOWN			pValues;
} TEXT_GET_ATTRS, *P_TEXT_GET_ATTRS;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				  	Messages Defined by Other Classes					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNewDefaults		takes P_TD_NEW, returns STATUS
	category: class message
	Initializes the NEW struct.

 In response to this message, clsText does the following:
//{
	pNew->object.cap				|= objCapCreate;

	memset(&(pNew->text), 0, sizeof(pNew->text));
	pNew->text.expectedSize			= 5;
	pNew->text.expectedTagCount		= 5;
//}
*/


/****************************************************************************
 msgNew		takes P_TD_NEW, returns STATUS
	category: class message
	Creates a new instance of clsText.
*/


//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  							Private									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * The message space is shared between clsTxtChar, clsTxtMark and
 * clsTxtBlock.  Messages which implicitly encode a version of their
 * attribute argument data structures are "TxtAttr" messages.  Subclasses
 * that make use of these data structures need to make similar arrangements
 * in their API.  The partitioning of the message space is:
 * 		TxtBlock	[0x0 ..0x20)
 * 		TxtMark		[0x20..0x40)
 * 		TxtChar		[0x40..0x60)
 * 		TxtAttr		[0x60..0xE0)  32 messages * 4 argument versions
 * 		unassigned	[0xE0..0x100)
*/

// Define must be on multiple lines to avoid breaking gensyms.
#define TBMakeMsg(i) \
	MakeMsg(clsText, i)
												// Next:  8; recycled: none
#define TMMakeMsg(i) \
	MakeMsg(clsText, 0x20+i)
												// Next: 13; recycled: none
#define TCMakeMsg(i) \
	MakeMsg(clsText, 0x40+i)
												// Next: 12; recycled: none
#define TAMakeMsg(aVer, i) \
	MakeMsg(clsText, 0x60+(0x20*aVer)+i)
												// Next: 7; recycled: none

#define VerOfTAMsg(msg)		((MsgNum(msg)-0x60) / 0x20)
#define taVersion	0
//REFGEN ENDIGNORE


/****************************************************************************
 msgTextChangeCount		takes S32, returns S32
	Passes back (and optionally sets) the textData's changeCount.

 Each instance of clsText keeps a monotonically increasing count of the
 number of changes that have been made to it (via msgTextModify).  In
 response to this message, a textData passes back that count.  The
 counter's value is always greater than or equal to 0.

 If the value of pArgs is:
	< 0:	the counter's current value is returned and the counter is 
			unchanged.
	maxS32:	the counter is incremented by one, and the new value returned.
	>= 0:	the counter is set to pArgs, and its previous value is returned.

 In general, clients should only increment the counter, not decrement it.
*/
#define	msgTextChangeCount	TCMakeMsg(0)


/****************************************************************************
 msgTextGet		takes TEXT_INDEX, returns STATUS
	Returns the character in a textData at the specified position.

 Return Value
	stsEndOfData:	pArgs->first is too large
	>= 0:			the 8 bit character is returned as the low byte of 
					the returned STATUS;  the high 3 bytes are zero.
*/
#define	msgTextGet			TCMakeMsg(1)


/****************************************************************************
 msgTextGetBuffer		takes P_TEXT_BUFFER, returns STATUS
	Passes back a contiguous range of characters from a textData.

 Use this message to get the values of several characters at a time.  This
 message is a high-performance alternative to msgTextGet.

 If pArgs->length > pArgs->bufLen, then up to bufLen characters are placed
 into pArgs->buf.

 Upon return, pArgs->bufUsed is set to the count of characters read, even
 if there was a problem with the request.
 
 Return Value
	stsBadParam:	pArgs->length was 0 or pArgs->bufLen was 0 or
					pArgs->buf was pNull.
	stsEndOfData:	pArgs->first is too large
	< stsOK:		some other error occurred.
*/
#define	msgTextGetBuffer		TCMakeMsg(5)


/****************************************************************************
 msgTextGetMetrics		takes P_TD_METRICS, returns STATUS
	Passes back the textData's metrics.
*/
#define	msgTextGetMetrics		TCMakeMsg(2)


/****************************************************************************
 msgTextLength		takes nothing, returns TEXT_INDEX
	Returns the number of characters stored in the textData.

 Return Value
 	< stsOK:	some error occurred.
	>= stsOK:	Cast the returned value to a TEXT_INDEX;  that's the number
				of characters.
*/
#define	msgTextLength			TCMakeMsg(3)


/****************************************************************************
 msgTextModify		takes P_TEXT_BUFFER, returns STATUS.
	Modifies the characters stored in the textData.

 Use this message to insert, delete or replace characters in a textData.

 In response to this message, the textData replaces the
 characters in the range [pArgs->first .. pArgs->first+pArgs->length)
 with the characters from pArgs->buf.  
 
 If pArgs->buf is pNull, the effect is a deletion. If pArgs->length is 0,
 the effect is an insertion.  Otherwise the effect is a replacement.  If
 pArgs->first is infTEXT_INDEX, the current length minus pArgs->length is
 substituted.  If pArgs->length is maxTEXT_INDEX, strlen(pArgs->buf) is
 substituted.

 Return Value
	stsReadOnly:	request refused because object is read only.
	stsOK:			modification successful.
*/
#define	msgTextModify			TCMakeMsg(4)


/****************************************************************************
 msgTextSetMetrics		takes P_TD_METRICS, returns STATUS
	Sets a textData's metrics.
*/
#define	msgTextSetMetrics		TCMakeMsg(6)


/****************************************************************************
 msgTextSpan		takes P_TEXT_SPAN, returns STATUS.
	Determines the range corresponding to the requested span.

 A span is a consecutive range of characters that share some common trait.
 Given a position and the desired span type, this message returns the range
 of the span.  For instance, a client can use this message to ask a
 textData to find the bounds of the word containing a position.

 Actually, this message can be used to find the start of one span and the
 end of another.  If pArgs->length is 1, then the start and end of the same
 span is returned.

 If the client only needs only the beginning or the end of the span, then
 pArgs->direction should be set to the needed end.  This substantially
 improves performance.

 Using this message, a textData can find the range of the following types of
 spans:
    -:  atomWSDelimit:	passes back a white-space delimited span
    -:  atomWord:		passes back a word span using the definitions in
                        tencode.h
	-:	atomSentence:
	-:	atomPara:
	-:	atomLine:
 
 pArgs->type specifies the desired span's type.

 pArgs->direction indicates whether the span should be searched for in
 preceding characters, succeeding characters, or both.  
 
 It is often useful to know something about the characters immediately
 preceding or succeeding the span.  This information is returned if 
 pArgs->needPrefix or pArgs->needSuffix (or both) are true.  Upon return,
 pArgs->prefixLength and/or pArgs->suffixLength identifies the appropriate
 characters.  
 
 pArgs->firstNormal and pArgs->lastNormal indicate whether the
 corresponding portions of the span are normal or abnormal characters for
 the span.  For instance, for atomWord, an "a" is a normal character, but
 an "!" is abnormal.

 Return Value
	stsBadParam:	Neither the two directions in pArgs->direction was on.
*/
#define msgTextSpan			TCMakeMsg(9)


/****************************************************************************
 msgTextSpanType		takes P_TEXT_SPAN, returns STATUS.
	Determines the span type of the specified range.

 In response to this message, a textData passes back the span type
 that corresponds to the range.
 
 The same range often has several span types.  For instance, all ranges
 have the span type atomChar.  All ranges that include a complete paragraph
 also have the span types atomChar, atomWord and atomSentence.
 When the passed-in range has multiple span types, the largest span type is
 returned.
 
 The span type ordering from smallest to largest is as follows.  This is
 also the complete list of span types returned in response to this message.
	-:	atomChar
	-:	atomWord
	-:	atomSentence
	-:	atomPara
	-:	atomDoc
*/
#define msgTextSpanType		TCMakeMsg(10)
	

//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextMarkAllocate	takes P_TEXT_MARK, returns STATUS
	Private.  For internal use only.  Subject to change without notice.
*/
#define	msgTextMarkAllocate	TMMakeMsg(1)
//REFGEN ENDIGNORE


//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextMarkFree		takes TM_ID, returns STATUS
	Private.  For internal use only.  Subject to change without notice.
*/
#define	msgTextMarkFree		TMMakeMsg(2)
//REFGEN ENDIGNORE


//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextMarkFindOverlap		takes P_TEXT_MARK, returns STATUS
	Private.  For internal use only.  Subject to change without notice.
*/
#define	msgTextMarkFindOverlap		TMMakeMsg(3)
//REFGEN ENDIGNORE


//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextMarkGet		takes P_TEXT_MARK, returns STATUS
	Private.  For internal use only.  Subject to change without notice.
*/
#define	msgTextMarkGet		TMMakeMsg(4)
//REFGEN ENDIGNORE


//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextMarkMinMax		takes P_TEXT_MARK, returns STATUS
	Private.  For internal use only.  Subject to change without notice.
*/
#define	msgTextMarkMinMax	TMMakeMsg(5)
//REFGEN ENDIGNORE


//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextMarkMaxMin		takes P_TEXT_MARK, returns STATUS
	Private.  For internal use only.  Subject to change without notice.
*/
#define	msgTextMarkMaxMin	TMMakeMsg(6)
//REFGEN ENDIGNORE


//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextMarkNext		takes P_TEXT_MARK, returns STATUS
	Private.  For internal use only.  Subject to change without notice.
*/
#define	msgTextMarkNext		TMMakeMsg(7)
//REFGEN ENDIGNORE


//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextMarkUpdate		takes P_TEXT_MARK, returns STATUS
	Private.  For internal use only.  Subject to change without notice.
*/
#define	msgTextMarkUpdate	TMMakeMsg(8)
//REFGEN ENDIGNORE


//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextGetTagOps		takes P_TEXT_TAG_OPS, returns STATUS
	Private.  For internal use only.  Subject to change without notice.
*/
#define	msgTextGetTagOps	TAMakeMsg(taVersion, 0)
//REFGEN ENDIGNORE


//REFGEN BEGINIGNORE
/*
 * Private.  For internal use only.
*/
#if defined(TXTDATA_PRIVATE) || defined(TXTD_SUB_INCLUDED)
#define	msgTextAttrsForPos				TAMakeMsg(taVersion, 5)
#define msgTextDefaultsFromLocalAttrs	TAMakeMsg(taVersion, 6)
#endif
//REFGEN ENDIGNORE


/****************************************************************************
 msgTextChangeAttrs		takes P_TEXT_CHANGE_ATTRS, returns STATUS
	Changes the attributes of the specified range.

 Clients use this message to change the formatting attributes of characters
 in a textData.  They can manipulate three types of attributes:
	-:	character attributes (indicated by atomChar)
	-:	paragraph attributes (indicated by atomPara)
	-:	tab attributes (indicated by atomParaTabs)

 The pArgs type for this message is P_TEXT_CHANGE_ATTRS.  This structure
 has a tag, which must be one of the three atoms mentioned above.  The
 structure also has two P_UNKNOWN fields:  pNewMask and pNewValues.  The
 true type of these two fields depends on the value of the tag.
//{
	tag				pNewValues type		pNewMask type
	========		================	================
	atomChar		P_TA_CHAR_ATTRS		P_TA_CHAR_MASK
	atomPara		P_TA_PARA_ATTRS		P_TA_PARA_MASK
	atomParaTabs	P_TA_MANY_TABS		none; always null
//}

 The mask field allows the client to change only some of the attributes.
 If the appropriate bit in the mask if off, then the value of the attribute
 is not changed.  To simplify initializing attribute and mask structures,
 textData has a few utility messages and functions:
    msgTextInitAttrs:	The client must set the tag pArgs->first.  In
                        response to this message, a textData initializes
                        pNewValues to the values in effect at pArgs->first
                        and sets all of the bits in the mask to zero.
    TextInitCharAttrs:	reads the default character attributes from the
                        process's resource list (using the resource id
                        textResDefaultCharAttrs), or sets all values to 0 if
                        the resource cannot be found.
    TextInitCharMask:	Turns off all bits in the mask
    TextInitParaAttrs:  Sets all values to 0.
	TextInitParaMask:	Turns off all bits in the mask

 If pArgs->first is the "magic value" textDefaultAttrs, the textData's
 default attributes are modified.

 If pArgs->tag is atomPara or atomParaTabs, then the passed-in range is 
 automatically extended to complete paragraph boundaries.  (The resulting
 range is passed back in pArgs->first and pArgs->length updated.)
 
 Return Value
	stsBadParam:	Either pArgs->tag or the range was invalid.  No 
					attributes have changed.
	< stsOK:		Some other error occurred.  No attributes have changed.
*/
#define msgTextChangeAttrs			TAMakeMsg(taVersion, 1)


/****************************************************************************
 msgTextClearAttrs		takes ATOM, returns STATUS
	Clears all attributes of the specified type to the default values.

 In response to this message, a textData clears all formatting for the
 specified type. This message is "all or nothing" -- no mask or range can
 be specified.

 The attributes have not changed  the return value is < stsOK:

 Return Value
	stsBadParam:	pArgs was invalid.  No attributes have changed.
	< stsOK:		Some other error occurred.  No attributes have changed.
*/
#define msgTextClearAttrs			TBMakeMsg(5)


/****************************************************************************
 msgTextEmbedObject		takes P_TEXT_EMBED_OBJECT, returns STATUS
	Embeds an object at a specified position.

 Each embedded object is represented by a character with the encoding value
 teEmbeddedObject.  (See tencode.h.)

 In response to this message, the textData inserts the embedded object
 anchor character and "remembers" the embedded object's id.
*/
#define	msgTextEmbedObject			TBMakeMsg(2)


/****************************************************************************
 msgTextExtractObject		takes OBJECT, returns STATUS
	Extracts the specified embedded object.

 In response to this message, the textData "forgets" the specified embedded
 object.  It also deletes the associated embedded object anchor character.
 
 Nothing is done to the object itself.  In particular, the client should
 probably msgWinExtract the object.
*/
#define	msgTextExtractObject			TBMakeMsg(4)


/****************************************************************************
 msgTextGetAttrs		takes P_TEXT_GET_ATTRS, returns STATUS
	Gets the attributes of the specified type.

 Clients can retrieve the attributes of a character in the textData using
 msgTextGetAttrs.

 The client specifies the type of attributes it is interested in by filling
 in pArgs->tag.  The client must set pArgs->pValues to point to a structure
 with the "real" type of the attributes corresponding to the tag.  This
 "real" type is described in the comments for msgTextChangeAttrs.

 The client also specifies the character whose attributes the client wants
 by specifying pArgs->first.  If pArgs->first is textDefaultAttrs then the
 default attribute values are returned.

 Return Value
	stsBadParam:	pArgs->tag is not valid
	stsEndOfData:	pArgs->first is too large
	stsOK:			the attribute values have been copied into pArgs->pValues
*/
#define msgTextGetAttrs				TAMakeMsg(taVersion, 2)


/****************************************************************************
 msgTextInitAttrs		takes P_TEXT_CHANGE_ATTRS, returns STATUS
	Initialize the attributes and mask before a msgTextChangeAttrs.

 The type of attributes is specified by pArgs->tag.  pArgs->pNewValues and
 pArgs->pNewMask must be set as appropriate to an invocation of
 msgTextChangeAttrs.

 If pArgs->first is textDefaultAttrs, the default attributes are used to
 initialize pArgs->pNewValues.  Otherwise the attributes in effect at
 pArgs->first are used.  All bits of pArgs->pNewMask are set to 0.

 Return Value
	stsBadParam:	Either pArgs->tag or the range was invalid.
	< stsOK:		Some other error occurred.  No change has been made
					to the attributes and mask.

 See Also
	msgTextChangeAttrs
*/
#define msgTextInitAttrs			TAMakeMsg(taVersion, 3)


/****************************************************************************
 msgTextPrintAttrs		takes P_TEXT_CHANGE_ATTRS, returns stsOK
	Prints the values of an attribute set and a mask.

 This message takes the same parameters as msgTextChangeAttrs and the pArgs
 must be filled in the same way.  In response to this message, a textData
 prints out a useful dump of the contents of pArgs.

 Internal Use Only:  If pArgs->first is txtPrvAttrs, then pArgs->pNewValues
 must be in the internal format.

 See Also
	msgTextChangeAttrs
*/
#ifdef DEBUG
#define msgTextPrintAttrs			TAMakeMsg(taVersion, 4)
#endif


/****************************************************************************
 msgTextRead			takes P_TEXT_READ, returns STATUS
	Inserts Ascii, RTF, etc. at the specified location.

 The textData reads data and inserts the data into itself.
 
 The fields of pArgs are:
    first:			the read text is inserted into the textData starting at
                    this position.  After a successful return, pArgs->first
					is position immediately after the inserted text.
    input:			the input source.  If pArgs->inputIsObject is true,
                    this field must hold a FILE_HANDLE object.  If
					pArgs->inputIsObject is false, then this field must
					hold a P_FILE.
    embeddedAction:	Client must set this to textEmbedInsert.  (Other
                    values are for internal use only.)
    freeAfter:		If true, then pArgs->input is freed after reading
                    successfully.
    inputIsObject:	describes the type of pArgs->input.
    format:			one of the file types defined in filetype.h, or 
                    fileTypeUndefined.  If the latter, the textData object
                    attempts to deduce the form at from the contents of the
                    data found in pArgs->input.

 The textData reads pArgs->input using the functions defined in stdio.h. 
 Thus, if pArgs->inputIsObject is true, pArgs->input must be an object
 which supports the stream protocol as used by stdio.

 Return Value
	stsReadOnly:	request refused because object is read only.
	stsNoMatch:		RTF error: first character of input is not "{" or
					format version > 1 or unrecognized font name.
	stsFailed:		StdioStreamBind() or fseek() failed.
	stsBadParam:	pArgs->format is invalid.
	stsFS...:		see <fs.h>.
	stsOK:			request completed successfully; pArgs->first updated.
*/
#define msgTextRead			TBMakeMsg(0)


/****************************************************************************
 msgTextWrite			takes P_TEXT_WRITE, returns STATUS
	Outputs the specified span as one of Ascii, RTF, etc.

 The fields of pArgs are:
    first:				first character of range to be written
    length:				length of range to be written
    output:				if null, the textData creates a P_FILE and returns
                        that handle.  If non-null, then this field is either
                        an object or a P_FILE, depending on the value of
                        outputIsObject.
    flags:				described below
    format:				one of the file types defined in filetype.h.
    outputIsObject:		If output is non-null and outputIsObject is true,
                        then output is an object.  If output is non-null
                        and outputIsObject is false, then output is a
                        P_FILE.

 Possible values for the flags field of a TEXT_WRITE are:
	twExtractEmbedded:	embedded objects in the specified span are
						extracted from their parent window.
    twTempFile:			if output is null, then a temporary file is created.
                        (Developer's Note:  If you're debugging the 
                        behavior of msgTextWrite, you probably don't want
                        to turn this flag on as your file will be deleted
                        before msgTextWrite returns.)
	twForUndo:			add additional information needed for supporting
						UNDO.

 Return Value
	stsBadParam:	pArgs->format is invalid.
	stsFailed:		StdioStreamBind() failed.
	stsFS...:		see <fs.h>.
	stsOK:			request completed successfully.
*/
#define msgTextWrite			TBMakeMsg(1)


//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextAllocateLink		takes P_TEXT_MARK, returns STATUS
	Private.  For internal use only.  Subject to change without notice.
*/
#define msgTextAllocateLink		TBMakeMsg(3)
//REFGEN ENDIGNORE


//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextFindLink			takes P_TEXT_MARK, returns STATUS
	Private.  For internal use only.  Subject to change without notice.
*/
#define msgTextFindLink			TBMakeMsg(6)
//REFGEN ENDIGNORE


//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextFreeLink			takes P_UNKNOWN, returns STATUS
	Private.  For internal use only.  Subject to change without notice.
*/
#define msgTextFreeLink			TBMakeMsg(7)
//REFGEN ENDIGNORE


/****************************************************************************
 msgTextEnumEmbeddedObjects		takes P_TEXT_ENUM_EMBEDDED, returns STATUS
	Enumerates the textData's embedded objects.

 There are two ways of enumerating the embedded objects:

 1) Get all the objects in one send.  The textData allocates an array of
 TEXT_EMBED_OBJECT elements and passes it back in pArgs->pItems.  You must
 OSHeapBlockFree() the array when you are done with it.  TEXT_ENUM_EMBEDDED
 is used as follows:

    first:		position at which you want to start the enumeration.  Use 0
                to start at the beginning of the data.

    length:		length of the range you want the enumeration to include.
                Use infTEXT_INDEX to go to the end of the data.

    flags:		Usually teeDefault.  Use teeFloat to get only floating
                embedded objects.  Use teeInline to get only in-line
                embedded objects.

    max:		Pass in 0.  The object passes back the number of items in the
                allocated block

    count:		Pass in maxU16.  The object passes back the number of items
                returned (same as max).

    pItems:		Pass in pNull.  The object passes back a pointer to the
                allocated block

 2) Get the objects a few at a time.  You repeatedly send
 msgTextEnumEmbeddedObjects re-using the same TEXT_ENUM_EMBEDDED structure.
 When the message returns stsEndOfData, there are no more objects in the
 enumeration.  You should set the fields of TEXT_ENUM_EMBEDDED only before
 the first call.  For successive calls you must not modify the fields.

	first:		Same as Case 1.
	length:		Same as Case 1.
	flags:		Same as Case 1.
	max:		number of objects the pItems block can hold.
    count:		Pass in the same value as max.  textData passes back the
                number of objects returned in block.  May be less than max
                for the last chunk, and is 0 when no further objects are
                left to enumerate.
	pItems:		pointer to a block that can hold at least max objects.

 Return Value
	stsOK:				next chunk of objects has been enumerated
	stsEndOfData:		no more objects to enumerate.  Passed back count is
						be zero.  If pItems was nil and max was 0, then no
						block has been allocated.
*/
#define msgTextEnumEmbeddedObjects			TMMakeMsg(9)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Notifications							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgTextAffected		takes P_TEXT_AFFECTED, returns STATUS.
	Notifies observers that a range of text has been affected.

 This message informs observers that the attributes of the range have been
 modified.
*/
#define	msgTextAffected			MsgNoError(TCMakeMsg(7))


/****************************************************************************
 msgTextCounterChanged		takes P_TEXT_COUNTER_CHANGED, returns STATUS.
	Notifies observers that textData's changeCount has been modified.

 The changeCount is normally incremented by 1 as a result of handling
 msgTextModify.  Observers here about these changes via msgTextReplaced and
 msgTextAffected notification messages.

 However, the changeCount can change in other ways.  For instance, the
 changeCount is rolled back as part of undoing certain operations.  Also,
 clients and/or subclasses can explicitly set the changeCount via
 magTextChangeCount.

 Whenever the changeCount changes in some way OTHER than a single increment
 by 1, msgTextCounterChanged is sent to the observers to allow them to
 synchronize any caches they keep based on the changeCount.
*/
#define	msgTextCounterChanged	MsgNoError(TCMakeMsg(11))


/****************************************************************************
 msgTextReplaced		takes P_TEXT_REPLACED, returns STATUS.
	Notifies observers that a range of text was replaced via msgTextModify.
*/
#define	msgTextReplaced			MsgNoError(TCMakeMsg(8))


//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextMarkAllocated		takes P_TM_ALLOCATED, returns STATUS.
	Private.  For internal use only.  Subject to change without notice.
*/
#define	msgTextMarkAllocated	MsgNoError(TMMakeMsg(10))
//REFGEN ENDIGNORE


//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextMarkFreed		takes P_TM_FREED, returns STATUS.
	Private.  For internal use only.  Subject to change without notice.
*/
#define	msgTextMarkFreed		MsgNoError(TMMakeMsg(11))
//REFGEN ENDIGNORE


//REFGEN BEGINIGNORE
/****************************************************************************
 msgTextMarkUpdated		takes P_TM_ALLOCATED, returns STATUS.
	Private.  For internal use only.  Subject to change without notice.
*/
#define	msgTextMarkUpdated		MsgNoError(TMMakeMsg(12))
//REFGEN ENDIGNORE


//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  							Private									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#define	msgTextGetTagOps1			TAMakeMsg(0, 0)
#define	msgTextAttrsForPos1			TAMakeMsg(0, 5)
#define msgTextChangeAttrs1			TAMakeMsg(0, 1)
#define msgTextGetAttrs1			TAMakeMsg(0, 2)
#define msgTextInitAttrs1			TAMakeMsg(0, 3)
#define msgTextPrintAttrs1			TAMakeMsg(0, 4)

#ifdef TXT_ATTRS_VER1
typedef TA_CHAR_ATTRS		TA_CHAR_ATTRS1;
typedef P_TA_CHAR_ATTRS 	P_TA_CHAR_ATTRS1;
typedef TA_PARA_ATTRS		TA_PARA_ATTRS1;
typedef P_TA_PARA_ATTRS 	P_TA_PARA_ATTRS1;
typedef TA_TABS				TA_TABS1;
typedef P_TA_TABS			P_TA_TABS1;
typedef TA_MANY_TABS		TA_MANY_TABS1;
typedef P_TA_MANY_TABS		P_TA_MANY_TABS1;
#endif


STATUS GLOBAL InitClsTextData(void);
STATUS GLOBAL InitClsTextChar(void);
STATUS GLOBAL InitClsTextMark(void);
STATUS GLOBAL InitClsTextBlock(void);


/****************************************************************************
 TextCompareByteRanges	returns S16
 	Returns an encoded description of how the two ranges overlap.
*/
S16 EXPORTED
TextCompareByteRanges(
	const BYTE_INDEX	aFirst,
	const BYTE_INDEX	aLastPlusOne,
	const BYTE_INDEX	bFirst,
	const BYTE_INDEX	bLastPlusOne);


/****************************************************************************
 TextIntersectByteRanges	returns BYTE_INDEX
	Computes the intersection of two ranges.
	
 This function returns the length of the intersection and *pIFirst contains
 the first position of the intersection.

*/
BYTE_INDEX EXPORTED
TextIntersectByteRanges(
	const BYTE_INDEX	aFirst,
	const BYTE_INDEX	aLength,
	const BYTE_INDEX	bFirst,
	const BYTE_INDEX	bLength,
	P_BYTE_INDEX		pIFirst);


/****************************************************************************
 TextUnionByteRanges	returns BYTE_INDEX
	Computes the union of two ranges.
	
 This function returns the length of the union and *pUFirst contains 
 the first position of the union.
*/
BYTE_INDEX EXPORTED
TextUnionByteRanges(
	const BYTE_INDEX	aFirst,
	const BYTE_INDEX	aLength,
	const BYTE_INDEX	bFirst,
	const BYTE_INDEX	bLength,
	P_BYTE_INDEX		pUFirst);

//REFGEN ENDIGNORE


#endif	// ifndef TXTDATA_INCLUDED
