/****************************************************************************
 File: tiff.h
 
 Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.52  $
   $Author:   dlow  $
     $Date:   30 Mar 1992 20:09:40  $

 This file contains the API definition for clsTiff (Tagged Image File Format).
 clsTiff inherits from clsObject.
 clsTiff provides decoding and display of TIFF file to a window.

 clsTiff remembers a pathname to a TIFF file; the file must be in the same
 location on redisplay. TIFF objects are not windows; they take a drawing
 context to repaint.

 clsTiff provides display of the black and white grey scale formats.
 It decodes compression types for packed data (type 1);
 Group3 (FAX) horizontial encoding (types 2 and 3);
 Pack Bits run-length (type 32773). Samples per pixel are limited to
 1, 2, 4, or 8. TIFF images must be grey scale; it does not support colormap
 or direct color (RGB) images. It supports tags for photometric interpretion,
 fill order, orientation, dot size, Intel & Motorola byte order.

 Common uses of clsTiff:

 clsTiff can be the data object for a clsView object. It is used by the Fax
 Viewer in this way to display fax images.
****************************************************************************/
#ifndef TIFF_INCLUDED
#define TIFF_INCLUDED

#ifndef PICSEG_INCLUDED
#include "picseg.h"
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNewDefaults								takes P_TIFF_NEW, returns STATUS
	category: class message
	Initializes a TIFF_NEW structure to default values.

 defaults:
 tiff.pName = pNull;
 tiff.imageFlags = sysDcImageFillWindow;
 tiff.rectangle = zeros;	
*/

/****************************************************************************
 msgNew								takes P_TIFF_NEW, returns STATUS
	category: class message
	Creates a new TIFF object, and optionaly opens its associated file.

 If imageFlags has the sysDcImageFillWindow flag set, msgNew will
 pass back the size of the image in mils in the rectangle member of the
 TIFF_NEW struct.
*/

typedef struct TIFF_STYLE {
	U16	save	: 1,		// false if reading and display; true for saving
	  	spare1	: 15;
	U16	spare2	: 16;
}TIFF_STYLE, * P_TIFF_STYLE;

typedef struct {
	P_U8				pName;		 // a pointer pathname of the file 
	SYSDC_IMAGE_FLAGS	imageFlags;	 // sysDcImageXXFilter and sysDcImageFillWindow 
	RECT32				rectangle;	 // display size of the tiff image in LUC
	TIFF_STYLE			style;
	S32					spare[3];
} TIFF_NEW_ONLY, * P_TIFF_NEW_ONLY;

#define	tiffNewFields		\
	objectNewFields			\
	TIFF_NEW_ONLY	tiff;

typedef struct TIFF_NEW {
	tiffNewFields
} TIFF_NEW, *P_TIFF_NEW;

/****  Status Codes	for msgNew ****/

/*
	stsTiffNumStrips:		returned if the number of strips is bad.
	stsTiffStripByteCount:	returned if the number of strip byte
							counts does not match the image length.
	stsTiffStripOffsets:	returned if there are no strip offsets.
	stsTiffImageTooLarge:	returned if the image is too large to
							display (32000 pixels by 32000 pixels).
	stsTiffByteCountZero:	returned the a byte count is zero.
	stsTiffBadName:			returned if pName is bad or pNull.
	stsFSNodeNotFound:		returned the TIFF file is not found.
	and status errors form	OSHeapBlockAlloc() 
*/

#define  stsTiffNumStrips			MakeStatus(clsTiff,0)
#define	 stsTiffStripByteCount		MakeStatus(clsTiff,1)
#define	 stsTiffStripOffsets		MakeStatus(clsTiff,2)
#define	 stsTiffImageTooLarge		MakeStatus(clsTiff,3)
#define	 stsTiffByteCountZero		MakeStatus(clsTiff,4)
#define	 stsTiffBadFormatId	 		MakeStatus(clsTiff,5)
#define	 stsTiffBadName		 		MakeStatus(clsTiff,6)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					clsPicSeg messages used by clsTiff					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 msgPicSegPaintObject			takes P_PIC_SEG_PAINT_OBJECT, returns STATUS
	Paints the Tiff to the drawing context object provided.

 Object Call either msgWinBeginPaint or msgWinBeginRepaint before
 using this message. A clsPicSeg object will send this message to
 any Tiff object in its display list. If the rectangle in
 P_PIC_SEG_PAINT_OBJECT is all zeros then the whole window is filled with the
 image.
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					clsTiff	Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgTiffGetMetrics 			   takes P_TIFF_METRICS, returns STATUS
	Passes back the metrics of the Tiff.
*/
#define msgTiffGetMetrics						MakeMsg(clsTiff, 1)

/****************************************************************************
 msgTiffSetMetrics			   takes P_TIFF_METRICS, returns STATUS
	Sets the metrics of the Tiff.
*/
#define msgTiffSetMetrics						MakeMsg(clsTiff, 2)

/****  Orientation defines  ****/
/*
 Valid values for metrics.orientation
*/
#define tiffOrientTopLeft		1	// 1st row top; 1st column left
#define tiffOrientTopRight		2	// 1st row top; 1st column right
#define tiffOrientBottomRight	3	// 1st row bottom; 1st column right
#define tiffOrientBottomLeft	4	// 1st row bottom; 1st column left
#define tiffOrientLeftTop		5	// 1st row left; 1st column top
#define tiffOrientRightTop		6 	// 1st row right; 1st column top
#define tiffOrientRightBottom	7	// 1st row right; 1st column bottom
#define tiffOrientLeftBottom	8	// 1st row left; 1st column bottom

/****  Compression types  ****/
/*
 Valid values for metrics.compression
*/
#define tiffCompPackedData	1		// 
#define tiffCompGroup3		2		// only horiz. encoding 
#define tiffCompFax			3		// only horiz. encoding w/EOL
#define tiffCompPackBits	32773	// Mac pack bits run-length 

/****  Rational  ****/
/*
 The ratio of two longs (num / dem).
*/
typedef struct {
	U32		num;
	U32		dem;
} RATIONAL, * P_RATIONAL;

/****  Metrics  ****/
/*
 The data read from the file tags.
*/
typedef struct {
	P_U8	 pFileName;			// the path for the file

	RECT32	 rectangle;			// the display rect
								//(zero width and height fills the window)

	SYSDC_IMAGE_FLAGS	imageFlags;

	U32		 newSubfileType;	// the tiff data read from the file
	U16		 SubfileType;		// 1 the only supported value
	U32		 width;				// number of pixels in the x dimension
	U32		 length;			// number of pixels in the y dimension
	U16		 bitsPerSample;		// number of bits per sample 1, 2, 4 or 8
	U16		 compression;		// the image compression type
	U16		 photometricInterpretation;// 0 - 0 black; highest value white
			 						   // 1 - highest value black; 0 white	
	U16		 fillOrder;			// bit order of image bytes
			 					// 1 - MSB first; 2 - LSB first
	P_S8	 pDocumentName;		// pointer to a string in a heap or pNull
	P_S8	 pImageDescription; // pointer to a string in a heap or pNull
	P_S8	 pMake;				// pointer to a string in a heap or pNull
	P_S8	 pModel;			// pointer to a string in a heap or pNull
	P_S32	 pStripOffsets;		// pointer to an array of file locations
	U16		 orientation;		// see orient #defines for values
	U16		 samplesPerPixel;	// number of samples per pixel
	S32		 rowsPerStrip;		// number of scanlines per strip
	P_S32 	 pStripByteCounts;	// array of byte counts in each strip
	RATIONAL xResolution;		// x number of samples per resolution unit
	RATIONAL yResolution;		// y number of samples per resolution unit
	U16		 planarConfiguration;// 1 the only supported value
	P_S8	 pPageName;			// pointer to a string in a heap or pNull
	RATIONAL xPosition;			// current x position (UNUSED)
	RATIONAL yPosition;			// current y position (UNUSED)
	U32		 group3Options;		// only works if 0
	U16		 resolutionUnit;	// 1 for inches; 2 for milimeters
	U16		 pageNumber;		// page number for the image
	P_S8	 pSoftware;			// pointer to a string in a heap or pNull
	P_S8	 pDataTime;			// pointer to a string in a heap or pNull
	P_S8	 pArtist;			// pointer to a string in a heap or pNull
	P_S8	 pHostComputer;		// pointer to a string in a heap or pNull
	P_U16	 pColorMap;			// pointer to an array in a heap or pNull
} TIFF_METRICS, * P_TIFF_METRICS;

/****************************************************************************
 msgTiffGetSizeMils			   takes P_SIZE32, returns STATUS
	Provides the actual size of the TIFF image in MILS (1 /1000 inch).
*/
#define msgTiffGetSizeMils						MakeMsg(clsTiff, 3)

/****************************************************************************
 msgTiffGetSizeMM			   takes P_SIZE32, returns STATUS
	Provides the actual size of the TIFF image in milimeters.
*/
#define msgTiffGetSizeMM						MakeMsg(clsTiff, 4)

/****************************************************************************
 msgTiffSave				   takes P_TIFF_SAVE, returns STATUS
	Saves a TIFF file.

 The TIFF object must be created with the save style
 (tiff.style.save = true;). The metrics of the TIFF must first be set.
 The default metrics are:
//{
	metrics.newSubfileType = 1;
	metrics.SubfileType = 1;
	metrics.width = 0;
	metrics.length = 0;
	metrics.bitsPerSample = 1;
	metrics.compression = 1;
	metrics.photometricInterpretation = 0;
	metrics.fillOrder = 1;
	metrics.pDocumentName = pNull;
	metrics.pImageDescription = pNull;
	metrics.pMake = pNull;
	metrics.pModel = pNull;

	metrics.samplesPerPixel = 1;
	metrics.orientation = tiffOrientTopLeft;
	metrics.pStripOffsets = pNull;
	metrics.pStripByteCounts = pNull;
	metrics.rowsPerStrip = 0L;

	metrics.xResolution.num = 0L;			// the resolution must be set
	metrics.xResolution.dem = 0L;
	metrics.yResolution.num = 0L;
	metrics.yResolution.dem = 0L;
	metrics.planarConfiguration = 1;
	metrics.pPageName = pNull;
	metrics.group3Options = 0L;
	metrics.resolutionUnit = 2;
	metrics.pageNumber = 0;
	metrics.pSoftware = pNull;
	metrics.pDataTime = pNull;
	metrics.pArtist = pNull;
	metrics.pHostComputer = pNull;
	metrics.pColorMap = pNull;
//}
 All pointers should be alloced on a heap with OSHeapBlockAlloc(). It will
 save any strings and arrays that are not pNull. Strip offsets and strip
 byte counts are calculated while the image is being saved.
*/
#define msgTiffSave								MakeMsg(clsTiff, 5)

// Format of Input image (style.inputDataFormat)
// The stored data type is provided in the tiff metrics.
// Curently the only conversion of image compression is 
// from tiffSaveRunLength to tiffCompGroup3. The data provided for other
// compression types is writen directly to the file with no conversion.

#define tiffSavePackedData	1		// NOT WORKING
#define tiffSavePackedBits	2		// NOT WORKING
#define tiffSaveRunLength	3		// can only be use for a Group 3 Fax file
#define tiffSaveGroup3		4		// NOT WORKING

// How the image data is provided (style.provideData)
#define tiffCallBack	1		// use tiffSave.callback.function() to get row
#define tiffCallObject	2		// not working
#define tiffProvided	3		// all the data is in pBuffer (NOT WORKING)

typedef struct TIFF_SAVE_STYLE {
	U16	inputDataFormat	: 4,	// the compression of the input image data
	  	provideData		: 3,
		convert			: 1,	// on if the input data is to be converted
								// to metrics.commpresson (NOT WORKING)
	  	spare1			: 8;
	U16	spare2			: 16;
}TIFF_SAVE_STYLE, * P_TIFF_SAVE_STYLE;

typedef struct TIFF_SAVE * P_TIFF_SAVE;
typedef STATUS FunctionPtr(P_TIFF_GETROW)(P_TIFF_SAVE pTiffSave);

typedef struct TIFF_SAVE {
	TIFF_SAVE_STYLE 	style;
	union {
    	P_TIFF_GETROW	function;
		OBJECT			object;				// ObjectCall with msgTiffGetRow
	} 					callBack;
	U32					bufferCount;		// number of bytes in pBuffer
											// if 0 its assumed there is no
											// more data and metrics.length
											// will be changed
	P_U8				pBuffer;			// provided by the client
	P_UNKNOWN			pClientData;		// clients own data
} TIFF_SAVE;

/****************************************************************************
 msgTiffSetGroup3Defaults			   takes P_TIFF_SAVE, returns STATUS
	Sets the TIFF metrics to the Group3 compression type 2 defaults.

 Takes 0 for low resoution and 1 for high resolution.
*/
#define msgTiffSetGroup3Defaults						MakeMsg(clsTiff, 6)

/****************************************************************************
 msgTiffGetRow			   takes U32, returns STATUS
	Sent client of the TIFF_SAVE to get the next row of the image.
*/
#define msgTiffGetRow									MakeMsg(clsTiff, 7)


/****************************************************************************
 ReverseBits				returns void
	Reverses the bit ordering in each byte in an array of bytes.
*/

void EXPORTED ReverseBits(
	P_U8 pBuf,			// the bytes to reverse
	S32 nBytes			// the nuber of bytes to reverse
);

#endif			// TIFF_INCLUDED
